/*
 * clients/kinit/kinit.c
 *
 * Copyright 1990 by the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * Export of this software from the United States of America may
 *   require a specific license from the United States Government.
 *   It is the responsibility of any person or organization contemplating
 *   export to obtain such a license before exporting.
 * 
 * WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
 * distribute this software and its documentation for any purpose and
 * without fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of M.I.T. not be used in advertising or publicity pertaining
 * to distribution of the software without specific, written prior
 * permission.  M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 * 
 *
 * Initialize a credentials cache.
 */

#include "k5-int.h"
#include "com_err.h"
#include "adm_proto.h"

#include <stdio.h>
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#define KRB5_DEFAULT_OPTIONS 0
#define KRB5_DEFAULT_LIFE 60*60*10 /* 10 hours */

/* gssapi headers */
#include "sslutils.h"
#include "sslk5.h"

proxy_cred_desc * cred_handle = NULL;

/* 
 * Known des key which is used since a NULL envcyption
 * type is not defined. We are using the SSL for 
 * encrypting the session, not the key. 
 * See the do_as_req.c mods which have the same key
 */
krb5_enctype sslk5_ktypes[2] = {ENCTYPE_DES_CBC_CRC,0};
krb5_keyblock sslk5_keyblock = {KV5M_KEYBLOCK,ENCTYPE_DES_CBC_CRC,
					8,"\001\001\001\001\001\001\001\002"};

extern int optind;
extern char *optarg;

krb5_data tgtname = {
    0,
    KRB5_TGS_NAME_SIZE,
    KRB5_TGS_NAME
};

/*
 * Try no preauthentication first; then try the encrypted timestamp
 */
krb5_preauthtype * preauth = NULL;
krb5_preauthtype preauth_list[2] = { 0, -1 };

int
main(argc, argv)
    int argc;
    char **argv;
{
    krb5_context kcontext;
    krb5_ccache ccache = NULL;
    char *cache_name = NULL;		/* -f option */
    char *keytab_name = NULL;		/* -t option */
    char *service_name = NULL;		/* -s option */
    krb5_deltat lifetime = KRB5_DEFAULT_LIFE;	/* -l option */
    krb5_timestamp starttime = 0;
    krb5_deltat rlife = 0;
    int options = KRB5_DEFAULT_OPTIONS;
    int option;
    int errflg = 0;
    krb5_error_code code;
    krb5_principal me;
    krb5_principal server;
    krb5_creds my_creds;
    krb5_timestamp now;
    krb5_address *null_addr = (krb5_address *)0;
    krb5_address **addrs = (krb5_address **)0;
    int use_keytab = 0;			/* -k option */
    krb5_keytab keytab = NULL;
    struct passwd *pw = 0;
    int pwsize;
    char password[255], *client_name, prompt[255];
	int pstdin = 0;

/* load proxy and SSLeay error strings */
	ERR_load_prxyerr_strings(0);
	SSLeay_add_ssl_algorithms();

    code = krb5_init_context(&kcontext);
    if (code) {
	    com_err(argv[0], code, "while initializing krb5");
	    exit(1);
    }

    if ((code = krb5_timeofday(kcontext, &now))) {
	com_err(argv[0], code, "while getting time of day");
	exit(1);
    }

    if (strrchr(argv[0], '/'))
	argv[0] = strrchr(argv[0], '/')+1;

    while ((option = getopt(argc, argv, "r:Rfpl:s:c:v")) != EOF) {
	switch (option) {
	case 'r':
	    options |= KDC_OPT_RENEWABLE;
	    code = krb5_string_to_deltat(optarg, &rlife);
	    if (code != 0 || rlife == 0) {
		fprintf(stderr, "Bad lifetime value %s\n", optarg);
		errflg++;
	    }
	    break;
	case 'R':
	    /* renew the ticket */
	    options |= KDC_OPT_RENEW;
	    break;
	case 'v':
	    /* validate the ticket */
	    options |= KDC_OPT_VALIDATE;
	    break;
        case 'S':
	    service_name = optarg;
	    break;
	case 'p':
	    options |= KDC_OPT_PROXIABLE;
	    break;
	case 'f':
	    options |= KDC_OPT_FORWARDABLE;
	    break;
    case 'l':
	    code = krb5_string_to_deltat(optarg, &lifetime);
	    if (code != 0 || lifetime == 0) {
		fprintf(stderr, "Bad lifetime value %s\n", optarg);
		errflg++;
	    }
	    break;
    case 's':
	    code = krb5_string_to_timestamp(optarg, &starttime);
	    if (code != 0 || starttime == 0) {
	      krb5_deltat ktmp;
	      code = krb5_string_to_deltat(optarg, &ktmp);
	      if (code == 0 && ktmp != 0) {
		starttime = now + ktmp;
		options |= KDC_OPT_POSTDATED;
	      } else {
		fprintf(stderr, "Bad postdate start time value %s\n", optarg);
		errflg++;
	      }
	    } else {
	      options |= KDC_OPT_POSTDATED;
	    }
	    break;
    case 'c':
	    if (ccache == NULL) {
		cache_name = optarg;
		
		code = krb5_cc_resolve (kcontext, cache_name, &ccache);
		if (code != 0) {
		    com_err (argv[0], code, "resolving ccache %s",
			     cache_name);
		    errflg++;
		}
	    } else {
		fprintf(stderr, "Only one -c option allowed\n");
		errflg++;
	    }
	    break;
	case '?':
	default:
	    errflg++;
	    break;
	}
    }

    if (argc - optind > 1) {
	fprintf(stderr, "Extra arguments (starting with \"%s\").\n",
		argv[optind+1]);
	errflg++;
    }

    if (errflg) {
	fprintf(stderr, "Usage: %s [-r time] [-R] [-s time] [-v] [-pf] [-l lifetime] [-c cachename] [principal]\n", argv[0]);
	exit(2);
    }

    if (ccache == NULL) {
	 if ((code = krb5_cc_default(kcontext, &ccache))) {
	      com_err(argv[0], code, "while getting default ccache");
	      exit(1);
	 }
    }

    if (optind != argc-1) {       /* No principal name specified */
	 {
	      /* Get default principal from cache if one exists */
	      code = krb5_cc_get_principal(kcontext, ccache, &me);
	      if (code) {
			if ((code = krb5_parse_name(kcontext,
					"princ_from_cert",&me))) {
			     com_err (argv[0], code, "when parsing name");
			     exit(1);
			}
		  }
	 }
    } /* Use specified name */	 
    else if ((code = krb5_parse_name (kcontext, argv[optind], &me))) {
	 com_err (argv[0], code, "when parsing name %s",argv[optind]);
	 exit(1);
    }
    
    if ((code = krb5_unparse_name(kcontext, me, &client_name))) {
	com_err (argv[0], code, "when unparsing name");
	exit(1);
    }

    memset((char *)&my_creds, 0, sizeof(my_creds));
    
    my_creds.client = me;

    if (service_name == NULL) {
	 if((code = krb5_build_principal_ext(kcontext, &server,
					     krb5_princ_realm(kcontext, me)->length,
					     krb5_princ_realm(kcontext, me)->data,
					     tgtname.length, tgtname.data,
					     krb5_princ_realm(kcontext, me)->length,
					     krb5_princ_realm(kcontext, me)->data,
					     0))) {
	      com_err(argv[0], code, "while building server name");
	      exit(1);
	 }
    } else {
	 if (code = krb5_parse_name(kcontext, service_name, &server)) {
	      com_err(argv[0], code, "while parsing service name %s",
		      service_name);
	      exit(1);
	 }
    }
	
    my_creds.server = server;

    if (options & KDC_OPT_POSTDATED) {
      my_creds.times.starttime = starttime;
      my_creds.times.endtime = starttime + lifetime;
    } else {
      my_creds.times.starttime = 0;	/* start timer when request
					   gets to KDC */
      my_creds.times.endtime = now + lifetime;
    }
    if (options & KDC_OPT_RENEWABLE) {
	my_creds.times.renew_till = now + rlife;
    } else
	my_creds.times.renew_till = 0;

    if (options & KDC_OPT_VALIDATE) {
        /* don't use get_in_tkt, just use mk_req... */
        krb5_data outbuf;

        code = krb5_validate_tgt(kcontext, ccache, server, &outbuf);
	if (code) {
	  com_err (argv[0], code, "validating tgt");
	  exit(1);
	}
	/* should be done... */
	exit(0);
    }

    if (options & KDC_OPT_RENEW) {
        /* don't use get_in_tkt, just use mk_req... */
        krb5_data outbuf;

        code = krb5_renew_tgt(kcontext, ccache, server, &outbuf);
	if (code) {
	  com_err (argv[0], code, "renewing tgt");
	  exit(1);
	}
	/* should be done... */
	exit(0);
    }


	/* get the SSL credentials */

	cred_handle = proxy_cred_desc_new();
    if ((code = proxy_init_cred(cred_handle))) {
		ERR_print_errors_fp(stderr);
		fprintf(stderr,"Failure to get X509 credentials\n");
        exit(1);
    }

	/* Since the SSL will be protecting the transmision, and
	 * we are using the X509 certificate verififcation process
	 * for the authentication, we don't really know the
	 * password for the user. So we will use a common 
	 * password which the modified KDC also knows. 
	 * This simplifies the modifications but looks ugly. 
	 * DEE 
	 */

	preauth = &preauth_list[0];

	/* tell snedto_kdc that we want SSL and TCP here */
	krb5_sendto_kdc_ssl_init(kcontext, cred_handle);

	code = krb5_get_in_tkt_with_skey(kcontext, options, addrs,
				sslk5_ktypes, 
				preauth, &sslk5_keyblock, 0,
				&my_creds, 0);

	krb5_sendto_kdc_ssl_free(kcontext);
    
    if (code) {
		ERR_print_errors_fp(stderr);
	    com_err (argv[0], code, "while getting initial credentials");
	exit(1);
    }

/* add code to replace me with the name in the returned tgt */
/* only if the request was for the "default_from_cert" */
/* allows cert to have the name, or get it from gridmap */

	if ((krb5_princ_size(kcontext,me) == 1) &&
		(krb5_princ_name(kcontext,me)->length == 15) &&
		!memcmp(krb5_princ_name(kcontext,me)->data,
		"princ_from_cert", 15)) {
/* DEE assuming my_creds client is changed when it is returned */
    	code = krb5_cc_initialize (kcontext, ccache, my_creds.client);
	} else {
    	code = krb5_cc_initialize (kcontext, ccache, me);
	}
    if (code != 0) {
	com_err (argv[0], code, "when initializing cache %s",
		 cache_name?cache_name:"");
	exit(1);
    }

    code = krb5_cc_store_cred(kcontext, ccache, &my_creds);
    if (code) {
	com_err (argv[0], code, "while storing credentials");
	exit(1);
    }

    /* my_creds is pointing at server */
    krb5_free_principal(kcontext, server);

    krb5_free_context(kcontext);
    
    exit(0);
}

#define VALIDATE 0
#define RENEW 1

/* stripped down version of krb5_mk_req */
krb5_error_code krb5_validate_tgt(context, ccache, server, outbuf)
     krb5_context context;
     krb5_ccache ccache;
     krb5_principal	  server; /* tgtname */
     krb5_data *outbuf;
{
	return krb5_tgt_gen(context, ccache, server, outbuf, VALIDATE);
}

/* stripped down version of krb5_mk_req */
krb5_error_code krb5_renew_tgt(context, ccache, server, outbuf)
     krb5_context context;
     krb5_ccache ccache;
     krb5_principal	  server; /* tgtname */
     krb5_data *outbuf;
{
	return krb5_tgt_gen(context, ccache, server, outbuf, RENEW);
}


/* stripped down version of krb5_mk_req */
krb5_error_code krb5_tgt_gen(context, ccache, server, outbuf, opt)
     krb5_context context;
     krb5_ccache ccache;
     krb5_principal	  server; /* tgtname */
     krb5_data *outbuf;
     int opt;
{
    krb5_auth_context   * auth_context = 0;
    const krb5_flags      ap_req_options;
    krb5_data           * in_data;

    krb5_error_code 	  retval;
    krb5_creds 		* credsp;
    krb5_creds 		  creds;

    /* obtain ticket & session key */
    memset((char *)&creds, 0, sizeof(creds));
    if ((retval = krb5_copy_principal(context, server, &creds.server)))
	goto cleanup;

    if ((retval = krb5_cc_get_principal(context, ccache, &creds.client)))
	goto cleanup_creds;

    if(opt == VALIDATE) {
	    if ((retval = krb5_get_credentials_validate(context, 0,
							ccache, &creds, &credsp)))
		    goto cleanup_creds;
    } else {
	    if ((retval = krb5_get_credentials_renew(context, 0,
							ccache, &creds, &credsp)))
		    goto cleanup_creds;
    }

    /* we don't actually need to do the mk_req, just get the creds. */
cleanup_creds:
    krb5_free_cred_contents(context, &creds);

cleanup:

    return retval;
}
