/*
 * sample_comm.c : 
 */ 


#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/time.h>
#include <arpa/inet.h>

#ifndef FREEBSD
#include <malloc.h>
#endif


#include <gssapi.h>

#include "sample_comm.h"

#define NAME_LENGTH 80

int connection(char *, unsigned int, char *);
int connect_to_server(char *, unsigned int);
int create_socket(unsigned int);
int setup_connection();
int establish_connection( int);

extern int errno; 



/*
 * Function: connect_to_server (client )
 *
 * Purpose: Opens a TCP connection to the name host and port.
 *
 * Arguments:
 *
 * 	host		(r) the target host name
 * 	port		(r) the target port, in host byte order
 *
 * Returns: the established socket file desciptor, or -1 on failure
 *
 * Effects:
 *
 * The host name is resolved with gethostbyname(), and the socket is
 * opened and connected.  If an error occurs, an error message is
 * displayed and -1 is returned.
 */
int connect_to_server(char *host, unsigned int port)
{
  char this_function[] = "connect_to_server";
  struct sockaddr_in saddr;
  struct hostent *hp;
  int s;
  int retval;
#ifdef DEBUG
  fprintf(stderr, "N %s: args\n\t host = %s, port = %d\n", 
	  this_function, host, port);
#endif /* DEBUG */     
  if ((hp = gethostbyname(host)) == NULL) {
    fprintf(stderr, "Unknown host: %s\n", host);
    return -1;
  }
     
  saddr.sin_family = hp->h_addrtype;
  memcpy((char *)&saddr.sin_addr, hp->h_addr, hp->h_length);
  saddr.sin_port = htons(port);

  if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("creating socket");
    return -1;
  }
  retval=connect(s, (struct sockaddr *)&saddr, sizeof(saddr));
  if (retval < 0) {
    perror("connecting to server");
    return -1;
  }
#ifdef DEBUG
  fprintf(stderr, "N %s connected socket %d to host %s\n",
	  this_function, s, host);
#endif /* DEBUG */ 
  return s;
} /* connect_to_server() */

/* client function */
int connection(char *host,unsigned int port, char *service_name)
{
  const char this_function[] = "connection()" ;
  int other_socket;
  if ((other_socket = connect_to_server(host, port)) < 0) {
    fprintf(stderr,"E %s: Couldn't connect to socket\n", this_function);
    return -1;
  }
  return other_socket;

} /* connection() */


/* server function */

/*
 * Function: create_socket
 *
 * Purpose: Opens a listening TCP socket.
 *
 * Arguments:
 *
 * 	port		(r) the port number on which to listen
 *
 * Returns: the listening socket file descriptor, or -1 on failure
 *
 * Effects:
 *
 * A listening socket on the specified port and created and returned.
 * On error, an error message is displayed and -1 is returned.
 */
int create_socket(unsigned int port)
{
     struct sockaddr_in saddr;
     int s;

     memset(&saddr,0,sizeof(saddr)); /* purify: umr */

     saddr.sin_family = AF_INET;
     saddr.sin_port = htons(port);
     saddr.sin_addr.s_addr = INADDR_ANY;

     if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	  perror("creating socket");
	  return -1;
     }
     {
	 int opt;
	 opt = 1;
	 setsockopt (s,SOL_SOCKET,SO_REUSEADDR,(char *)&opt,sizeof(opt));
     }

     if (bind(s, (struct sockaddr *) &saddr, sizeof(saddr)) < 0) {
	  perror("binding socket");
	  return -1;
     }
     if (listen(s, 5) < 0) {
	  perror("listening on socket");
	  return -1;
     }
     return s;
} /* create_socket() */

/* server functonn */
int setup_connection(int port) {
    int my_socket;

    if ((my_socket = create_socket(port)) < 0)
	exit(1);
    return my_socket;
} /* setup_connection() */

/* server function */
int establish_connection(int my_socket) {
  char this_function[] ="establish_connection";
  int other_socket;
  struct sockaddr_in net_client;
  struct hostent *hostptr;
  int addr_len = sizeof(struct sockaddr);
  int retval;
    
  while ((( retval =
	    accept(my_socket, (struct sockaddr *) (&net_client), 
		   &addr_len)) == -1) && errno == -1)
    ;
  if (retval == -1)
    return retval;
  hostptr = 
    gethostbyaddr((char *) &(net_client.sin_addr.s_addr), 4, AF_INET);
  if (hostptr == NULL)
    fprintf(stderr,"N %s:after succesful accept(), client host is UNKNOWN\n",
	    this_function);
  else
    fprintf(stderr, "N %s: after succesful accept(), client host is %s\n",
	    this_function, (*hostptr).h_name);
  other_socket = retval;
  return other_socket;
} /* establish_connection() */



int 
send_message(int peer_socket,
	     gss_ctx_id_t context_handle,
	     gss_buffer_t input_message_buffer,
	     int wrap_flag,
	     int encrypt_flag)
{

  const char this_function[] = "send_message()" ;
  gss_buffer_desc output_message_buffer_desc ;
  gss_buffer_t output_message_buffer = &output_message_buffer_desc ;

  OM_uint32 minor_status ;

  if (wrap_flag) {
    int conf_state ; 
    gss_wrap(&minor_status,
	     context_handle,
	     encrypt_flag,
	     GSS_C_QOP_DEFAULT,
	     input_message_buffer,
	     &conf_state,
	     output_message_buffer) ;

#ifdef DEBUG
    fprintf(stderr, 
	    "N %s: Sending wrapped data: [%d] \n", 
	    this_function, 
	    output_message_buffer->length) ;
#endif

  } else {
    output_message_buffer->length = input_message_buffer->length ;
    output_message_buffer->value = 
      (void*) malloc(output_message_buffer->length) ;

    if (output_message_buffer->value == NULL) {
      fprintf(stderr,
	      "E %s: Could not allocate enough memory!\n", this_function) ;
      return -1; 
    }
    memcpy(output_message_buffer->value, 
	   input_message_buffer->value, 
	   output_message_buffer->length) ;
  }


  {
    int net_buf_size, num_written;

    net_buf_size = htonl(output_message_buffer->length);
    num_written = write(peer_socket,
			(void *) &net_buf_size, 
			sizeof(net_buf_size));
    if (num_written != sizeof(net_buf_size) ) {
      fprintf(stderr, 
	      "E %s: Buffersize send_to_server. %d bytes written\n",
	      this_function, num_written) ;
      return -1 ;
    }

    num_written = write(peer_socket,
			output_message_buffer->value,
			output_message_buffer->length);

    if (num_written != output_message_buffer->length) {
      fprintf(stderr, 
	      "E %s: Message not sent correctly.  Only %d bytes written.\n", 
	      this_function, num_written) ;
      return -1 ;
    }

#ifdef DEBUG
    fprintf(stderr, 
	    "N %s: Message sent [%d].\n", 
	    this_function, num_written) ;
#endif
    gss_release_buffer(&minor_status, output_message_buffer) ;
    return num_written ;
  }
}

int 
receive_message(int peer_socket,
	     gss_ctx_id_t context_handle,
	     gss_buffer_t output_message_buffer,
	     int unwrap_flag)
{
  const char this_function[] = "receive_message()" ;  
  OM_uint32 minor_status ;  
  gss_buffer_desc input_message_buffer_desc ;
  gss_buffer_t input_message_buffer = &input_message_buffer_desc; 

  {
    int num_read ;
    int bufsize;
        
    num_read= read(peer_socket, (void *) &bufsize, sizeof(bufsize));
    if (num_read != sizeof(bufsize)) {
      fprintf(stderr, 
	      "E %s: Incorrect buf_size read: [%d]\n",
	      this_function, 
	      num_read);
      return -1 ;
    }
    input_message_buffer->length = ntohl(bufsize);
    input_message_buffer->value = (void*) malloc(input_message_buffer->length);
    if (input_message_buffer->value == NULL) {
      fprintf(stderr,
	      "E %s: Could not allocate enough memory!\n", this_function) ;
      exit(1); 
    }        
    num_read= read(peer_socket, 
		   (void *) input_message_buffer->value, 
                   input_message_buffer->length);

    /* unwrap the mesage */
    if (unwrap_flag) {

      int conf_state ;
      gss_qop_t qop_state ;


#ifdef DEBUG
      fprintf(stderr, 
	      "N %s: Received a wrapped packet: [%d]\n",
	      this_function, 
	      input_message_buffer->length) ;
#endif

      gss_unwrap(&minor_status,
		 context_handle,
		 input_message_buffer,
		 output_message_buffer,
		 &conf_state,
		 &qop_state) ;
#ifdef DEBUG
      fprintf(stderr, 
	      "N %s: Message unwrapped.  It was determined to be: %s\n",
	      this_function, 
	      conf_state?"Encrypted":"Clear text") ;
#endif
    } else {
      output_message_buffer->length = input_message_buffer->length ;
      output_message_buffer->value = 
	(void*) malloc(output_message_buffer->length ) ;
      if (output_message_buffer->value == NULL) {
	fprintf(stderr,
		"E %s: Could not allocate enough memory!\n", this_function) ;
	return -1 ; 
      }
      memcpy(output_message_buffer->value,
	     input_message_buffer->value,
	     output_message_buffer->length) ;
    }
#ifdef DEBUG
    fprintf(stderr, 
            "N %s: Received message: [%d]\n",
            this_function, 
            output_message_buffer->length) ;
#endif

    gss_release_buffer(&minor_status, input_message_buffer) ;
    return num_read ;
  }

} /* receive_message */
