#!/bin/sh
######################################################################
#
# make_gnu_regex.sh
#
# Convert regex.[ch] from the GNU regex library into gnu_regex.[ch] as
# compiled and used by Globus. Specifically we do the following:
#
#  - Prepend all the function names with gaa_regex_ so that there
#    are no conflicts if someone tried to link with both the globus
#    libraries and the gnu regex libraries.
#
#  - Have gnu_regex.c look for gnu_regex.h instead of regex.h
#
#  - Make printchar() a macro for putchar(). printchar() is a debugging
#    function that for some reason was put into a different file,
#    so instead of including it as well, we'll just work around it.
#
# This was written for version 0.12 of the GNU regex library available
# from ftp://prep.ai.mit.edu/pub/gnu/regex/
#
# To convert:
#   - Rename regex.h to gnu_regex.h.in
#   - Rename regex.c to gnu_regex.c.in
#   - Run this script to create gnu_regex.c
#
######################################################################

# Exit on any error
set -e
######################################################################
#
# Defaults
#

cat=cat
sed=sed

sed_script=gnu_regex.sed

######################################################################
#
# Create output files with preamble
#

cat > gnu_regex.c <<EOF

/*
 * gnu_regex.c autogenerated from GNU regex.c by make_gnu_regex.sh
 * 
 */

/* Start Globus prefix code */

/* Use malloc() instead of alloca for portability */
#ifndef REGEX_MALLOC
#define REGEX_MALLOC
#endif

/* End Globus prefix code */
EOF

cat > gnu_regex.h <<EOF
/*
 * gnu_regex.h autogenerated from GNU regex.h by make_gnu_regex.sh
 * 
 */

EOF
######################################################################
#
# Make sed script
#

$cat > $sed_script <<EOF
#
# Rename all the functions so that they don't conflict if someone
# links with globus and the gnu regex library
#
s/print_fastmap/gaa_regex_print_fastmap/g
s/print_partial_compiled_pattern/gaa_regex_print_partial_compiled_pattern/g
s/print_compiled_pattern/gaa_regex_print_compiled_pattern/g
s/print_double_string/gaa_regex_print_double_string/g
s/re_set_syntax/gaa_regex_re_set_syntax/g
s/regex_compile/gaa_regex_regex_compile/g
s/re_compile_fastmap/gaa_regex_re_compile_fastmap/g
s/re_set_registers/gaa_regex_re_set_registers/g
s/re_search/gaa_regex_re_search/g
s/re_search_2/gaa_regex_re_search_2/g
s/re_match/gaa_regex_re_match/g
s/re_match_2/gaa_regex_re_match_2/g
s/re_compile_pattern/gaa_regex_re_compile_pattern/g
s/re_comp/gaa_regex_re_comp/g
s/re_exec/gaa_regex_re_exec/g
s/regcomp/gaa_regex_regcomp/g
s/regexec/gaa_regex_regexec/g
s/regerror/gaa_regex_regerror/g
s/regfree/gaa_regex_regfree/g
#
# Raname regex.h to gnu_regex.h
#
s/regex\.h/gnu_regex.h/g
#
# Ugh. printchar() is a wrapper around putchar() that was put into
# a seperate file for some reason. We'll just use putchar()
#
s/extern void printchar ();/\#define printchar(c) putchar(c)/g

EOF

######################################################################
#
# Parse input file
#

echo "Using $sed to create gnu_regex.c"
$sed -f $sed_script gnu_regex.c.in >> gnu_regex.c

echo "Using $sec to create gnu_regex.h"
$sed -f $sed_script gnu_regex.h.in >> gnu_regex.h

######################################################################
#
# Success
#

echo "Success"
exit 0


