/******************************************************************************
gram_client_multi_test.c

Description:
    Test of the gram client API
    synopsys: gram_client_multi_test [-n number of iterations]
                                     -g gatekeeper_file
                                     -r rsl_file
    submit the rsls in the rsl_file to the gatekeepers in the gatekeeper,
    submitting at most 1 per gatekeeper at any time. If a number of iterations
    is given, then the multi-test will submit all rsls that many times.
    Each iteration ends when all rsls which have been submitted enter the
    done or failed state.

    Tests handling of multiple allowed callbacks, multiple outstanding
    requests. Can run in threaded or non-threaded environment.

    Can also be used to farm out independent jobs to a set of gatekeepers
    as they become available. Multiple jobs can be sent to a gatekeeper
    by specifying the gatekeeper contact multiple times in the gatekeeper_file.

CVS Information:

    $Source: /home/globdev/CVS/globus-current/Globus/ResourceManagement/gram/tests/gram_client_multi_test.c,v $
    $Date: 1998/11/30 16:57:03 $
    $Revision: 1.14 $
    $Author: bester $
******************************************************************************/

/******************************************************************************
                             Include header files
******************************************************************************/
#include "globus_common.h"
#include "globus_gram_client.h"

/******************************************************************************
                               Type definitions
******************************************************************************/
typedef struct globus_l_monitor_s
{
    globus_mutex_t 	mutex;
    globus_cond_t  	cond;

    /* these two are the condition for completion:
     * all started == true
     * outstanding == 0
     */
    globus_bool_t  	all_started;
    int			outstanding;
    /*
     * Lists of gatekeepers read from the config file, could by
     * dynamic, if we had desire/time . Currently these need to be
     * gatekeeper contact strings 'host:port:id'
     * active gatekeepers is a  hash job_contact -> gatekeeper contact string
     * free_gatekeepers is a  list of globus_l_gatekeeper_t's
     */
    globus_fifo_t	free_gatekeepers;
    globus_hashtable_t	active_gatekeepers;

    /*
     * Lists of RSLs to distribute. They should be generic enough
     * to run on any platform.
     * pending_rsls is a list of char *'s
     * finished_rsls is a list of char *'s
     */
    globus_fifo_t	pending_rsls;
    globus_fifo_t	finished_rsls;

    /* number of times to cycle through rsls  */
    int num_repeats;
} globus_l_monitor_t;

#define monitor_done_pred(m) (m->all_started == GLOBUS_TRUE && m->outstanding == 0)

typedef struct globus_l_gatekeeper_s
{
    char *gatekeeper_contact;
    char *callback_contact;
    char *job_contact;
    char *rsl;
    globus_l_monitor_t *monitor;
} globus_l_gatekeeper_t;

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static void
globus_l_multitest_callback(void * user_callback_arg,
			    char * job_contact,
			    int state,
			    int errorcode);

static int
globus_l_multitest_fill_fifo(FILE *fp,
			     globus_fifo_t *fifo);

static int
globus_l_multitest_init(int argc,
			char **argv,
			globus_l_monitor_t **monitorp);

static int
globus_l_multitest_monitor_init(globus_l_monitor_t *monitor);

static int
globus_l_multitest_destroy(globus_l_monitor_t **monitor);
static void
globus_l_monitor_reset(globus_l_monitor_t *monitor);

/******************************************************************************
                       Define module specific variables
******************************************************************************/
#define GLOBUS_L_STRING_MAX     256
#define GLOBUS_L_MONITOR_ENTER() globus_mutex_lock(&monitor->mutex)
#define GLOBUS_L_MONITOR_EXIT() globus_mutex_unlock(&monitor->mutex)


/******************************************************************************
Function:	main()

Description:

Parameters:

Returns:
******************************************************************************/
int
main(int argc,
     char ** argv)
{
    int 		rc;
    int 		num_rsls;
    int 		infinite;
    globus_l_monitor_t *monitor;
    int			i;
    
    rc = globus_l_multitest_init(argc, argv, &monitor);

    if(rc != GLOBUS_SUCCESS)
    {
	return 1;
    }
    
    GLOBUS_L_MONITOR_ENTER();
    num_rsls = globus_fifo_size(&monitor->pending_rsls);
    if(monitor->num_repeats == 0)
    {
	infinite = GLOBUS_TRUE;
    }
    else
    {
	infinite = GLOBUS_FALSE;
    }

    for(i = 0; i < monitor->num_repeats || infinite; i++)
    {
	while (!monitor_done_pred(monitor))
	{
	    monitor->all_started = GLOBUS_FALSE;
	    
	    while((/* can't start another job */
		(globus_fifo_empty(&monitor->free_gatekeepers) ||
		 globus_fifo_empty(&monitor->pending_rsls)) &&
		/* not done */
		(globus_fifo_size(&monitor->finished_rsls) != num_rsls)))
	    {
		globus_cond_wait(&monitor->cond,
				 &monitor->mutex);
	    }
	    if(globus_fifo_size(&monitor->finished_rsls) == num_rsls)
	    {
		monitor->all_started = GLOBUS_TRUE;
		continue;
	    }
	    else /* can start job*/
	    {
		globus_l_gatekeeper_t * globus_l_waiting_job;
		char *dummy;
		
		globus_l_waiting_job =
		    globus_malloc(sizeof(globus_l_gatekeeper_t));
		
		globus_l_waiting_job->gatekeeper_contact = (char *) globus_fifo_dequeue(&monitor->free_gatekeepers);
		globus_l_waiting_job->rsl = (char *) globus_fifo_dequeue(&monitor->pending_rsls);
		globus_l_waiting_job->job_contact = "";
		globus_l_waiting_job->monitor = monitor;
		
		monitor->outstanding++;

		GLOBUS_L_MONITOR_EXIT();

		rc = globus_gram_client_callback_allow(globus_l_multitest_callback,
						       globus_l_waiting_job,
						       &globus_l_waiting_job->callback_contact);
		
		rc = globus_gram_client_job_request(globus_l_waiting_job->gatekeeper_contact,
						    globus_l_waiting_job->rsl,
						    GLOBUS_GRAM_CLIENT_JOB_STATE_ALL,
						    globus_l_waiting_job->callback_contact,
						    &dummy);
		GLOBUS_L_MONITOR_ENTER();
		
	    }
	} /* endwhile */

	/*reset to start over*/
	globus_l_monitor_reset(monitor);
    }

    GLOBUS_L_MONITOR_EXIT();

    globus_mutex_destroy(&monitor->mutex);
    globus_cond_destroy(&monitor->cond);


    globus_module_deactivate_all();

    return(0);
}

static int
globus_l_multitest_init(int argc,
			char **argv,
			globus_l_monitor_t **monitorp)
{
    char c;
    int rc;
    int num_repeats=1;
    globus_l_monitor_t *monitor;
    extern char *optarg;
    FILE *gatekeeper_fp=GLOBUS_NULL, *rsl_fp=GLOBUS_NULL;
    char *usage_string = 
	"gram_client_multi_test [-n num_repeats] -r rsl_file -g gatekeeper_file\n";

    rc = globus_module_activate(GLOBUS_COMMON_MODULE);

    if(rc != GLOBUS_SUCCESS)
    {
	return rc;
    }
    (*monitorp) = globus_malloc(sizeof(globus_l_monitor_t));

    if(*monitorp == GLOBUS_NULL)
    {
	return GLOBUS_FAILURE;
    }
    
    monitor = *monitorp;

    while((c = getopt(argc, argv, "n:g:r:")) != EOF)
    {
	switch(c)
	{
	case 'n':
	    num_repeats = atoi(optarg);
	    if(num_repeats < 0)
	    {
		globus_libc_printf("Error, -n argument must be >= 0\n");
		goto early_exit;
	    }
	    break;
	case 'g':
	    gatekeeper_fp = fopen(optarg, "r");
	    if(gatekeeper_fp == GLOBUS_NULL)
	    {
		globus_libc_printf("Error cannot open gatekeeper file %s\n",
				   optarg);
		goto early_exit;
	    }
	    break;
	case 'r':
	    rsl_fp = fopen(optarg, "r");
	    if(rsl_fp == GLOBUS_NULL)
	    {
		globus_libc_printf("Error cannot open rsl file %s\n",
				   optarg);
		goto early_exit;
	    }
	    break;
	case '?':
	    globus_libc_printf("Invalid argument %c\n",
			       c);
	    goto early_exit;
	}
    }

    if(gatekeeper_fp == GLOBUS_NULL ||
       rsl_fp == GLOBUS_NULL)
    {
	globus_libc_printf(usage_string);
	goto early_exit;
    }

    rc = globus_l_multitest_monitor_init(monitor);

    if(rc != GLOBUS_SUCCESS)
    {
	goto early_exit;
    }
    rc = globus_l_multitest_fill_fifo(gatekeeper_fp,
				      &monitor->free_gatekeepers);
    if(rc != GLOBUS_SUCCESS)
    {
	goto early_exit;
    }


    rc = globus_l_multitest_fill_fifo(rsl_fp,
				      &monitor->pending_rsls);
    monitor->num_repeats = num_repeats;

    if(rc != GLOBUS_SUCCESS)
    {
	goto early_exit;
    }

    fclose(rsl_fp);
    fclose(gatekeeper_fp);

    rc = globus_module_activate(GLOBUS_GRAM_CLIENT_MODULE);
    if(rc == GLOBUS_SUCCESS)
    {
	return GLOBUS_SUCCESS;
    }

 early_exit:
    globus_module_deactivate(GLOBUS_COMMON_MODULE);
    return GLOBUS_FAILURE;
}

static void
globus_l_multitest_callback(void * user_callback_arg,
			    char * job_contact,
			    int state,
			    int errorcode)
{
    globus_l_monitor_t *monitor;
    globus_l_gatekeeper_t *gk;

    printf("globus_l_multitest_callback():\njob_contact=%s, state=%i, errorcode=%i\n", 
           job_contact, state, errorcode);

    gk = (globus_l_gatekeeper_t *) user_callback_arg;
    monitor = gk->monitor;

    GLOBUS_L_MONITOR_ENTER();

    switch(state)
    {
    case GRAM_JOB_STATE_PENDING:
	printf("Got GRAM_JOB_STATE_PENDING\n");
        break;
    case GRAM_JOB_STATE_ACTIVE:
	printf("Got GRAM_JOB_STATE_ACTIVE\n");
        break;
    case GRAM_JOB_STATE_FAILED:
	printf("Got GRAM_JOB_STATE_FAILED\n");
	monitor->outstanding--;
	globus_hashtable_remove(&monitor->active_gatekeepers,
				(void *) job_contact);
	globus_fifo_enqueue(&monitor->finished_rsls,
			    (void *) gk->rsl);
	globus_fifo_enqueue(&monitor->free_gatekeepers,
			    (void *) gk->gatekeeper_contact);
	GLOBUS_L_MONITOR_EXIT();

	globus_gram_client_callback_disallow(gk->callback_contact);
	
	GLOBUS_L_MONITOR_ENTER();
	globus_free(gk);
	break;
    case GRAM_JOB_STATE_DONE:
	printf("Got GRAM_JOB_STATE_DONE\n");
	monitor->outstanding--;
	globus_hashtable_remove(&monitor->active_gatekeepers,
				(void *) job_contact);
	globus_fifo_enqueue(&monitor->finished_rsls,
			   (void *) gk->rsl);
	globus_fifo_enqueue(&monitor->free_gatekeepers,
			   (void *) gk->gatekeeper_contact);
	GLOBUS_L_MONITOR_EXIT();

	globus_gram_client_callback_disallow(gk->callback_contact);
	
	GLOBUS_L_MONITOR_ENTER();
	globus_free(gk);
	break;
    }

    globus_cond_signal(&monitor->cond);

    GLOBUS_L_MONITOR_EXIT();
}


/******************************************************************************
Function:	globus_l_multitest_fill_fifo()

Description:

Parameters:

Returns:
******************************************************************************/
static int
globus_l_multitest_fill_fifo(FILE *fp,
                             globus_fifo_t *fifo)
{
  char *line_buffer = globus_malloc(GLOBUS_L_STRING_MAX);

  if(line_buffer == GLOBUS_NULL)
  {
      return GLOBUS_FAILURE;
  }

  while(fgets(line_buffer, GLOBUS_L_STRING_MAX - 1, fp))
  {
      if(line_buffer[0] == '#' ||
	 line_buffer[0] == '\n')
      {
	  continue;
      }
      if(line_buffer[strlen(line_buffer) - 1] == '\n')
      {
	  line_buffer[strlen(line_buffer) - 1] = '\0';
      }
      globus_fifo_enqueue(fifo,
			  (void *) strdup(line_buffer));
  }

  globus_free(line_buffer);

  return GLOBUS_SUCCESS;
} /* globus_l_multitest_fill_fifo() */

/******************************************************************************
Function:	globus_l_multitest_monitor_init()

Description:

Parameters:

Returns:
******************************************************************************/
static int
globus_l_multitest_monitor_init(globus_l_monitor_t *monitor)
{
    if(globus_mutex_init(&monitor->mutex,
			 (globus_mutexattr_t *) GLOBUS_NULL) != GLOBUS_SUCCESS)
    {
	return GLOBUS_FAILURE;
    }
    if(globus_cond_init(&monitor->cond,
			(globus_condattr_t *) GLOBUS_NULL) != GLOBUS_SUCCESS)
    {
	return GLOBUS_FAILURE;
    }
    if(globus_hashtable_init(&monitor->active_gatekeepers,
			     100,
			     globus_hashtable_string_hash,
			     globus_hashtable_string_keyeq) != GLOBUS_SUCCESS)
    {
	return GLOBUS_FAILURE;
    }
    if(globus_fifo_init(&monitor->free_gatekeepers) != GLOBUS_SUCCESS)
    {
	return GLOBUS_FAILURE;
    }

   if(globus_fifo_init(&monitor->pending_rsls) != GLOBUS_SUCCESS)
    {
	return GLOBUS_FAILURE;
    }
    if(globus_fifo_init(&monitor->finished_rsls) != GLOBUS_SUCCESS)
    {
	return GLOBUS_FAILURE;
    }
    monitor->all_started = GLOBUS_FALSE;
    monitor->outstanding = 0;
    return GLOBUS_SUCCESS;
} /* globus_l_multitest_monitor_init() */

static int
globus_l_multitest_destroy(globus_l_monitor_t **monitor)
{
    return GLOBUS_FALSE;
}

static void
globus_l_monitor_reset(globus_l_monitor_t *monitor)
{
    monitor->all_started = GLOBUS_FALSE;

    while(!globus_fifo_empty(&monitor->finished_rsls))
    {
	char *tmp_rsl = (char *) globus_fifo_dequeue(&monitor->finished_rsls);
	globus_fifo_enqueue(&monitor->pending_rsls,
			    (void *) tmp_rsl);
    }
}
