/******************************************************************************
grami_ggg_accept.c
 
Description:
	 Globus GSSAPI Glue accept security context

	This routine acceps a GSSAPI security context and 
	is called by the gram_gatekeeper. It isolates 
	the GSSAPI from the rest of the gram code. 

Input required :
	*globus_client name who requested authentication 
	a get token routine 
	first arg for the get 
	a send token routine 
	first arg for the send

	No knowledge of gss-api is assumed outside of these
	glue routines. But the return code is the GSS
	major status or 0. 


CVS Information:
 
	$Source: /home/globdev/CVS/globus-current/Globus/ResourceManagement/gram/common/grami_ggg_accept.c,v $
	$Date: 1998/02/16 15:29:25 $
	$Revision: 1.7 $
	$Author: dengert $
******************************************************************************/

#ifdef GSS_AUTHENTICATION
/******************************************************************************
                             Include header files
******************************************************************************/
#include <gssapi.h>
#include <malloc.h>
#include "grami_ggg.h"
#include <stdio.h>

/******************************************************************************
                               Type definitions
******************************************************************************/
/******************************************************************************
                          Module specific prototypes
******************************************************************************/
/******************************************************************************
                       Define module specific variables
******************************************************************************/
static gss_cred_id_t credential_handle;

int (*gss_user_supplied_setenv)(char *, char *,int);

/******************************************************************************
Function:   grami_ggg_accept()
Description:
	Called by hte gatekeeper to get the acceptors credential
Parameters:
Returns:
	0 on sucess
******************************************************************************/
int
grami_ggg_accept_first()
{
    int rc;
    OM_uint32 mstatus; 

	/* 
	 * As a kludge for now, the gssapi_ssleay proxy_marshal routine
	 * need to set the environment.
	 * So we will set the pointer to the setenv here 
	 *
	 */
	gss_user_supplied_setenv = grami_setenv;

    credential_handle = GSS_C_NO_CREDENTIAL;

    rc = gss_acquire_cred(&mstatus,
            GSS_C_NO_NAME,
            GSS_C_INDEFINITE,
            GSS_C_NO_OID_SET,
            GSS_C_ACCEPT,
            &credential_handle,
            NULL,
            NULL) ;

    return (rc);
}
/******************************************************************************
Function:   grami_ggg_accept()
Description:
	This routine acceps a GSSAPI security context and 
	is called by the gram_gatekeeper. It isolates 
	the GSSAPI from the rest of the gram code. 

Parameters:
	*globus_client_name to return who requested authentication 
	ggg_server_name - NULL or the globus server principal
	a get token routine 
	first arg for the get token routine 
	a send token routine 
	first arg for the send token routine

Returns:
	0 upon sucess, and the name who requested authentication
	!=0 an error code 

******************************************************************************/
int 
grami_ggg_accept(char ** ggg_client_name,
               char * ggg_server_name,
               int (*ggg_get_token)(void *, void **, size_t *), 
               void *ggg_get_context,
               int (*ggg_send_token)(void *, void *, size_t),
               void *ggg_send_context)
{

  int context_established = 0;

  OM_uint32 major_status = GSS_S_COMPLETE;
  OM_uint32 minor_status = 0;
  OM_uint32 minor_status2 = 0;
  OM_uint32 message_context = 0;
  gss_ctx_id_t context_handle = GSS_C_NO_CONTEXT;
  gss_buffer_desc input_token_desc;
  gss_buffer_t input_token = &input_token_desc;
  gss_channel_bindings_t input_chan_bindings = NULL;
  gss_name_t client_name = NULL;    
  gss_OID mech_type;
  gss_buffer_desc output_token_desc;
  gss_buffer_t output_token = &output_token_desc;
  OM_uint32  ret_flags;
  OM_uint32  time_req;
  gss_cred_id_t deleg_cred;
  char * cp;

  gss_buffer_desc tmp_name_desc;
  gss_buffer_t tmp_name = &tmp_name_desc;

  gss_name_t g_client_name = NULL;
  gss_buffer_desc status_string_desc;
  gss_buffer_t status_string = &status_string_desc;


  input_token->length = 0;
  output_token->length = 0;
  status_string->length = 0;
  tmp_name->length = 0;
  if (ggg_client_name)
    *ggg_client_name = NULL;

  /* 
   * As a kludge for now, the gssapi_ssleay proxy_marshal routine
   * need to set the environment.
   * So we will set the pointer to the setenv here 
   *
   */
  gss_user_supplied_setenv = grami_setenv;

 /* if we want to overrid the default server name */

 if (ggg_server_name) {

   gss_name_t serv_name = NULL;
   gss_buffer_desc tmp_buffer_desc;
   gss_buffer_t tmp_buffer = &tmp_buffer_desc;

   tmp_buffer->value = ggg_server_name;
   tmp_buffer->length = strlen(ggg_server_name);
 
    major_status = gss_import_name(&minor_status,
					tmp_buffer,
			        0 /*(gss_OID) GSS_C_NT_HOSTBASED_SERVICE*/,
                    &serv_name);

    major_status = gss_acquire_cred(&minor_status,
                                    serv_name,
                                    GSS_C_INDEFINITE,
                                    GSS_C_NO_OID_SET,
                                    GSS_C_ACCEPT,
                                    &credential_handle,
                                    NULL,
                                    NULL);
  }

  do {
    if ((ggg_get_token(ggg_get_context,
                      &input_token->value,
                      &input_token->length)) < 0) {
       major_status = GSS_S_FAILURE;
       minor_status = 256+2;
       break;
    }
    major_status = gss_accept_sec_context(&minor_status,
                                        &context_handle,
                                        credential_handle,
                                        input_token,
                                        input_chan_bindings,
                                        &client_name,
                                        &mech_type,
                                        output_token,
                                        &ret_flags,
                                        &time_req,
                                        &deleg_cred);

    if (output_token->length != 0) {
      if ((ggg_send_token(ggg_send_context, 
                   output_token->value,
                   output_token->length)) < 0) {
        major_status = GSS_S_FAILURE;
        minor_status = 256+1;
      }
      gss_release_buffer(&minor_status,
                          output_token);
    }
    if (GSS_ERROR(major_status)) {
      if (context_handle != GSS_C_NO_CONTEXT)
         gss_delete_sec_context(&minor_status,
                                &context_handle,
                                GSS_C_NO_BUFFER);
        break;
    }
  } while (major_status & GSS_S_CONTINUE_NEEDED);

  if (input_token->length >0) {
    free(input_token->value); /* alloc done by g_get_token */
    input_token->length = 0;
  }

  if (major_status == GSS_S_COMPLETE) {
    if (client_name) {
      gss_buffer_desc tmp_buffer_desc;
      gss_buffer_t tmp_buffer = &tmp_buffer_desc;

      if ((major_status = gss_export_name(&minor_status,
                                   client_name,
                                   tmp_buffer)) == GSS_S_COMPLETE) {
       
        cp = (char *)malloc(tmp_buffer->length+1);
        if (cp) {
          memcpy(cp, tmp_buffer->value, tmp_buffer->length);
          cp[tmp_buffer->length] = '\0';
          *ggg_client_name = cp;
        } else {
          major_status = GSS_S_FAILURE;
          minor_status = 256+3;
        }
      }
      gss_release_buffer(&minor_status, tmp_buffer);
    } else {
      *ggg_client_name = "zero length client";
    }
  }
  /* return GSS status or zero if all ok. */
  if (major_status != GSS_S_COMPLETE) { 
    if (gss_display_status(&minor_status2,
                           major_status,
                           GSS_C_GSS_CODE,
                           GSS_C_NO_OID,
                           &message_context,
                           status_string) == GSS_S_COMPLETE) {
      fprintf(stderr,"%.*s\n",
                      status_string->length,
                      status_string->value);
      gss_release_buffer(&minor_status2,
                         status_string);
      if (gss_display_status(&minor_status2,
                             minor_status,
                             GSS_C_MECH_CODE,
                             GSS_C_NO_OID,
                             &message_context,
                             status_string) == GSS_S_COMPLETE) {
        fprintf(stderr,"%.*s\n",
                     status_string->length,
                     status_string->value);
        gss_release_buffer(&minor_status2, status_string);
      }
    }

    /* for now or in the minor status */
    return (major_status|minor_status);
  }
  else
    return 0;
}
#endif /* GSS_AUTHENTICATION */
