
#include <sys/types.h>

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>

#include "nexus.h"

/* include hdrs needed to print messages and manipulate strings 
 * (duroc internal functions) */
#include "duroc-common.h"
#include "utils.h"
#include "rsl.h"

/* include hdrs needed to use client and server libs */
#include "globus_duroc_control.h"
#include "globus_gass_server_ez.h"
#include "globus_common.h"

extern int _nx_debug_level;

/********************************************************************
 *
 * s_globus_duroc_callback_func
 *
 ********************************************************************/

static
char *s_state_string (int state)
{
  if ( state == GLOBUS_DUROC_JOB_STATE_DONE )
    return "GLOBUS_DUROC_JOB_STATE_DONE";
  else if ( state == GLOBUS_DUROC_JOB_STATE_FAILED ) 
    return "GLOBUS_DUROC_JOB_STATE_FAILED";
  else if ( state == GLOBUS_DUROC_JOB_STATE_ACTIVE )
    return "GLOBUS_DUROC_JOB_STATE_ACTIVE";
  else if ( state == GLOBUS_DUROC_JOB_STATE_PENDING ) 
    return "GLOBUS_DUROC_JOB_STATE_PENDING";
  else
    return "unknown state code";
}

static
void s_globus_duroc_callback_func (void *user_arg, char *job_contact,
			   int state, int errorcode)
{
  errorcode = ((state==GLOBUS_DUROC_JOB_STATE_FAILED) 
	       ? errorcode 
	       : GLOBUS_DUROC_SUCCESS);

  utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "DUROC JOB callback: job %s, state %s, errorcode %d.\n",
		 job_contact, s_state_string(state), errorcode);
}

/********************************************************************
 *
 * s_gram_fault_callback_func
 *
 ********************************************************************/

static
int s_globus_duroc_fault_callback_func (void *user_arg, int fault_code)
{
  utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
	       "duroc-request tool received nexus fault code %d\n",
	       fault_code);

  return 0;
}


/********************************************************************
 *
 * main
 *
 ********************************************************************/

int main (int argc, char* argv[])
{
  char *server_contact;
  char *request_string = NULL;
  char *request_file = NULL;
  char *callback_contact = NULL;
  int   interactive = 0;
  int   untrusting = 0;
  int   verbose = 1;
  int   no_more_options = 0;
  int   usage_error = 0;
  int   use_gass = 0; 
  unsigned short gass_port = 0;

  int err;
  globus_fifo_t argsq;
  int i;

  err = globus_module_activate (GLOBUS_COMMON_MODULE);
  assert (!err);

  globus_fifo_init (&argsq);

  /* put all args in queue */
  for (i=1; i<argc; i++) {
    globus_fifo_enqueue (&argsq, (void *) argv[i]);
  }


  while ( ! globus_fifo_empty (&argsq) ) {
    char *flag;

    flag = (char *) globus_fifo_dequeue (&argsq);

    if ( no_more_options 
	 || (flag[0]!='-') ) {
      /* last arg must be request string
       * (after '--' flag or not a '-' flag) */

      if ( request_string != NULL ) {
	fprintf (stderr, "%s: too many request strings specified!\n",
		 argv[0]);
	usage_error = 1;
      }
      else
	request_string = strdup (flag);
    }
    else {
      /* option flag */
      if ( flag[1]=='i' ) {
	interactive = 1;
      }
      else if ( flag[1]=='f' ) {
	char *filename;
	
	filename = (char *) globus_fifo_dequeue (&argsq);
	
	if ( request_file != NULL ) {
	  fprintf (stderr, "%s: too many request files specified!\n", 
		   argv[0]);
	  usage_error = 1;
	}
	else
	  request_file = strdup (filename);
      }
      else if ( flag[1]=='q' ) {
	verbose = 0;
      }
      else if ( flag[1]=='-' ) {
	no_more_options = 1;
      }
      else if ( flag[1]=='s' ) {
	use_gass = 1;
      }
      else {
	/* unrecognized flag */
	fprintf (stderr, "%s: unrecognized flag %s!\n",
		 argv[0], flag);
	usage_error = 1;
      }
    }
  }

  if ( (request_string!=NULL)
       && (request_file!=NULL) ) {
    fprintf (stderr,
	     "%s: cannot specify both request string and request file!\n",
	     argv[0]);
    usage_error = 1;
  }

  if ( request_file!=NULL ) {
    FILE *fp;
      
    fp = fopen (request_file, "r");
    if ( fp != NULL ) {
      int i;
      int c;
      int len = 0;

      fseek(fp, 0, SEEK_SET);
      while ( fgetc(fp) != EOF ) {
	len ++;
      }
      fseek(fp, 0, SEEK_SET);
	
      request_string = (char *) malloc (sizeof (char)
					* (len + 1));
      i=0;

      while ( (i<=len) 
	      && ((c=fgetc(fp))
		  != EOF) ) {
	request_string[i] = c;
	i++;
      }
      request_string[i] = '\0';
      
      free (request_file);
      request_file = NULL;

      fclose (fp);
    }
    else {
      fprintf (stderr,
	       "%s: cannot open file %s!\n",
	       argv[0], request_file);
      usage_error = 1;
    }
  }

  if ( (request_string==NULL)
       && (request_file==NULL) ) {
    fprintf (stderr,
	     "%s: must specify a request string or request file!\n",
	     argv[0]);
    usage_error = 1;
  }

  if ( usage_error ) {
    fprintf (stderr, 
	     "usage: duroc-request [-i] [-q] [-s] "
	     "{ {-f file} | [--] {request-string} }\n");
    return -1;
  }
  else /* request_string != NULL */ {
    /* the code below uses the request_string directly */
  }

  if ( verbose == 0 ) {
    globus_set_diagnostics_file (NULL);
  }

  err = globus_module_activate (GLOBUS_NEXUS_MODULE);
  assert (!err);

  nexus_enable_fault_tolerance(s_globus_duroc_fault_callback_func, NULL);

  if ( use_gass ) {
    /* transform the RSL to send "free" output streams to our
     * stdout/stderr via gass, rather than to /dev/null */
    globus_duroc_rsl_ast_t request_ast;
    char *gass_server_url = NULL;

    request_ast = globus_duroc_rsl_parse (request_string);

    err = globus_gass_server_ez_init (&gass_port, 
				      &gass_server_url,
				      GLOBUS_GASS_SERVER_EZ_LINE_BUFFER
				      | GLOBUS_GASS_SERVER_EZ_STDOUT_ENABLE
				      | GLOBUS_GASS_SERVER_EZ_STDERR_ENABLE,
				      NULL);
    assert (!err);
    assert (gass_server_url!=NULL);

    globus_duroc_rsl_add_output_defaults (request_ast, gass_server_url);

    globus_free (gass_server_url);
    gass_server_url = NULL;

    request_string = globus_duroc_rsl_unparse (request_ast);
  }

  {
    globus_duroc_control_t control;
    char *job_contact = NULL;
    int results_count;
    int *results;

    globus_module_activate (GLOBUS_DUROC_CONTROL_MODULE);
    assert (!err);

    err = globus_duroc_control_init (&control);
    assert (!err);

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG || verbose,
		 "making duroc request: %s\n",
		 request_string);

    err = globus_duroc_control_job_request (&control,
				     request_string,
				     0, 
				     callback_contact, 
				     &job_contact,
				     &results_count,
				     &results);
    utils_debug (GLOBUS_DUROC_DEBUG_FLAG || verbose, 
		 "duroc request status: %d\n"
		 "duroc job contact: \"%s\"\n",
		 err,
		 ((err==GLOBUS_DUROC_SUCCESS)
		  ? job_contact
		  : ""));
    if ( err == GLOBUS_DUROC_SUCCESS ) {
      int i;

      utils_debug (GLOBUS_DUROC_DEBUG_FLAG || verbose,
		   "duroc subrequests' status:\n");
      for (i=0; i<results_count; i++) {
	utils_debug (GLOBUS_DUROC_DEBUG_FLAG || verbose,
		     "    subrequest %d: %d\n",
		     i, results[i]);
      }

      globus_free (results);
    }
    else {
      utils_debug (GLOBUS_DUROC_DEBUG_FLAG || verbose,
		   "duroc request failed, exiting.\n");

      goto hard_exit;
    }

    if (interactive) {
      int flags;

      nexus_stdio_lock ();
      fprintf (stdout,
	       "entering interactive control mode...\n\n");
      nexus_stdio_unlock ();

      flags = fcntl (1 /* stdin */, F_SETFL, O_NONBLOCK);

      /* loop for commands */
      while (1) {
	int i;
	int     subjob_count;
	char ** subjob_labels;
	int   * subjob_states;

	/* print job state summary */

	err = globus_duroc_control_subjob_states (&control,
					   job_contact,
					   &subjob_count,
					   &subjob_states,
					   &subjob_labels);
	assert (!err);

	nexus_stdio_lock ();

	fprintf (stdout, "subjob states:\n");

	for (i=0; i<subjob_count; i++) {
	  fprintf (stdout,
		   "subjob >>%s<< %s\n",
		   (subjob_labels[i] ? subjob_labels[i] : "(none)"),
		   ((subjob_states[i]
		     ==GLOBUS_DUROC_SUBJOB_STATE_PENDING)
		    ? "PEND"
		    : ((subjob_states[i]
			==GLOBUS_DUROC_SUBJOB_STATE_ACTIVE)
		       ? "pend ACTIVE"
		       : ((subjob_states[i]
			   ==GLOBUS_DUROC_SUBJOB_STATE_CHECKED_IN)
			  ? "pend active CHECKIN"
			  : ((subjob_states[i]
			      ==GLOBUS_DUROC_SUBJOB_STATE_RELEASED)
			     ? "pend active checkin RUN"
			     : (subjob_states[i]
				==GLOBUS_DUROC_SUBJOB_STATE_DONE)
			     ? "DONE"
			     : "FAILED")))));
	  globus_free (subjob_labels[i]);
	  subjob_labels[i] = NULL;
	}

	fprintf (stdout, "end subjob states.\n\n");
	
	globus_free (subjob_states);
	globus_free (subjob_labels);

	/* prompt for user command */

	fprintf (stdout, "\n"
		 "enter command \"Dlabel\" (delete labeled subjob),\n"
		 "              \"K\" (kill entire job),\n"
		 "              \"C\" (commit current job),\n"
		 "           or \"Q\" (quit request tool)\n\n");
	
	nexus_stdio_unlock ();
	
	/* get user command and perform background processing */
	{
	  globus_fifo_t input;

	  err = globus_fifo_init (&input); assert (!err);
	  
	  while ( globus_fifo_empty (&input) ) {
	    ssize_t size;
	    char buf[1];

	    nexus_stdio_lock ();

	    size = read (1 /* stdin */, buf, 1 /* one byte */);

	    nexus_stdio_unlock ();
	    
	    if (size == 1) {
	      /* queue up data just read */
	      globus_fifo_enqueue (&input, (void *) (int) buf[0]);
	    }
	    else if (size < 0) {
	      /* no input ready */
	      nexus_poll();
	    }
	    else {
	      /* eof? */

	      nexus_stdio_lock ();
	      fprintf (stdout, "eof reached. exiting.\n");
	      nexus_stdio_unlock ();
	      
	      goto user_exit;
	    }
	  }
	  
	  if ( ((char) (int) globus_fifo_peek (&input)) == 'C' ) {
	    nexus_stdio_lock ();
	    fprintf (stdout, "C commit requested\n");
	    nexus_stdio_unlock ();
	    globus_fifo_dequeue (&input);

	    nexus_stdio_lock ();
	    fprintf (stdout,
		     "releasing barrier at user's request...\n");
	    nexus_stdio_unlock ();

	    err = globus_duroc_control_barrier_release (&control,
						 job_contact,
						 GLOBUS_TRUE);

	    nexus_stdio_lock ();
	    fprintf (stdout,
		     "release returned %s.\n",
		     (err ? "failure" : "success"));
	    nexus_stdio_unlock ();
	  }
	  else if ( ((char) (int) globus_fifo_peek (&input)) == 'Q' ) {
	    nexus_stdio_lock ();
	    fprintf (stdout, "Q quit requested\n");
	    nexus_stdio_unlock ();
	    globus_fifo_dequeue (&input);

	    goto user_exit;
	  }
	  else if ( ((char) (int) globus_fifo_peek (&input)) == 'K' ) {
	    nexus_stdio_lock ();
	    fprintf (stdout, "K kill job requested\n");
	    nexus_stdio_unlock ();
	    globus_fifo_dequeue (&input);

	    nexus_stdio_lock ();
	    fprintf (stdout, "canceling job at user's request...\n");
	    nexus_stdio_unlock ();

	    err = globus_duroc_control_job_cancel (&control,
					    job_contact);

	    nexus_stdio_lock ();
	    fprintf (stdout,
		     "cancel returned %s.\n",
		     (err ? "failure" : "success"));
	    nexus_stdio_unlock ();
	  }
	  else if ( ((char) (int) globus_fifo_peek (&input)) == 'D' ) {
	    /* get subjob label..
	     * all characters up to but not including newline */
	    int newline_read = 0;
	    ssize_t size;
	    char buf[1];
	    char *subjob_label;

	    globus_fifo_dequeue (&input); /* throw out 'D' */

	    while ( ! newline_read ) {
	      size = read (1 /* stdin */, buf, 1 /* one byte */);
	      if (size == 1) {
		if ( buf[0] != '\n' ) {
		  globus_fifo_enqueue (&input, (void *) (int) buf[0]);
		}
		else {
		  newline_read = 1;
		}
	      }
	      else if ( size == -1 ) {
		/* no input ready */
	      }
	      else {
		/* eof? */
		nexus_stdio_lock ();
		fprintf (stdout, "eof reached. exiting.\n");
		nexus_stdio_unlock ();
		    
		goto user_exit;
	      }
	    }

	    {
	      int len;

	      len = globus_fifo_size (&input);
	      if (len>0) {
		subjob_label = globus_malloc (sizeof(char)
					      * (len + 1));
		for (i=0; i<len; i++) {
		  subjob_label[i] = (char) (int) globus_fifo_dequeue (&input);
		}
		subjob_label[len] = '\0';
	      }
	      else {
		subjob_label = "";
	      }
	    }

	    nexus_stdio_lock ();
	    fprintf (stdout, "D delete subjob >>%s<< requested\n",
		     subjob_label);
	    nexus_stdio_unlock ();

	    err = globus_duroc_control_subjob_delete (&control,
					       job_contact,
					       subjob_label);

	    nexus_stdio_lock ();
	    if (!err) {
	      fprintf (stdout, "subjob >>%s<< deleted\n", subjob_label);
	    }
	    else {
	      fprintf (stdout, "subjob >>%s<< deletion failed "
		       "(code %d)\n", subjob_label, err);
	    }
	    nexus_stdio_unlock ();
	  }
	  else {
	    /* unknown character, reissue prompt */
	    globus_fifo_dequeue (&input);
	  }
	}
	nexus_poll ();
      }

    after_loop2: /* eliminate pesky dead-code compiler warning */
      ;
    }
    else {
      utils_debug (GLOBUS_DUROC_DEBUG_FLAG || verbose,
		   "releasing barrier in automatic mode...\n");

      globus_duroc_control_barrier_release (&control,
				     job_contact,
				     GLOBUS_TRUE);

      if ( untrusting ) {
	/* can't trust GRAM to tell us when we're done 
	 * so wait for user to kill us */

	utils_debug (GLOBUS_DUROC_DEBUG_FLAG || verbose,
		     "looping forever (can't trust termination signals)\n");

	while (1) {
	  nexus_poll_blocking ();
	}
      }
      else {
	utils_debug (GLOBUS_DUROC_DEBUG_FLAG || verbose,
		     "waiting for job termination\n");

	while (1) {
	  int i;
	  int     subjob_count;
	  char ** subjob_labels;
	  int   * subjob_states;
	  int not_terminated = 0;

	  /* poll for job state */

	  err = globus_duroc_control_subjob_states (&control,
					     job_contact,
					     &subjob_count,
					     &subjob_states,
					     &subjob_labels);
	  assert (!err);
	  
	  for (i=0; i<subjob_count; i++) {
	    if ( (subjob_states[i] != GLOBUS_DUROC_SUBJOB_STATE_DONE)
		 && (subjob_states[i] != GLOBUS_DUROC_SUBJOB_STATE_FAILED) )
	      not_terminated = 1;
	    
	    globus_free (subjob_labels[i]);
	    subjob_labels[i] = NULL;
	  }
	  
	  globus_free (subjob_states);
	  globus_free (subjob_labels);
	  
	  if ( not_terminated ) 
	    nexus_poll_blocking ();
	  else
	    goto job_terminated;
	}
	
      job_terminated:
	goto user_exit;
      }
    }
  }

hard_exit:
  err = 1;
  goto final_exit;

user_exit:
  err = 0;
  goto final_exit;

final_exit:

  if ( use_gass ) {
    globus_gass_server_ez_shutdown (gass_port);
  }

  globus_module_deactivate (GLOBUS_DUROC_CONTROL_MODULE);
  globus_module_deactivate (GLOBUS_NEXUS_MODULE);

  free (request_string);

  return err;
}
