
#include <stdio.h>

#include <assert.h>

#include <netdb.h>
#include <sys/param.h>
#include <sys/types.h>
#include <unistd.h>

#include "globus_common.h"
#include "globus_nexus.h"
#include "globus_gram_myjob.h"
#include "globus_duct_runtime.h"

#include "globus_duroc_runtime.h"
#include "globus_duroc_bootstrap.h"

#include "utils.h"



/********************************************************************
 *
 * main
 *
 ********************************************************************/

int main ()
{
  int err;
  int local_subjob_index;

  globus_fifo_t fifo_test;

  globus_fifo_init (&fifo_test);

  globus_fifo_enqueue (&fifo_test, (void *) 1);
  globus_fifo_enqueue (&fifo_test, (void *) 2);
  globus_fifo_enqueue (&fifo_test, (void *) 5);
  globus_fifo_enqueue (&fifo_test, (void *) 3);
  globus_fifo_enqueue (&fifo_test, (void *) 4);
  globus_fifo_enqueue (&fifo_test, (void *) 6);

  globus_fifo_remove (&fifo_test, (void *) 5);
  globus_fifo_remove (&fifo_test, (void *) 6);

  utils_fprintf (stderr, "fifo_test:");
  while (! globus_fifo_empty (&fifo_test) ) {
    utils_fprintf (stderr, " %ld", (long) globus_fifo_dequeue (&fifo_test));
  }
  utils_fprintf (stderr, "\n");

  globus_fifo_destroy (&fifo_test);


  /* intialize and start nexus */
  err = globus_module_activate (GLOBUS_NEXUS_MODULE);
  assert (!err);

  nexus_enable_fault_tolerance (NULL, NULL);
  utils_fprintf (stderr, "duroc-test-app: "
		 "nexus activated\n");

  utils_fprintf (stderr, "beginning uninitialized API tests...\n");

  utils_fprintf (stderr, "  globus_duroc_runtime_inter_subjob_send()... ");
  err = globus_duroc_runtime_inter_subjob_send (-1, "tag", 
						4, (globus_byte_t *) "data");
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));

  utils_fprintf (stderr, "  globus_duroc_runtime_inter_subjob_receive()... ");
  err = globus_duroc_runtime_inter_subjob_receive ("tag", NULL, NULL);
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));

  utils_fprintf (stderr, "  globus_duroc_runtime_inter_subjob_structure()... ");
  err = globus_duroc_runtime_inter_subjob_structure (NULL, NULL, NULL);
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));

  utils_fprintf (stderr, "  globus_duroc_runtime_intra_subjob_rank()... ");
  err = globus_duroc_runtime_intra_subjob_rank (NULL);
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));
  
  utils_fprintf (stderr, "  globus_duroc_runtime_intra_subjob_size()... ");
  err = globus_duroc_runtime_intra_subjob_size (NULL);
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));
  
  utils_fprintf (stderr, "  globus_duroc_runtime_intra_subjob_send()... ");
  err = globus_duroc_runtime_intra_subjob_send (-1, "tag", 
						4, (globus_byte_t *) "data");
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));

  utils_fprintf (stderr, "  globus_duroc_runtime_intra_subjob_receive()... ");
  err = globus_duroc_runtime_intra_subjob_receive ("tag", NULL, NULL);
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));

  utils_fprintf (stderr, "completed uninitialized API tests\n");


  err = globus_module_activate (GLOBUS_DUCT_RUNTIME_MODULE);
  assert (!err);

  err = globus_module_activate (GLOBUS_GRAM_MYJOB_MODULE);
  assert (!err);

  err = globus_module_activate (GLOBUS_DUROC_RUNTIME_MODULE);
  assert (!err);

  utils_fprintf (stderr, "duroc-test-app: "
		 "duroc_runtime activated\n");

  globus_duroc_runtime_barrier ();

  utils_fprintf (stderr, "duroc-test-app: "
		 "duroc_runtime_barrier returned\n");

  err = globus_module_deactivate (GLOBUS_DUROC_RUNTIME_MODULE);
  assert (!err);

  err = globus_module_deactivate (GLOBUS_GRAM_MYJOB_MODULE);
  assert (!err);
 
  err = globus_module_deactivate (GLOBUS_DUCT_RUNTIME_MODULE);
  assert (!err);

  utils_fprintf (stderr, "duroc-test-app: "
		 "duroc_runtime deactivated\n");

  utils_fprintf (stderr, "beginning wild standalone API tests...\n");

  utils_fprintf (stderr, "  globus_duroc_runtime_inter_subjob_send()... ");
  err = globus_duroc_runtime_inter_subjob_send (-1, "tag", 
						4, (globus_byte_t *) "data");
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));

  utils_fprintf (stderr, "  globus_duroc_runtime_inter_subjob_structure()... ");
  err = globus_duroc_runtime_inter_subjob_structure (NULL, NULL, NULL);
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));

  utils_fprintf (stderr, "  globus_duroc_runtime_intra_subjob_rank()... ");
  err = globus_duroc_runtime_intra_subjob_rank (NULL);
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));
  
  utils_fprintf (stderr, "  globus_duroc_runtime_intra_subjob_size()... ");
  err = globus_duroc_runtime_intra_subjob_size (NULL);
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));
  
  utils_fprintf (stderr, "  globus_duroc_runtime_intra_subjob_send()... ");
  err = globus_duroc_runtime_intra_subjob_send (-1, "tag", 
						4, (globus_byte_t *) "data");
  utils_fprintf (stderr, "%d %s\n", err, globus_duroc_error_string (err));

  utils_fprintf (stderr, "completed wild standalone API tests\n");


  /* the application body */
  {
    err = globus_module_activate (GLOBUS_DUROC_BOOTSTRAP_MODULE);
    assert (!err);

    utils_fprintf (stderr, "duroc-test-app: "
		   "duroc_bootstrap activated\n");

    {
      char *hostname;
      int i;
      char *my_info;
      int subjob_count;
      int subjob_id;
      char **subjob_info_array;

      {
	i = 256;
	hostname = (char*) nexus_malloc (sizeof(char)*i);
	while ( gethostname (hostname, i) != 0 ) {
	  nexus_free (hostname);
	  i *= 2;
	  hostname = (char*) nexus_malloc (sizeof(char)*i);
	}
      }

      my_info = nexus_malloc (sizeof(char)
			      *(utils_strlen ("SUBJOB (pid XXXXXX) ")
				+ utils_strlen (hostname)
				+ 1));
      assert (my_info!=NULL);
      
      utils_sprintf (my_info, 
		     "SUBJOB %s (pid %d)", 
		     hostname, getpid());

      nexus_free (hostname);

      utils_fprintf (stderr, "duroc-test-app: "
		     "performing subjob_exchange\n");
      err = globus_duroc_bootstrap_subjob_exchange (my_info,
					     &subjob_count,
					     &subjob_id,
					     &subjob_info_array);

      local_subjob_index = subjob_id;

      if (err) {
	utils_fprintf (stderr, "duroc-test-app: "
		       "subjob_exchange returned error %d\n", err);
      }
      else {
	utils_fprintf (stderr, "duroc-test-app: %s: "
		       "subjob_exchange returned index %d of %d...\n",
		       my_info, subjob_id, subjob_count);
	if ( subjob_info_array != NULL ) {
	  int subjob;
	  for (subjob=0; subjob<subjob_count; subjob++) {
	    utils_fprintf (stderr, 
			   "duroc-test-app: "
			   "%s (subjob %d of %d): "
			   "subjob_info[%d] == \"%s\"\n",
			   my_info,
			   subjob_id, subjob_count,
			   subjob, 
			   subjob_info_array[subjob]);
	  }
	  if ( subjob_id==0 ) {
	    FILE *out;
	    nexus_stdio_lock ();
	    out = fopen ("duroc-test-app.out", "a");
	    if ( out!=NULL ) {
	      for (subjob=0; subjob<subjob_count; subjob++) {
		fprintf (out, 
			 "%s (subjob %d of %d): "
			 "subjob_info[%d] == \"%s\"\n",
			 my_info,
			 subjob_id, subjob_count,
			 subjob, 
			 subjob_info_array[subjob]);
	      }
	      fclose (out);
	    }
	    nexus_stdio_unlock ();
	  }
	}
      }
    }

    {
      nexus_endpointattr_t local_epattr;
      nexus_endpoint_t local_ep;
      nexus_startpoint_t local_sp;

      int process_count;
      nexus_startpoint_t * sp_array;

      nexus_startpoint_t sp_copy;

      err = nexus_endpointattr_init (&local_epattr);
      assert (!err);

      err = nexus_endpoint_init (&local_ep,
				 &local_epattr);
      assert (!err);

      err = nexus_startpoint_bind (&local_sp, &local_ep);
      assert (!err);

      err = nexus_startpoint_copy (&sp_copy, &local_sp);
      assert (!err);

      err = nexus_startpoint_destroy (&sp_copy);
      assert (!err);

      utils_fprintf (stderr, "duroc-test-app: "
		     "performing fill_master_sp_vector...\n");

      globus_duroc_bootstrap_master_sp_vector (&local_sp,
					&process_count,
					&sp_array);

      utils_fprintf (stderr, "duroc-test-app: "
		     "master_sp_vector returned\n");

      if ( sp_array != NULL ) {
	int i;
	
	utils_fprintf (stderr, "duroc-test-app: "
		       "master_sp_vector process_count %d\n",
		       process_count);

	assert (nexus_startpoint_to_current_context (&(sp_array[0])));
	for (i=1; i<process_count; i++) {
	  assert (! nexus_startpoint_to_current_context (&(sp_array[i])));
	}

	{
	  FILE *out;
	  nexus_stdio_lock ();
	  out = fopen ("duroc-test-app.out", "a");
	  if ( out!=NULL ) {
	    fprintf (out, 
		     "master_sp_vector returned in master: "
		     "%d processes reported\n",
		     process_count);
	    fclose (out);
	  }
	  nexus_stdio_unlock ();
	}
      }
      else {
	assert (process_count == 0);
      }

      sp_array = NULL;
      process_count = -1;
      
      {
	int ordered_subjob_index;
	char *ordered_subjob_index_string;

	ordered_subjob_index_string = getenv ("SUBJOB_INDEX");
	if (ordered_subjob_index_string) {
	  utils_fprintf (stderr, "using user-supplied SUBJOB_INDEX=\"%s\"\n", 
			 ordered_subjob_index_string);
	  nexus_stdio_lock ();
	  sscanf (ordered_subjob_index_string, "%d", &ordered_subjob_index);
	  nexus_stdio_unlock ();
	}
	else {
	  ordered_subjob_index = local_subjob_index;
	}
	

	globus_duroc_bootstrap_ordered_master_sp_vector (&local_sp,
							 ordered_subjob_index,
							 &process_count,
							 &sp_array);
	
	utils_fprintf (stderr, "duroc-test-app: "
		       "ordered_master_sp_vector returned\n");
	
	if ( sp_array != NULL ) {
	  int i;
	  
	  utils_fprintf (stderr, "duroc-test-app: "
			 "ordered_master_sp_vector process_count %d\n",
			 process_count);
	  
	  assert (nexus_startpoint_to_current_context (&(sp_array[0])));
	  for (i=1; i<process_count; i++) {
	    assert (! nexus_startpoint_to_current_context (&(sp_array[i])));
	  }
	  
	  {
	    FILE *out;
	    nexus_stdio_lock ();
	    out = fopen ("duroc-test-app.out", "a");
	    if ( out!=NULL ) {
	      fprintf (out, 
		       "ordered_master_sp_vector returned in master: "
		       "%d processes reported\n",
		       process_count);
	      fclose (out);
	    }
	    nexus_stdio_unlock ();
	  }
	}
	else {
	  assert (process_count == 0);
	}
      }
    }

    globus_module_deactivate (GLOBUS_DUROC_BOOTSTRAP_MODULE);

    utils_fprintf (stderr, "duroc-test-app: "
		   "duroc_bootstrap deactivated\n");
  }

  globus_module_deactivate (GLOBUS_NEXUS_MODULE);

  fprintf (stderr, "duroc-test-app: nexus deactivated\n");
  fprintf (stderr, "duroc-test-app: exiting\n");

  return 0;
}
