
/********************************************************************
 *
 * This file implements thread-safe utility routines
 *
 ********************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <assert.h>

#include "globus_duroc_common.h"
#include "globus_common.h"
#include "globus_gram_client.h"

#include "nexus.h"

#include "utils.h"

int
globus_duroc_error_is_gram_client_error (int error_code)
{
  return ( error_code >= GLOBUS_DUROC_ERROR_GRAM_ERROR_BASE );
}

int
globus_duroc_error_get_gram_client_error (int error_code)
{
  return ( error_code - GLOBUS_DUROC_ERROR_GRAM_ERROR_BASE );
}

const char *
globus_duroc_error_string (int err)
{
  const char * result;

  switch (err) {
  case GLOBUS_DUROC_SUCCESS: 
    result = "operation successful"; break;
  case GLOBUS_DUROC_ERROR_INVALID_REQUEST: 
    result = "an invalid request was made"; break;
  case GLOBUS_DUROC_ERROR_INVALID_PARAMETER: 
    result = "an invalid argument was provided"; break;
  case GLOBUS_DUROC_ERROR_DUCT_FAILED: 
    result = "an unexpected globus_duct library error occurred"; break;
  case GLOBUS_DUROC_ERROR_INIT_FAILED: 
    result = "an initialization step failed"; break;
  case GLOBUS_DUROC_ERROR_UNKNOWN_LABEL: 
    result = "the provided RSL 'label' value is unknown"; break;
  case GLOBUS_DUROC_ERROR_NEXUS_FAILED: 
    result = "an unexpected globus_nexus library error occurred"; break;
  case GLOBUS_DUROC_ERROR_MALLOC_FAILED: 
    result = "memory allocation failed"; break;
  case GLOBUS_DUROC_ERROR_INTERNAL_FAILURE: 
    result = "an unexpected globus_duroc library error occurred"; 
    break;
  case GLOBUS_DUROC_ERROR_INVALID_RSL: 
    result = "the provided RSL request was not in the expected format"; break;
  case GLOBUS_DUROC_ERROR_INVALID_MULTIREQ: 
    result = "the provided RSL multi-request was not in the expected format";
    break;
  case GLOBUS_DUROC_ERROR_GRAM_FAILED: 
    result = "an unexpected globus_gram library error occurred"; break;
  case GLOBUS_DUROC_ERROR_INVALID_MANAGER_CONTACT: 
    result = "the provided RSL 'resourceManagerContact' value is incorrect";
    break;
  case GLOBUS_DUROC_ERROR_ALREADY_RELEASED: 
    result = "the job barrier has already been released"; break;
  case GLOBUS_DUROC_ERROR_ALREADY_CANCELLED: 
    result = "the job has already been cancelled"; break;
  case GLOBUS_DUROC_ERROR_BAD_START_TYPE: 
    result = "the provided RSL 'subjobStartType' value is incorrect"; break;
  case GLOBUS_DUROC_ERROR_BAD_COMMS_TYPE: 
    result = "the provided RSL 'subjobCommsType' value is incorrect"; break;
  case GLOBUS_DUROC_ERROR_NOT_INITIALIZED: 
    result = "the required initialization has not been performed"; break;
  case GLOBUS_DUROC_ERROR_INVALID_OPERATION: 
    result = "the requested operation is invalid"; break;
  case GLOBUS_DUROC_ERROR_DUPLICATE_SUBJOB_LABEL: 
    result = "the same RSL 'label' value was supplied for more than one subjob"; 
    break;
  case GLOBUS_DUROC_ERROR_PROTOCOL_VERSION_MISMATCH: 
    result = "incompatible globus_duroc library versions were detected";
    break;
  case GLOBUS_DUROC_ERROR_INVALID_CHECKIN: 
    result = "an invalid barrier check-in message was received"; break;
  default:
    if ( globus_duroc_error_is_gram_client_error (err) ) {
      result = globus_gram_client_error_string (
		globus_duroc_error_get_gram_client_error (err));
    }
    else {
      result = "an unknown failure occurred"; 
    }
    break;
  }

  return result;
}


int utils_sprintf(char *s, const char *format,  ...)
{
  va_list pa;

  int res;

  nexus_stdio_lock ();

  va_start(pa, format);
  res = vsprintf (s, format, pa);
  va_end(pa);

  nexus_stdio_unlock ();

  return res;
}

int utils_fprintf(FILE *fp, const char *format,  ...)
{
  va_list pa;

  int res;

  nexus_stdio_lock ();

  va_start(pa, format);
  res = vfprintf (fp, format, pa);
  va_end(pa);

  nexus_stdio_unlock ();

  return res;
}

int utils_strlen (const char *str)
{
  int len = 0;

  while (str[len] != '\0') len++;

  return len;
}

char *utils_strdup (const char *src)
{
  int i, len;
  char *dst;

  if (src==NULL) return NULL;

  for (len=0; src[len]!='\0'; len++) /* count */;

  dst = globus_malloc (sizeof(char)*(len + 1));
  assert (dst!=NULL);

  for (i=0; i<len; i++) {
    dst[i] = src[i];
  }
  dst[len] = '\0';

  return dst;
}

int utils_streq (const char *s1, const char *s2)
/* assume ASCII */
{
  int i=0;
  
  while ( (s1[i] != '\0')
	  && (s2[i] != '\0') ) {
    if ( s1[i] != s2[i] ) return 0;

    i++;
  }

  if ( s1[i] != s2[i] ) return 0;

  return 1;
}

#define s_abs_int(x) \
(((x) < 0) ? (0 - (x)) : (x))

#define s_alpha(c) \
     ( ( ((c) >= 'a') && ((c) <= 'z') ) \
       || ( ((c) >= 'A') && ((c) <= 'Z') ) )

#define s_down_case(c) \
     ( ( ((c) >= 'A') && ((c) <= 'Z') ) \
       ? ((c) - ('A' - 'a')) \
       : (c) )

int utils_streq_ci (const char *s1, const char *s2)
{
  int i=0;
  
  while ( (s1[i] != '\0')
	  && (s2[i] != '\0') ) {
    if ( s_alpha (s1[i]) && s_alpha (s2[i]) ) {
      if ( s_down_case (s1[i]) != s_down_case (s2[i]) )
	return 0;
    }
    else if ( s1[i] != s2[i] ) return 0;

    i++;
  }

  if ( s1[i] != s2[i] ) return 0;

  return 1;
}

void utils_debug (int pred_flag, const char*format, ...)
{
  va_list pa;

  if ( pred_flag ) {
    va_start(pa, format);
    nexus_stdio_lock ();
    vfprintf (stderr,
	      format, pa);
    nexus_stdio_unlock ();
    va_end(pa);
  }
}

#define globus_l_hex_ascii_to_int(digit)        \
( ((digit) >= '0' && (digit) <= '9')            \
  ? ((int) ((digit) - '0'))                     \
  : ( ((digit) >= 'A' && (digit) <= 'F')        \
      ? ((int) ((digit) - 'A' + 0xA))           \
      : ( ((digit) >= 'a' && (digit) <= 'f')    \
	  ? ((int) ((digit) - 'a' + 0xA))       \
	  : (assert ( 0 ), 0))))

void
globus_l_duroc_hex_decode_byte_array (const char *    src_hex_array,
				      int             byte_count,
				      globus_byte_t * dst_byte_array)
{
  int byte;

  assert ( dst_byte_array!=NULL );
  assert ( src_hex_array!=NULL );
  assert ( (utils_strlen (src_hex_array) % 2) == 0 );
  assert ( (utils_strlen (src_hex_array) / 2) == byte_count );

  for (byte=0; byte < byte_count; byte++) {
    char high_digit, low_digit;
    int  high_val, low_val;

    high_digit = src_hex_array[2*byte+0];
    low_digit  = src_hex_array[2*byte+1];

    high_val = globus_l_hex_ascii_to_int (high_digit);
    low_val  = globus_l_hex_ascii_to_int (low_digit);

    assert ( high_val >= 0 && high_val < 16 );
    assert ( low_val  >= 0 && low_val  < 16 );

    dst_byte_array[byte] = ((globus_byte_t) (16*high_val + low_val));
  }
}

#define globus_l_hex_int_to_ascii(digit)        \
( ((digit) >= 0 && (digit) <= 9)                \
  ? ((char) ((digit) + '0'))                    \
  : ( ((digit) >= 0xA && (digit) <= 0xF)        \
      ? ((char) ((digit) - 0xA + 'A'))          \
      : (assert ( 0 ), 0)))

void
globus_l_duroc_hex_encode_byte_array (const globus_byte_t * src_byte_array,
				      int                   byte_count,
				      char *                dst_hex_array)
{
  int byte;

  assert ( src_byte_array!=NULL );
  assert ( dst_hex_array!=NULL );

  for (byte=0; byte < byte_count; byte++) {
    int  val;
    char high_digit, low_digit;
    char coded[3];
    int  high_val, low_val;

    val = (int) src_byte_array[byte];

    assert ( val >= 0 && val < 256 );

    high_val = val / 16;
    low_val  = val % 16;

    high_digit = globus_l_hex_int_to_ascii (high_val);
    low_digit  = globus_l_hex_int_to_ascii (low_val);

    coded[0] = high_digit; coded[1] = low_digit; coded[2] = '\0';

    dst_hex_array[2*byte+0] = high_digit;
    dst_hex_array[2*byte+1] = low_digit;
  }

  dst_hex_array[2*byte] = '\0';
}


