
#include <assert.h>

#include "nexus.h"
#include "globus_gram_client.h"

#include "subjob.h"
#include "job_monitor.h"
#include "control.h"

#include "duroc-common.h"


#define s_subjob_init_mutex_err(err) \
(globus_duroc_at_error("mutex_init", err), \
 /* GLOBUS_DUROC_ERROR_NEXUS_FAILED */ \
 globus_error_put (globus_object_construct (GLOBUS_ERROR_TYPE_COMMUNICATION_FAILED)))

#define s_subjob_init_cond_err(err) \
     (globus_duroc_at_error("cond_init", err), \
/* GLOBUS_DUROC_ERROR_NEXUS_FAILED */ \
 globus_error_put (globus_object_construct (GLOBUS_ERROR_TYPE_COMMUNICATION_FAILED)))

#define s_subjob_init_label_err(err) \
     (globus_duroc_at_error("strdup", err), \
/* GLOBUS_DUROC_ERROR_MALLOC_FAILED */ \
 globus_error_put (globus_object_construct (GLOBUS_ERROR_TYPE_ACCESS_FAILED)))

#define s_subjob_init_serialno_err(err) \
     (globus_duroc_at_error("make_subjob_serialno", err), \
     /* GLOBUS_DUROC_ERROR_INTERNAL_FAILURE */ \
globus_error_put (globus_object_construct (GLOBUS_ERROR_TYPE_ABORTED)))

#define s_subjob_init_duct_err(err) \
     (globus_duroc_at_error("duct_config_data_init", err), \
/* GLOBUS_DUROC_ERROR_DUCT_FAILED */ \
globus_error_put (globus_object_construct (GLOBUS_ERROR_TYPE_COMMUNICATION_FAILED)))

#define s_subjob_init_link_monitor_err(err) \
     (globus_duroc_at_error("monitor_link_subjob", err), err)

/*
 * initialize subjob record,
 * bind subjob record into job_monitor
 *
 * if this call succeeds, the record will be scheduled for 
 * reference-count garbage-collection.
 * the space at (*subjobp) MUST have been globus_malloc'd!!
 */
globus_result_t
globus_duroc_control_i_subjob_init (globus_duroc_control_t      * controlp,
			     globus_duroc_job_monitor_t  * job_monitorp,
			     globus_duroc_subjob_t       * subjobp,
			     char                 * subjob_label,
			     globus_duroc_start_method_t   starttype,
			     globus_duroc_comms_method_t   commstype)
{
  int err;
  globus_result_t res;

  if ( (controlp==NULL) || (job_monitorp==NULL)
       || (subjobp==NULL) ) { 
    return globus_error_put (
		     globus_object_construct (GLOBUS_ERROR_TYPE_BAD_DATA));
  }

  subjobp->ref_count = 0;

  err = nexus_mutex_init (&(subjobp->mutex), NULL);
  if (err) {
    res = s_subjob_init_mutex_err (err);
    goto subjob_init_mutex_error;
  }

  err = nexus_cond_init (&(subjobp->cond), NULL);
  if (err) {
    res = s_subjob_init_cond_err (err);
    goto subjob_init_cond_error;
  }

  subjobp->contact = NULL;
  subjobp->state = GLOBUS_DUROC_SUBJOB_STATE_PENDING;
  subjobp->label = NULL;
  subjobp->start_type = starttype;
  subjobp->comms_type = commstype;
  subjobp->checked_in = GLOBUS_FALSE;
  subjobp->told_to_run = GLOBUS_FALSE;

  assert ( (starttype==GLOBUS_DUROC_START_STRICT)
	   || (starttype==GLOBUS_DUROC_START_LOOSE)
	   || (starttype==GLOBUS_DUROC_START_NONE) );

  assert ( (commstype==GLOBUS_DUROC_COMMS_BLOCKING)
	   || (commstype==GLOBUS_DUROC_COMMS_NONBLOCKING)
	   || (commstype==GLOBUS_DUROC_COMMS_NONE) );

  if ( subjob_label ) subjobp->label = utils_strdup (subjob_label);
  if ((subjobp->label==NULL)
      && (subjob_label!=NULL)) {
    res = s_subjob_init_label_err (0);
    goto subjob_init_label_error;
  }

  subjobp->serialno = globus_duroc_control_i_job_monitor_make_subjob_no (
								job_monitorp);
  if (subjobp->serialno <= 0) {
    res = s_subjob_init_serialno_err (subjobp->serialno);
    goto subjob_init_serialno_error;
  }

  res = globus_duroc_control_i_job_monitor_link_subjob (job_monitorp, 
							subjobp,
							subjob_label);
  if (err) {
    res = s_subjob_init_link_monitor_err (res);
    goto subjob_init_link_monitor_error;
  }

  return GLOBUS_SUCCESS;

  /* subjob_init_error_clauses:

  globus_duroc_control_i_job_monitor_unlink_subjob (job_monitorp, subjobp); */
 subjob_init_link_monitor_error:

 subjob_init_serialno_error:

  globus_free ((char *) subjobp->label);
 subjob_init_label_error:

  nexus_cond_destroy (&(subjobp->cond));
 subjob_init_cond_error:

  nexus_mutex_destroy (&(subjobp->mutex));
 subjob_init_mutex_error:

  return res;
}

#define s_register_contact_job_lock_err(err) \
(globus_duroc_at_error ("mutex lock", err), GLOBUS_DUROC_ERROR_NEXUS_FAILED)
#define s_register_contact_lock_err(err) \
(globus_duroc_at_error ("mutex lock", err), GLOBUS_DUROC_ERROR_NEXUS_FAILED)
#define s_register_contact_rereg_err(err) \
(globus_duroc_at_error ("contact already registered", 0), \
 GLOBUS_DUROC_ERROR_INTERNAL_FAILURE)
#define s_register_contact_strdup_err(err) \
(globus_duroc_at_error ("strdup", 0), GLOBUS_DUROC_ERROR_MALLOC_FAILED)
#define s_register_contact_monitor_err(err) \
(globus_duroc_at_error ("job_monitor link gram", err), GLOBUS_DUROC_ERROR_INTERNAL_FAILURE)
#define s_register_contact_control_err(err) \
(globus_duroc_at_error ("control link gram", err), GLOBUS_DUROC_ERROR_INTERNAL_FAILURE)

/* 
 * add GRAM contact to subjob record,
 * link into job monitor and control
 */
int 
globus_duroc_control_i_subjob_register_contact (globus_duroc_control_t      *controlp,
					 globus_duroc_job_monitor_t *job_monitorp,
					 globus_duroc_subjob_t      *subjobp,
					 const char          *contact)
{
  int err;
  int job_serialno;
  int subjob_serialno;

  if ( (controlp==NULL) || (job_monitorp==NULL) || (subjobp==NULL) 
       || (contact==NULL) ) 
    return GLOBUS_DUROC_ERROR_INVALID_PARAMETER;

  err = nexus_mutex_lock (&(job_monitorp->mutex));
  if (err) {
    err = s_register_contact_job_lock_err (err);
    goto register_contact_job_lock_error;
  }

  job_serialno = job_monitorp->serialno;

  err = nexus_mutex_unlock (&(job_monitorp->mutex)); assert (!err);
  
  err = nexus_mutex_lock (&(subjobp->mutex));
  if (err) {
    err = s_register_contact_lock_err (err);
    goto register_contact_subjob_lock_error;
  }
  
  if ( subjobp->contact != NULL ) {
    err = s_register_contact_rereg_err (0);
    goto register_contact_rereg_error;
  }
  
  subjobp->contact = utils_strdup (contact);
  if ( subjobp->contact == NULL ) {
    err = s_register_contact_strdup_err (0);
    goto register_contact_strdup_error;
  }
  
  subjob_serialno = subjobp->serialno;

  err = nexus_mutex_unlock (&(subjobp->mutex)); assert (!err);

  err = globus_duroc_control_i_job_monitor_link_gram (job_monitorp, 
					       contact, 
					       subjob_serialno);
  if (err) {
    err = s_register_contact_monitor_err (err);
    goto register_contact_monitor_error;
  }
  
  err = globus_duroc_control_i_control_link_gram (controlp, contact, job_serialno);
  if (err) {
    err = s_register_contact_control_err (err);
    goto register_contact_control_error;
  }

  return GLOBUS_DUROC_SUCCESS;

  /* register_contact_error_clauses:
  
  globus_duroc_control_i_control_unlink_gram (controlp, contact); */
 register_contact_control_error:

  globus_duroc_control_i_job_monitor_unlink_gram (job_monitorp, contact);
 register_contact_monitor_error:
  
  globus_free ((char *) subjobp->contact);
  subjobp->contact = NULL;
 register_contact_strdup_error:
  
 register_contact_rereg_error:
  
  err = nexus_mutex_unlock (&(subjobp->mutex)); assert (!err);
 register_contact_subjob_lock_error:
  
 register_contact_job_lock_error:
  
  return err;
}


void 
globus_duroc_control_i_subjob_state_update (globus_duroc_control_t * controlp,
				     globus_duroc_job_monitor_t *job_monitorp,
				     globus_duroc_subjob_t * subjobp,
				     int state,
				     int errorcode)
{
  int err;
  globus_bool_t need_poll = GLOBUS_FALSE;

  err = nexus_mutex_lock (&(subjobp->mutex)); assert (!err);

  if ( state == GRAM_JOB_STATE_PENDING ) {
    /* no new information */
  }
  else if ( state == GRAM_JOB_STATE_ACTIVE ) {
    if ( subjobp->state == GLOBUS_DUROC_SUBJOB_STATE_PENDING ) {
      need_poll = GLOBUS_TRUE;
      subjobp->state = GLOBUS_DUROC_SUBJOB_STATE_ACTIVE;
    }
  }
  else if ( state == GRAM_JOB_STATE_DONE ) {
    need_poll = GLOBUS_TRUE;
    subjobp->state = GLOBUS_DUROC_SUBJOB_STATE_DONE;
  }
  else if ( state == GRAM_JOB_STATE_FAILED ) {
    need_poll = GLOBUS_TRUE;
    subjobp->state = GLOBUS_DUROC_SUBJOB_STATE_FAILED;
    subjobp->errorcode = errorcode;
  }
  else {
    globus_duroc_at_error ("unknown GRAM state", state);
  }

  err = nexus_mutex_unlock (&(subjobp->mutex)); assert (!err);

  if ( need_poll == GLOBUS_TRUE ) 
    globus_duroc_control_i_job_monitor_poll (controlp,
				      job_monitorp);
}
				     

/*
 * reverse of init operation, free resources
 * unbind subjob record
 *
 * kill caller's reference,
 * destroy if ref_count hits zero
 */
void globus_duroc_control_i_subjob_destroy (globus_duroc_control_t      *controlp,
				     globus_duroc_job_monitor_t *job_monitorp,
				     globus_duroc_subjob_t      **subjobpp)
{
  int err;
  globus_duroc_subjob_t *subjobp;

  subjobp = *subjobpp;

  if ( (controlp==NULL) || (job_monitorp==NULL) || (subjobp==NULL) ) return;

  err = nexus_mutex_lock (&(subjobp->mutex)); assert (!err);

  if ( subjobp->contact != NULL ) {
    globus_duroc_control_i_control_unlink_gram (controlp, subjobp->contact);
    globus_duroc_control_i_job_monitor_unlink_gram (job_monitorp, subjobp->contact);
  }

  err = nexus_mutex_unlock (&(subjobp->mutex)); assert (!err);

  globus_duroc_control_i_job_monitor_unlink_subjob (job_monitorp, subjobp);

  err = nexus_mutex_lock (&(subjobp->mutex)); assert (!err);

  if ( subjobp->contact != NULL ) {
    globus_free ((char *) subjobp->contact);
    subjobp->contact = NULL;
  }

  *subjobpp = NULL;

  subjobp->ref_count -= 1;

  err = nexus_mutex_unlock (&(subjobp->mutex)); assert (!err);

  if ( subjobp->ref_count > 0 ) {
    return;
  }
  else {
    return;
  }

#if 0
  globus_free ((char *) subjobp->label);

  nexus_cond_destroy (&(subjobp->cond));

  nexus_mutex_destroy (&(subjobp->mutex));

  /* subjobs are reference-count garbage-collected */
  globus_free (subjobp);

  return;
#endif
}

