
#include <assert.h>
#include "nexus.h"
#include "globus_gram_client.h"
#include "globus_common.h"

#include "globus_duroc_control.h"

#include "duroc-common.h"

#include "control.h"
#include "subjob.h"
#include "job_monitor.h"

#include "rsl.h"


/* #define USE_THREADED_GRAM_CLIENT_REQUEST 1 */
/* #define  USE_NONBLOCKING_GRAM_CLIENT_REQUEST 1 */

static int 
globus_duroc_control_i_subjob_add (globus_duroc_control_t     * controlp,
			    globus_duroc_job_monitor_t * job_monitorp,
			    globus_duroc_rsl_ast_t       subjob_ast);

globus_module_descriptor_t globus_duroc_control_module = 
{
  "globus_duroc_control",
  globus_duroc_control_activate,
  globus_duroc_control_deactivate,
  GLOBUS_NULL /* at exit */,
  GLOBUS_NULL /* get_pointer_func */
};


#define s_job_submit_parse_err(err) \
(globus_duroc_at_error("request parse", NULL), GLOBUS_DUROC_ERROR_INVALID_RSL)
#define s_job_submit_count_err(err) \
    (globus_duroc_at_error("count RSL subjobs", err), GLOBUS_DUROC_ERROR_INVALID_MULTIREQ)
#define s_job_submit_malloc_err(err) \
    (globus_duroc_at_error("globus_malloc", err), GLOBUS_DUROC_ERROR_MALLOC_FAILED)
#define s_job_submit_monitor_init_err(err) \
    (globus_duroc_at_error("job_monitor init", err), GLOBUS_DUROC_ERROR_INIT_FAILED)
#define s_job_submit_subjob_split_err(err) \
    (globus_duroc_at_error("split RSL subjob ASTs", err), \
     GLOBUS_DUROC_ERROR_INVALID_MULTIREQ)


typedef struct globus_duroc_subjob_request_thread_s {
  volatile int * resultp;
  globus_cond_t * condp;
  globus_mutex_t * mutexp;
  volatile int * threads_pendingp;
  globus_duroc_control_t * controlp;
  globus_duroc_job_monitor_t * job_monitorp;
  globus_duroc_rsl_ast_t ast;
} globus_duroc_subjob_request_thread_t;

void *
globus_duroc_request_thread_func (void * user_arg)
{
  globus_duroc_subjob_request_thread_t * thread_state;

  thread_state = ((globus_duroc_subjob_request_thread_t *)
		  user_arg);
  assert (thread_state!=NULL);

#ifdef USE_NONBLOCKING_GRAM_CLIENT_REQUEST

#else
  /* make blocking request call that generates a job contact or error */
  *(thread_state->resultp) = globus_duroc_control_i_subjob_add (
					thread_state->controlp,
					thread_state->job_monitorp,
					thread_state->ast);

  globus_mutex_lock (thread_state->mutexp);
  *(thread_state->threads_pendingp) -= 1;
  globus_cond_broadcast (thread_state->condp);
  globus_mutex_unlock (thread_state->mutexp);
#endif /* USE_NONBLOCKING_GRAM_CLIENT_REQUEST */

  return NULL;
}

/*
 * for each job request:
 * -- create empty job monitor
 * -- enter job monitor into radix demux table
 * -- for each subjob in request:
 * -- -- subjob start subroutine
 * -- return status code
 */
int 
globus_duroc_control_job_request (globus_duroc_control_t  * controlp,
				  const char              * description,
				  int                       job_state_mask,
				  const char              * callback_contact,
				  char                   ** job_contactp,
				  int                     * subjob_countp,
				  volatile int           ** subjob_resultsp)
{
  int                   err;
  globus_duroc_rsl_ast_t       rsl_ast;
  int                   subjob_count;
  globus_duroc_job_monitor_t * job_monitorp;
  globus_list_t       * subjob_asts;

  GLOBUS_IGNORE job_state_mask;
  GLOBUS_IGNORE callback_contact;

  if ( (controlp == NULL)
       || (description == NULL)
       || (job_contactp == NULL)
       || (subjob_countp == NULL)
       || (subjob_resultsp == NULL) ) 
    return GLOBUS_DUROC_ERROR_INVALID_PARAMETER;

  *job_contactp = NULL;
  *subjob_resultsp = NULL;

  rsl_ast = globus_duroc_rsl_parse (description);
  if (rsl_ast==NULL) {
    err = s_job_submit_parse_err (NULL);
    goto job_submit_parse_error;
  }

  subjob_count = globus_duroc_rsl_multicount (rsl_ast);
  if (subjob_count < 1) {
    err = s_job_submit_count_err (subjob_count);
    goto job_submit_count_error;
  }

  *subjob_countp = subjob_count;

  *subjob_resultsp = globus_malloc (sizeof(int) * subjob_count);
  if ((*subjob_resultsp)==NULL) {
    err = s_job_submit_malloc_err (err);
    goto job_submit_codes_malloc_error;
  }

  job_monitorp = globus_malloc (sizeof(globus_duroc_job_monitor_t));
  if (job_monitorp==NULL) {
    err = s_job_submit_malloc_err (err);
    goto job_submit_monitor_malloc_error;
  }

  err = globus_duroc_control_i_job_monitor_init (controlp, job_monitorp);
  if (err) {
    err = s_job_submit_monitor_init_err (err);
    goto job_submit_monitor_init_error;
  }

  subjob_asts = globus_duroc_rsl_subrequests_list (rsl_ast);
  if (subjob_asts==NULL) {
    err = s_job_submit_subjob_split_err (0);
    goto job_submit_subjob_split_error;
  }

  {
    int      i;
    globus_list_t * subjob_asts_iter;
    globus_cond_t cond;
    globus_mutex_t mutex;
    volatile int threads_pending = 0;

    globus_cond_init (&cond, NULL);
    globus_mutex_init (&mutex, NULL);

    subjob_asts_iter = subjob_asts;

    for (i=0; i<subjob_count; i++) {
      globus_duroc_rsl_ast_t subjob_ast;
      globus_duroc_subjob_request_thread_t * thread_state;
#ifdef USE_THREADED_GRAM_CLIENT_REQUEST
      globus_thread_t ignored_threadid;
#endif

      subjob_ast = (globus_duroc_rsl_ast_t) globus_list_first (subjob_asts_iter);
      assert (subjob_ast!=NULL);
      subjob_asts_iter = globus_list_rest (subjob_asts_iter);

      thread_state = globus_malloc (
			    sizeof (globus_duroc_subjob_request_thread_t));
      assert (thread_state!=NULL);
      thread_state->mutexp = &mutex;
      thread_state->condp = &cond;
      thread_state->threads_pendingp = &threads_pending;
      thread_state->resultp = &(*subjob_resultsp)[i];
      thread_state->controlp = controlp;
      thread_state->job_monitorp = job_monitorp;
      thread_state->ast = subjob_ast;

      globus_mutex_lock (&mutex);
      threads_pending += 1;
      globus_mutex_unlock (&mutex);

#ifdef USE_THREADED_GRAM_CLIENT_REQUEST
      err = globus_thread_create (&ignored_threadid,
				  NULL /* threadattr */,
				  globus_duroc_request_thread_func,
				  (void *) thread_state);
#else
      err = -1;
#endif /* USE_THREADED_GRAM_CLIENT_REQUEST */

      if (err) {
	/* no thread spawned so do work here */
	globus_duroc_request_thread_func ((void *) thread_state);
      }
    }

    globus_mutex_lock (&mutex);
    while ( threads_pending > 0 ) {
      globus_cond_wait (&cond, &mutex);
    }
    globus_mutex_unlock (&mutex);

    globus_cond_destroy (&cond);
    globus_mutex_destroy (&mutex);

    assert (globus_list_empty (subjob_asts_iter));
  }

  globus_duroc_rsl_subrequests_list_free (subjob_asts);

  err = globus_duroc_control_job_contact (controlp, job_monitorp, job_contactp);
  assert (!err);

  globus_duroc_control_i_job_monitor_release (controlp, &job_monitorp);

  globus_duroc_rsl_free (rsl_ast);

  return GLOBUS_DUROC_SUCCESS;


  /* job_submit_error_clauses:

  globus_duroc_rsl_subrequests_list_free (subjob_asts); */
 job_submit_subjob_split_error:

  globus_duroc_control_i_job_monitor_destroy (controlp, job_monitorp);
 job_submit_monitor_init_error:

  globus_free (job_monitorp);
 job_submit_monitor_malloc_error:

  globus_free ((int *) *subjob_resultsp);
 job_submit_codes_malloc_error:

  /* no resources */
 job_submit_count_error:

  globus_duroc_rsl_free (rsl_ast);
 job_submit_parse_error:

  return err;
}



static int
s_subjob_augment_ast (globus_duroc_control_t     * controlp,
		      globus_duroc_job_monitor_t * job_monitorp,
		      globus_duroc_subjob_t      * subjobp,
		      globus_duroc_rsl_ast_t       subjob_request)
{
  int err;
  char * checkin_contact;
  char * binding;
  globus_duct_control_t * duct_controlp;
  char * duct_checkin_contact;
  globus_bool_t need_checkin_info;
  globus_bool_t need_duct_info;

  err = nexus_mutex_lock (&(subjobp->mutex)); assert (!err);

  if ( subjobp->start_type != GLOBUS_DUROC_START_NONE ) {
    need_checkin_info = GLOBUS_TRUE;


    binding = globus_malloc (sizeof(char)
			     * (utils_strlen ("XXXXXXXXXXXXXXXX")
				+ 1));
    assert (binding!=NULL);
    err = utils_sprintf (binding, "%x",
			 subjobp->serialno);
    err = globus_duroc_rsl_setenv (subjob_request, 
				   "GLOBUS_DUROC_SUBJOB_SERIALNO",
				   binding);
    globus_free (binding);

  }
  else {
    need_checkin_info = GLOBUS_FALSE;
  }

  if ( subjobp->comms_type != GLOBUS_DUROC_COMMS_NONE ) {
    need_duct_info = GLOBUS_TRUE;
  }
  else {
    need_duct_info = GLOBUS_FALSE;
  }

  err = nexus_mutex_unlock (&(subjobp->mutex)); assert (!err);

  if ( need_checkin_info ) {
    err = globus_duroc_control_i_control_contact_lsp (controlp, &checkin_contact);
    assert (!err);

    err = globus_duroc_rsl_setenv (subjob_request, 
				   "GLOBUS_DUROC_CHECKIN_CONTACT",
				   checkin_contact);
    globus_free (checkin_contact);
  }

  err = nexus_mutex_lock (&(job_monitorp->mutex)); assert (!err);

  if ( need_duct_info ) {
    duct_controlp = &(job_monitorp->duct_control);
  }

  if ( need_checkin_info ) {
    binding = globus_malloc (sizeof(char)
			     * (utils_strlen ( "XXXXXXXXXXXXXXXX")
				+ 1));
    assert (binding!=NULL);
    err = utils_sprintf (binding, "%x",
			 job_monitorp->serialno);
    err = globus_duroc_rsl_setenv (subjob_request, 
				   "GLOBUS_DUROC_JOB_SERIALNO",
				   binding);
    globus_free (binding);
  }

  err = nexus_mutex_unlock (&(job_monitorp->mutex)); assert (!err);

  if ( need_duct_info ) {
    err = globus_duct_control_contact_lsp (duct_controlp, &duct_checkin_contact);
    assert (!err);

    err = globus_duroc_rsl_setenv (subjob_request,
				   "GLOBUS_DUROC_DUCT_CONTACT",
				   duct_checkin_contact);
  }

  err = nexus_mutex_lock (&(subjobp->mutex)); assert (!err);

  if ( need_duct_info ) {
    binding = globus_malloc (sizeof(char)
			     * (utils_strlen ("XXXXXXXXXXXXXXXX")
				+ 1));
    assert (binding!=NULL);
    err = utils_sprintf (binding, "%x",
			 subjobp->serialno);
    err = globus_duroc_rsl_setenv (subjob_request,
				   "GLOBUS_DUROC_DUCT_ID",
				   binding);
    globus_free (binding);
  }

  err = nexus_mutex_unlock (&(subjobp->mutex)); assert (!err);


  return GLOBUS_DUROC_SUCCESS;
}

#define s_subjob_add_ast_copy_err(err) \
(globus_duroc_at_error ("RSL copy", err), GLOBUS_DUROC_ERROR_MALLOC_FAILED)
#define s_subjob_add_getlabel_err(err) \
(globus_duroc_at_error ("RSL get label", err), GLOBUS_DUROC_ERROR_INVALID_RSL)
#define s_subjob_add_getstarttype_err(err) \
(globus_duroc_at_error ("RSL get subjobStartType", err), GLOBUS_DUROC_ERROR_INVALID_RSL)
#define s_subjob_add_bad_starttype_err(err) \
(globus_duroc_at_error ("RSL bad subjobStartType", 0), GLOBUS_DUROC_ERROR_BAD_START_TYPE)
#define s_subjob_add_getcommstype_err(err) \
(globus_duroc_at_error ("RSL get subjobCommType", err), GLOBUS_DUROC_ERROR_INVALID_RSL)
#define s_subjob_add_bad_commstype_err(err) \
(globus_duroc_at_error ("RSL bad subjobCommType", 0), GLOBUS_DUROC_ERROR_BAD_COMMS_TYPE)
#define s_subjob_add_malloc_err(err) \
(globus_duroc_at_error ("globus_malloc", err), GLOBUS_DUROC_ERROR_MALLOC_FAILED)
#define s_subjob_add_init_err(err) \
(globus_duroc_at_error ("subjob init", err), err)
#define s_subjob_add_augment_err(err) \
(globus_duroc_at_error ("subjob augment RSL", err), GLOBUS_DUROC_ERROR_INTERNAL_FAILURE)
#define s_subjob_add_unparse_err(err) \
(globus_duroc_at_error ("subjob unparse RSL", err), GLOBUS_DUROC_ERROR_INTERNAL_FAILURE)
#define s_subjob_add_globus_gram_submit_err(err) \
(GLOBUS_DUROC_ERROR_GRAM_ERROR_BASE+(err))
#define s_subjob_add_register_err(err) \
(globus_duroc_at_error ("register contact", err), GLOBUS_DUROC_ERROR_INTERNAL_FAILURE)
#define s_subjob_add_globus_gram_contact_err(err) \
(globus_duroc_at_error ("RSL resourceManagerContact", 0), \
 GLOBUS_DUROC_ERROR_INVALID_MANAGER_CONTACT)
#define s_subjob_add_globus_gram_request_err(err) \
(globus_duroc_at_error ("RSL strip contact", 0), GLOBUS_DUROC_ERROR_INTERNAL_FAILURE)

/*
 * subjob add subroutine:
 * -- add subjob to job monitor
 * -- issue GRAM request
 * -- -- subjob check-in contact info in environment
 * -- -- inter-subjob DUCT contact info in environment
 * -- enter GRAM job data into job monitor
 * -- enter GRAM job ID into radix demux table
 * -- enter GRAM job ID into callback demux table [process deferrals]
 */
static int 
globus_duroc_control_i_subjob_add (globus_duroc_control_t     * controlp,
			    globus_duroc_job_monitor_t * job_monitorp,
			    globus_duroc_rsl_ast_t       subjob_ast)
{
  int                    err;
  globus_result_t        res;
  globus_duroc_subjob_t       * subjobp;
  globus_duroc_rsl_ast_t        subjob_globus_gram_request_ast;
  char                 * subjob_label;
  char                 * subjob_starttype_string;
  globus_duroc_start_method_t   subjob_starttype;
  char                 * subjob_commstype_string;
  globus_duroc_comms_method_t   subjob_commstype;
  char                 * subjob_request_string;
  char                 * broker_contact;
  char                 * subjob_contact;

  err = globus_duroc_rsl_label (subjob_ast, &subjob_label);
  if (err) {
    err = s_subjob_add_getlabel_err (err);
    goto s_subjob_add_getlabel_error;
  }

  err = globus_duroc_rsl_starttype (subjob_ast, &subjob_starttype_string);
  if (err) {
    err = s_subjob_add_getstarttype_err (err);
    goto s_subjob_add_getstarttype_error;
  }

  if ( utils_streq (subjob_starttype_string, "strict-barrier") )
    subjob_starttype = GLOBUS_DUROC_START_STRICT;
  else if ( utils_streq (subjob_starttype_string, "loose-barrier") )
    subjob_starttype = GLOBUS_DUROC_START_LOOSE;
  else if ( utils_streq (subjob_starttype_string, "no-barrier") )
    subjob_starttype = GLOBUS_DUROC_START_NONE;
  else {
    err = s_subjob_add_bad_starttype_err (0);
    goto s_subjob_add_bad_starttype_error;
  }

  err = globus_duroc_rsl_commstype (subjob_ast, &subjob_commstype_string);
  if (err) {
    err = s_subjob_add_getcommstype_err (err);
    goto s_subjob_add_getcommstype_error;
  }

  if ( utils_streq (subjob_commstype_string, "blocking-join") )
    subjob_commstype = GLOBUS_DUROC_COMMS_BLOCKING;
  else if ( utils_streq (subjob_commstype_string, "independent") )
    subjob_commstype = GLOBUS_DUROC_COMMS_NONE;
  else {
    err = s_subjob_add_bad_commstype_err (0);
    goto s_subjob_add_bad_commstype_error;
  }

  subjobp = (globus_duroc_subjob_t *) globus_malloc (sizeof(globus_duroc_subjob_t));
  if (subjobp==NULL) {
    err = s_subjob_add_malloc_err (err);
    goto s_subjob_add_subjob_malloc_error;
  }

  res = globus_duroc_control_i_subjob_init (controlp, 
				     job_monitorp, 
				     subjobp, 
				     subjob_label,
				     subjob_starttype,
				     subjob_commstype);
  if (res) {
    err = s_subjob_add_init_err ( (long) res);
    goto s_subjob_add_subjob_init_error;
  }

  broker_contact = globus_duroc_rsl_globus_gram_contact (subjob_ast);
  if (broker_contact==NULL) {
    err = s_subjob_add_globus_gram_contact_err (0);
    goto s_subjob_add_globus_gram_contact_error;
  }

  subjob_globus_gram_request_ast = globus_duroc_rsl_strip_globus_duroc_fields (subjob_ast);
  if (subjob_globus_gram_request_ast==NULL) {
    err = s_subjob_add_globus_gram_request_err (0);
    goto s_subjob_add_globus_gram_request_error;
  }

  err = s_subjob_augment_ast (controlp, job_monitorp, subjobp, 
			      subjob_globus_gram_request_ast);
  if (err) {
    err = s_subjob_add_augment_err (err);
    goto s_subjob_add_subjob_augment_error;
  }

  subjob_request_string = globus_duroc_rsl_unparse (subjob_globus_gram_request_ast);
  if (subjob_request_string==NULL) {
    err = s_subjob_add_unparse_err (err);
    goto s_subjob_add_subjob_unparse_error;
  }

  utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
	       "\n"
	       "submit subjob >>%s<< gram job >>%s<< to manager >>%s<< ...\n"
	       "\n",
	       (subjob_label ? subjob_label : "(none)"),
	       subjob_request_string,
	       broker_contact);

  globus_duroc_control_i_control_open_gram (controlp);

#ifdef USE_NONBLOCKING_GRAM_CLIENT_REQUEST
  err = globus_gram_client_job_request_nonblocking ();
#else
  err = globus_gram_client_job_request (broker_contact,
			  subjob_request_string,
			  GRAM_JOB_STATE_ALL,
			  controlp->subjob_callback_contact,
			  &subjob_contact);
#endif /* USE_NONBLOCKING_GRAM_CLIENT_REQUEST */

  globus_free (subjob_request_string);

  if ( err ) {
    err = s_subjob_add_globus_gram_submit_err (err);
    goto s_subjob_add_globus_gram_submit_error;
  }

#ifdef USE_NONBLOCKING_GRAM_CLIENT_REQUEST
  /* the asynch status callback does registration 
   * and error-handling for us */
#else
  assert (subjob_contact!=NULL);

  utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
	       "\n"
	       "registering subjob >>%s<< gram contact >>%s<< ...\n"
	       "\n",
	       (subjob_label ? subjob_label : "(none)"),
	       subjob_contact);

  err = globus_duroc_control_i_subjob_register_contact (controlp, 
						 job_monitorp, 
						 subjobp,
						 subjob_contact);
  if (err) {
    err = s_subjob_add_register_err (err);
    goto s_subjob_add_register_globus_gram_error;
  }
#endif /* USE_NONBLOCKING_GRAM_CLIENT_REQUEST */
  
  globus_duroc_rsl_free (subjob_globus_gram_request_ast);
  globus_free ((char *) subjob_label);
  globus_free (subjob_contact);

  return GLOBUS_DUROC_SUCCESS;


 s_subjob_add_register_globus_gram_error:

  globus_gram_client_job_cancel (subjob_contact);
  globus_free (subjob_contact);
 s_subjob_add_globus_gram_submit_error:

 s_subjob_add_subjob_unparse_error:
  
 s_subjob_add_subjob_augment_error:

  globus_duroc_rsl_free (subjob_globus_gram_request_ast);
 s_subjob_add_globus_gram_request_error:

 s_subjob_add_globus_gram_contact_error:
    
  globus_duroc_control_i_subjob_destroy (controlp, job_monitorp, &subjobp);
 s_subjob_add_subjob_init_error:

  globus_free (subjobp);
 s_subjob_add_subjob_malloc_error:

 s_subjob_add_bad_commstype_error:

  globus_free ((char *) subjob_commstype_string);
 s_subjob_add_getcommstype_error:
 s_subjob_add_bad_starttype_error:

  globus_free ((char *) subjob_starttype_string);
 s_subjob_add_getstarttype_error:

  globus_free ((char *) subjob_label);
 s_subjob_add_getlabel_error:

  return err;
}

#define s_subjob_add_job_contact_err(err) \
(globus_duroc_at_error ("job lookup", err), GLOBUS_DUROC_ERROR_INVALID_PARAMETER)
#define s_subjob_add_parse_err(err) \
(globus_duroc_at_error ("subjob request parse", NULL), GLOBUS_DUROC_ERROR_INVALID_RSL)

/*
 * for each subjob add request:
 * -- subjob start subroutine
 */
int 
globus_duroc_control_subjob_add (globus_duroc_control_t  * controlp,
			  const char       * job_contact,
			  char             * subjob_request_string)
{
  int                   err;
  globus_duroc_job_monitor_t * job_monitorp;
  globus_duroc_rsl_ast_t       subjob_ast;

  if (controlp==NULL) return GLOBUS_DUROC_ERROR_INVALID_PARAMETER;
  
  err = globus_duroc_control_i_job_lookup (controlp, job_contact, &job_monitorp);
  if (err) {
    err = s_subjob_add_job_contact_err (err);
    goto subjob_add_job_contact_error;
  }

  subjob_ast = globus_duroc_rsl_parse (subjob_request_string);
  if (subjob_ast==NULL) {
    err = s_subjob_add_parse_err (NULL);
    goto subjob_add_parse_error;
  }

  /* do standard add operation, propogate result */
  err = globus_duroc_control_i_subjob_add (controlp, job_monitorp, subjob_ast);

  /* release our reference to the monitor */
  globus_duroc_control_i_job_monitor_release (controlp, &job_monitorp);
  
  globus_duroc_rsl_free (subjob_ast);
 subjob_add_parse_error:

  globus_duroc_control_i_job_monitor_release (controlp, &job_monitorp);
 subjob_add_job_contact_error:

  return err;
}


#define s_subjob_delete_job_contact_err(err) \
(globus_duroc_at_error ("job lookup", err), GLOBUS_DUROC_ERROR_INVALID_PARAMETER)

/*
 * for each subjob delete request:
 * -- map to job_monitor
 * -- subjob delete subroutine
 */
int 
globus_duroc_control_subjob_delete (globus_duroc_control_t  * controlp,
			     const char       * job_contact,
			     const char       * subjob_label)
{
  int err;
  globus_duroc_job_monitor_t *job_monitorp;

  if (controlp==NULL) return GLOBUS_DUROC_ERROR_INVALID_PARAMETER;
  
  err = globus_duroc_control_i_job_lookup (controlp, job_contact, &job_monitorp);
  if (err) {
    err = s_subjob_delete_job_contact_err (err);
    goto subjob_delete_job_contact_error;
  }

  /* do standard delete operation, propogate result */
  err = globus_duroc_control_i_subjob_delete (controlp, job_monitorp, subjob_label);

  globus_duroc_control_i_job_monitor_release (controlp, &job_monitorp);
  
 subjob_delete_job_contact_error:

  return err;
}

#define s_barrier_release_job_contact_err(err) \
(globus_duroc_at_error ("job lookup", err), GLOBUS_DUROC_ERROR_INVALID_PARAMETER)

/*
 * for each commit request:
 * -- enter commit status into job monitor
 * -- -- no_checkin handled as WAIT, FAIL or PRUNE?
 * -- -- terminated handled as FAIL or PRUNE?
 * -- send RUN ? [if committed and complete check-in]
 */
int 
globus_duroc_control_barrier_release (globus_duroc_control_t * controlp,
			       const char      * job_contact,
			       globus_bool_t     wait_for_checkins)
{
  int err;
  globus_duroc_job_monitor_t *job_monitorp;

  if ( (controlp==NULL) )
    return GLOBUS_DUROC_ERROR_INVALID_PARAMETER;
  
  err = globus_duroc_control_i_job_lookup (controlp, job_contact, &job_monitorp);
  if (err) {
    err = s_barrier_release_job_contact_err (err);
    goto barrier_release_job_contact_error;
  }

  /* do internal release operation, propogate result */
  err = globus_duroc_control_i_job_monitor_barrier_release (controlp,
						     job_monitorp,
						     wait_for_checkins);

  globus_duroc_control_i_job_monitor_release (controlp, &job_monitorp);
  
 barrier_release_job_contact_error:

  return err;
}

#define s_cancel_job_contact_err(err) \
(globus_duroc_at_error ("job lookup", err), GLOBUS_DUROC_ERROR_INVALID_PARAMETER)

int
globus_duroc_control_job_cancel (globus_duroc_control_t * controlp,
			  const char      * contact)
{
  int err;
  globus_duroc_job_monitor_t *job_monitorp;

  err = globus_duroc_control_i_job_lookup (controlp, contact, &job_monitorp);
  if (err) {
    err = s_cancel_job_contact_err (err);
    goto cancel_job_contact_error;
  }

  /* do internal release operation, propogate result */
  err = globus_duroc_control_i_job_cancel (controlp,
				    job_monitorp);

  globus_duroc_control_i_job_monitor_release (controlp, &job_monitorp);
  
 cancel_job_contact_error:

  return err;
}

#define s_states_job_contact_err(err) \
(globus_duroc_at_error ("job lookup", err), GLOBUS_DUROC_ERROR_INVALID_PARAMETER)

int
globus_duroc_control_subjob_states (globus_duroc_control_t  * controlp,
			     const char       * job_contact,
			     int              * subjob_countp,
			     int             ** subjob_statesp,
			     char            *** subjob_labelsp)
{
  int err;
  globus_duroc_job_monitor_t *job_monitorp;

  err = globus_duroc_control_i_job_lookup (controlp, job_contact, &job_monitorp);
  if (err) {
    err = s_states_job_contact_err (err);
    goto job_states_contact_error;
  }

  /* do internal release operation, propogate result */
  err = globus_duroc_control_i_subjob_states (controlp,
				       job_monitorp,
				       subjob_countp,
				       subjob_statesp,
				       subjob_labelsp);

  globus_duroc_control_i_job_monitor_release (controlp, &job_monitorp);
    
 job_states_contact_error:

  return err;
}

#define s_contacts_job_contact_err(err) \
(globus_duroc_at_error ("job lookup", err), GLOBUS_DUROC_ERROR_INVALID_PARAMETER)

int
globus_duroc_control_subjob_contacts (globus_duroc_control_t  * controlp,
			     const char        * job_contact,
			     int               * subjob_countp,
			     char            *** subjob_contactsp,
			     char            *** subjob_labelsp)
{
  int err;
  globus_duroc_job_monitor_t *job_monitorp;

  err = globus_duroc_control_i_job_lookup (controlp, job_contact, &job_monitorp);
  if (err) {
    err = s_contacts_job_contact_err (err);
    goto contacts_contact_error;
  }

  /* do internal release operation, propogate result */
  err = globus_duroc_control_i_subjob_contacts (controlp,
				       job_monitorp,
				       subjob_countp,
				       subjob_contactsp,
				       subjob_labelsp);

  globus_duroc_control_i_job_monitor_release (controlp, &job_monitorp);
    
 contacts_contact_error:

  return err;
}
