
#include "globus_duroc_bootstrap.h"

#include <stdio.h>
#include <assert.h>
#include <string.h>

#include "globus_common.h"
#include "duroc-common.h"
#include "globus_duroc_runtime.h"
#include "utils.h"
#include "globus_gram_myjob.h"
#include "nexus.h"

globus_module_descriptor_t globus_duroc_bootstrap_module = 
{
  "globus_duroc_bootstrap",
  globus_duroc_bootstrap_activate,
  globus_duroc_bootstrap_deactivate,
  NULL /* atexit */,
  NULL /* get_pointer_func */
};


int
globus_duroc_bootstrap_activate (void)
{
  if ( globus_module_activate (GLOBUS_COMMON_MODULE) != GLOBUS_SUCCESS ) 
    goto activate_common_module_error;

  if ( globus_module_activate (GLOBUS_DUROC_RUNTIME_MODULE) 
       != GLOBUS_SUCCESS ) 
    goto activate_duroc_runtime_module_error;

  return GLOBUS_SUCCESS;

activate_duroc_runtime_module_error:
  globus_module_deactivate (GLOBUS_COMMON_MODULE);

activate_common_module_error:
  return GLOBUS_FAILURE;
}

int
globus_duroc_bootstrap_deactivate (void)
{
  int rc;

  rc = GLOBUS_SUCCESS;

  if ( globus_module_deactivate (GLOBUS_DUROC_RUNTIME_MODULE)
       != GLOBUS_SUCCESS )
    rc = GLOBUS_FAILURE;

  if ( globus_module_deactivate (GLOBUS_COMMON_MODULE) != GLOBUS_SUCCESS )
    rc = GLOBUS_FAILURE;

  return rc;
}


static int subjob_exchange_round = 0;

int
globus_duroc_bootstrap_subjob_exchange (const char   * local_info,
				 int          * subjob_countp,
				 int          * local_indexp,
				 char       *** subjob_info_arrayp)
{
  int err;
  int gram_rank;
  int gram_size;

  subjob_exchange_round++;

  if ( local_info == NULL ) local_info = "";

  if ( (subjob_countp == NULL)
       || (local_indexp == NULL)
       || (subjob_info_arrayp == NULL) ) return -1;

  err = globus_duroc_runtime_intra_subjob_rank (&gram_rank);
  if (err) return 1;

  err = globus_duroc_runtime_intra_subjob_size (&gram_size);
  if (err) return 2;

  utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
	       "globus_duroc_bootstrap_subjob_exchange-%d "
	       " gram_rank=%d  gram_size=%d  (local %s)\n",
	       subjob_exchange_round,
	       gram_rank, gram_size,
	       ((gram_rank==0) ? "master" : "slave"));

  (*subjob_countp) = -1;
  (*local_indexp) = -1;
  (*subjob_info_arrayp) = NULL;
  
  if ( gram_rank == 0 ) {
    int             local_subjob_addr;
    int             remote_subjob_count;
    int           * remote_subjob_addrs;
    int i;

    err = globus_duroc_runtime_inter_subjob_structure (&local_subjob_addr,
						&remote_subjob_count,
						&remote_subjob_addrs);
    if (err) return 4;

    (*subjob_info_arrayp) = globus_malloc (sizeof (char *)
					   * (remote_subjob_count + 1));
    if ( (*subjob_info_arrayp) == NULL ) {
      err = 6;
      goto malloc_error;
    }

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "globus_duroc_bootstrap_subjob_exchange-%d "
		 "subjob=%d has remote_subjob_count=%d\n",
		 subjob_exchange_round,
		 local_subjob_addr, remote_subjob_count);

    if ( remote_subjob_count > 0 ) {
      globus_list_t * addr_list;
      globus_list_t * sorted_addr_list;

      addr_list = NULL;
      sorted_addr_list = NULL;

      /* 
       * compute sorted-address list
       */
      err = globus_list_insert (&addr_list,
				(void *) (long) local_subjob_addr);
      assert (!err);

      for (i=0; i<remote_subjob_count; i++) {
	err = globus_list_insert (&addr_list,
				  (void *) (long) (remote_subjob_addrs[i]));
	assert (!err);
      }

      sorted_addr_list = globus_list_sort (addr_list,
					   globus_list_int_less,
					   NULL);

      globus_list_free (addr_list); 
      addr_list = NULL;

      /*
       * using the just-computed sorted_addr_list, perform exchange
       */
      {
	int             dest_addr;
	int             send_index;
	int             receive_index;
	globus_list_t * our_addr_node;
	globus_list_t * dest_addr_node;
	char          * tag;
	
	assert ( (remote_subjob_count + 1)
		 == globus_list_size (sorted_addr_list) );

	our_addr_node = globus_list_search (sorted_addr_list,
					    (void *) (long) local_subjob_addr);
	assert (our_addr_node!=NULL);

	(*subjob_countp) = remote_subjob_count + 1;
	(*local_indexp) = (globus_list_size (sorted_addr_list)
			   - globus_list_size (our_addr_node));

	/* put own entry into the info array */
	(*subjob_info_arrayp)[*local_indexp] = utils_strdup (local_info);

	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "globus_duroc_bootstrap_subjob_exchange-%d "
		     "subjob=%d has exchange_index=%d/%d\n",
		     subjob_exchange_round,
		     local_subjob_addr, 
		     (*local_indexp), 
		     globus_list_size (sorted_addr_list));

	/* send all messages to the node after us in the "ring" */
	dest_addr_node = globus_list_rest (our_addr_node);
	if ( dest_addr_node == NULL ) {
	  /* we are the last node in the list.
	   * "next" node is the first in the list. */
	  dest_addr_node = sorted_addr_list;
	}
	dest_addr = (int) (long) globus_list_first (dest_addr_node);

	/* first we can send our own */
	send_index = (*local_indexp);

	/* first receive the one previous to our own */
	receive_index = (((*local_indexp) > 0)
			 ? ((*local_indexp) - 1)
			 : ((*subjob_countp) - 1));

	tag = globus_malloc (sizeof(char *)
			     * (utils_strlen ("subjob exchange round  index ")
				+ 16 /* hex-encoded 64-bit int */
				+ 16 /* hex-encoded 64-bit int */
				+ 1 /* nul terminator */));
	assert (tag!=NULL);

	/* send/recv each entry */
	for (i=0; i<remote_subjob_count; i++) {
	  globus_byte_t   send_buf[GRAM_MYJOB_MAX_BUFFER_LENGTH];
	  globus_byte_t * receive_buf;
	  int len;
	  
	  /* send the one we have */
	  utils_sprintf (tag, "subjob exchange round %x index %x",
			 subjob_exchange_round,
			 send_index);

	  len = (utils_strlen ( (*subjob_info_arrayp)[send_index] )
		 + 1 /* nul terminator */);

	  assert ( len <= GRAM_MYJOB_MAX_BUFFER_LENGTH );


	  utils_sprintf ((char *)send_buf, "%s", 
			 (*subjob_info_arrayp)[send_index]);

	  err = globus_duroc_runtime_inter_subjob_send (dest_addr,
							tag,
							len,
							send_buf);
	  if (err) {
	    /* fatal error */
	    err = 7;
	    goto exchange_send_error;
	  }

	  /* receive a new one */
	  utils_sprintf (tag, "subjob exchange round %x index %x",
			 subjob_exchange_round,
			 receive_index);

	  err = globus_duroc_runtime_inter_subjob_receive (tag,
						    &len,
						    &receive_buf);
	  assert (!err);

	  (*subjob_info_arrayp)[receive_index] 
	    = utils_strdup ((char *)receive_buf);

	  assert ( len > 0 );
	  assert ( utils_strlen ((char *)receive_buf)
		   == (len-1) );

	  /* next time send the new one */
	  send_index = receive_index;

	  /* next time receive the one previous to the last new one */
	  receive_index = ((receive_index > 0)
			   ? (receive_index - 1)
			   : ((*subjob_countp) - 1));
	}

	globus_free (tag);
	tag = NULL;
      }
    
      globus_list_free (sorted_addr_list);
    }
    else {
      /* we're the only subjob */
      (*subjob_countp) = 1;
      (*local_indexp) = 0;
      (*subjob_info_arrayp)[0] = strdup (local_info);

      utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		   "globus_duroc_bootstrap_subjob_exchange-%d "
		   "subjob=%d has exchange_index=%d/%d\n",
		   subjob_exchange_round,
		   local_subjob_addr, 
		   (*local_indexp), 
		   1);
    }

    /* signal slaves to unblock */
    for (i=1; i<gram_size; i++) {
      globus_byte_t buf[1];
      
      buf[0] = 0;
      globus_duroc_runtime_intra_subjob_send (i, 
				       "subjob exchange unblock",
				       1, buf);
    }

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "globus_duroc_bootstrap_subjob_exchange-%d "
		 "subjob=%d returning success\n",
		 subjob_exchange_round,
		 local_subjob_addr);

    return 0;

  exchange_send_error: 
  malloc_error:

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "globus_duroc_bootstrap_subjob_exchange-%d "
		 "subjob=%d returning error %d\n",
		 subjob_exchange_round,
		 local_subjob_addr,
		 err);

    globus_free ( remote_subjob_addrs );
    return err;
  }
  else {
    globus_byte_t buf[GRAM_MYJOB_MAX_BUFFER_LENGTH];
    int len;
    
    /* wait for our local master to unblock us */
    globus_duroc_runtime_intra_subjob_receive ("subjob exchange unblock",
					&len, buf);
    assert (len==1);
    assert (buf[0]==0);

    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		 "globus_duroc_bootstrap_subjob_exchange-%d "
		 "gram_rank=%d returning success\n",
		 subjob_exchange_round,
		 gram_rank);

    return 0;
  }
}


int
globus_duroc_bootstrap_linearize_startpoint (nexus_startpoint_t  * sp,
				      char               ** lsp_string)
{
  int                  err;
  nexus_startpoint_t   sp_copy;
  globus_byte_t        bbuff[GRAM_MYJOB_MAX_BUFFER_LENGTH];
  globus_byte_t      * ptr;
  int                  len;
 
  len = 0;

   err = nexus_startpoint_copy (&sp_copy, 
                               sp); 
  assert (!err);
 
  ptr = bbuff;
  utils_sprintf ((char *)ptr, "%d", nexus_user_format());
  while ( (*ptr)!='\0' ) ptr++; ptr++; /* d d ... d \0 MESG */
  nexus_user_put_startpoint_transfer (&ptr, &sp_copy, 1);
  len = (int) (ptr - bbuff);
  assert (len<=GLOBUS_DUCT_MAX_MSG_LENGTH);
 
  /* now hex-encode the buffer and prepend 'LSP' */
  
  (*lsp_string) = globus_malloc (sizeof(char) 
				 * (3 /* LSP */
				    + (2 * len) + 1 /* hex buff */
				    + 1 /* \0 */));
  assert ( (*lsp_string)!=NULL );
 
  err = utils_sprintf ( (*lsp_string), "LSP"); assert (err==3);
  globus_l_duroc_hex_encode_byte_array (bbuff, len, (*lsp_string)+3);
  
  /* contact has form:
   *   >L S P hd hd ... hd<
   * "LSP" prefix identifies this as linearized startpoint
   * hex digit substring should be hex_decoded to obtain:
   *   >d d ... d \0 user-sp<
   * "d d ... d" is the user-buffer format
   * user-sp is the startpoint
   */
  return 0;
}

int 
globus_duroc_bootstrap_extract_linearized_startpoint (nexus_startpoint_t * spp, 
					       const char         * sp_msg)
{
  int err;

  if ( (spp==NULL)
       || (sp_msg==NULL) ) return -1;

  if ( (sp_msg[0] == 'U')
       && (sp_msg[1] == 'R')
       && (sp_msg[2] == 'L') ) {
    /* an attachment URL */
    return -1;
  }
  else if ( (sp_msg[0] == 'L')
	    && (sp_msg[1] == 'S')
	    && (sp_msg[2] == 'P') ) {
    /* a linearized startpoint */
    globus_byte_t   bbuff[GRAM_MYJOB_MAX_BUFFER_LENGTH];
    globus_byte_t * ptr;
    int           format = 0;

    /* sp_msg has form:
     *   >L S P hd hd ... hd<
     * "LSP" prefix identifies this as linearized startpoint
     * hex digit substring should be hex_decoded to obtain:
     *   >d d ... d \0 user-sp<
     *   >d d ... d< is the user-buffer format tag in decimal digits
     *   >user-sp< is the startpoint in user-buffer format
     */

    assert ( (utils_strlen (sp_msg+3) % 2) == 0 );

    globus_l_duroc_hex_decode_byte_array (sp_msg+3, 
			       utils_strlen (sp_msg+3) / 2,
			       bbuff);

    ptr = bbuff;
    err = nexus_stdio_lock(); assert (!err);
    err = sscanf ((char *)ptr, "%d", &format); assert (err==1);
    err = nexus_stdio_unlock(); assert (!err);
    while ( (*ptr)!='\0' ) ptr++; ptr++; /* d d ... d \0 user-sp */
    nexus_user_get_startpoint (&(ptr), spp, 1, format);

    return 0;
  }
  else return -1;
}


int
globus_duroc_bootstrap_master_sp_vector (nexus_startpoint_t  * local_sp, 
				  int                 * job_size,
				  nexus_startpoint_t ** sp_array)
{
  int err;
  int subjob_count;
  int local_subjob_index;
  char ** subjob_info_array;

  err = globus_duroc_bootstrap_subjob_exchange ("", 
						&subjob_count, 
						&local_subjob_index,
						&subjob_info_array);
  if (err) return -1;

  if ( subjob_info_array != NULL ) {
    int i;
    for (i=0; i<subjob_count; i++) {
      globus_free (subjob_info_array[i]);
    }
    globus_free (subjob_info_array);
  }

  return globus_duroc_bootstrap_ordered_master_sp_vector (local_sp,
							  local_subjob_index,
							  job_size,
							  sp_array);
}

static int fill_ordered_sp_vector_round = 0;

int
globus_duroc_bootstrap_ordered_master_sp_vector (nexus_startpoint_t  * local_sp, 
					  int                   subjob_index,
					  int                 * job_size,
					  nexus_startpoint_t ** sp_array)
{
  int err;
  int gram_rank;
  int gram_size;

  fill_ordered_sp_vector_round++;

  err = globus_duroc_runtime_intra_subjob_rank (&gram_rank);
  if (err) {
    err = 1;
    goto vector_error;
  }

  err = globus_duroc_runtime_intra_subjob_size (&gram_size);
  if (err) {
    err = 2;
    goto vector_error;
  }

  utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
	       "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
	       " gram_rank=%d  gram_size=%d\n",
	       fill_ordered_sp_vector_round,
	       gram_rank, gram_size);

  if ( gram_rank == 0 ) {
    char ** local_sp_msg_array;
    int gram_slave;

    local_sp_msg_array = globus_malloc (sizeof(char *)
					* gram_size);

    err = globus_duroc_bootstrap_linearize_startpoint (local_sp,
						&(local_sp_msg_array[0]));
    assert (!err);

    /* receive startpoint messages from local slaves */
    {
      char * tag;

      tag = globus_malloc (sizeof(char)
			   * (utils_strlen ("fill_ordered_sp_vector: round "
					    "XXXXXXXXXXXXXXXX"
					    " local sp "
					    "XXXXXXXXXXXXXXXX")
			    + 1));
      assert (tag!=NULL);

      for (gram_slave=1; gram_slave < gram_size; gram_slave++) {
	char sp_msg[GRAM_MYJOB_MAX_BUFFER_LENGTH];
	int  sp_msg_len;
	
	utils_sprintf (tag, 
		       "fill_ordered_sp_vector: round %x local sp %x", 
		       fill_ordered_sp_vector_round,
		       gram_slave);

	globus_duroc_runtime_intra_subjob_receive (tag,
					    &sp_msg_len,
					    (globus_byte_t *) sp_msg);
	assert ( (utils_strlen (sp_msg) + 1) == sp_msg_len );
	
	local_sp_msg_array[gram_slave] = utils_strdup (sp_msg);
	assert ( local_sp_msg_array[gram_slave] != NULL );
      }
      
      globus_free (tag);
    }

    /* compute subjob addressing */
    {
      int i;
      int   local_subjob_addr;
      int   master_subjob_addr;
      int   remote_subjob_count;
      int * remote_subjob_addrs;

      globus_list_t * sorted_addr_list;

      err = globus_duroc_runtime_inter_subjob_structure (&local_subjob_addr,
						  &remote_subjob_count,
						  &remote_subjob_addrs);
      if (err) {
	err = 3;
	goto vector_error;
      }

      utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		   "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
		   "subjob=%d has remote_subjob_count=%d\n",
		   fill_ordered_sp_vector_round,
		   local_subjob_addr,
		   remote_subjob_count);

      /* order addressing based on user-supplied subjob indices */
      {
	char * subjob_index_string;
	int subjob_count;
	int local_index;
	char ** subjob_index_string_array;
	globus_hashtable_t addr_mapt;

	subjob_index_string = globus_malloc (sizeof(char)
					     *( 32 + 2 ));
	assert (subjob_index_string!=NULL);
    
	utils_sprintf (subjob_index_string, "%x|%x", 
		       subjob_index, local_subjob_addr);

	err = globus_duroc_bootstrap_subjob_exchange (subjob_index_string,
					       &subjob_count,
					       &local_index,
					       &subjob_index_string_array);
	if (err) {
	  err = 4;
	  goto vector_error;
	}

	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
		     "subjob=%d has local_index=%d, user_index=%d\n",
		     fill_ordered_sp_vector_round,
		     local_subjob_addr,
		     local_index,
		     subjob_index);

	for (i=0; i<subjob_count; i++) {
	  utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		       "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
		       "subjob=%d i_s_a[%d]=\"%s\"\n",
		       fill_ordered_sp_vector_round,
		       local_subjob_addr,
		       i, subjob_index_string_array[i]);
	}

	sorted_addr_list = NULL;

	globus_hashtable_init (&addr_mapt,
			       16 /* zero info default */,
			       globus_hashtable_int_hash,
			       globus_hashtable_int_keyeq);

	for (i=0; i<subjob_count; i++) {
	  if ( i==local_index ) {
	    err = globus_hashtable_insert (&addr_mapt,
					   (void *) (long) subjob_index,
					   (void *) (long) local_subjob_addr);
	    assert (!err);

	    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
			 "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
			 "subjob=%d mapping user_index=%d to subjob_addr=%d\n",
			 fill_ordered_sp_vector_round,
			 local_subjob_addr,
			 local_index,
			 local_subjob_addr);
	  }
	  else {
	    int index;
	    int addr;

	    nexus_stdio_lock();
	    sscanf (subjob_index_string_array[i], "%x|%x", 
		    &index, &addr);
	    nexus_stdio_unlock();
	    err = globus_hashtable_insert (&addr_mapt,
					   (void *) (long) index,
					   (void *) (long) addr);
	    assert (!err);

	    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
			 "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
			 "subjob=%d mapping user_index=%d to subjob_addr=%d\n",
			 fill_ordered_sp_vector_round,
			 local_subjob_addr,
			 index,
			 addr);
	  }
	}

	/* build list in reverse */
	for (i=(subjob_count-1); i>=0; i--) {
	  globus_list_insert (&sorted_addr_list,
			      globus_hashtable_lookup (&addr_mapt,
						       (void *) (long) i));
	}

	globus_hashtable_destroy (&addr_mapt);
      }


      master_subjob_addr = (int) (long) globus_list_first (sorted_addr_list);

      if ( local_subjob_addr == master_subjob_addr ) {
	/* receive startpoint arrays from local masters */
	globus_list_t * subjob_iter;
	globus_hashtable_t local_sp_msg_arrayst;
	globus_hashtable_t local_sizest;
	int process_count = 0;
	nexus_startpoint_t *process_sp_array;

	err = globus_hashtable_init (&local_sp_msg_arrayst,
				     16 /* zero info default */,
				     globus_hashtable_int_hash,
				     globus_hashtable_int_keyeq);

	err = globus_hashtable_init (&local_sizest,
				     16 /* zero info default */,
				     globus_hashtable_int_hash,
				     globus_hashtable_int_keyeq);

	for (subjob_iter = sorted_addr_list;
	     ! globus_list_empty (subjob_iter);
	     subjob_iter = globus_list_rest (subjob_iter)) {
	  int current_subjob_addr;
	  
	  current_subjob_addr = (int) (long) globus_list_first (subjob_iter);

	  if ( current_subjob_addr == local_subjob_addr ) {
	    char ** local_sp_msg_array_copy;

	    local_sp_msg_array_copy = globus_malloc (sizeof (char *)
						     * gram_size);
	    assert (local_sp_msg_array_copy!=NULL);

	    for (gram_slave=0; gram_slave < gram_size; gram_slave++) {
	      local_sp_msg_array_copy[gram_slave]
		= utils_strdup (local_sp_msg_array[gram_slave]);
	      assert (local_sp_msg_array_copy[gram_slave] != NULL);
	    }
	    
	    err = globus_hashtable_insert (&local_sp_msg_arrayst,
					   (void *) (long) local_subjob_addr,
					   (void *) local_sp_msg_array_copy);
	    assert (!err);

	    err = globus_hashtable_insert (&local_sizest,
					   (void *) (long) local_subjob_addr,
					   (void *) (long) gram_size);
	    assert (!err);

	    utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
			 "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
			 "subjob=%d (master): subjob %d has size %d\n",
			 fill_ordered_sp_vector_round,
			 local_subjob_addr,
			 current_subjob_addr,
			 gram_size);

	    process_count += gram_size;
	  }
	  else {
	    int subjob_size;

	    {
	      char * subjob_size_tag;
	      char * subjob_size_msg;
	
	      int subjob_size_msg_len;

	      subjob_size_tag 
		= globus_malloc (sizeof(char)
				 * (utils_strlen ("fill_ordered_sp_vector: "
						  "round XXXXXXXXXXXXXXXX "
						  "subjob XXXXXXXXXXXXXXXX "
						  "size")
				    + 1));
	      assert (subjob_size_tag!=NULL);
	      
	      utils_sprintf (subjob_size_tag, 
			     "fill_ordered_sp_vector: round %x "
			     "subjob %x size",
			     fill_ordered_sp_vector_round,
			     current_subjob_addr);
	      
	      err = globus_duroc_runtime_inter_subjob_receive (
					 subjob_size_tag,
					 &subjob_size_msg_len,
					 (globus_byte_t **) &subjob_size_msg);
	      assert (!err);
	      assert ( (utils_strlen (subjob_size_msg) + 1)
		       == subjob_size_msg_len );

	      nexus_stdio_lock ();
	      err = sscanf (subjob_size_msg, "%x", &subjob_size);
	      nexus_stdio_unlock ();
	      assert (err==1);

	      globus_free (subjob_size_msg);
	      globus_free (subjob_size_tag);

	      err = globus_hashtable_insert (&local_sizest,
					   (void *) (long) current_subjob_addr,
					   (void *) (long) subjob_size);
	      assert (!err);

	      utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		   "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
			   "subjob=%d (master): subjob %d has size %d\n",
			   fill_ordered_sp_vector_round,
			   local_subjob_addr,
			   current_subjob_addr,
			   subjob_size);

	      process_count += subjob_size;
	    }
	    {
	      char * subjob_array_tag;
	      char ** subjob_sp_msg_array;

	      subjob_sp_msg_array = globus_malloc (sizeof(char *)
		   *( (int) (long) 
		      globus_hashtable_lookup (&local_sizest,
					       ((void *) (long)
						current_subjob_addr))));

	      subjob_array_tag 
		= globus_malloc (sizeof(char)
				 * (utils_strlen ("fill_ordered_sp_vector: "
						  "round XXXXXXXXXXXXXXXX "
						  "subjob XXXXXXXXXXXXXXXX "
						  "process XXXXXXXXXXXXXXXX "
						  "sp")
				    + 1));
	      assert (subjob_array_tag!=NULL);
	      
	      for (gram_slave=0; gram_slave<subjob_size; gram_slave++) {
		int subjob_sp_msg_len;
	      
		utils_sprintf (subjob_array_tag, 
			       "fill_ordered_sp_vector: round %x "
			       "subjob %x "
			       "process %x sp",
			       fill_ordered_sp_vector_round,
			       current_subjob_addr,
			       gram_slave);
		
		err = globus_duroc_runtime_inter_subjob_receive (
		   subjob_array_tag,
		   &subjob_sp_msg_len,
		   (globus_byte_t **) &(subjob_sp_msg_array[gram_slave]));
		assert (!err);
		assert ( (utils_strlen (subjob_sp_msg_array[gram_slave]) + 1)
			 == subjob_sp_msg_len );
	      }

	      err = globus_hashtable_insert (&local_sp_msg_arrayst,
				     (void *) (long) current_subjob_addr,
				     (void *) subjob_sp_msg_array);
	      assert (!err);
	    }
	  }
	}
	
	process_sp_array = globus_malloc (sizeof (nexus_startpoint_t)
					  * process_count);
	assert (process_sp_array!=NULL);

	{
	  int subjob_base_process_num = 0;

	  for (subjob_iter = sorted_addr_list;
	       ! globus_list_empty (subjob_iter);
	       subjob_iter = globus_list_rest (subjob_iter)) {
	    int current_subjob_addr;
	    char ** subjob_sp_array;
	    int subjob_size;
	    
	    current_subjob_addr = (int) (long) globus_list_first (subjob_iter);

	    subjob_size 
	      = ((int) (long) 
		 globus_hashtable_lookup (&local_sizest,
					  (void *) (long) current_subjob_addr));
	    assert (subjob_size>0);
	    globus_hashtable_remove (&local_sizest, 
				     (void *) (long) current_subjob_addr);


	    subjob_sp_array 
	      = ((char **)
		 globus_hashtable_lookup (&local_sp_msg_arrayst,
				  (void *) (long) current_subjob_addr));
	    assert (subjob_sp_array!=NULL);
	    globus_hashtable_remove (&local_sp_msg_arrayst, 
				     (void *) (long) current_subjob_addr);


	    for (gram_slave=0; gram_slave < subjob_size; gram_slave++) {
	      err = globus_duroc_bootstrap_extract_linearized_startpoint (
				   &(process_sp_array[(subjob_base_process_num
						       + gram_slave)]),
				   subjob_sp_array[gram_slave]);
	      assert (!err);

	      globus_free (subjob_sp_array[gram_slave]);
	    }
	    
	    globus_free (subjob_sp_array);

	    subjob_base_process_num += subjob_size;
	  }

	}

	globus_hashtable_destroy (&local_sp_msg_arrayst);
	globus_hashtable_destroy (&local_sizest);

	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
		     "subjob=%d (master), user_index=%d, "
		     "pcount=%d, sp_array=%x\n",
		     fill_ordered_sp_vector_round,
		     local_subjob_addr,
		     subjob_index,
		     process_count,
		     (long) process_sp_array);

	if (job_size) (*job_size) = process_count;
	if (sp_array) (*sp_array) = process_sp_array;
      }
      else {
	/* send local startpoint array to master subjob */
	{
	  char * subjob_size_tag;
	  char * subjob_size_msg;
	
	  subjob_size_tag 
	    = globus_malloc (sizeof(char)
			     * (utils_strlen ("fill_ordered_sp_vector: "
					      "round XXXXXXXXXXXXXXXX "
					      "subjob XXXXXXXXXXXXXXXX "
					      "size")
				+ 1));
	  assert (subjob_size_tag!=NULL);

	  utils_sprintf (subjob_size_tag, 
			 "fill_ordered_sp_vector: round %x "
			 "subjob %x size",
			 fill_ordered_sp_vector_round,
			 local_subjob_addr);

	  subjob_size_msg = globus_malloc (sizeof(char)
				   * (utils_strlen ("XXXXXXXXXXXXXXXX")));
	  assert (subjob_size_msg!=NULL);

	  utils_sprintf (subjob_size_msg, "%x", gram_size);

	  err = globus_duroc_runtime_inter_subjob_send (
				     master_subjob_addr,
				     subjob_size_tag,
				     utils_strlen (subjob_size_msg) + 1,
				     (globus_byte_t *)subjob_size_msg);
	  if (err) {
	    err = 7;
	    goto vector_error;
	  }

	  globus_free (subjob_size_tag);
	  globus_free (subjob_size_msg);
	}
	
	{
	  char * subjob_array_tag;

	  subjob_array_tag 
	    = globus_malloc (sizeof(char)
			     * (utils_strlen ("fill_ordered_sp_vector: "
					      "round XXXXXXXXXXXXXXXX "
					      "subjob XXXXXXXXXXXXXXXX "
					      "process XXXXXXXXXXXXXXXX "
					      "sp")
				+ 1));
	  assert (subjob_array_tag!=NULL);

	  for (gram_slave=0; gram_slave<gram_size; gram_slave++) {
	    utils_sprintf (subjob_array_tag, 
			   "fill_ordered_sp_vector: round %x "
			   "subjob %x "
			   "process %x sp",
			   fill_ordered_sp_vector_round,
			   local_subjob_addr,
			   gram_slave);
	    
	    err = globus_duroc_runtime_inter_subjob_send (
		       master_subjob_addr,
		       subjob_array_tag,
		       utils_strlen (local_sp_msg_array[gram_slave]) + 1,
		       (globus_byte_t *) local_sp_msg_array[gram_slave]);
	    if (err) {
	      err = 8;
	      goto vector_error;
	    }
	  }
	}

	utils_debug (GLOBUS_DUROC_DEBUG_FLAG,
		     "globus_duroc_bootstrap_ordered_master_sp_vector-%d "
		     "subjob=%d (slave), user_index=%d, "
		     "pcount=%d, sp_array=%x%s\n",
		     fill_ordered_sp_vector_round,
		     local_subjob_addr,
		     subjob_index,
		     0,
		     0x0,
		     (sp_array
		      ? ""
		      : ", user sp_array ptr is NULL!") );

	if (job_size) (*job_size) = 0;
	if (sp_array) (*sp_array) = NULL;
      }

      globus_list_free (sorted_addr_list);
      sorted_addr_list = NULL;
    }
    
    /* free local array */
    for (gram_slave=0; gram_slave < gram_size; gram_slave++) {
      globus_free (local_sp_msg_array[gram_slave]);
    }
    globus_free (local_sp_msg_array);

    /* effect a barrier */
    {
      int i;
      int count;
      int index;
      char ** info_array;

      globus_duroc_bootstrap_subjob_exchange ("", &count, &index, &info_array);
      if ( info_array != NULL ) {
	for (i=0; i<count; i++) {
	  globus_free (info_array[i]);
	}
	globus_free (info_array);
      }
    }

    return 0;
  }
  else {
    /* send startpoint to local master */
    char * sp_msg;
    char * tag;

    err = globus_duroc_bootstrap_linearize_startpoint (local_sp, &sp_msg);
    assert (!err);

    tag = globus_malloc (sizeof (char)
		 * (utils_strlen ("fill_ordered_sp_vector: round  local sp "
				  "XXXXXXXXXXXXXXXX"
				  "XXXXXXXXXXXXXXXX")
		    + 1));
    assert (tag!=NULL);

    utils_sprintf (tag, "fill_ordered_sp_vector: round %x local sp %x", 
		   fill_ordered_sp_vector_round,
		   gram_rank);

    globus_duroc_runtime_intra_subjob_send (0, 
				     tag,
				     strlen(sp_msg) + 1, 
				     (globus_byte_t *) sp_msg);

    globus_free (sp_msg);
    globus_free (tag);

    /* matching call for "compute subjob addressing" exchange above */
    {
      int dummy;
      int dummy2;
      char ** dummy3;

      globus_duroc_bootstrap_subjob_exchange ("", &dummy, &dummy2, &dummy3);
      assert (dummy3==NULL);
    }

    /* effect a barrier */
    {
      int dummy;
      int dummy2;
      char ** dummy3;

      globus_duroc_bootstrap_subjob_exchange ("", &dummy, &dummy2, &dummy3);
      assert (dummy3==NULL);
    }

    if (job_size) (*job_size) = 0;
    if (sp_array) (*sp_array) = NULL;
    return 0;
  }

vector_error:
  (*job_size) = 0;
  (*sp_array) = NULL;
  return err;
}


/* inter- and intra-subjob message tags */
#define SUBJOB_MASTER_TO_SUBJOB0_MASTER_T "subjob mstr to subjob0 mstr topology"
#define SUBJOB0_MASTER_TO_SUBJOB_MASTER_T "subjob0 mstr to subjob mstr topology"
#define SUBJOB_MASTER_TO_SLAVE_T          "subjob mstr to slave topology"

#define SUBJOB_SLAVE_TO_MASTER_D          "subjob slave to master data"
#define SUBJOB_MASTER_TO_SUBJOB_MASTER_D  "subjob master to subjob master data"
#define SUBJOB_MASTER_TO_SLAVE_D          "subjob master to slave data"

#define MY_GRANK_STR_SIZE 10


/***************************************/
/* local utility function declarations */
/***************************************/

static void copy_byte_array_from_tagged_buff(char *tagged_intrabuff, 
					    int bufflen, 
					    char **outbuffs, 
					    int *outbufflens,
					    int *g_rank);

/*
 * globus_duroc_bootstrap_all_to_all_distribute_bytearray(
 *                                        globus_byte_t *inbuff, 
 *                                        int inbufflen, 
 *                                        int *nprocs,
 *                                        int *my_grank,
 *                                        globus_byte_t ***outbuffs,
 *                                        int **outbufflens);
 *
 * MUST be called by EVERY proc, each supplying inbuff,inbufflen
 * (may be NULL,0).
 *
 * fills the last 4 args as follows:
 *     nprocs  - total number of procs 
 *     my_grank - unique id in range [0, nprocs-1]. 
 *                preserves order of your subjob as it appears in envoking RSL,
 *                i.e., my_rank will be greater than all procs of 
 *                subjobs appearing before mine in envoking RSL and 
 *                less than all procs of subjobs appearing after mine.
 *     outbuffs - malloc'd a vector (length nprocs) where each element
 *                points to a (also malloc'd) copy of the byte array 
 *                supplied by a particular proc such that 
 *                outbuffs[i] is a copy of the byte array of
 *                the proc assigned my_grank=i (each proc should
 *                find a copy of the byte array they supplied in
 *                outbuffs[my_grank])
 *     outbufflens - malloc'd a vector (length nprocs) of ints where
 *                   outbufflens[i] is the length of the byte array
 *                   outbuffs[i].
 *                
 */

/* NICK: should return 'globus-like' rc ... not 'void' */

void globus_duroc_bootstrap_all_to_all_distribute_bytearray(
					    globus_byte_t *inbuff, 
					    int inbufflen, 
					    int *nprocs,
					    int *my_grank,
					    globus_byte_t ***outbuffs,
					    int **outbufflens)
{
    int rank_in_my_subjob;
    int my_subjob_size;    /* used by subjobmasters only */
    int nsubjobs;          /* used by subjobmasters only */
    int *subjob_addresses; /* used by subjobmasters only */

    globus_module_activate(GLOBUS_DUROC_BOOTSTRAP_MODULE);

    globus_duroc_runtime_intra_subjob_rank(&rank_in_my_subjob);

    globus_i_duroc_get_topology(
	rank_in_my_subjob, 
	&my_subjob_size,
	&subjob_addresses,
	nprocs, 
	&nsubjobs,
	my_grank);

    /* NICK: exiting on these failed mallocs is not right thing */
    /*       to do.  should set error rc and return. fix that   */
    /*       later when i learn more about globus rc stuff.     */
    if (!(*outbuffs = 
	(globus_byte_t **) globus_libc_malloc(*nprocs*sizeof(globus_byte_t *))))
    {
	globus_libc_fprintf(stderr, 
	    "ERROR: failed malloc of %ld bytes\n", 
	    *nprocs*sizeof(globus_byte_t *));
	exit(1);
    } /* endif */
    if (!(*outbufflens = (int *) globus_libc_malloc(*nprocs*sizeof(int))))
    {
	globus_libc_fprintf(stderr, 
	    "ERROR: failed malloc of %ld bytes\n", *nprocs*sizeof(int));
	exit(1);
    } /* endif */

    globus_i_duroc_distribute_byte_array(
	inbuff,
	inbufflen,
	rank_in_my_subjob,
	my_subjob_size,
	subjob_addresses,
	*nprocs,
	nsubjobs,
	*my_grank,
	*outbuffs,
	*outbufflens);

    /************/
    /* clean-up */
    /************/

    if (!rank_in_my_subjob)
	/* subjob master */
	globus_libc_free(subjob_addresses);

    globus_module_deactivate(GLOBUS_DUROC_BOOTSTRAP_MODULE);

} /* end NICK_globus_duroc_bootstrap_all_to_all_distribute_bytearray() */


/*
 * void
 * globus_i_duroc_get_topology(
 *     int				rank_in_my_subjob, 
 *     int *				my_subjob_size, 
 *     int **				subjob_addresses,
 *     int *				nprocs, 
 *     int *				nsubjobs, 
 *     int *				my_grank)
 *
 * MUST be called by EVERY proc, each supplying rank_in_my_subjob
 * rank_in_my_subjob==0 -> subjobmaster, else subjobslave
 *
 * fills the remaining args:
 *     my_subjob_size  - populated for subjobmasters only, 
 *                       nprocs (including master) in their subjob 
 *     subjob_addresses - malloc'd and filled for OTHER subjobmasters only
 *                        inter_subjob_addr's of other subjobmasters
 *                        my subjob_addr NOT included (so njobs-1)
 *     nsubjobs - populated for subjobmasters only.
 *                total number of procs
 *     nprocs - total number of procs
 *     my_grank - my rank in [0, nprocs-1]
 *     
 */

/* NICK: should return 'globus-like' rc ... not 'void' */

void
globus_i_duroc_get_topology(
    int					rank_in_my_subjob, 
    int *				my_subjob_size, 
    int **				subjob_addresses,
    int *				nprocs, 
    int *				nsubjobs, 
    int *				my_grank)
{
    char topology_buff[GRAM_MYJOB_MAX_BUFFER_LENGTH];
    char *buff;
    int bufflen;
    int i;
    int sj0_master_idx;    /* used by subjobmasters only */
    int *job_sizes;        /* used by subjobmaster 0 only */
    int *g_ranks;          /* used by subjobmaster 0 only */

    if (rank_in_my_subjob)
    {
	/* subjob slave */

	globus_duroc_runtime_intra_subjob_receive(
	    SUBJOB_MASTER_TO_SLAVE_T,         /* tag */
	    &bufflen,                         /* nbytes received? */
	    (globus_byte_t *) topology_buff);  /* message */

	sscanf(topology_buff, "%d %d", nprocs, my_grank);
    }
    else
    {
	/* subjob master */
	int duroc_subjobmaster_rank;
	int my_subjob_addr;
	int rsl_subjob_rank;
	char *rsl_subjob_rank_env_var;

	globus_duroc_runtime_intra_subjob_size(my_subjob_size);
	globus_duroc_runtime_inter_subjob_structure(&my_subjob_addr,
						    nsubjobs,
						    subjob_addresses);
	/* finding index of master subjob 0, he */
	/* is the one with the lowest address   */
	for (i = 0, sj0_master_idx = -1, duroc_subjobmaster_rank = 0; 
	    i < *nsubjobs; i ++)
	{
	    if ((sj0_master_idx == -1 && (*subjob_addresses)[i] < my_subjob_addr)
	    || (sj0_master_idx != -1 
		    && (*subjob_addresses)[i] < (*subjob_addresses)[sj0_master_idx]))
		sj0_master_idx = i;
	    if ((*subjob_addresses)[i] < my_subjob_addr)
		duroc_subjobmaster_rank ++;
	} /* endfor */
	/* globus_duroc_runtime_inter_subjob_structure reports the */
	/* number of REMOTE subjobs (*other* than the subjob i'm   */
	/* master of).  to get the TOTAL number of subjobs in this */
	/* run i must increment the value reported by              */
	/* globus_duroc_runtime_inter_subjob_structure             */
	(*nsubjobs) ++;

	/* NICK: should not exit here ... should set globus-like rc. */
	if (!(rsl_subjob_rank_env_var=getenv("GLOBUS_DUROC_SUBJOB_INDEX")))
	{
	    globus_libc_fprintf(stderr, 
"ERROR: required environment variable GLOBUS_DUROC_SUBJOB_INDEX not set.\n");
	    globus_libc_fprintf(stderr, 
"       Each subjob in envoking RSL must have GLOBUS_DUROC_SUBJOB_INDEX\n");
	    globus_libc_fprintf(stderr, 
"       set to rank (0, 1, 2, ...) of subjob as it appears in the envoking RSL.\n");
	    exit(1);
	} /* endif */
	rsl_subjob_rank = atoi(rsl_subjob_rank_env_var);
	if (rsl_subjob_rank < 0 || rsl_subjob_rank >= *nsubjobs)
	{
	    globus_libc_fprintf(stderr, 
	"ERROR: env variable GLOBUS_DUROC_SUBJOB_INDEX %d must be >= 0 and\n", 
		rsl_subjob_rank);
	    globus_libc_fprintf(stderr, 
		"ERROR: less than the number of subjobs %d for this run.\n", 
		*nsubjobs);
	    exit(1);
	} /* endif */

	if (duroc_subjobmaster_rank)
	{
	    /* NOT master of subjob 0 */

	    sprintf(topology_buff,
		    "%d %d %d",
		    duroc_subjobmaster_rank,
		    rsl_subjob_rank,
		    *my_subjob_size);

	    globus_duroc_runtime_inter_subjob_send(
		(*subjob_addresses)[sj0_master_idx],  /* dest */
		SUBJOB_MASTER_TO_SUBJOB0_MASTER_T, /* tag */
		strlen(topology_buff)+1,            /* nbytes */
		(globus_byte_t *) topology_buff);  /* data */

	    globus_duroc_runtime_inter_subjob_receive(
		SUBJOB0_MASTER_TO_SUBJOB_MASTER_T, /* tag */
		&bufflen,                          /* nbytes received? */
		(globus_byte_t **) &buff);         /* message */

	    sscanf(buff, "%d %d", nprocs, my_grank);

	    globus_libc_free(buff);
	}
	else
	{
	    /* master of subjob 0 */

	    int j;
	    int temp;
	    /* vectors len nsubjobs, all indexed by duroc_subjobmaster_rank */
	    int *rsl_ranks; /* received from other subjob masters */

	    /* NICK: exiting on these failed mallocs is not right thing */
	    /*       to do.  should set error rc and return. fix that   */
	    /*       later when i learn more about globus rc stuff.     */
	    if (!(rsl_ranks = 
		(int *) globus_libc_malloc(*nsubjobs*sizeof(int))))
	    {
		globus_libc_fprintf(stderr, 
		    "ERROR: failed malloc of %ld bytes\n", 
		    *nsubjobs*sizeof(int));
		exit(1);
	    } /* endif */
	    if (!(job_sizes = 
		(int *) globus_libc_malloc(*nsubjobs*sizeof(int))))
	    {
		globus_libc_fprintf(stderr, 
		    "ERROR: failed malloc of %ld bytes\n", 
		    *nsubjobs*sizeof(int));
		exit(1);
	    } /* endif */
	    if (!(g_ranks = (int *) globus_libc_malloc(*nsubjobs*sizeof(int))))
	    {
		globus_libc_fprintf(stderr, 
		    "ERROR: failed malloc of %ld bytes\n", 
		    *nsubjobs*sizeof(int));
		exit(1);
	    } /* endif */

	    /* need to sort subjob_addresses so that i may associate */
	    /* (using incoming duroc_subjobmaster_rank) incoming     */
	    /* rsl_subjob_rank and my_subjob_size with dest addr     */
	    /* in subjob_addresses                                   */
	    for (i = 1; i < *nsubjobs-1; i ++)
	    {
		for (j = i; j > 0; j --)
		{
		    if ((*subjob_addresses)[j] < (*subjob_addresses)[j-1])
		    {
			temp = (*subjob_addresses)[j];
			(*subjob_addresses)[j] = (*subjob_addresses)[j-1];
			(*subjob_addresses)[j-1] = temp;
		    } /* endif */
		} /* endfor */
	    } /* endfor */

	    /* rsl_ranks[] and job_sizes[] are indexed by  */
	    /* duroc_subjobmaster_rank, and i know that my */
	    /* duroc_subjobmaster_rank==0                  */
	    rsl_ranks[0] = rsl_subjob_rank;
	    job_sizes[0] = *my_subjob_size;

	    for (i = 1; i < *nsubjobs; i ++)
	    {
		int ranks, sizes;
		
		/*
		 * receiving 3 longs from other subjob master
                 *    duroc_subjobmaster_rank (used to index job_sizes[] 
                 *                             and rsl_ranks[])
                 *    rsl_subjob_rank 
                 *    my_subjob_size
                 */
		globus_duroc_runtime_inter_subjob_receive(
		    SUBJOB_MASTER_TO_SUBJOB0_MASTER_T, /* tag */
		    &bufflen,                          /* nbytes received? */
		    (globus_byte_t **) &buff);         /* message */

		sscanf(buff, "%d %d %d", &j, &ranks, &sizes);
		rsl_ranks[j] = ranks;
		job_sizes[j] = sizes;
		
		globus_libc_free(buff);
	    } /* endfor */

	    /* calculating nprocs and everyones' g_rank based */
	    /* on rsl_rank and job_sizes ...                  */
	    /* mygrank = sum job_size for all rsl_ranks       */
	    /*           that are less than mine              */
	    for (i = 0, *nprocs = 0; i < *nsubjobs; i ++)
	    {
		(*nprocs) += job_sizes[i];
		for (g_ranks[i] = 0, j = 0; j < *nsubjobs; j ++)
		if (rsl_ranks[i] > rsl_ranks[j])
		    g_ranks[i] += job_sizes[j];
	    } /* endfor */
	    *my_grank = g_ranks[0];

	    /* sending other subjob masters nprocs and their g_rank */
	    for (i = 0; i < *nsubjobs-1; i ++)
	    {
		sprintf(topology_buff, "%d %d", *nprocs, g_ranks[i+1]);
		globus_duroc_runtime_inter_subjob_send(
		    (*subjob_addresses)[i],               /* dest */
		    SUBJOB0_MASTER_TO_SUBJOB_MASTER_T, /* tag */
		    strlen(topology_buff)+1,           /* nbytes */
		    (globus_byte_t *) topology_buff);  /* data */
	    } /* endfor */

	    globus_libc_free(rsl_ranks);
	    globus_libc_free(job_sizes);
	    globus_libc_free(g_ranks);

	} /* endif */

	/* all subjob masters sending nprocs and g_rank to their slaves */
	for (i = 1; i < *my_subjob_size; i ++)
	{
	    sprintf(topology_buff, "%d %d", *nprocs, (*my_grank) + i);
	    globus_duroc_runtime_intra_subjob_send(
		i,                                /* dest */
		SUBJOB_MASTER_TO_SLAVE_T,         /* tag */
		strlen(topology_buff)+1,          /* nbytes */
		(globus_byte_t *) topology_buff); /* data */
	} /* endfor */
    } /* endif */

} /* end globus_i_duroc_get_topology() */

void
globus_i_duroc_distribute_byte_array(
    globus_byte_t *			inbuff,
    int 				inbufflen,
    int 				rank_in_my_subjob,
    int 				my_subjob_size,
    int *				subjob_addresses,
    int 				nprocs,
    int 				nsubjobs,
    int 				my_grank,
    globus_byte_t **			outbuffs,
    int *				outbufflens)
{

    int i;
    int j;
    char *buff;
    int bufflen;
    char tagged_intrabuff[GRAM_MYJOB_MAX_BUFFER_LENGTH];
    int g_rank;

    /* everyone needs to copy their byte array into outbuffs */
    if (!(outbuffs[my_grank] = (globus_byte_t *) globus_libc_malloc(inbufflen)))
    {
	globus_libc_fprintf(stderr, 
	    "ERROR: failed malloc of %d bytes\n", inbufflen);
	exit(1);
    } /* endif */
    memcpy(outbuffs[my_grank], inbuff, inbufflen);
    outbufflens[my_grank] = inbufflen;

    /* tagging and copying my byte array for distribution */
    sprintf(tagged_intrabuff, "%d", my_grank);
    memcpy(tagged_intrabuff+MY_GRANK_STR_SIZE, inbuff, inbufflen);

    if (rank_in_my_subjob)
    {
	/* subjob slave */

	/* send my byte array to my master */
	globus_duroc_runtime_intra_subjob_send(
	    0,                                /* dest */
	    SUBJOB_SLAVE_TO_MASTER_D,         /* tag */
	    MY_GRANK_STR_SIZE+inbufflen,      /* nbytes */
	    (globus_byte_t *) tagged_intrabuff); /* data */
	
	/* receiving all other byte arrays from my master */
	for (i = 0; i < nprocs-1; i ++)
	{
	    globus_duroc_runtime_intra_subjob_receive(
		SUBJOB_MASTER_TO_SLAVE_D, /* tag */
		&bufflen,                 /* nbytes received? */
		(globus_byte_t *) tagged_intrabuff);        /* message */

	    copy_byte_array_from_tagged_buff(tagged_intrabuff, 
					    bufflen, 
					    (char **) outbuffs, 
					    outbufflens,
					    &g_rank);
	} /* endfor */
    }
    else 
    {
	/* subjob master */

	/* sending my byte array to all my slaves */
	for (i = 1; i < my_subjob_size; i ++)
	{
	    globus_duroc_runtime_intra_subjob_send(
		i,                                /* dest */
		SUBJOB_MASTER_TO_SLAVE_D,         /* tag */
		MY_GRANK_STR_SIZE+inbufflen,      /* nbytes */
		(globus_byte_t *) tagged_intrabuff);  /* data */
	} /* endfor */

	/* sending my byte array to other subjob masters */
	for (i = 0; i < nsubjobs-1; i ++)
	{
	    globus_duroc_runtime_inter_subjob_send(
		subjob_addresses[i],                 /* dest */
		SUBJOB_MASTER_TO_SUBJOB_MASTER_D,    /* tag */
		MY_GRANK_STR_SIZE+inbufflen,         /* nbytes */
		(globus_byte_t *) tagged_intrabuff); /* data */
	} /* endfor */

	/* receiving byte arrays from my slaves.  making a copy */
	/* for myself, distributing it to my other slaves, and  */
	/* finally other subjob masters                         */
	for (i = 0; i < my_subjob_size-1; i ++)
	{
	    globus_duroc_runtime_intra_subjob_receive(
		SUBJOB_SLAVE_TO_MASTER_D, /* tag */
		&bufflen,                 /* nbytes received? */
		(globus_byte_t *) tagged_intrabuff);        /* message */

	    /* making copy for myself */
	    copy_byte_array_from_tagged_buff(tagged_intrabuff, 
					    bufflen, 
					    (char **) outbuffs, 
					    outbufflens,
					    &g_rank);

	    /* send to other of slaves */
	    for (j = 1; j < my_subjob_size; j ++)
	    {
		if (my_grank+j != g_rank)
		{
		    /* message originated at slave other than this one */
		    globus_duroc_runtime_intra_subjob_send(
			j,                        /* dest */
			SUBJOB_MASTER_TO_SLAVE_D, /* tag */
			bufflen,                  /* nbytes */
			(globus_byte_t *) tagged_intrabuff);        /* data */
		} /* endif */
	    } /* endfor */

	    /* sending to other subjob masters */
	    for (j = 0; j < nsubjobs-1; j ++)
	    {
		globus_duroc_runtime_inter_subjob_send(
		    subjob_addresses[j],                 /* dest */
		    SUBJOB_MASTER_TO_SUBJOB_MASTER_D,    /* tag */
		    bufflen,                             /* nbytes */
		    (globus_byte_t *) tagged_intrabuff); /* data */
	    } /* endfor */
	} /* endfor */

	/* receiving from other subjob masters, making a copy for */
	/* myself, distributing to my slaves */
	for (i = 0; i < nprocs-my_subjob_size; i ++)
	{
	    globus_duroc_runtime_inter_subjob_receive(
		SUBJOB_MASTER_TO_SUBJOB_MASTER_D, /* tag */
		&bufflen,                         /* nbytes received? */
		(globus_byte_t **) &buff);         /* message */

	    /* make copy for myself */
	    copy_byte_array_from_tagged_buff(buff, 
					    bufflen, 
					    (char **) outbuffs, 
					    outbufflens,
					    &g_rank);

	    /* distribute to all my slaves */
	    for (j = 1; j < my_subjob_size; j ++)
	    {
		globus_duroc_runtime_intra_subjob_send(
		    j,                        /* dest */
		    SUBJOB_MASTER_TO_SLAVE_D, /* tag */
		    bufflen,                  /* nbytes */
		    (globus_byte_t *) buff);                    /* data */
	    } /* endfor */

	    globus_libc_free(buff);

	} /* endfor */
    } /* endif */

} /* end globus_i_duroc_distribute_byte_array() */


/***************************/
/* Local Utility Functions */
/***************************/

static void copy_byte_array_from_tagged_buff(char *tagged_intrabuff, 
					    int bufflen, 
					    char **outbuffs, 
					    int *outbufflens,
					    int *g_rank)
{
    int len;
    
    sscanf(tagged_intrabuff, "%d", g_rank);

    if (len = bufflen - MY_GRANK_STR_SIZE)
    {
	if (!(outbuffs[*g_rank] = (char *) globus_libc_malloc(len)))
	{
	    globus_libc_fprintf(stderr,
		"ERROR: failed malloc of %d bytes\n", len);
	    exit(1);
	} /* endif */

	memcpy(outbuffs[*g_rank], 
	    tagged_intrabuff + MY_GRANK_STR_SIZE,
	    len);
    }
    else
	outbuffs[*g_rank] = (char *) NULL;

    outbufflens[*g_rank] = len;

} /* end copy_byte_array_from_tagged_buff() */
