/**********************************************************************
test_utp.c

Test functionality of the Unnamed Timing Package (UTP).
**********************************************************************/

#include <stdio.h>
#include <unistd.h>

#include "globus_utp.h"

/*
The tests: create 5 timers, use them like this:

timer 0: use 3 times, accumulate about 5 seconds; disable and stop
	(produces no warning); name it "Timer 0".
timer 1: use once, then reset (looks like it wasn't used at all);
	start/stop it when all are disabled; name it "Timer \"one\"".
timer 2: enable, stop before starting; name it "Timer 2.0".
timer 3: start without stopping
timer 4: disable, start/stop it, enable, start/stop to accumulate about 1
	second
timer 5: try to use it (it is bogus)

Also disable all timers, start/stop timer 1, enable all again.

(All times accumulated using sleep(3).)

Then globus_utp_write_file("utp_dump_test_txt") and exit().
*/

int main()
{
    double accumTime0;
    int prec;

    if (globus_module_activate(GLOBUS_UTP_MODULE) != GLOBUS_SUCCESS)
    {
	fprintf(stderr, "ERROR: Failed to activate utp module\n");
	return(1);
    }
	
    if (globus_utp_init(5, GLOBUS_UTP_MODE_SHARED))
    {
	fprintf(stderr, "globus_utp_init() failed, exiting\n");
	return 1;
    }

    globus_utp_set_attribute("expected-result-%s", "0", "about %d", 5);
    globus_utp_set_attribute("expected-name-0%s", "", "Timer 0");
    globus_utp_set_attribute("expected-count-0%s", "", "%d", 3);
    globus_utp_set_attribute("expected-result-%s", "1", "0");
    globus_utp_set_attribute("expected-name-%s", "1", "Timer \\\"one\\\"");
    globus_utp_set_attribute("expected-count-1%s", "", "0");
    globus_utp_set_attribute("expected-result-%s", "2", "0");
    globus_utp_set_attribute("expected-name-2%s", "", "Timer 2.000000");
    globus_utp_set_attribute("expected-count-2%s", "", "0");
    globus_utp_set_attribute("expected-result-%s", "3", "about %.1f", 4.0);
    globus_utp_set_attribute("expected-name-3%s", "", "\\\"\\\"");
    globus_utp_set_attribute("expected-count-3%s", "", "%.0f", 1.0);
    globus_utp_set_attribute("expected-result-4%s", "", "about %.1lf", 1.0);
    globus_utp_set_attribute("expected-name-4%s", "", "\\\"\\\"");
    globus_utp_set_attribute("expected-count-4%s", "", "1");

    globus_utp_set_attribute("expected-message-1%s", "",
			     "globus_utp_stop_timer: timer 2 not running");
    globus_utp_set_attribute("expected-message-2%s", "",
			     "globus_utp_start_timer: illegal timer 5");
    globus_utp_set_attribute("expected-message-3%s", "",
			     "Accumulated 5.015066 seconds in timer 0.");
    globus_utp_set_attribute("expected-message-4%s", "",
			     "globus_utp_write_file: timer 3 still running");

    globus_utp_name_timer(0, "Timer 0");
    globus_utp_name_timer(1, "Timer \\\"%s\\\"", "one");
    globus_utp_name_timer(2, "Timer %f", 2.0);

    globus_utp_disable_timer(4);
    globus_utp_start_timer(4);
    globus_utp_stop_timer(4);
    globus_utp_enable_timer(4);

    globus_utp_start_timer(0);
    globus_utp_start_timer(4);
    sleep(1);
    globus_utp_stop_timer(0);
    globus_utp_stop_timer(4);

    globus_utp_start_timer(1);
    sleep(1);
    globus_utp_stop_timer(1);
    globus_utp_reset_timer(1);
    globus_utp_disable_all_timers();
    globus_utp_start_timer(1);
    globus_utp_stop_timer(1);
    globus_utp_enable_all_timers();

    globus_utp_enable_timer(2);
    globus_utp_stop_timer(2);
    globus_utp_start_timer(3);
    globus_utp_start_timer(5);

    globus_utp_start_timer(0);
    sleep(3);
    globus_utp_stop_timer(0);

    globus_utp_start_timer(0);
    sleep(1);
    globus_utp_stop_timer(0);
    globus_utp_disable_timer(0);
    globus_utp_stop_timer(0);

    globus_utp_get_accum_time(0, &accumTime0, &prec);
    printf("Accumulated %.*f seconds in timer 0.\n", prec,
	   accumTime0);

    globus_utp_write_file("utp_dump_test.txt");

    if (globus_module_deactivate(GLOBUS_UTP_MODULE) != GLOBUS_SUCCESS)
    {
	fprintf(stderr, "ERROR: Failed to deactivate utp module\n");
	return(1);
    }
	
    return 0;
}
