/*
 * $Author: toonen $
 * $RCSfile: globus_netlogger.c,v $
 * $Revision: 1.2 $
 * $Locker:  $
 * $Date: 1999/09/14 01:44:37 $
 */


/*
 * logging.c     NetLogger routines
 *
 * creates log messages in ULM (Universal Logger Message) format, as
 *	defined by the IETF Internet-Draft
 *	(see: ftp://ds.internic.net/internet-drafts/draft-abela-ulm-02.txt)
 *
 * The ULM format is as follows:
 *   DATE=YYYYMMDDhhmmss.38298 HOST=host PROG=progname LVL=level F1= F2=......
 *
 *    where LVL = "Emergency" / "Alert" / "Error" / "Warning" / "Auth" / "Security"
 *                    / "Usage" / "System" / "Important" / "Debug";
 *
 *	NetLogger by default uses the levels "Usage" for general timing information,
 *		and "System" for system events like TCP retransmissions
 *
 *    and where F1, F2, etc are any number of user defined fields.
 *
 *  ULM recommends that field names are 4 or less characters.
 *
 * NetLogger always adds the following fields:
 *		NL.EVNT=event_name 
 *		NL.EVNT: NetLogger event name
 *
 * Sample ULM NetLogger message:
 *   DATE=19980401103010.28294 HOST=dpss.lbl.gov PROG=dpss_mast LVL=Usage NL.EVNT=ISS_MASTER_IN
 *		DPSS.SES=1 DPSS.BID=15.98.144.0
 *
 *  for more info see: http://www-didc.lbl.gov/NetLogger
 *
 *    Brian Tierney, LBNL
 *
 */

#ifndef lint
static char rcsid[] =
"@(#) $Id: globus_netlogger.c,v 1.2 1999/09/14 01:44:37 toonen Exp $";
#endif

#include <stdio.h>
#include <stdarg.h>
#include <syslog.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/param.h>
#include <stdlib.h>

#if defined(GLOBUS_NETLOGGER_ENABLED)
#   include "globus_netlogger.h"
#else
#   include "logging.h"
#endif


#define MAXLINE 255



 /*
  * depending on what you are logging, and how important it is to see things
  * happening in "real time", fflush might slow you down too much. Use with
  * caution!
  */
/*
#define FLUSH_RESULTS
*/

/* these routines are defined in the DPSS utils lib, but are also included here
	for standalone usage.  */
/*
 * #define HAVE_DPSS
 */

#ifndef HAVE_DPSS

char  *hname = NULL;
char  *pname = NULL;
static int writen(int, char *, int);
static void my_status(char *,...);
static void err_ret(char *,...);
static void my_perror();
static char *sys_err_str();

#endif
#if defined(GLOBUS_IO_ENABLED)
globus_io_handle_t * SecureNetLogOpen(char *,unsigned short);
#endif

#ifdef CORRECT_TIMESTAMPS
static void Netlog_correcttime(NLhandle * lp);
#endif				/* CORRECT_TIMESTAMPS */


/***********************************************************/

NLhandle *
NetLoggerOpen(int method, char *p_name, char *filename, char *hostname, int port)
{
    NLhandle *lp = (NLhandle *) malloc(sizeof(NLhandle));
    char     *hn = NULL, *shn = NULL;
    struct hostent *hp = NULL;
    int       pid;

    if (p_name == NULL)		/* must set to something, so use pid */
    {
	pid = (int) getpid();
	p_name = (char *) malloc(32);
	sprintf(p_name, "%d", pid);
    }
#ifdef CORRECT_TIMESTAMPS
    lp->tshift_sec = 0;
    lp->tshift_usec = 0;
#endif				/* CORRECT_TIMESTAMPS */

    hn = (char *) malloc(MAXHOSTNAMELEN);
    shn = (char *) malloc(MAXHOSTNAMELEN);
    /* get the local host name (probably does not include the domain) */
    if (gethostname(shn, MAXHOSTNAMELEN) < 0)
    {
	perror("gethostname");
	strcpy(shn, "N/A");
    }
    /* now lookup the full hostname from the name server */
    if ((hp = gethostbyname(shn)) == NULL)
    {
	perror("gethostbyname");
	strcpy(hn, "N/A");
    } else
    {
	/* now copy out the full hostname for the hostnet record */
	strncpy(hn, hp->h_name, MAXHOSTNAMELEN);	/* use strncpy to guard
							 * again bad DNS entries */
    }

    lp->num_events = 0;
    lp->hold = 0;

    switch (method)
    {
    case NL_FILE:
    case NL_FILE_MEM:
	if (!(lp->log_fp = LogOpen(filename, hostname)))	/* hostname arg
							 * overloaded to also
							 * indicate append mode */
	    return NULL;
	lp->method = method;
	lp->pname = strdup(p_name);
	lp->hname = strdup(hn);
	lp->level = strdup("Usage");
	lp->log_socket = -1;
	if (method == NL_FILE_MEM)
	{
	    lp->hold = 1;
	    lp->event_buffer = (char **) calloc(NL_MAX_BUFFER, sizeof(char *));
	}
	break;

    case NL_SYSLOG:
    case NL_SYSLOG_MEM:
	lp->method = method;
	lp->pname = strdup(p_name);
	lp->hname = strdup(hn);
	lp->level = strdup("Usage");
	lp->log_fp = NULL;
	lp->log_socket = -1;
	SysLogOpen(p_name, LOG_LOCAL0);
	if (method == NL_SYSLOG_MEM)
	{
	    lp->hold = 1;
	    lp->event_buffer = (char **) calloc(NL_MAX_BUFFER, sizeof(char *));
	}
	break;

    case NL_HOST:
    case NL_HOST_MEM:
	lp->method = method;
	lp->pname = strdup(p_name);
	lp->hname = strdup(hn);
	lp->level = strdup("Usage");
	lp->log_fp = NULL;
#if defined(GLOBUS_IO_ENABLED)
	if (use_globus())
	{
		lp->handle = SecureNetLogOpen(hostname,port);
		lp->log_socket = -1;
	} else {
		lp->log_socket = NetLogOpen(hostname, port);
		lp->handle = NULL;
	}
#else /* old style */
	lp->log_socket = NetLogOpen(hostname, port);
#endif /* defined(GLOBUS_IO_ENABLED) */

#ifdef CORRECT_TIMESTAMPS
	/* set time correction values based on the time on the netlogd host */
	Netlog_correcttime(lp);
#endif				/* CORRECT_TIMESTAMPS */
	if (method == NL_HOST_MEM)
	{
	    lp->hold = 1;
	    lp->event_buffer = (char **) calloc(NL_MAX_BUFFER, sizeof(char *));
	}
	break;

    default:
	err_ret("Error: Unknown NetLogger method.");
	lp = NULL;
	break;
    }
    free(hn);
    free(shn);
    return lp;
}

/***********************************************************/
int
NetLoggerGTWrite(NLhandle * lp, long sec, long usec, char *keyword,
		 char *fmt,...)
{
    char     *func = "NetLoggerGTWrite";
    int       rval = 1, n, size;
    va_list   args;
    struct tm *tm = NULL;
    char     *outbuf, *outbuf2, *date, *date2;

    /* check for bad time stamps */
    if (sec <= 0)
    {
	my_status("%s: Error: Keyword: %s; Bad time value: seconds = %d", func,
		  keyword, sec);
	return -1;
    }
    outbuf = (char *) calloc(1024, sizeof(char));
    outbuf2 = (char *) calloc(1024, sizeof(char));
    date = (char *) calloc(MAXLINE, sizeof(char));
    date2 = (char *) calloc(MAXLINE, sizeof(char));

#ifdef CORRECT_TIMESTAMPS
    if (lp->tshift_usec != 0)
    {
	usec += lp->tshift_usec;
	if (usec >= 1000000)
	{
	    usec -= 1000000;
	    sec++;
	} else if (usec < 0)
	{
	    usec += 1000000;
	    sec--;
	}
    }
    sec += lp->tshift_sec;
#endif

    /* build date string from the arg "sec" using "Univeral Coord. time" */
    if ((tm = gmtime((time_t *) & sec)) == NULL)
	perror("localtime error");

    /* output date in this form: YYYYMMDDHHMMSS  */
    if (strftime(date, MAXLINE, "%Y%m%d%H%M%S", tm) == 0)
	perror("strftime error");

    /* add microsec component to DATE field */
    sprintf(date2, "%s.%06d", date, (int) usec);

    va_start(args, fmt);
    vsprintf(outbuf, fmt, args);
    va_end(args);

    sprintf(outbuf2, "DATE=%s HOST=%s PROG=%s LVL=%s NL.EVNT=%s %s \n",
	    date2, lp->hname, lp->pname, lp->level, keyword, outbuf);

    switch (lp->method)
    {
    case NL_FILE:
	fprintf(lp->log_fp, "%s", outbuf2);
#ifdef FLUSH_RESULTS
	fflush(lp->log_fp);
#endif
	break;

    case NL_SYSLOG:
	syslog(LOG_INFO, "%s", outbuf2);
	break;

    case NL_HOST:

#if defined(GLOBUS_IO_ENABLED)
	if (lp->handle != NULL) {
		if (SecureNetLogSend(lp->handle, outbuf2))
			break;
		else /* error */
		    return -1;
	}
#endif 
	/* write size, then write message */
	size = htonl(strlen(outbuf2));
	n = writen(lp->log_socket, (char *) &size, sizeof(int));
	if (n != sizeof(int))
	{
	    my_status("%s: Error sending Netlogger message to socket.", func);
	    return -1;
	}
	n = writen(lp->log_socket, outbuf2, strlen(outbuf2));
	if (n != strlen(outbuf2))
	{
	    my_status("%s: Error sending Netlogger message to socket.", func);
	    return -1;
	}
	break;

    case NL_FILE_MEM:
    case NL_SYSLOG_MEM:
    case NL_HOST_MEM:
	/* copy to buffer for NetLoggerFlush call later */
	lp->event_buffer[lp->num_events] = strdup(outbuf2);
	lp->num_events++;
	if (lp->num_events >= NL_MAX_BUFFER)
	{
	    my_status("NetLogger buffer full. Flushing Buffer.");
	    NetLoggerFlush(lp);
	}
	break;

    default:
	err_ret("Error: Unknown NetLogger method.");
	rval = -1;
	break;
    }
    free(outbuf);
    free(outbuf2);
    free(date);
    free(date2);

    return rval;
}

/***********************************************************/
/* flush NetLogger message from memory to the specified destination */
int
NetLoggerFlush(NLhandle * lp)
{
    int       i, size, sz, n;
    char     *func = "NetLoggerFlush";

    my_status("Flushing events from NetLogger Buffer.");

    for (i = 0; i < lp->num_events; i++)
    {
	switch (lp->method)
	{
	case NL_FILE_MEM:
	    fprintf(lp->log_fp, "%s", lp->event_buffer[i]);
	    break;

	case NL_SYSLOG_MEM:
	    syslog(LOG_INFO, "%s", lp->event_buffer[i]);
	    break;

	case NL_HOST_MEM:
#if defined(GLOBUS_IO_ENABLED)
		if (lp->handle != NULL) {
			if (SecureNetLogSend(lp->handle, lp->event_buffer[i]))
				break;
			else /* error */
			    return -1;
	}
#endif 
	    /* write size, then write message */
	    size = strlen(lp->event_buffer[i]);
	    sz = htonl(size);
	    n = writen(lp->log_socket, (char *) &sz, sizeof(int));
	    if (n != sizeof(int))
	    {
		my_status("%s: Error sending Netlogger message to socket.", func);
		return -1;
	    }
	    n = writen(lp->log_socket, lp->event_buffer[i], size);
	    if (n != size)
	    {
		my_status("%s: Error sending Netlogger message to socket.", func);
		return -1;
	    }
	    break;

	default:
	    err_ret("Error: Unknown NetLogger method.");
	    return -1;
	    break;
	}
	free(lp->event_buffer[i]);
    }
#ifdef FLUSH_RESULTS
    if (lp->method == NL_FILE_MEM)
	fflush(lp->log_fp);
#endif
    lp->num_events = 0;
    return 1;
}

/***********************************************************/
 /*
  * build and write a NetLogger message.
  * 
  * For example: DATE=19980401103010 HOST=dpss.lbl.gov PROG=dpss_mast LVL=Usage
  * NL.EVNT=ISS_MASTER_IN NL.SEC=889644821 NL.USEC=333739 DPSS.SES=1
  * DPSS.BID=15.98.144.0
  */

int
NetLoggerWrite(NLhandle * lp, char *keyword, char *fmt,...)

{
    char     *func = "NetLoggerWrite";
    int       rval = 1, n, size;
    va_list   args;
    char     *outbuf, *outbuf2, *date, *date2;
    struct timeval tv;
    struct tm *tm = NULL;
    long      etime;

    /* get system time */
    gettimeofday(&tv, 0);

    outbuf = (char *) calloc(1024, sizeof(char));
    outbuf2 = (char *) calloc(1024, sizeof(char));
    date = (char *) calloc(MAXLINE, sizeof(char));
    date2 = (char *) calloc(MAXLINE, sizeof(char));

#ifdef CORRECT_TIMESTAMPS
    if (lp->tshift_usec != 0)
    {
	tv.tv_usec += lp->tshift_usec;
	if (tv.tv_usec >= 1000000)
	{
	    tv.tv_usec -= 1000000;
	    tv.tv_sec++;
	} else if (tv.tv_usec < 0)
	{
	    tv.tv_usec += 1000000;
	    tv.tv_sec--;
	}
    }
    tv.tv_sec += lp->tshift_sec;
#endif

    /* build date string from the arg "sec" using "Univeral Coord. time" */
    if ((tm = gmtime((time_t *) & tv.tv_sec)) == NULL)
	perror("localtime error");

    /* output date in this form: YYYYMMDDHHMMSS  */
    if (strftime(date, MAXLINE, "%Y%m%d%H%M%S", tm) == 0)
	perror("strftime error");

    /* add microsec component to DATE field */
    sprintf(date2, "%s.%06d", date, (int) tv.tv_usec);

    va_start(args, fmt);
    vsprintf(outbuf, fmt, args);
    va_end(args);

    sprintf(outbuf2, "DATE=%s HOST=%s PROG=%s LVL=%s NL.EVNT=%s %s \n",
	    date2, lp->hname, lp->pname, lp->level, keyword, outbuf);

    switch (lp->method)
    {
    case NL_FILE:
	fprintf(lp->log_fp, "%s", outbuf2);
#ifdef FLUSH_RESULTS
	fflush(lp->log_fp);
#endif
	break;

    case NL_SYSLOG:
	syslog(LOG_INFO, "%s", outbuf2);
	break;

    case NL_HOST:
#if defined(GLOBUS_IO_ENABLED)
	if (lp->handle != NULL) {
		if (SecureNetLogSend(lp->handle, outbuf2))
			break;
		else /* error */
		    return -1;
	}
#endif 
	/* write size, then write message */
	size = htonl(strlen(outbuf2));
	n = writen(lp->log_socket, (char *) &size, sizeof(int));
	if (n != sizeof(int))
	{
	    my_status("%s: Error sending Netlogger message to socket.", func);
	    return -1;
	}
	n = writen(lp->log_socket, outbuf2, strlen(outbuf2));
	if (n != strlen(outbuf2))
	{
	    my_status("%s: Error sending Netlogger message to socket.", func);
	    return -1;
	}
	break;

    case NL_FILE_MEM:
    case NL_SYSLOG_MEM:
    case NL_HOST_MEM:
	/* copy to buffer for NetLoggerFlush call later */
	lp->event_buffer[lp->num_events] = strdup(outbuf2);
	lp->num_events++;
	if (lp->num_events >= NL_MAX_BUFFER)
	{
	    my_status("NetLogger buffer full. Flushing Buffer.");
	    NetLoggerFlush(lp);
	}
	break;

    default:
	err_ret("Error: Unknown NetLogger method.");
	rval = -1;
	break;
    }
    free(outbuf);
    free(outbuf2);
    free(date);
    free(date2);

    return rval;
}

/***********************************************************/
int
NetLoggerSetLevel(NLhandle * lp, char *level)
{
    lp->level = strdup(level);
    return 1;
}

/***********************************************************/
int
NetLoggerClose(NLhandle * lp)
{
    char     *func = "NetLoggerClose";
    int       rval = 1;

    if (lp->num_events > 1)
	/* flush any events still in memory */
	NetLoggerFlush(lp);

    switch (lp->method)
    {
    case NL_FILE:
    case NL_FILE_MEM:
	LogClose(lp->log_fp);
	break;

    case NL_SYSLOG:
    case NL_SYSLOG_MEM:
	SysLogClose();
	break;

    case NL_HOST:
    case NL_HOST_MEM:
#if defined(GLOBUS_IO_ENABLED)
	if(lp->handle != NULL) {
		SecureNetLogClose(&(lp->handle));
		break;
	}
#endif 
	NetLogClose(lp->log_socket);
	break;

    default:
	err_ret("%s: Error: Unknown NetLogger method.", func);
	rval = -1;
	break;
    }

    return rval;
}

/***********************************************************/
FILE     *
LogOpen(char *logfile_name, char *mode)
{
    FILE     *fp = NULL;

    if (logfile_name == NULL)
	return NULL;

    if (mode == NULL) 
        mode = "w";

    if (!strcmp(mode, "a"))
    {
	my_status("NetLogger: appending data to file %s", logfile_name);
	if ((fp = fopen(logfile_name, "a")) == NULL)
	{
	    fprintf(stderr, "Error openning log file %s \n", logfile_name);
	    return NULL;
	}
    } else
    {
	if ((fp = fopen(logfile_name, "w")) == NULL)
	{
	    fprintf(stderr, "Error openning log file %s \n", logfile_name);
	    return NULL;
	}
    }
    return fp;
}

/***********************************************************/
int
SysLogOpen(char *prog_name, int syslog_facility)
{
    /* syslog_facility can be LOG_USER, LOG_LOCAL0, etc. See 'man syslog' */

    openlog(prog_name, LOG_PID, syslog_facility);

    /* this call has no error return vals */
    return 1;
}

/***********************************************************/
/* written by Jean Tedesco, ANL */
/* NOTE: might also want a version of this call that just warns the user if the
   clocks are out of sync  -blt  */

#ifdef CORRECT_TIMESTAMPS
void
Netlog_correcttime(NLhandle * lp)
{
  char *func = "Netlog_correcttime";
    struct Timebuf
    {
	int       size;
	char      buf[44];	/* let transmit the value in ascii, life is
				 * easier */
	/* 44 char is more than long enought to store 2 long integers        */
    }         timebuf;
    struct timeval tvbuff[3];
    long      total_shift = 0;
    long      trans;
    long      tshift;		/* tshift can not be bigger than 2^33-1 =
				 * 8589 sec (2h23min) */
    int       i, n;
    ldiv_t    shift;
#define NBTRY 100

    for (i = 0; i < NBTRY; i++)
    {
	/* write a size of -1 to trigger the time correction function */
	timebuf.size = htonl(-1);
	/* get the time */
	gettimeofday(&tvbuff[0], 0);	/* tv.tv_sec, tv.tv_usec */
#if defined(GLOBUS_IO_ENABLED)
	if ( lp->handle != NULL )
	{
		if (SecureNetLogSend(lp->handle, timebuf))
		{
			n = 0;
		} else {
		     n = sizeof(timebuf);
		}

	} else {
		n = writen(lp->log_socket, (char *) &timebuf, sizeof(timebuf));
	}
#else
	n = writen(lp->log_socket, (char *) &timebuf, sizeof(timebuf));
#endif
	if (n != sizeof(timebuf))
	{
	    my_status("NetLog: Error sending Netlogger time synch. message");
	    lp->tshift_sec = 0;
	    lp->tshift_usec = 0;
	    return;
	}

#if defined(GLOBUS_IO_ENABLED)
	if ( lp->handle != NULL )
	{
		if (SecureNetLogRead(lp->handle, timebuf.buf, 44))
		{ /* error */
			n=0;
		} else {
			n = 44;
		}

	} else {
		n = readn(lp->log_socket, timebuf.buf, 44);
	}
#else
	n = readn(lp->log_socket, timebuf.buf, 44);
#endif
	if (n != 44)
	{
	    my_status("NetLog: Error reading Netlogger time synch. message");
	    lp->tshift_sec = 0;
	    lp->tshift_usec = 0;
	    return;
	}
	do {
	  char pbuf[45];
	  memcpy(pbuf,timebuf.buf,44);
	  pbuf[44]='\0';
	} while(0);
	/* get the time */
	gettimeofday(&tvbuff[2], 0);	/* tv.tv_sec, tv.tv_usec  */

	/*
	 * Extract timestamp from a string.
	 */
	if ( sizeof(time_t) == sizeof(long) ) {
	  sscanf(timebuf.buf, "%ld %ld",
		 &(tvbuff[1].tv_sec),
		 &(tvbuff[1].tv_usec));
	}
	else if ( sizeof(time_t) == sizeof(int) ) {
	  sscanf(timebuf.buf, "%d %ld",
		 &(tvbuff[1].tv_sec),
		 &(tvbuff[1].tv_usec));
	}
	else {
	  fprintf(stderr,"%s: Cannot format time synch msg: "
		  "time_t is not an integer or long integer\n", func);
	  exit(1);
	}

	/* calculate aproximation of the transmission time             */
	/* lets hope the transmission time is equal in both directions */
	trans = ((tvbuff[2].tv_sec - tvbuff[0].tv_sec) * 1000000 +
		 (tvbuff[2].tv_usec - tvbuff[0].tv_usec)) / 2;
	tshift = ((tvbuff[1].tv_sec - tvbuff[0].tv_sec) * 1000000 +
		  (tvbuff[1].tv_usec - tvbuff[0].tv_usec)) - trans;
	/* I do average here to not overflow total_shift */
	total_shift += tshift / NBTRY;
    }
#ifdef DEBUG_TIME
    printf("me: %ld sec\n", tvbuff[0].tv_sec);
    printf("me: %ld usec\n", tvbuff[0].tv_usec);
    printf("him: %ld sec\n", tvbuff[1].tv_sec);
    printf("him: %ld usec\n", tvbuff[1].tv_usec);
    printf("mea: %ld sec\n", tvbuff[2].tv_sec);
    printf("mea: %ld usec\n", tvbuff[2].tv_usec);
    printf("trans: %ld usec\n", trans);

    printf("Shift: %ld usec\n", total_shift);
#endif

    shift = ldiv(total_shift, 1000000L);
    lp->tshift_sec = shift.quot;
    lp->tshift_usec = shift.rem;

#ifdef DEBUG_TIME
    printf("Shift: %ld sec\n", lp->tshift_sec);
    printf("Shift: %ld usec\n", lp->tshift_usec);
#endif
}				/* correcttime() */

#endif				/* CORRECT_TIMESTAMPS */

/***********************************************************/
int
NetLogOpen(char *hostname, int port)
{
    struct hostent *hp = NULL;
    struct sockaddr_in sin;
    int       log_socket, i = 1;

    my_status("opening connect to netlog message server host %s, port %d",
	      hostname, port);
    if (hostname == NULL)
    {
	my_status("Error: hostname not set!");
	return -1;
    }
    if ((hp = gethostbyname(hostname)) == 0)
    {
	perror("gethostbyname");
	return -1;
    }
    memset((void *) &sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = ((struct in_addr *) (hp->h_addr))->s_addr;
    sin.sin_port = htons(port);

    if ((log_socket = socket(AF_INET, SOCK_STREAM, 0)) == -1)
    {
	perror("socket");
	return -1;
    }
#ifdef FLUSH_RESULTS
    if (setsockopt(log_socket, IPPROTO_TCP, TCP_NODELAY, (char *) &i, (int) sizeof(int)) < 0)
	perror("setsockopt:tcp_nodelay");
#endif


    if (connect(log_socket, (struct sockaddr *) & sin, sizeof(sin)) == -1)
    {
	perror("connect");
	close(log_socket);
	log_socket = -1;
	return -1;
    }
    return log_socket;
}

/***********************************************************/
int
LogClose(FILE * fp)
{
    fclose(fp);
    return 1;
}

/***********************************************************/
int
SysLogClose(void)
{
    closelog();
    return 1;
}

/***********************************************************/
int
NetLogClose(int log_socket)
{
    close(log_socket);
    return 1;
}

/*******************************************************************/
/* added by Martin Reddy, SRI:  non-varargs version of write calls */
/*******************************************************************/
int
NetLoggerGTWriteString(NLhandle * lp, long sec, long usec, char *keyword,
		       char *data)
{
    char     *func = "NetLoggerGTWriteString";
    int       rval = 1, n, size;
    char     *outbuf, *date, *date2;
    struct tm *tm = NULL;

    outbuf = (char *) calloc(1024, sizeof(char));
    date = (char *) calloc(MAXLINE, sizeof(char));
    date2 = (char *) calloc(MAXLINE, sizeof(char));

    /* build date string from the arg "sec" using "Univeral Coord. time" */
    if ((tm = gmtime((time_t *) & sec)) == NULL)
	perror("localtime error");

    /* output date in this form: YYYYMMDDHHMMSS  */
    if (strftime(date, MAXLINE, "%Y%m%d%H%M%S", tm) == 0)
	perror("strftime error");
    /* add microsec component to DATE field */
    sprintf(date2, "%s.%06d", date, (int) usec);

#ifdef CORRECT_TIMESTAMPS
    usec += lp->tshift_usec;
    if (usec >= 1000000)
    {
	usec -= 1000000;
	sec++;
    } else if (usec < 0)
    {
	usec += 1000000;
	sec--;
    }
    sec += lp->tshift_sec;
#endif

    sprintf(outbuf, "DATE=%s HOST=%s PROG=%s LVL=%s NL.EVNT=%s %s \n",
	    date2, lp->hname, lp->pname, lp->level, keyword, data);

    switch (lp->method)
    {
    case NL_FILE:
	fprintf(lp->log_fp, "%s", outbuf);
#ifdef FLUSH_RESULTS
	fflush(lp->log_fp);
#endif
	break;

    case NL_SYSLOG:
	syslog(LOG_INFO, "%s", outbuf);
	break;

    case NL_HOST:
#if defined(GLOBUS_IO_ENABLED)
	if (lp->handle != NULL) {
		if (SecureNetLogSend(lp->handle, outbuf))
			break;
		else /* error */
		    return -1;
	}
#endif 
	/* write size, then write message */
	size = htonl(strlen(outbuf));
	my_status("** buflen=%d Size of integer=%d\n", size, sizeof(int));
	n = writen(lp->log_socket, (char *) &size, sizeof(int));
	if (n != sizeof(int))
	{
	    my_status("%s: Error sending Netlogger message to socket.", func);
	    return -1;
	}
	n = writen(lp->log_socket, outbuf, strlen(outbuf));
	if (n != strlen(outbuf))
	{
	    my_status("%s: Error sending Netlogger message to socket.", func);
	    return -1;
	}
	break;

    default:
	err_ret("Error: Unknown NetLogger method.");
	rval = -1;
	break;
    }
    free(outbuf);
    free(date);
    free(date2);

    return rval;
}

/*******************************************************************/

int
NetLoggerWriteString(NLhandle * lp, char *keyword, char *data)
{
    char     *func = "NetLoggerWriteString()";
    struct timeval tv;

    /* get system time */
    gettimeofday(&tv, 0);

    return NetLoggerGTWriteString(lp, tv.tv_sec, tv.tv_usec, keyword, data);
}

/*******************************************************************/

/* NOT DONE: This routine not finished, but contain most of the
	required components to make it easy to finish it if someone
	finds it usefull
*/
float
ulm_elapsed_time(char *t1, char *t2)
{
    char     *d1, *d2, *date;
    struct timeval tv1, tv2;
    struct tm *tm = NULL;
    float     etime;

    /* test code to see how long it take to convert date back again */
    /* results: it take about 2 ms: which I think is too long! */

    /* convert time t1 */
    memset(tm, 0, sizeof(struct tm));
    sscanf(t1, "%4d%2d%2d%2d%2d%2d", &(tm->tm_year), &(tm->tm_mon),
	   &(tm->tm_mday), &(tm->tm_hour), &(tm->tm_min), &(tm->tm_sec));
    tm->tm_year = tm->tm_year - 1900;
    tv1.tv_sec = (long) mktime(tm);

    /* convert time t2 */
    memset(tm, 0, sizeof(struct tm));
    sscanf(t2, "%4d%2d%2d%2d%2d%2d", &(tm->tm_year), &(tm->tm_mon),
	   &(tm->tm_mday), &(tm->tm_hour), &(tm->tm_min), &(tm->tm_sec));
    tm->tm_year = tm->tm_year - 1900;
    tv2.tv_sec = (long) mktime(tm);

    etime = ((tv2.tv_sec * 1000000) + tv2.tv_usec) -
	((tv1.tv_sec * 1000000) + tv1.tv_usec);

    return etime;
}


/*******************************************************************/
/*******************************************************************/

/********* routines from utils.c for standalone version of
   the NetLogger Library  */

#ifndef HAVE_DPSS

/***********************************************************/
static void
my_status(char *fmt,...)	/* use this one with X-windows programs to
				 * avoid name conflict */
{
    va_list   args;



    if (pname != NULL && hname != NULL)
	fprintf(stderr, "%s@%s: ", pname, hname);
    else if (pname != NULL)
	fprintf(stderr, "%s: ", pname);

    va_start(args, fmt);
    vfprintf(stderr, fmt, args);
    va_end(args);

    fputc('\n', stderr);
    fflush(stderr);

    return;

}

/***********************************************************/

/* err_ret(str, arg1, arg2...): print message and return */
static void
err_ret(char *fmt,...)
{
    va_list   args;

    if (pname != NULL && hname != NULL)
	fprintf(stderr, "%s@%s: ", pname, hname);
    else if (pname != NULL)
	fprintf(stderr, "%s: ", pname);


    va_start(args, fmt);
    vfprintf(stderr, fmt, args);
    va_end(args);

    my_perror();
    fflush(stdout);
    fflush(stderr);

    return;
}

/*************************************************************/
static void
my_perror()
{
#if !defined(__FreeBSD__)
    char     *sys_err_str();
#endif

    fprintf(stderr, " %s\n", sys_err_str());
}

/**************************************************************/

static int
writen(fd, ptr, nbytes)		/* use for writing n characters to a socket */
    register int fd;
    register char *ptr;
    register int nbytes;
{
    int       nleft, nwritten;

    nleft = nbytes;
    while (nleft > 0)
    {
	nwritten = write(fd, ptr, nleft);
	if (nwritten < 0)
	    return (nwritten);	/* error */

	nleft -= nwritten;
	ptr += nwritten;
    }
    return (nbytes - nleft);
}

/**************************************************************/
static int
readn(register int fd, register char *ptr, register int nbytes)
{
    int       nleft, nread;

    nleft = nbytes;
    while (nleft > 0)
    {
	nread = read(fd, ptr, nleft);
	if (nread < 0)
	    return (nread);	/* error */

	nleft -= nread;
	ptr += nread;
    }
    return (nbytes - nleft);
}

/**************************************************************/
extern int errno;
extern int sys_nerr;
#if !defined(__FreeBSD__) && !defined(__linux__)
extern char *sys_errlist[];
#endif
/**************************************************************/
static char *
sys_err_str()
{
    static char msgstr[200];

    if (errno != 0)
    {
	if (errno > 0 && errno < sys_nerr)
	    sprintf(msgstr, "(%s)", sys_errlist[errno]);
	else
	    sprintf(msgstr, "(errno = %d)", errno);
    } else
    {
	msgstr[0] = '\0';
    }

    return (msgstr);
}

#endif

#if defined(GLOBUS_IO_ENABLED)
/**************************************************************/
int
use_globus()
{
    if(getenv("GLOBUS_NETLOGGER_USE_SSL") == NULL )
        return 0;
    return 1;
}
/**************************************************************/
int
error_stuff(globus_result_t result, char *msg)
{
    globus_object_t *err = GLOBUS_NULL;
    char     *errstring = GLOBUS_NULL;

    if (result != GLOBUS_SUCCESS) {
        err = globus_error_get(result);

        if (!globus_io_eof(err)) {
            errstring = globus_object_printable_to_string(err);
            globus_libc_printf("%s failed: %s\n", msg, errstring);
        }
        return 1;
    } else {
/*
        globus_libc_printf("%s ok\n", msg);
*/
    }
    return 0;

}
/**************************************************************/
globus_io_handle_t *
SecureNetLogOpen( char *host, unsigned short port)
{
   globus_io_attr_t attr;
   globus_io_handle_t * handle;
   globus_io_secure_authorization_data_t auth_data;
   globus_result_t result;

   handle = (globus_io_handle_t *)calloc(1,sizeof(globus_io_handle_t));

   /* active modules */
   globus_module_activate(GLOBUS_COMMON_MODULE);
   globus_module_activate(GLOBUS_IO_MODULE);

   /* setup auth */
   globus_io_tcpattr_init(&attr);
   globus_io_attr_set_secure_authentication_mode( &attr,
                                GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI,
                                                  GSS_C_NO_CREDENTIAL);

   globus_io_secure_authorization_data_initialize(&auth_data);
   /* XXX what does this get set to? Needs to be supplied!!*/
   globus_io_secure_authorization_data_set_identity( &auth_data,
                              "/C=US/O=Globus/O=Lawrence Berkeley National "
                              "Laboratory/OU=Future Technologies Group/CN="
                              "dpss program");

    globus_io_attr_set_secure_authorization_mode( &attr,
                               GLOBUS_IO_SECURE_AUTHORIZATION_MODE_IDENTITY,
                                                 &auth_data);

     result = globus_io_tcp_connect( host,
                                   (unsigned short) port,
                                   &attr,
                                   handle);

     if (error_stuff(result, "connect"))
        return NULL;

    globus_io_tcpattr_destroy(&attr);

    return(handle);
}
/**************************************************************/
/* return 1 on good send, 0 on error */
int 
SecureNetLogSend(globus_io_handle_t * handle, char *buf)
{
  globus_result_t result;
  globus_size_t  nbytes;
  globus_size_t len;

  len = strlen(buf);

  if (len <= 0 )
	return 0;

   result = globus_io_write(handle, (unsigned char *)buf, len, &nbytes);

  if (error_stuff(result, "SecureNetLogSend")) {
       return 0;
   }

   if (nbytes != len )
   {
	fprintf(stderr,"error with output %d instead of %d\n", nbytes, len);
	return 0;
   }
    return 1;
}
/**************************************************************/
/* return 1 on good send, 0 on error */
int 
SecureNetLogRead(globus_io_handle_t * handle, char *buf, int size)
{
  globus_result_t result;
  globus_size_t  nbytes;

   result = globus_io_read(handle, (unsigned char *)buf, size, size, &nbytes);

  if (error_stuff(result, "SecureNetLogRead")) {
       return 0;
   }

   if (nbytes != size )
   {
	fprintf(stderr,"error with output %d instead of %d\n", nbytes, size);
	return 0;
   }
    return 1;
}
/**************************************************************/
int
SecureNetLogClose(globus_io_handle_t *handle)
{
    globus_io_close(handle);
    globus_module_deactivate(GLOBUS_IO_MODULE);
    globus_module_deactivate(GLOBUS_COMMON_MODULE);
}
#endif
