# This is a perl library script
#
# This is a parser for the language used to describe Globus Object
# Definitions.  This language follows the convention used in the RFCs
# pertaining to directory servers. We have, however, made several additions.
#
#
# Grammar Specification 
#     (Locations for semantic actions are denoted by <<sa()>>,
#      the parameters passed are identified by $1, $2, ...
#      These numbers identify the token position on the RHS
#      the rule.
#
#    root               ::= class { class }
#  
#    class              ::= type OBJECT CLASS  
#                           opt_subclass
#                           dit_spec(type -> RD_Name, formalName)
#                           must_contain(type, RD_Name  -> attrs) 
#                           may_contain(type, formalName -> attrs)       <<sa()>>
#
#
#     opt_subclass
#              ::=   SUBCLASS OF <parent>                 <<sa($1,$6)>>
#              ::=
#                   
#
#
#    Directory Information Tree Spec (DIT).
#      Indicate the position of this object within the dit.  
#      Note that this has no effect on the ojbect definition within
#      ldap (save the rdn name).  its primary purpose is documentation 
#      _only_.
#    ---------------------------------------------------------------
#    dit_spec(objectType -> RD_Name, formalName)  
#                       ::= rdn_spec(objectType ->RD_Name, formalName)                          
#                           child_of(objectType,RD_Name,formalName)            <<sa()>>
#
#
#    rdn_spec(objectType -> RD_Name, formalName)
#                       ::= RDN "=" RD_Name  
#                           "(" formalName ")"                 <<sa($3, $5)>>
#
#                       ::= RDN "=" RD_Name                    <<sa($3, $5)>>
#
#    child_of(objectType,RD_Name,formalName)  
#                       ::= CHILD OF                           <<sa()>>
#                           "{" dit_list(objectType, RD_Name, formalName)
#                           "}"                                <<sa()>>
#
#                       ::=                                    <<sa()>>
#
#
#    dit_list(objectType, RD_Name, formalName)
#                       ::=  name_list(',')
#                            { ";"                            <<sa(objectType, DN, formalName>> 
#                              name_list(',') }               <<sa(objectType, DN, formalName>>
#
#
#
#    STUFF TO INDICATE THE ATTRIBUTES ASSOCIATED WITH AN OBJECT
#    ----------------------------------------------------------
#    must_contain(type, RD_Name->attrs) 
#                       ::= MUST CONTAIN                   <<sa()>> 
#                           "{" pair_list(type->attr) "}"  <<sa(RD_Name)>>
#
#                       ::=                                <<sa()>>
#  
#    may_contain(type, formalName->attrs)
#                       ::= MAY CONTAIN                    <<sa()>>
#                           "{" pair_list(type ->attr) "}" <<sa(formalName)>>
#
#                       ::=                                <<sa()>>
#  
#    pair_list(type->attr)  ::=  pair { "," <<sa()>> pair }    <<sa()>>
#  
#    pair(->name)       ::=  name_list('|') "::" <type>    <<sa($1, $3>>
#  
#    name_list(sep->names)
#                       ::=  <name> { sep  <name> }        <<sa()>>
#
#
#
#  
# By providing the appropriate semantic actions, a translator can be
# constructed using this parser.
#
# The following rountines are assumed to be defined 
#      (within the "main" namespace).
#
#  sa_class_header, sa_class_closer,
#
#  sa_dit_closer,
#
#  sa_RDN_name,
#  sa_childof_header, sa_childof_closer, sa_no_childof,
#  sa_dit_list_separator,
#  sa_dit_list_last,
#
#  sa_must_header,  sa_must_closer,  sa_no_must,
#  sa_may_header,   sa_may_closer,   sa_no_may,
#  sa_pair_list_separator,
#  sa_pair_list_last,
#  sa_pair,
#  sa_comment
#
#  The semantic action "sa_comment" is associated with comments, and is
#  directly called by the lexicon  (removeWhite).  As such, the semantic
#  action can be called at any point in time within the grammar.
#  Comments begin with a sharp (#) and end by a newline




# parser_ln is a global variable used to report
# the line on which the first error is encountered.
package Parser;

#Package Globals (Symbol Table Stuff)
#  %oc_id                : oc name -->  unique Symbol Table number 
#  $num_oc               : the number of Object Classes
#  %dit_object{oc}       : returns TRUE or FALSE
#  %must_attributes{oc}  : space separated list of attributes
#  %may_attributes{oc}   : space separated list of attributes
#  %types{oc:attr}       : the type of an object's attribute
#  %parent{oc}           : parent of a object class
#  %rdn{oc}              : Relative Distinguished Name

reset(%oc_id);
reset($num_oc);
$num_oc = 1;

reset(%dit_object);
reset(%must_attributes);
reset(%may_attributes);
reset(%types);
reset(%parent);

#Package Locals
$parser_ln = 0;


sub main'parser {
  local($name) = '';

  while (!eof(STDIN)) {
    $name = &lookahead;
    if ($name eq 'ERROR') {
      #Maybe end of file with comments
      return 'ERROR';
    }
    if ($name ne '') {        
      $result = &class;
    }
  }
}




sub class {
   local($type_name, $parent) = ('', '');
   local($rdn_name, $formalName) = ('', '');

   $type_name = &next_token;
   &match("OBJECT");   &match("CLASS");


   if ($type_name ne 'GlobusTop') {
     &match("SUBCLASS"); &match("OF");
     $parent = &next_token;
   } else {
     if (&lookahead eq 'SUBCLASS') {
	 print STDERR "GlobusTop does not have a parent: $parser_ln\n";
     }
   } 

   ###########################
   # action for class header

   $oc_id{$type_name} = $num_oc++;
   $parent{$type_name} = $parent;

   if ($oc_id{$parent} == 0 && $type_name ne 'GlobusTop') {
      print STDERR "You must specify $parent before $type_name:",
            "  $parser_ln\n";
   }

   &'sa_class_header($type_name, $parent);


   ############################

   #This is to provide documentation
   #The rdn name must added to the MUST clause
   #the formalName is added to the MAY clause

   if ($type_name ne 'GlobusTop') {
     ($rdn_name, $formalName) = &dit_spec($type_name);
   }

   $must_attributes{$type_name} = &must_contain($type_name, $rdn_name);
   $may_attributes{$type_name}  = &may_contain($type_name, $formalName);


   ###########################
   # action for class closer
   
   &'sa_class_closer($type_name);

   ###########################
 
}



sub dit_spec {
    local($objectType) = @_;
    local($RD_Name, $formalName) = ('', '');


    ($RD_Name, $formalName) = &rdn_spec($objectType);
    $rdn{$objectType} = $RD_Name;

    &child_of($objectType, $RD_Name, $formalName);

    &'sa_dit_closer;

    return $RD_Name, $formalName;
}



sub rdn_spec {
    local($objectType) = @_;                   # Input  Values
    local($commonName, $formalName) = ('', '');  # Return Values


    &match('RDN');
    &match('=');
    
    $commonName = &next_token;
    $types{"$objectType:$commonName"} = "cis";

    if (&lookahead eq '(' ) {
	&match('(');
	$formalName = &next_token;
	&match(')');
        $types{"$objectType:$formalName"} = "cis";
    } else {
	$formalName = $commonName;
    }
    
    ###########################
    # action for "RDN name"
    
    &'sa_RDN_name($commonName, $formalName);

    ###########################


    return $commonName, $formalName;
}


sub child_of {
    local($objectType, $RD_Name, $formalName)  = @_;  
    #the base Object type for the RDN name;


    if (&lookahead eq 'CHILD') {
	&match("CHILD");
	&match("OF");

	###########################
	# action for "child-of header"

	$dit_object{$objectType} = 'TRUE';
	&'sa_childof_header();

        ###########################

        &match("{");
        &dit_list($objectType, $RD_Name, $formalName);
        &match("}");

	##########################
	# action for "child-of closer"
       
	&'sa_childof_closer();

        ###########################

    } else {

	###########################
	# action for "child-of header"

	$dit_object{$RD_Name} = 'FALSE';       
	&'sa_no_childof();

        ###########################
    }
}


sub dit_list {
    local($objectType, $RD_Name, $formalName) = @_;   

    local($semi, $dit_parent, $DN) = ('', '', '');

    $dit_parent = &name_list(',');

    $semi = &lookahead;

    while ($semi eq ';') {
	&match(";");

	if (&lookahead eq "}") {
	    #I have seen an optional ";" at the end
	    print STDERR "Saw superfluous \";\" on line: ", $parser_ln, "\n";
	    last;
	}

	###########################
	# action for dit separator
	
        $DN = $formalName . " " . $dit_parent;
	&'sa_dit_list_separator($objectType, $DN, $formalName);

        ###########################

        $dit_parent = &name_list(',');
        $semi = &lookahead;

   }

   ###########################
   # action for last dit

   if ($formalName eq '') {  
     $DN = $RD_Name . " " . $dit_parent;
   } else {
     $DN = $formalName . " " . $dit_parent;
   }

   &'sa_dit_list_last($objectType, $DN, $formalName);

   ###########################

}





sub must_contain #($type_name, $rdn_name->attributes)
{
    local($type_name, $rdn_name) = @_;
    local($attributes) = '';  #space separated list

    $attributes = $rdn_name;
    if (&lookahead eq 'MUST') {
        &match("MUST");
        &match("CONTAIN");
       
        ###########################
        # action for "must header"
       
        &'sa_must_header($rdn_name);
 
        ###########################

        &match("{");
        if ($attributes ne '') {
           $attributes .= ' ';
        }
        $attributes .= &pair_list($type_name);

        &match("}");
       
        ###########################
        # action for "must closer"
       
        &'sa_must_closer($rdn_name);

        ###########################


    } else {
       ;

       ###########################
       # action for no "must header"

       &'sa_no_must($rdn_name);

       ###########################
     
   }

   return $attributes;

}


sub may_contain #(type_name->attributes)
{
    local($type_name, $formalName) = @_;
    local($attributes) = '';  #space separated list


    $attributes = $formalName;
    if (&lookahead eq 'MAY') {
	&match("MAY");
	&match("CONTAIN");

	###########################
	# action for "may header"
	
	&'sa_may_header($formalName);

        ###########################

        &match("{");
        if ($attributes ne '') {
           $attributes .= ' ';
        }
        $attributes .= &pair_list($type_name);

        &match("}");

        ###########################
        # action for "may closer"
     
        &'sa_may_closer($formalName);

	###########################

    } else {
	;

	###########################
	# action for empty "may header"
	
	&'sa_no_may($formalName);

        ###########################
    }

   return $attributes;

}



sub pair_list #(type_name->attributes)
{
   local($type_name)  = @_;  # Name of the current object type
   local($attributes) = '';  # space separated list

   $attributes = &pair;

   while (&lookahead eq ',') {
     &match(",");

     if (&lookahead eq "}") {
	 #I have seen an optional "," at the end
	 print STDERR "Saw superfluous \",\" on line: ", $parser_ln, "\n";
	 last;
     }

     ###########################
     # action for pair separator
     
     &'sa_pair_list_separator();

     ###########################

     $attributes .= " " . &pair;
   }

   ###########################
   # action for last pair
  
   &'sa_pair_list_last();
     
   ###########################

   return $attributes;
}



sub pair #($objectClass->attribute)
{
   local($objectClass) = @_;
   local($names, $separator, $type) = ('', '', '');

   $names = &name_list("|");

   &match("::");

   $type = &next_token;

   #Note that this breaks if we start using the  "|"  in the input
   $types{"$objectClass:$names"} = $type;

   ###########################
   # action for pair name
   
   &'sa_pair($names, $type);

   ###########################

  return $names

}


sub name_list {
    local($separator) = @_;
    local($name) = '';
    local($token) = '';

    $name = &next_token;
    $token = &lookahead;
   
    while ($token eq $separator) {
	&match($separator);
     
	$name .= " " .  &next_token;
	$token = &lookahead;
    }
    
    return $name;
}





##################################################################
#  LEXICON ROUNTINES : 
#    next_token  : function that obtains next token
#    lookahead   : function that peaks at the next token
#    match       : routine that detemines if the next_token 
#                  matches the intended string
#    removeWhite : reads in lines from <STDIN>, strips comments, 
#                  and  removes leading white space. 
#
#################################################################

sub next_token {
   local($token) = '';   

   &removeWhite;

   #Check for a AlphaNumeric
   if (/^(\w+)/) {
      s/^(\w+)//, $token = $1;
   }

   # Perl's poor man case statement

   if ($token eq '') {
     # look for special tokens
     SWITCH : {
       $token = '::', last SWITCH if s/^:://;
       $token = '{' , last SWITCH if s/^{//;
       $token = '}' , last SWITCH if s/^}//;
       $token = '(' , last SWITCH if s/^\(//;
       $token = ')' , last SWITCH if s/^\)//;
       $token = ',' , last SWITCH if s/^,//;
       $token = ';' , last SWITCH if s/^;//;
       $token = '|' , last SWITCH if s/^\|//;
       $token = '=' , last SWITCH if s/^=//;

       #insert new tokens at this point

       #default
       $token = 'ERROR'
     }
   }

   return $token;
}

sub lookahead {

   # Read the next token
   $lookahead = &next_token;

   # Now put it back
   s/^/$lookahead /;

   return $lookahead;
}



sub match {
  # Given the first (and only) input parameter
  # verify if it matches the next token;

  local($pattern) = @_;
  local($token) = '';

  $token = &next_token;
  if ($token ne $pattern) {
      print STDERR "Unmatch token \"$token\" on line: $parser_ln\n";
      $pattern .= tr/a-z/A-Z/;
      if ($token ne $pattern) {
	  die "\n\nUnable to resolve\n\n";
      }
  }
  return 'TRUE';
}




sub removeWhite {

    while ($_ !~ /^\S/) {
	# Continue to loop until the 
	# leading charactor is not a white space

	if ($_ eq '') {             # Read the next line
	    if (eof(STDIN)) {  
		return;
	    }
	    $_ = <STDIN>;
	    $parser_ln++;
	}

        # Remove all leading white space
	s/^\s+//;

	# Remove and preprocess any comments
	if ( /^\#.*$/ ) {                
	    local($comment);
	    s/^(\#.*)$//, $comment = $1;   
	    &'sa_comment($comment);
        }

    }
}




1;  # loaded;

