/* ldapmodify.c - generic program to modify or add entries using LDAP */

#include "globus_common.h"

#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_SYS_FILE_H
#include <sys/file.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#ifdef HAVE_SYS_SIGNAL_H
#include <sys/signal.h>
#endif
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include <lber.h>
#include <ldap.h>
#include <ldif.h>

static char	*prog;
static char	*binddn = NULL;
static char	*passwd = NULL;
static char	*ldaphost = NULL;
static int	ldapport = 0;
static int      timeout  = -1;
static int	new, replace, not, verbose, contoper, force, valsfromfiles;
static LDAP	*ld;

#define safe_realloc( ptr, size )	( ptr == NULL ? malloc( size ) : \
					 realloc( ptr, size ))

#define LDAPMOD_MAXLINE		4096

/* strings found in replog/LDIF entries (mostly lifted from slurpd/slurp.h) */
#define T_REPLICA_STR		"replica"
#define T_DN_STR		"dn"
#define T_CHANGETYPESTR         "changetype"
#define T_ADDCTSTR		"add"
#define T_MODIFYCTSTR		"modify"
#define T_DELETECTSTR		"delete"
#define T_MODRDNCTSTR		"modrdn"
#define T_MODOPADDSTR		"add"
#define T_MODOPREPLACESTR	"replace"
#define T_MODOPDELETESTR	"delete"
#define T_MODSEPSTR		"-"
#define T_NEWRDNSTR		"newrdn"
#define T_DELETEOLDRDNSTR	"deleteoldrdn"

/* string found in libldap/ldap-int.h */
#define LDAP_LDAP_REF_STR         "ldap://"
#define LDAP_LDAP_REF_LEN         7
#define LDAP_REF_STR              "Referral:\n"
#define LDAP_REF_STR_LEN          10

static int process_ldapmod_rec ( char *rbuf );
static int process_ldif_rec ( char *rbuf );
static void addmodifyop ( LDAPMod ***pmodsp, int modop, char *attr,
	char *value, int vlen );
static int domodify ( char *dn, LDAPMod **pmods, int newentry );
static int dodelete ( char *dn );
static int domodrdn ( char *dn, char *newrdn, int deleteoldrdn );
static void freepmods ( LDAPMod **pmods );
static int fromfile ( char *path, struct berval *bv );
static char *read_one_record ( FILE *fp );

static void set_ld_timeout(LDAP  *ld, int timeout_val, char *env_string);
static char *check_for_referral(char *sptr, char *sstr);
static int get_port(char *s);


int
main( int argc, char **argv )
{
    char		*infile, *rbuf, *start, *p, *q;
    FILE		*fp;
    int			rc, i, use_ldif, authmethod, want_bindpw, debug;
    char                *rptr=NULL;
    char		*usage = "usage: %s [-abcknrvWF] [-d debug-level] [-h ldaphost] [-T timeout ][-p ldapport] [-D binddn] [-w passwd] [ -f file | < entryfile ]\n";

    if (( prog = strrchr( argv[ 0 ], '/' )) == NULL ) {
	prog = argv[ 0 ];
    } else {
	++prog;
    }
    new = ( strcmp( prog, "ldapadd" ) == 0 );

    infile = NULL;
    not = verbose = valsfromfiles = want_bindpw = debug = 0;
    authmethod = LDAP_AUTH_SIMPLE;

    while (( i = getopt( argc, argv, "WFabckKnrtvh:p:D:w:d:f:T:" )) != EOF ) {
	switch( i ) {
	case 'a':	/* add */
	    new = 1;
	    break;
	case 'b':	/* read values from files (for binary attributes) */
	    valsfromfiles = 1;
	    break;
	case 'c':	/* continuous operation */
	    contoper = 1;
	    break;
	case 'r':	/* default is to replace rather than add values */
	    replace = 1;
	    break;
	case 'k':	/* kerberos bind */
#ifdef HAVE_KERBEROS
		authmethod = LDAP_AUTH_KRBV4;
#else
		fprintf (stderr, "%s was not compiled with Kerberos support\n", argv[0]);
#endif
	    break;
	case 'K':	/* kerberos bind, part 1 only */
#ifdef HAVE_KERBEROS
		authmethod = LDAP_AUTH_KRBV41;
#else
		fprintf (stderr, "%s was not compiled with Kerberos support\n", argv[0]);
#endif
	    break;
	case 'F':	/* force all changes records to be used */
	    force = 1;
	    break;
	case 'h':	/* ldap host */
	    ldaphost = strdup( optarg );
	    break;
	case 'D':	/* bind DN */
	    binddn = strdup( optarg );
	    break;
	case 'w':	/* password */
	    passwd = strdup( optarg );
	    break;
	case 'd':
#ifdef LDAP_DEBUG
	    debug |= atoi( optarg );
#else /* LDAP_DEBUG */
	    fprintf( stderr, "%s: compile with -DLDAP_DEBUG for debugging\n",
		    prog );
#endif /* LDAP_DEBUG */
	    break;
	case 'f':	/* read from file */
	    infile = strdup( optarg );
	    break;
	case 'p':
	    ldapport = atoi( optarg );
	    break;
	case 'n':	/* print adds, don't actually do them */
	    ++not;
	    break;
	case 'v':	/* verbose mode */
	    verbose++;
	    break;
        case 'T':               /* timeout */
            timeout = atoi( optarg );
            break;
	case 'W':
		want_bindpw++;
		break;
	default:
	    fprintf( stderr, usage, prog );
	    exit( 1 );
	}
    }

    if ( argc - optind != 0 ) {
	fprintf( stderr, usage, prog );
	exit( 1 );
    }

    if ( infile != NULL ) {
	if (( fp = fopen( infile, "r" )) == NULL ) {
	    perror( infile );
	    exit( 1 );
	}
    } else {
	fp = stdin;
    }

	if ( debug ) {
		ldap_debug = debug;
	}

#ifdef SIGPIPE
#undef SIGNAL
#ifdef HAVE_SIGSET
#define SIGNAL sigset
#else
#define SIGNAL signal
#endif
	(void) SIGNAL( SIGPIPE, SIG_IGN );
#endif

    if ( !not ) {
	if (( ld = ldap_init( ldaphost, ldapport )) == NULL ) {
	    perror( "ldap_init" );
	    exit( 1 );
	}

	ld->ld_deref = LDAP_DEREF_NEVER;	/* this seems prudent */

        set_ld_timeout(ld, timeout, "GRID_INFO_TIMEOUT");

	if (want_bindpw)
		passwd = getpass("Enter LDAP Password: ");

	if ( ldap_bind_s( ld, binddn, passwd, authmethod ) != LDAP_SUCCESS ) {
	    ldap_perror( ld, "ldap_bind" );
	    exit( 1 );
	}
/************************************************************************
   Because openldap's client library does simple find in handling referrals
 we had to find the right host and port to bind from here with auth/passwd
 setup correctly before following through with the modify.
   We are assuming this is alway single referral per result and ber_buf is
 replaced every time a ldap_init is called 
   The validity of the content of ld->ld_sb.sb_ber.ber_buf could change if
 server gives back different thing. But with netscape4.1. It seems to contain
 the dn and 'immediate referral string' if a referral is encountered or else
 just contains some ber control sequence. So, this should be considered a 
 temporary workaround at most.                                         Mei
*************************************************************************/
        while(rptr=check_for_referral(rptr, ld->ld_sb.sb_ber.ber_buf)) {
         char *dptr=strstr(rptr, ":");
         int cnt=0;

         /* grab the host + port and ld_matched */
         if(ldaphost) free(ldaphost);
         *dptr='\0';
         ldaphost=strdup(rptr);
         *dptr=':';
         dptr++;
         ldapport=get_port(dptr);

         ldap_unbind(ld);
         /* grab the host + port and ld_matched */
         if (( ld = ldap_init( ldaphost, ldapport )) == NULL ) {
            perror( "ldap_init" );
            exit( 1 );
         }
         ld->ld_deref = LDAP_DEREF_NEVER;        /* this seems prudent */
         rptr=NULL;

         /* try to bind again */
         if ( ldap_bind_s( ld, binddn, passwd, authmethod ) != LDAP_SUCCESS ) {
             ldap_perror( ld, "ldap_bind" );
             exit( 1 );
         }
        }
    }

    rc = 0;

    while (( rc == 0 || contoper ) &&
		( rbuf = read_one_record( fp )) != NULL ) {
	/*
	 * we assume record is ldif/slapd.replog if the first line
	 * has a colon that appears to the left of any equal signs, OR
	 * if the first line consists entirely of digits (an entry id)
	 */
	use_ldif = ( p = strchr( rbuf, ':' )) != NULL &&
		( q = strchr( rbuf, '\n' )) != NULL && p < q &&
		(( q = strchr( rbuf, '=' )) == NULL || p < q );

	start = rbuf;

	if ( !use_ldif && ( q = strchr( rbuf, '\n' )) != NULL ) {
	    for ( p = rbuf; p < q; ++p ) {
		if ( !isdigit( (unsigned char) *p )) {
		    break;
		}
	    }
	    if ( p >= q ) {
		use_ldif = 1;
		start = q + 1;
	    }
	}

	if ( use_ldif ) {
	    rc = process_ldif_rec( start );
	} else {
	    rc = process_ldapmod_rec( start );
	}

	free( rbuf );
    }

    if ( !not ) {
	ldap_unbind( ld );
    }

    exit( rc );

	/* UNREACHABLE */
	return(0);
}


static int
process_ldif_rec( char *rbuf )
{
    char	*line, *dn, *type, *value, *newrdn, *p;
    int		rc, linenum, vlen, modop, replicaport;
    int		expect_modop, expect_sep, expect_ct, expect_newrdn;
    int		expect_deleteoldrdn, deleteoldrdn;
    int		saw_replica, use_record, new_entry, delete_entry, got_all;
    LDAPMod	**pmods;

    new_entry = new;

    rc = got_all = saw_replica = delete_entry = expect_modop = 0;
    expect_deleteoldrdn = expect_newrdn = expect_sep = expect_ct = 0;
    linenum = 0;
    deleteoldrdn = 1;
    use_record = force;
    pmods = NULL;
    dn = newrdn = NULL;

    modop = LDAP_MOD_ADD; 

    while ( rc == 0 && ( line = str_getline( &rbuf )) != NULL ) {
	++linenum;
	if ( expect_sep && strcasecmp( line, T_MODSEPSTR ) == 0 ) {
	    expect_sep = 0;
	    expect_ct = 1;
	    continue;
	}
	
	if ( str_parse_line( line, &type, &value, &vlen ) < 0 ) {
	    fprintf( stderr, "%s: invalid format (line %d of entry: %s\n",
		    prog, linenum, dn == NULL ? "" : dn );
	    rc = LDAP_PARAM_ERROR;
	    break;
	}

	if ( dn == NULL ) {
	    if ( !use_record && strcasecmp( type, T_REPLICA_STR ) == 0 ) {
		++saw_replica;
		if (( p = strchr( value, ':' )) == NULL ) {
		    replicaport = 0;
		} else {
		    *p++ = '\0';
		    replicaport = atoi( p );
		}
		if ( strcasecmp( value, ldaphost ) == 0 &&
			replicaport == ldapport ) {
		    use_record = 1;
		}
	    } else if ( strcasecmp( type, T_DN_STR ) == 0 ) {
		if (( dn = strdup( value )) == NULL ) {
		    perror( "strdup" );
		    exit( 1 );
		}
		expect_ct = 1;
	    }
	    continue;	/* skip all lines until we see "dn:" */
	}

	if ( expect_ct ) {
	    expect_ct = 0;
	    if ( !use_record && saw_replica ) {
		printf( "%s: skipping change record for entry: %s\n\t(LDAP host/port does not match replica: lines)\n",
			prog, dn );
		free( dn );
		return( 0 );
	    }

	    if ( strcasecmp( type, T_CHANGETYPESTR ) == 0 ) {
		if ( strcasecmp( value, T_MODIFYCTSTR ) == 0 ) {
			new_entry = 0;
			expect_modop = 1;
		} else if ( strcasecmp( value, T_ADDCTSTR ) == 0 ) {
			new_entry = 1;
		} else if ( strcasecmp( value, T_MODRDNCTSTR ) == 0 ) {
		    expect_newrdn = 1;
		} else if ( strcasecmp( value, T_DELETECTSTR ) == 0 ) {
		    got_all = delete_entry = 1;
		} else {
		    fprintf( stderr,
			    "%s:  unknown %s \"%s\" (line %d of entry: %s)\n",
			    prog, T_CHANGETYPESTR, value, linenum, dn );
		    rc = LDAP_PARAM_ERROR;
		}
		continue;
	    } else if ( new ) {		/*  missing changetype => add */
		new_entry = 1;
		modop = LDAP_MOD_ADD;
	    } else {
		expect_modop = 1;	/* missing changetype => modify */
	    }
	}

	if ( expect_modop ) {
	    expect_modop = 0;
	    expect_sep = 1;
	    if ( strcasecmp( type, T_MODOPADDSTR ) == 0 ) {
		modop = LDAP_MOD_ADD;
		continue;
	    } else if ( strcasecmp( type, T_MODOPREPLACESTR ) == 0 ) {
		modop = LDAP_MOD_REPLACE;
		continue;
	    } else if ( strcasecmp( type, T_MODOPDELETESTR ) == 0 ) {
		modop = LDAP_MOD_DELETE;
		addmodifyop( &pmods, modop, value, NULL, 0 );
		continue;
	    } else {	/* no modify op:  use default */
		modop = replace ? LDAP_MOD_REPLACE : LDAP_MOD_ADD;
	    }
	}

	if ( expect_newrdn ) {
	    if ( strcasecmp( type, T_NEWRDNSTR ) == 0 ) {
		if (( newrdn = strdup( value )) == NULL ) {
		    perror( "strdup" );
		    exit( 1 );
		}
		expect_deleteoldrdn = 1;
		expect_newrdn = 0;
	    } else {
		fprintf( stderr, "%s: expecting \"%s:\" but saw \"%s:\" (line %d of entry %s)\n",
			prog, T_NEWRDNSTR, type, linenum, dn );
		rc = LDAP_PARAM_ERROR;
	    }
	} else if ( expect_deleteoldrdn ) {
	    if ( strcasecmp( type, T_DELETEOLDRDNSTR ) == 0 ) {
		deleteoldrdn = ( *value == '0' ) ? 0 : 1;
		got_all = 1;
	    } else {
		fprintf( stderr, "%s: expecting \"%s:\" but saw \"%s:\" (line %d of entry %s)\n",
			prog, T_DELETEOLDRDNSTR, type, linenum, dn );
		rc = LDAP_PARAM_ERROR;
	    }
	} else if ( got_all ) {
	    fprintf( stderr,
		    "%s: extra lines at end (line %d of entry %s)\n",
		    prog, linenum, dn );
	    rc = LDAP_PARAM_ERROR;
	} else {
	    addmodifyop( &pmods, modop, type, value, vlen );
	}
    }

    if ( rc == 0 ) {
	if ( delete_entry ) {
	    rc = dodelete( dn );
	} else if ( newrdn != NULL ) {
	    rc = domodrdn( dn, newrdn, deleteoldrdn );
	} else {
	    rc = domodify( dn, pmods, new_entry );
	}

	if ( rc == LDAP_SUCCESS ) {
	    rc = 0;
	}
    }

    if ( dn != NULL ) {
	free( dn );
    }
    if ( newrdn != NULL ) {
	free( newrdn );
    }
    if ( pmods != NULL ) {
	freepmods( pmods );
    }

    return( rc );
}


static int
process_ldapmod_rec( char *rbuf )
{
    char	*line, *dn, *p, *q, *attr, *value;
    int		rc, linenum, modop;
    LDAPMod	**pmods;

    pmods = NULL;
    dn = NULL;
    linenum = 0;
    line = rbuf;
    rc = 0;

    while ( rc == 0 && rbuf != NULL && *rbuf != '\0' ) {
	++linenum;
	if (( p = strchr( rbuf, '\n' )) == NULL ) {
	    rbuf = NULL;
	} else {
	    if ( *(p-1) == '\\' ) {	/* lines ending in '\' are continued */
		memcpy( p - 1, p, strlen( p ) + 1 );
		rbuf = p;
		continue;
	    }
	    *p++ = '\0';
	    rbuf = p;
	}

	if ( dn == NULL ) {	/* first line contains DN */
	    if (( dn = strdup( line )) == NULL ) {
		perror( "strdup" );
		exit( 1 );
	    }
	} else {
	    if (( p = strchr( line, '=' )) == NULL ) {
		value = NULL;
		p = line + strlen( line );
	    } else {
		*p++ = '\0';
		value = p;
	    }

	    for ( attr = line;
		  *attr != '\0' && isspace( (unsigned char) *attr ); ++attr ) {
		;	/* skip attribute leading white space */
	    }

	    for ( q = p - 1; q > attr && isspace( (unsigned char) *q ); --q ) {
		*q = '\0';	/* remove attribute trailing white space */
	    }

	    if ( value != NULL ) {
		while ( isspace( (unsigned char) *value )) {
		    ++value;		/* skip value leading white space */
		}
		for ( q = value + strlen( value ) - 1; q > value &&
			isspace( (unsigned char) *q ); --q ) {
		    *q = '\0';	/* remove value trailing white space */
		}
		if ( *value == '\0' ) {
		    value = NULL;
		}

	    }

	    if ( value == NULL && new ) {
		fprintf( stderr, "%s: missing value on line %d (attr is %s)\n",
			prog, linenum, attr );
		rc = LDAP_PARAM_ERROR;
	    } else {
		 switch ( *attr ) {
		case '-':
		    modop = LDAP_MOD_DELETE;
		    ++attr;
		    break;
		case '+':
		    modop = LDAP_MOD_ADD;
		    ++attr;
		    break;
		default:
		    modop = replace ? LDAP_MOD_REPLACE : LDAP_MOD_ADD;
		}

		addmodifyop( &pmods, modop, attr, value,
			( value == NULL ) ? 0 : strlen( value ));
	    }
	}

	line = rbuf;
    }

    if ( rc == 0 ) {
	if ( dn == NULL ) {
	    rc = LDAP_PARAM_ERROR;
	} else if (( rc = domodify( dn, pmods, new )) == LDAP_SUCCESS ) {
	    rc = 0;
	}
    }

    if ( pmods != NULL ) {
	freepmods( pmods );
    }
    if ( dn != NULL ) {
	free( dn );
    }

    return( rc );
}


static void
addmodifyop( LDAPMod ***pmodsp, int modop, char *attr, char *value, int vlen )
{
    LDAPMod		**pmods;
    int			i, j;
    struct berval	*bvp;

    pmods = *pmodsp;
    modop |= LDAP_MOD_BVALUES;

    i = 0;
    if ( pmods != NULL ) {
	for ( ; pmods[ i ] != NULL; ++i ) {
	    if ( strcasecmp( pmods[ i ]->mod_type, attr ) == 0 &&
		    pmods[ i ]->mod_op == modop ) {
		break;
	    }
	}
    }

    if ( pmods == NULL || pmods[ i ] == NULL ) {
	if (( pmods = (LDAPMod **)safe_realloc( pmods, (i + 2) *
		sizeof( LDAPMod * ))) == NULL ) {
	    perror( "safe_realloc" );
	    exit( 1 );
	}
	*pmodsp = pmods;
	pmods[ i + 1 ] = NULL;
	if (( pmods[ i ] = (LDAPMod *)calloc( 1, sizeof( LDAPMod )))
		== NULL ) {
	    perror( "calloc" );
	    exit( 1 );
	}
	pmods[ i ]->mod_op = modop;
	if (( pmods[ i ]->mod_type = strdup( attr )) == NULL ) {
	    perror( "strdup" );
	    exit( 1 );
	}
    }

    if ( value != NULL ) {
	j = 0;
	if ( pmods[ i ]->mod_bvalues != NULL ) {
	    for ( ; pmods[ i ]->mod_bvalues[ j ] != NULL; ++j ) {
		;
	    }
	}
	if (( pmods[ i ]->mod_bvalues =
		(struct berval **)safe_realloc( pmods[ i ]->mod_bvalues,
		(j + 2) * sizeof( struct berval * ))) == NULL ) {
	    perror( "safe_realloc" );
	    exit( 1 );
	}
	pmods[ i ]->mod_bvalues[ j + 1 ] = NULL;
	if (( bvp = (struct berval *)malloc( sizeof( struct berval )))
		== NULL ) {
	    perror( "malloc" );
	    exit( 1 );
	}
	pmods[ i ]->mod_bvalues[ j ] = bvp;

	if ( valsfromfiles && *value == '/' ) {	/* get value from file */
	    if ( fromfile( value, bvp ) < 0 ) {
		exit( 1 );
	    }
	} else {
	    bvp->bv_len = vlen;
	    if (( bvp->bv_val = (char *)malloc( vlen + 1 )) == NULL ) {
		perror( "malloc" );
		exit( 1 );
	    }
	    memcpy( bvp->bv_val, value, vlen );
	    bvp->bv_val[ vlen ] = '\0';
	}
    }
}


static int
domodify( char *dn, LDAPMod **pmods, int newentry )
{
    int			i, j, k, notascii, op;
    struct berval	*bvp;

    if ( pmods == NULL ) {
	fprintf( stderr, "%s: no attributes to change or add (entry %s)\n",
		prog, dn );
	return( LDAP_PARAM_ERROR );
    }

    if ( verbose ) {
	for ( i = 0; pmods[ i ] != NULL; ++i ) {
	    op = pmods[ i ]->mod_op & ~LDAP_MOD_BVALUES;
	    printf( "%s %s:\n", op == LDAP_MOD_REPLACE ?
		    "replace" : op == LDAP_MOD_ADD ?
		    "add" : "delete", pmods[ i ]->mod_type );
	    if ( pmods[ i ]->mod_bvalues != NULL ) {
		for ( j = 0; pmods[ i ]->mod_bvalues[ j ] != NULL; ++j ) {
		    bvp = pmods[ i ]->mod_bvalues[ j ];
		    notascii = 0;
		    for ( k = 0; (unsigned long) k < bvp->bv_len; ++k ) {
			if ( !isascii( bvp->bv_val[ k ] )) {
			    notascii = 1;
			    break;
			}
		    }
		    if ( notascii ) {
			printf( "\tNOT ASCII (%ld bytes)\n", bvp->bv_len );
		    } else {
			printf( "\t%s\n", bvp->bv_val );
		    }
		}
	    }
	}
    }

    if ( newentry ) {
	printf( "%sadding new entry %s\n", not ? "!" : "", dn );
    } else {
	printf( "%smodifying entry %s\n", not ? "!" : "", dn );
    }

    if ( !not ) {
	/* We'll always try to modify the object in place. If this
	   fails, we'll do an incremental add.
	 */
        i = ldap_modify_s( ld, dn, pmods );

	if( i == LDAP_NO_SUCH_OBJECT ||
            i == LDAP_ALREADY_EXISTS ||
            i == LDAP_TYPE_OR_VALUE_EXISTS) {
	    i = ldap_inc_add_s( ld, dn, pmods );
	} 

	if ( i != LDAP_SUCCESS ) {
	    ldap_perror( ld, newentry ? "ldap_add" : "ldap_modify" );
	} else if ( verbose ) {
	    printf( "modify complete\n" );
	}
    } else {
	i = LDAP_SUCCESS;
    }

    putchar( '\n' );

    return( i );
}


static int
dodelete( char *dn )
{
    int	rc;

    printf( "%sdeleting entry %s\n", not ? "!" : "", dn );
    if ( !not ) {
	if (( rc = ldap_inc_delete_s( ld, dn )) != LDAP_SUCCESS ) {
	    ldap_perror( ld, "ldap_delete" );
	} else if ( verbose ) {
	    printf( "delete complete" );
	}
    } else {
	rc = LDAP_SUCCESS;
    }

    putchar( '\n' );

    return( rc );
}


static int
domodrdn( char *dn, char *newrdn, int deleteoldrdn )
{
    int	rc;

    if ( verbose ) {
	printf( "new RDN: %s (%skeep existing values)\n",
		newrdn, deleteoldrdn ? "do not " : "" );
    }

    printf( "%smodifying rdn of entry %s\n", not ? "!" : "", dn );
    if ( !not ) {
	if (( rc = ldap_modrdn2_s( ld, dn, newrdn, deleteoldrdn ))
		!= LDAP_SUCCESS ) {
	    ldap_perror( ld, "ldap_modrdn" );
	} else {
	    printf( "modrdn completed\n" );
	}
    } else {
	rc = LDAP_SUCCESS;
    }

    putchar( '\n' );

    return( rc );
}



static void
freepmods( LDAPMod **pmods )
{
    int	i;

    for ( i = 0; pmods[ i ] != NULL; ++i ) {
	if ( pmods[ i ]->mod_bvalues != NULL ) {
	    ber_bvecfree( pmods[ i ]->mod_bvalues );
	}
	if ( pmods[ i ]->mod_type != NULL ) {
	    free( pmods[ i ]->mod_type );
	}
	free( pmods[ i ] );
    }
    free( pmods );
}


static int
fromfile( char *path, struct berval *bv )
{
	FILE		*fp;
	long		rlen;
	int		eof;

	if (( fp = fopen( path, "r" )) == NULL ) {
	    	perror( path );
		return( -1 );
	}

	if ( fseek( fp, 0L, SEEK_END ) != 0 ) {
		perror( path );
		fclose( fp );
		return( -1 );
	}

	bv->bv_len = ftell( fp );

	if (( bv->bv_val = (char *)malloc( bv->bv_len )) == NULL ) {
		perror( "malloc" );
		fclose( fp );
		return( -1 );
	}

	if ( fseek( fp, 0L, SEEK_SET ) != 0 ) {
		perror( path );
		fclose( fp );
		return( -1 );
	}

	rlen = fread( bv->bv_val, 1, bv->bv_len, fp );
	eof = feof( fp );
	fclose( fp );

	if ( (unsigned long) rlen != bv->bv_len ) {
		perror( path );
		free( bv->bv_val );
		return( -1 );
	}

	return( bv->bv_len );
}


static char *
read_one_record( FILE *fp )
{
    int         len;
    char        *buf, line[ LDAPMOD_MAXLINE ];
    int		lcur, lmax;

    lcur = lmax = 0;
    buf = NULL;

    while (( fgets( line, sizeof(line), fp ) != NULL ) &&
            (( len = strlen( line )) > 1 )) {
        if ( lcur + len + 1 > lmax ) {
            lmax = LDAPMOD_MAXLINE
		    * (( lcur + len + 1 ) / LDAPMOD_MAXLINE + 1 );
	    if (( buf = (char *)safe_realloc( buf, lmax )) == NULL ) {
		perror( "safe_realloc" );
		exit( 1 );
	    }
        }
        strcpy( buf + lcur, line );
        lcur += len;
    }

    return( buf );
}

/*******************************************************************/
/*  The above is a direct copy of the ldapmodify.c code taken from */
/*  the OpendLDAP distibution.                                     */
/*  The rest of this file is stuff just to support the incremental */
/*  update of Globus objects                                       */
/*******************************************************************/
/*******************************************************************
*
* Copyright (c) 1997 USC/ISI 
* ID: mds-modify.c
* Creation Date: 7/97
* Revision: $Revision: 1.15 $
* Last Modification Date: 7/97
* Author: Steven Fitzgerald
*
* Purpose: Do perform incremental creation and deletion 
* of DIT objects. Place holders ar inserted to keep the 
* form of the DIT consistent until future operation can 
* insert/delete the appropriate objects.
*
*******************************************************************/

#define MATCH 0
#define GlobusStub_val "GlobusStub"


#define get_rdn_type(dest, source)                      \
    { char * end_rdn;                                   \
      end_rdn = strchr((dest), '='); (*end_rdn) = '\0'; \
      (source) = strdup((dest));                        \
      (*end_rdn) = '=';                                 \
    }


static int        add_GlobusStub_object( LDAP *ld, char *dn);
static LDAPMod ** build_GlobusStub_attrs(char *dn);
static char     * find_parent_dn (char *dn);
static void       adjust_modifyops( LDAPMod ** pmods, char *dn);



/********************************************************************/
/*  Procedure:  ldap_inc_add_s                                      */
/*  Purpose:    Performs an ldap_add_s, but on certain types of     */
/*              it tries to recover. Recover is performed by        */
/*              modifying the DIT structure to allow the new object */
/*              to be accepted                                      */
/*                                                                  */
/*  Conditions: LDAP_NO_SUCH_OBJECT                                 */
/*              The parent does not exist so it tries to create one */
/*                                                                  */
/*              LDAP_ALREADY_EXISTS                                 */
/*              The object might have been created via the recover  */
/*              process of LDAP_NO_SUCH_OBJECT.  Try to replace the */
/*              object instead of add                               */
/********************************************************************/
int
ldap_inc_add_s( LDAP *ld, char *dn, LDAPMod **attrs )
{
  int             msgid;
  LDAPMessage     *res;
  int              result;
  char            *parent_dn;

  if ( (msgid = ldap_add( ld, dn, attrs )) == -1 )
    return( ld->ld_errno );

  if ( ldap_result( ld, msgid, 1, (struct timeval *) NULL, &res ) == -1 )
    return( ld->ld_errno );

  result = ldap_result2error( ld, res, 1 );

  /* Based on the error value do some special processing */
  switch (result) { 
       case LDAP_NO_SUCH_OBJECT :
	 /* Assuming that at least one interior node is missing */

	 if ( verbose ) {
	   printf( "creating parent entry %s as a GlobusStub\n", dn);
	 }

	 parent_dn = find_parent_dn(dn);
	 result = add_GlobusStub_object(ld, parent_dn);

	 if (result == LDAP_SUCCESS) {
	   /* We were able to build the interior nodes  */
           if ( result != LDAP_SUCCESS ) {
              ldap_perror( ld, "ldap_inc_add_s" );
           } else if (verbose ) {
              printf( "creation of GlobusStub complete\n" );
           }

	   return ldap_add_s(ld, dn, attrs);

	 }
	 break;

       case LDAP_ALREADY_EXISTS : 
         /*  Check to see if its a GlobusStub  */
	 {
	     /* Replace the GlobusStub */
	     if ( verbose ) {
	       printf( "modifying GlobusStub\n" );
	     }

	     /* Go through the attrs structure and change    */
	     /* the modop value for object class and the rdn */
	     adjust_modifyops(attrs, dn); 

	     result = ldap_modify_s(ld, dn, attrs);

	     if (result == LDAP_SUCCESS) {
	       if ( verbose ) {
		 printf("  modification of GlobusStub complete\n" );
	       }
	     } else {
	       ldap_perror(ld, "ldap_inc_add_s");
	     }
	 }
	 break;

       default:
	 break;
    }

  ld->ld_errno = result;
  return( result );

}
	


/********************************************************************/
/*  Procedure:  ldap_inc_delete_s                                   */
/*  Purpose:    Performs an ldap_delete_s, but on non-leaf nodes    */
/*              it tries to recover. Recover is performed by        */
/*              inserting a GlobusStub in its place.                */
/*                                                                  */
/********************************************************************/
int 
ldap_inc_delete_s(LDAP *ld, char * dn) {

  int       result = LDAP_SUCCESS;
  char     *rdn    = NULL;
  LDAPMod **pmods  = NULL;

  result = ldap_delete_s( ld, dn );

  if (result ==  LDAP_NOT_ALLOWED_ON_NONLEAF ) {
    /* Lets put in a GlobusStub */
    char        *filter  = "objectclass=*";
    int         types_attrs = 0;
    LDAPMessage *res, *entry;
    BerElement  *berptr;
    char        *attr_type;
	
    get_rdn_type(dn, rdn);

    result = ldap_search_s(ld, dn, LDAP_SCOPE_BASE, 
			   filter, NULL, types_attrs, &res);
    if (result != LDAP_SUCCESS) {
      ldap_perror(ld, "ldap_inc_add_s");
    }

    if (verbose) {
      printf( "replacing entry \"%s\" with as a GlobusStub\n", dn);
    }

    entry = ldap_first_entry(ld, res);  /* There can only be one result */

    /* build an an attribut list that                    */
    /*  1. replace objectClass with top and GlobusStub   */
    /*  2. deletes all non rdn values                    */

    addmodifyop(&pmods, LDAP_MOD_REPLACE, "objectclass", "top", 3);
    addmodifyop(&pmods, LDAP_MOD_REPLACE, "objectclass", 
	    GlobusStub_val, strlen(GlobusStub_val));

    attr_type = ldap_first_attribute(ld, entry, &berptr);
    while (attr_type != NULL)  {
      if ((strcasecmp(attr_type, "objectclass") != MATCH) &&
          (strcasecmp(attr_type, rdn) != MATCH)) {
	addmodifyop(&pmods, LDAP_MOD_DELETE, attr_type, NULL, 0);
      }

      attr_type = ldap_next_attribute(ld, entry, berptr);
    } 

    result = ldap_modify_s(ld, dn, pmods);

    if ((result == LDAP_SUCCESS) && verbose) {
      printf( "replace ment of GlobusStub complete\n" );
    }

    /* Note that ldap_next_attribute frees up berptr */
    ldap_msgfree(res);

    free (rdn);
    freepmods(pmods);
  }
  
  return result;
}




/********************************************************************/
/*  Procedure: add_GlobusStub                                       */
/*  Purpose:   Given a DN, it                                       */
/*             1) builds a stub object                              */
/*             2) incrementally adds the stub object                */
/*  Bugs:      It stops processing when the RDN is an orgainization */
/*             or an organizational unit.  It should probable stop  */
/*             when it hits the DEFAULT_BASE.                       */
/********************************************************************/
static int 
add_GlobusStub_object( LDAP *ld, char *dn) {

  char    ** rdns;
  LDAPMod ** attrs;
  int        ret_value;

  /* if the parent dn starts with "ou=", "o=", or "c=",
     we have attempted to go to far, return error        */

  if ( (strncmp("ou=", dn, 3) == MATCH) ||
       (strncmp("o=",  dn, 2) == MATCH) ||
       (strncmp("c=",  dn, 2) == MATCH) ){
    return -1;
  }

  attrs = build_GlobusStub_attrs(dn);
  ret_value = ldap_inc_add_s(ld, dn, attrs);
   
  if ( attrs != NULL) {
    freepmods( attrs );
  }

  return ret_value;
}


/********************************************************************/
/*  Procedure: build_GlobusStub_attrs                               */
/*  Purpose:   Builds a set of attributes for a GlobusStub.  It     */
/*             ensures the the appropriate RDN is specified.        */
/*                                                                  */
/********************************************************************/
static LDAPMod ** 
build_GlobusStub_attrs(char *dn){

  LDAPMod ** pmods   = NULL;
  char     * stub    = GlobusStub_val;
  char    ** rdns;
  char     * rdn_type;
  char     * rdn_value;

  rdns   = ldap_explode_dn(dn, 0);   

  /* Find the "=", set it to null, and step over it */
  rdn_type     = rdns[0];
  rdn_value    = strchr(rdns[0], '=');

  if (!rdn_value) {
      ldap_perror(ld, " build_GlobusStub_attrs() got a bogus RDN with no =; this one won't be handled");
      return NULL;
  }

  (*rdn_value) = '\0'; 
  rdn_value++;  
  
  addmodifyop( &pmods,  LDAP_MOD_ADD, 
	       "objectClass",  "top", strlen("top"));
  addmodifyop( &pmods,  LDAP_MOD_ADD, 
	       "objectClass",  stub, strlen(stub));
  addmodifyop( &pmods,  LDAP_MOD_ADD, 
	       rdn_type, rdn_value, strlen(rdn_value));

  rdn_value--;
  (*rdn_value) = '=';

  ldap_value_free(rdns);

  return pmods;
}



/********************************************************************/
/*  Procedure: find_parent_dn                                       */
/*  Purpose:   Given a DN, it strips off the RDN.  Thus providing   */
/*             the DN of its parent within the DIT.                 */
/*  Bugs:      1) Does not handle quoted strings                    */
/*             2) Does not handle nulls (\0)                        */
/********************************************************************/
static char * 
find_parent_dn (char *dn)
{
  /* Another way to do this is to use the ldap_explode_dn routine. */
  /* However, no ldap_implode_dn routine exists */
  char * c_ptr;

  do {
    c_ptr = strpbrk(dn, ",;\"\\"); /* the special cases */
    if ( (*c_ptr) == '"' ) {
      ldap_perror(ld, "Unable to handle quotes");
    }
    if ( (*c_ptr) == '\\' ) {
        c_ptr ++;
        if ( (*c_ptr) == '\0' ) {
	   ldap_perror(ld, "Unable to handle nulls (\\0)");
	}
    }
  } while ( (*c_ptr) != ',' && (*c_ptr) != ';' );
  c_ptr ++;  /* skip over the ',' or ';' */
  
  while ( isspace( *c_ptr) )
    ++c_ptr;
  
  return c_ptr;
}



/********************************************************************/
/*  Procedure: adjust_modifyops                                     */
/*  Purpose:   For a given modification structure, it modifies the  */
/*             the operation value to be                            */
/*                REPLACE for  objectclass and rdn,                 */
/*                ADD     for  all others                           */
/*             This is necessary when we attempt to update a        */
/*             GlobusStub.                                          */
/********************************************************************/

static void  adjust_modifyops( LDAPMod ** pmods, char *dn){

  int  i;
  int  modop;
  char *rdn;

  modop = LDAP_MOD_REPLACE | LDAP_MOD_BVALUES;

  get_rdn_type(dn, rdn);

  if ( pmods != NULL ) {
    for ( i=0 ; pmods[ i ] != NULL; ++i) {
      if ( strcasecmp( pmods[ i ]->mod_type, "objectclass" ) == MATCH ) {
	pmods[ i ]->mod_op  = modop;
	continue;
      }
      if ( strcasecmp( pmods[ i ]->mod_type, rdn ) == MATCH ) {
	pmods[ i ]->mod_op  = modop;
	continue;
      }

      /* Make sure the subsequent call to ldap_modify has good values */
      pmods[ i ]->mod_op = LDAP_MOD_ADD | LDAP_MOD_BVALUES;
    }
  }

  free(rdn);

  return;
}

void set_ld_timeout(LDAP  *ld, int timeout_val, char *env_string)
{

  if(timeout_val > 0) {
    ld->ld_timeout=timeout_val;
    } else { /* check the environment variable */
      char *tmp=getenv(env_string);
      int tmp_int=0;
      if(tmp) tmp_int=atoi(tmp);
      if(tmp_int>0) ld->ld_timeout=tmp_int;
  }
}

char *check_for_referral(char *sptr, char *sstr)
{
  char *some_string;
  if(sptr==NULL)
    some_string=sstr;
    else some_string=sptr;

  sptr=NULL;
  if(strlen(some_string) > (LDAP_LDAP_REF_LEN+LDAP_REF_STR_LEN+1)) {
    sptr=strstr(some_string,LDAP_REF_STR);
    if(sptr) {
      sptr=strstr(some_string,LDAP_LDAP_REF_STR);
      sptr=sptr+LDAP_LDAP_REF_LEN;
      } else sptr=NULL;
  }

  if(sptr) {
    return sptr;
  } else return NULL;
}


static int is_num(char i)
{
  switch (i){
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9' : return 1;
    default: break;
  }
  return 0;
}

int get_port(char *s)
{
  int port=0;
  char c;
  while(is_num(*s)) {
    c = *s;
    port=port*10+atoi(&c);
    s++;
  }
  return port;
}




