/*
************************************************************************
**  GUSTO Heartbeat Monitor Local Monitor (HBMLM) program             **
************************************************************************
*/
/* TODO:
 *   - convert all lists to globus_list
 *   - put setup_tcp/udp_port code into hbm_common
 *   - Add number of datacollectors to the checkpoint information
 *   - Add more logging of events, i.e. when reg/unreg messages get
 *     processed, commits, cancel msgs, possibly add a flag to log all
 *     heartbeats...for debugging
 *   - Whenever status changes to shutdown/unregister for any of the
 *     clients, send a HB messages so the information propagates to the
 *     datacollector.
 *   - double check all return values
 */

#define HBM_LM_C


/*
#define DEBUG_MSGS
#define DEBUG_STDERR
*/
#ifdef DEBUG_MSGS
#define DEBUGF1(x)      globus_libc_fprintf( hbmlm_log_file_fptr,(x))
#define DEBUGF2(x,y)    globus_libc_fprintf( hbmlm_log_file_fptr,(x),(y))
#define DEBUGF3(x,y,z)  globus_libc_fprintf( hbmlm_log_file_fptr,(x),(y),(z))
#else
#define DEBUGF1(x)
#define DEBUGF2(x,y)
#define DEBUGF3(x,y,z)
#endif

#include "globus_args.h"
#include "globus_common.h"
#include "globus_libc.h"
#include "globus_strptime.h"

#include "globus_hbm_codes.h"
#include "globus_hbm_defaults.h"

#include "hbm.h"

#if !defined(PID_MAX) && defined(MAXPID)
#   define PID_MAX MAXPID
#endif

/*
** #define HBM_EXPERIMENT  is included in hbm.h
*/

/*
**  Argument fields.
*/
const char* usage_short =
    "globus_hbm_localmonitor -chkpt <path> [ -log <path> ] [ -ps <path> ] "
    "[ -rptinterval # ] -status <path>\n\n";

const char* usage_long =
    "\n"
    "globus_hbm_localmonitor options\n"
    "    Where options are:\n"
    "        -chkpt <path>    :  <path/filename> base for checkpoint file\n"
    "                            if a directory, then the filename used "
    "will be\n"
    "                                globus_hbm_localmonitor.checkpoint\n"
    "        -log   <path>    :  <path/filename> base for checkpoint file\n"
    "                            if a directory, then the filename used "
    "will be\n"
    "                                globus_hbm_localmonitor.log\n"
    "      [ -ps    <path> ]  :  <path/filename> for ps program\n"
    "                            if a directory, then the filename used "
    "will be\n"
    "                                ps\n"
    "      [ -rptinterval # ] :  default report interval\n"
    "        -status <path>   :  <path/filename> base for status file\n"
    "                            if a directory, then the filename used "
    "will be\n"
    "                                globus_hbm_localmonitor.status\n\n";

#define ARGS_ID_CHKPT           1
#define ARGS_ID_LOG             2
#define ARGS_ID_PS              3
#define ARGS_ID_RPTINTERVAL     4
#define ARGS_ID_STATUS          5
#define ARGS_ID_MAX             5

static char*    args_names_chkpt[]      = { "-chkpt",
                                            GLOBUS_NULL };
static char*    args_names_log[]        = { "-log",
                                            GLOBUS_NULL };
static char*    args_names_ps[]         = { "-ps",
                                            GLOBUS_NULL };
static char*    args_names_rptinterval[] =
                                          { "-rptinterval",
                                            GLOBUS_NULL };
static char*    args_names_status[]     = { "-status",
                                            GLOBUS_NULL };

static globus_args_valid_predicate_t
                args_validate_function_rptinterval[] =
                                          { globus_validate_int };

static globus_validate_int_parms_t
                args_validate_int_ge0_parms =
                                          { GLOBUS_VALIDATE_INT_MIN,
                                            0,
                                            0 };

static void*    args_validate_parms_rptinterval[] =
                          { (void *) &args_validate_int_ge0_parms };

static globus_args_option_descriptor_t
                args_option_list[] =      { { ARGS_ID_CHKPT,
                                              args_names_chkpt,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_LOG,
                                              args_names_log,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_PS,
                                              args_names_ps,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_RPTINTERVAL,
                                              args_names_rptinterval,
                                              1,
                                              args_validate_function_rptinterval,
                                              args_validate_parms_rptinterval },
                                            { ARGS_ID_STATUS,
                                              args_names_status,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL } };

globus_list_t*  args_options_found_list;
globus_args_option_instance_t*
                args_option;

globus_bool_t   arg_got_option_chkpt    = GLOBUS_FALSE;
globus_bool_t   arg_got_option_log      = GLOBUS_FALSE;
globus_bool_t   arg_got_option_ps       = GLOBUS_FALSE;
globus_bool_t   arg_got_option_rptinterval =
                                          GLOBUS_FALSE;
globus_bool_t   arg_got_option_status   = GLOBUS_FALSE;

char*           arg_option_fname_chkpt_str =
                                          GLOBUS_NULL;
char*           arg_option_fname_log_str =
                                          GLOBUS_NULL;
char*           arg_option_fname_status_str =
                                          GLOBUS_NULL;
char*           arg_option_ps_program_name_str =
                                          GLOBUS_NULL;

int             arg_option_rptinterval  = HBMLM_DEFAULT_RPT_INTERVAL;


#if !defined(MAXPATHLEN)
#define MAXPATHLEN PATH_MAX
#endif

char*           hbmlm_log_fname_str     = GLOBUS_NULL;
char*           hbmlm_ps_command_str    = GLOBUS_NULL;
char*           hbmlm_ps_program_str    = GLOBUS_NULL;
char*           hbmlm_status_fname_str  = GLOBUS_NULL;

static FILE*    hbmlm_log_file_fptr     = GLOBUS_NULL;


struct hbmlm_lm_fields_s {
    struct in_addr                      LM_hostIPNum;
    char*                               LM_hostIPNum_str;
    int                                 LM_hostNameLen;
    char*                               LM_hostName_str;
    char*                               LM_procNameFromPS_str;
    u_short                             LM_portNumReg;
    u_short                             LM_portNumRpt;
    struct sockaddr_in                  LM_addr_reg;
    struct sockaddr_in                  LM_addr_rpt;
    int                                 LM_defaultReportInterval;
    int                                 LM_ClientsCt;
    char*                               LM_ckptFileName_str;
    char*                               LM_ckptFileNameWk_str;
    UTCtime                             LM_ckptTime;
};
typedef struct hbmlm_lm_fields_s hbmlm_lm_fields_t;


/* global variables and initialization */

static globus_callback_handle_t
                globus_hbmlm_callback_handle;


static hbmlm_cl_list_head_t
                hbmlm_cl_list;

static hbmlm_cl_list_head_t
                shadow_cl_list;

static hbmlm_lm_fields_t
                hbmlm_lm_info;

struct timeval  time_curr               = { 0, 0 };
struct timeval  time_to_monitor_write;

static char*    hbmlm_proc_name_str     = GLOBUS_NULL;

static int      quit_now                = 0;

static int      sock_fd_reg;
static int                      sock_fd_rpt;

static long     time_conv_diff;

#ifdef HBM_EXPERIMENT
    int                 retcd_time;

    struct timeval      time_monitoring_duration,
                        time_monitoring_start,
                        time_monitoring_stop;

    struct rusage       rusage_children,
                        rusage_hbm_before_processing,
                        rusage_hbm_after_processing,
                        rusage_hbm_used;

    struct timeval      time_hbm_before_total,
                        time_hbm_after_total,
                        time_hbm_used_total;

    int                 getrusage_retcd;
#endif


/* local function declarations */

static hbmlm_cl_fields_t*
globus_l_hbmlm_add_cl_entry(
                hbmlm_cl_list_head_t*   cl_list,
                hbmlm_cl_fields_t*      cl_fields,
                hbmlm_cldc_fields_t*    cldc_fields,
                globus_bool_t*          duplicate_client );

static globus_bool_t
globus_l_hbmlm_ckpt_restore_extract_lm_fields_from_ckpt_string(
                char*,
                int* );

static globus_bool_t
globus_l_hbmlm_ckpt_restore_validate_hdr_fields_from_ckpt_string(
                char* );

static int
globus_l_hbmlm_commit_shadow_cl_list(
                void );

static int
globus_l_hbmlm_create_write_status_file(
                char* );

static void
globus_l_hbmlm_delete_and_free_cl_list(
                hbmlm_cl_list_head_t* );

static void
globus_l_hbmlm_delete_cl_entries_shutdown_procs(
                void );

static int
globus_l_hbmlm_extract_and_process_cl_reg_message(
                char*,
                int,
                char* );

static int
globus_l_hbmlm_extract_and_process_cl_unreg_message(
                char*,
                int );

static int
globus_l_hbmlm_extract_cl_fields_from_ckpt_string(
                char*                   string,
                hbmlm_cl_fields_t*      cl_fields );

static int
globus_l_hbmlm_extract_cldc_fields_from_ckpt_string(
                char*                   string,
                hbmlm_cl_fields_t*      cl_fields,
                hbmlm_cldc_fields_t*    cldc_fields );

static hbmlm_cldc_fields_t*
globus_l_hbmlm_find_cldc_fields_entry(
                hbmlm_cl_fields_t*,
                struct sockaddr_in );

static void
globus_l_hbmlm_format_send_rptmsg(
                hbmlm_cl_fields_t*,
                hbmlm_cldc_fields_t* );

static void
globus_l_hbmlm_free_cl_fields(
                hbmlm_cl_fields_t* );

static void
globus_l_hbmlm_free_cldc_fields(
                hbmlm_cldc_fields_t* );

static int
globus_l_hbmlm_get_arguments(
                globus_list_t* );

static UTCtime
globus_l_hbmlm_get_boot_time();

static int
globus_l_hbmlm_get_LM_machine_info( void );

static globus_bool_t
globus_l_hbmlm_poll(
                globus_abstime_t*       time_stop,
                void*                   user_args );

static void
globus_l_hbmlm_process_reg_unreg_msg(
                int );

static void
globus_l_hbmlm_process_rpt_msg(
                int );

static void
globus_l_hbmlm_read_ckpointfile( void );

static int
globus_l_hbmlm_send_ack(
                int                     read_fd,
                int                     ret_code );

static void
globus_l_hbmlm_send_rpt_msgs( void );

static int
globus_l_hbmlm_set_up_tcp_port(
                int*,
                struct sockaddr_in* );

static int
globus_l_hbmlm_set_up_udp_port(
                int*,
                struct sockaddr_in* );

RETSIGTYPE
globus_l_hbmlm_shutdown_quit(
                int );

static void
globus_l_hbmlm_write_ckpointfile(
                void );

/**********************************************************************
 *
 *  Module activation structure
 *
 **********************************************************************
*/
extern globus_module_descriptor_t       globus_i_hbm_localmonitor_module;

#define GLOBUS_HBM_LOCALMONITOR_MODULE (&globus_i_hbm_localmonitor_module)

/*
************************************************************************
**                                                                    **
**  Procedure Definitions                                             **
**                                                                    **
************************************************************************
*/

static hbmlm_cl_fields_t*
globus_l_hbmlm_add_cl_entry(
                hbmlm_cl_list_head_t*   cl_list_ptr,
                hbmlm_cl_fields_t*      cl_fields_new_ptr,
                hbmlm_cldc_fields_t*    cldc_fields_new_ptr,
                globus_bool_t*          duplicate )
{
    hbmlm_cl_fields_t*
                cl_fields_ptr           = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields_ptr         = GLOBUS_NULL;

    if( duplicate != GLOBUS_NULL )
    {
        *duplicate = GLOBUS_FALSE;
    }
    if(( cl_fields_ptr = globus_i_hbm_find_cl_fields_entry(
                                        cl_list_ptr,
                                        cl_fields_new_ptr->CL_procPID )) ==
                                GLOBUS_NULL )
    {
/*      This is a new client -- add to the list.  */
        for( cl_fields_ptr = cl_list_ptr->head;
             cl_fields_ptr;
             cl_fields_ptr = cl_fields_ptr->next )
        {
            if( cl_fields_ptr->CL_procPID > cl_fields_new_ptr->CL_procPID )
            {
                break;
            }
        }
        ListInsertBefore( cl_list_ptr, cl_fields_new_ptr, cl_fields_ptr );
        cl_fields_ptr = cl_fields_new_ptr;
    }
    else
    {
        if( duplicate != GLOBUS_NULL )
        {
            *duplicate = GLOBUS_TRUE;
        }
        if( cl_fields_new_ptr != cl_fields_ptr )
        {
/*
**          The client entry already exists and
**              is distinct from the one passed in --
**                  Update the old record with the new values (if different).
*/
            DEBUGF2(    "\n%s:  Registration message received for "
                        "registered process",
                        hbmlm_proc_name_str );
            DEBUGF2(    "\n%s:    Original entry updated\n",
                        hbmlm_proc_name_str );

            if( strcmp( cl_fields_ptr->CL_procName_str,
                          cl_fields_new_ptr->CL_procName_str ))
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Name change in registration message for "
                        "registered process\n"
                        "    Switching to new name\n"
                        "        Old name:  %s\n"
                        "        New name:  %s\n\n",
                        hbmlm_proc_name_str,
                        cl_fields_ptr->CL_procName_str,
                        cl_fields_new_ptr->CL_procName_str );

                globus_free(
                        cl_fields_ptr->CL_procName_str );
                cl_fields_ptr->CL_procName_str =
                                        cl_fields_new_ptr->CL_procName_str;
                cl_fields_new_ptr->CL_procName_str = GLOBUS_NULL;
            }
            cl_fields_ptr->CL_procStatus =
                                        cl_fields_new_ptr->CL_procStatus;
            cl_fields_ptr->CL_blockedTime =
                                        cl_fields_new_ptr->CL_blockedTime;
            cl_fields_ptr->CL_cpuSecs = cl_fields_new_ptr->CL_cpuSecs;
            cl_fields_ptr->CL_updated = cl_fields_new_ptr->CL_updated;

            globus_l_hbmlm_free_cl_fields( cl_fields_new_ptr );
            cl_fields_new_ptr = GLOBUS_NULL;
        }
    }

/*
**  Client has been added if it didn't exist,
**  and cl_fields_ptr points to the (new or old ) entry in the list.
*/
    if( cldc_fields_new_ptr != GLOBUS_NULL )
    {
/*
**      This add routine is used by restore from checkpoint file.
**      Need to make this check to ensure that all updates were
**      appropriately performed.
*/
        if(   ( cldc_fields_new_ptr->CL_shutdownType == 0 )
           && (   ( cl_fields_ptr->CL_procStatus ==
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
               || ( cl_fields_ptr->CL_procStatus ==
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL )
               || ( cl_fields_ptr->CL_procStatus ==
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED )))
        {
            cldc_fields_new_ptr->CL_shutdownType =
                                        cl_fields_ptr->CL_procStatus;
            cldc_fields_new_ptr->CL_shutdownTime = time_curr.tv_sec;
            cldc_fields_new_ptr->CL_shutdownMsgCt = 0;
        }

        /* look for data collector match for this PID */
        cldc_fields_ptr = globus_l_hbmlm_find_cldc_fields_entry(
                                        cl_fields_ptr,
                                        cldc_fields_new_ptr->CL_DCaddrRpt );
        if( cldc_fields_ptr != GLOBUS_NULL )
        {
            /* This data collector is already registered */

            /* This re-registration may be for a shutdown process.
             * We need to copy the new status/shutdown information
             * (to reset it. )
             */
            cldc_fields_ptr->CL_shutdownType =
                                        cldc_fields_new_ptr->CL_shutdownType;
            cldc_fields_ptr->CL_shutdownMsgCt =
                                        cldc_fields_new_ptr->CL_shutdownMsgCt;
            cldc_fields_ptr->CL_shutdownTime =
                                        cldc_fields_new_ptr->CL_shutdownTime;

            /*
            ** New registration should *always* generate increase
            ** in the message string number, even if the string
            ** hasn't changed.
            ** (User may depend on it to generate registration event
            ** at the data collector ).
            */
            cldc_fields_ptr->CL_DCMsgNum++;

            globus_hbm_free( cldc_fields_ptr->CL_DCMsgString );

            cldc_fields_ptr->CL_DCMsgString =
                                cldc_fields_new_ptr->CL_DCMsgString;
            cldc_fields_new_ptr->CL_DCMsgString = GLOBUS_NULL;

            globus_l_hbmlm_free_cldc_fields( cldc_fields_new_ptr );
            cldc_fields_new_ptr = GLOBUS_NULL;
        }
        else /* add the new datacollector for this PID */
        {
            for ( cldc_fields_ptr = cl_fields_ptr->CL_dclist.head;
                  cldc_fields_ptr;
                  cldc_fields_ptr = cldc_fields_ptr->next )
            {
                if( strcmp( cldc_fields_ptr->CL_DCHostIPNum_str,
                            cldc_fields_new_ptr->CL_DCHostIPNum_str ) > 0 )
                    break;
            }
            ListInsertBefore( &(cl_fields_ptr->CL_dclist ),
                              cldc_fields_new_ptr,
                              cldc_fields_ptr );
        }
    }

    return cl_fields_ptr;
}


static globus_bool_t
globus_l_hbmlm_ckpt_restore_extract_lm_fields_from_ckpt_string(
                char*                   buf_ptr,
                int*                    cl_entry_ct_ptr )
{
    char*       token                   = GLOBUS_NULL;
    int         port_tmp                = 0;

/*
**  LM_hostIPNum
*/
    token = strtok( buf_ptr, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        Invalid Checkpoint file.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    if( strcmp( token, hbmlm_lm_info.LM_hostIPNum_str ) != 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        LM_hostIPNum for different host, "
                        "file skipped.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*
**  LM_hostName
*/
    token = strtok( GLOBUS_NULL, ";" );
    if(   ( token == GLOBUS_NULL )
       || ( strcmp( token, hbmlm_lm_info.LM_hostName_str ) != 0 ))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        Invalid Checkpoint file.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*
**  LM_portNumReg
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        HBMLM checkpoint data incomplete, "
                        "file skipped.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    port_tmp = atoi( token );
    if( port_tmp <= 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        HBMLM checkpoint data invalid, "
                        "file skipped.\n"
                        "        Registration port <= 0: [%d]\n\n",
                        hbmlm_proc_name_str,
                        port_tmp);

        return GLOBUS_FAILURE;
    }
    hbmlm_lm_info.LM_portNumReg = (u_short) port_tmp;

/*
**  LM_portNumRpt (port number for sending heartbeat reports to DCs)
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        HBMLM checkpoint data incomplete, "
                        "file skipped.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    port_tmp = atoi( token );
    if( port_tmp <= 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [07] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        HBMLM checkpoint data invalid, "
                        "file skipped.\n"
                        "        Heartbeat port <= 0: [%d]\n\n",
                        hbmlm_proc_name_str,
                        port_tmp);

        return GLOBUS_FAILURE;
    }
    hbmlm_lm_info.LM_portNumRpt = (u_short) port_tmp;

/*
**  LM_defaultReportInterval
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [08] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        HBMLM checkpoint data incomplete, "
                        "file skipped.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*
**  LM_ClientsCt (number of client processes )
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [09] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        Corrupt Checkpoint File.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    *cl_entry_ct_ptr = atoi( token );
    if( *cl_entry_ct_ptr < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [10] in "
                        "globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string():\n"
                        "        Warning -- client count < 0, using 0.\n\n",
                        hbmlm_proc_name_str );

        *cl_entry_ct_ptr = 0;
    }

    return GLOBUS_SUCCESS;
}


static globus_bool_t
globus_l_hbmlm_ckpt_restore_validate_hdr_fields_from_ckpt_string(
                char*                   buf_ptr )
{
    char*       token                   = GLOBUS_NULL;

    UTCtime     boot_time,
                checkpoint_time;

    token = strtok( buf_ptr, "\n" );
    checkpoint_time = globus_i_hbm_convert_string_to_UTCtime(
                        token,
                        time_conv_diff );
    if( checkpoint_time == 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_ckpt_restore_validate_"
                        "hdr_fields_from_ckpt_string():\n"
                        "        Invalid HBMLM Checkpoint file, "
                        "file skipped.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    boot_time = globus_l_hbmlm_get_boot_time();
    if( boot_time > checkpoint_time )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_ckpt_restore_validate_"
                        "hdr_fields_from_ckpt_string():\n"
                        "        HBMLM Checkpoint file is from before "
                        "last reboot, file skipped.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    else if( boot_time == 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_ckpt_restore_validate_"
                        "hdr_fields_from_ckpt_string():\n"
                        "        Error getting boot time.\n"
                        "        Restoring from checkpoint file anyway.\n\n",
                        hbmlm_proc_name_str );
    }

    return GLOBUS_SUCCESS;
}


static int
globus_l_hbmlm_commit_shadow_cl_list()
{
    hbmlm_cl_fields_t*      cl_fields_ptr = GLOBUS_NULL;

    hbmlm_cldc_list_head_t  cl_dclist_wk;
    hbmlm_cldc_fields_t*    cldc_fields_ptr = GLOBUS_NULL;

    globus_bool_t           duplicate_client;
    int                     rc;

    rc = GLOBUS_SUCCESS;

    while( shadow_cl_list.head != GLOBUS_NULL )
    {
        cl_fields_ptr = shadow_cl_list.head;
        ListRemove(&shadow_cl_list, cl_fields_ptr );

/*
**      Ensure that we have DC entries for the new client.
*/
        if( cl_fields_ptr->CL_dclist.head == GLOBUS_NULL )
        {
            rc = GLOBUS_FAILURE;
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_commit_shadow_cl_list():\n"
                        "        hbmlm_registration commit FAILURE for "
                        "PID=%u\n\n",
                        hbmlm_proc_name_str,
                        cl_fields_ptr->CL_procPID );

            globus_l_hbmlm_free_cl_fields( cl_fields_ptr );
        }
        else
        {
            cl_dclist_wk.head = cl_fields_ptr->CL_dclist.head;
            cl_dclist_wk.tail = cl_fields_ptr->CL_dclist.tail;
            cl_fields_ptr->CL_dclist.head = cl_fields_ptr->CL_dclist.tail =
                                        GLOBUS_NULL;
            while( cl_dclist_wk.head != GLOBUS_NULL )
            {
                cldc_fields_ptr = cl_dclist_wk.head;
                ListRemove( &cl_dclist_wk, cldc_fields_ptr );

                cl_fields_ptr = globus_l_hbmlm_add_cl_entry(
                                        &hbmlm_cl_list,
                                        cl_fields_ptr,
                                        cldc_fields_ptr,
                                        &duplicate_client );

                if( duplicate_client == GLOBUS_FALSE )
                {
                    hbmlm_lm_info.LM_ClientsCt++;
                }
            }
        }
    }

    return rc;
}


static int
globus_l_hbmlm_create_write_status_file(
                char*                   status_file_name_ptr )
{
    int         status_fd;
    char*       status_file_name_wk_ptr = GLOBUS_NULL;
    char        status_buff[64];

    int         length = 0;

/*
**  First check the status file name.
*/
    if( status_file_name_ptr == GLOBUS_NULL )
    {
        return GLOBUS_FAILURE;
    }
    if( *status_file_name_ptr == '\0' )
    {
        return GLOBUS_FAILURE;
    }

/*
**  Get rid of the old file if it exists.
*/
    unlink( status_file_name_ptr );

/*
**  Want to make sure that no process reads the incomplete file,
**  so we will use procedure similar to that for checkpoint files.
**
**  First get a work file name for creating the file.
**
**  Then create the file under the work file name.
**
**  Then link the real name to the file and unlink the work name.
*/
    if(( status_file_name_wk_ptr =
                        globus_malloc(
                                        strlen( status_file_name_ptr ) +
                                        4 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    globus_libc_sprintf(
                        status_file_name_wk_ptr,
                        "%s_wk",
                        status_file_name_ptr );

/*
**  creat the file and write the status information to it:
**      pid
**      process name
**      register/unregister port.
*/
    if(( status_fd = creat( status_file_name_wk_ptr, 0644 )) == -1 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );
        globus_free(    status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }

/*
**  Now write the data to the status file.
*/

    if( write(          status_fd,
                        "LocalMonitorPID:  ",
                        17 ) != 17 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );

        close( status_fd );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }
    globus_libc_sprintf(
                        status_buff,
                        "%d\n",
                        (int) getpid());
    length = strlen( status_buff );
    if( write(          status_fd,
                        status_buff,
                        length ) != length )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );

        close( status_fd );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }

    if( write(          status_fd,
                        "LocalMonitorProcName:  ",
                        22 ) != 22 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );

        close( status_fd );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }
    length = strlen( hbmlm_lm_info.LM_procNameFromPS_str );
    if( write(          status_fd,
                        hbmlm_lm_info.LM_procNameFromPS_str,
                        length ) != length )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );

        close( status_fd );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }

    if( write(          status_fd,
                        "\nLocalMonitorPortNumReg:  ",
                        25 ) != 25 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [07] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );

        close( status_fd );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }
    globus_libc_sprintf(
                        status_buff,
                        "%hu\n",
                        hbmlm_lm_info.LM_portNumReg );
    length = strlen( status_buff );
    if( write(          status_fd,
                        status_buff,
                        length ) != length )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [08] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );

        close( status_fd );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }

    if( write(          status_fd,
                        "LocalMonitorPSCommand:  ",
                        23 ) != 23 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [09] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );

        close( status_fd );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }
    length = strlen( hbmlm_ps_command_str );
    if( write(          status_fd,
                        hbmlm_ps_command_str,
                        length ) != length )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [10] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );

        close( status_fd );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }
    if( write(          status_fd,
                        "\n",
                        1 ) != 1 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [09] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error creating status work file \"%s\","
                        "aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_wk_ptr );

        close( status_fd );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }


    close( status_fd );

    if( link(   status_file_name_wk_ptr,
                status_file_name_ptr ) != 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [08] in "
                        "globus_l_hbmlm_create_write_status_file():\n"
                        "        Error linking status file name \"%s\"\n"
                        "                   to status work file \"%s\"\n"
                        "        Aborting.\n\n",
                        hbmlm_proc_name_str,
                        status_file_name_ptr,
                        status_file_name_wk_ptr );
        unlink( status_file_name_wk_ptr );
        globus_free( status_file_name_wk_ptr );

        return GLOBUS_FAILURE;
    }

    unlink( status_file_name_wk_ptr );

    globus_free( status_file_name_wk_ptr );

    return GLOBUS_SUCCESS;
}


static void
globus_l_hbmlm_delete_and_free_cl_list(
                hbmlm_cl_list_head_t*   cl_list_ptr )
{
    hbmlm_cl_fields_t*      cl_fields_ptr = GLOBUS_NULL;

    while( cl_list_ptr->head != GLOBUS_NULL )
    {
        cl_fields_ptr = cl_list_ptr->head;
        ListRemove( cl_list_ptr, cl_fields_ptr );
        globus_l_hbmlm_free_cl_fields( cl_fields_ptr );
    }
    cl_list_ptr->head = cl_list_ptr->tail = GLOBUS_NULL;

    return;
}


static void
globus_l_hbmlm_delete_cl_entries_shutdown_procs()
{
    hbmlm_cl_fields_t*
                cl_fields               = GLOBUS_NULL;
    hbmlm_cl_fields_t*
                cl_tmp                  = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields             = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_tmp                = GLOBUS_NULL;

    cl_fields = hbmlm_cl_list.head;
    while( cl_fields != GLOBUS_NULL )
    {
        cldc_fields = cl_fields->CL_dclist.head;
        while( cldc_fields != GLOBUS_NULL )
        {
            if( cldc_fields->CL_shutdownMsgCt >= 5 )
            {
                cldc_tmp = cldc_fields->next;
                ListRemove( &( cl_fields->CL_dclist ), cldc_fields );
                globus_l_hbmlm_free_cldc_fields( cldc_fields );
                cldc_fields = cldc_tmp;
            }
            else
            {
                cldc_fields = cldc_fields->next;
            }
        }
        if( cl_fields->CL_dclist.head == GLOBUS_NULL )
        {
            cl_tmp = cl_fields->next;
            ListRemove( &hbmlm_cl_list, cl_fields );
            globus_l_hbmlm_free_cl_fields( cl_fields );
            hbmlm_lm_info.LM_ClientsCt--;
            cl_fields = cl_tmp;
            break;
        }
        else
        {
            cl_fields = cl_fields->next;
        }
    }

    return;
}


static int
globus_l_hbmlm_extract_and_process_cl_reg_message(
                char*                   msg_buf,
                int                     reg_code,
                char*                   msg_end )
{
    hbmlm_cl_fields_t*
                cl_fields               = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields             = GLOBUS_NULL;
    char*       ip_addr_str             = GLOBUS_NULL;

/*
**  Create the structures to hold message data.
*/
/*
**  Initialize all pointers.
*/
    if( ( cl_fields = globus_malloc( sizeof( hbmlm_cl_fields_t ))) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_reg_message():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_HBM_OUT_OF_MEMORY;
    }
    memset(             (void *) cl_fields,
                        0,
                        sizeof( hbmlm_cl_fields_t ));

    cl_fields->prev = cl_fields->next = GLOBUS_NULL;
    cl_fields->CL_procPID = 0;
    cl_fields->CL_procName_str = GLOBUS_NULL;
    cl_fields->CL_procStatus = GLOBUS_HBM_PROCSTATUS_UNKNOWN;
    cl_fields->CL_blockedTime = 0;
    cl_fields->CL_cpuSecs = 0;
    cl_fields->CL_updated = GLOBUS_FALSE;
    cl_fields->CL_dclist.head = cl_fields->CL_dclist.tail = GLOBUS_NULL;

    if( ( cldc_fields = globus_malloc( sizeof( hbmlm_cldc_fields_t ))) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_reg_message():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        globus_l_hbmlm_free_cl_fields( cl_fields );

        return GLOBUS_HBM_OUT_OF_MEMORY;
    }
    memset(             (void *) cldc_fields,
                        0,
                        sizeof( hbmlm_cldc_fields_t ));

    cldc_fields->next = cldc_fields->prev = GLOBUS_NULL;
    cldc_fields->CL_procNameRpt_str = GLOBUS_NULL;
    cldc_fields->CL_DCHostIPNum_str = GLOBUS_NULL;
    cldc_fields->CL_DCMsgString = GLOBUS_NULL;

/*
**  extract RegCprocessPID
*/
    UnpackUInt32( msg_buf, cl_fields->CL_procPID );
    msg_buf += NUM_PACKED_BYTES;

/*
**  extract RegCprocessName
*/
    if( ( cl_fields->CL_procName_str =
                            globus_malloc( strlen( msg_buf ) + 1 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_reg_message():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        globus_l_hbmlm_free_cl_fields( cl_fields );
        globus_l_hbmlm_free_cldc_fields( cldc_fields );

        return GLOBUS_FAILURE;
    }
    strcpy( cl_fields->CL_procName_str, msg_buf );
    msg_buf += ( strlen( msg_buf ) + 1 );

/*
**  extract RegCreportName
*/
    if( ( cldc_fields->CL_procNameRpt_str =
                            globus_malloc( strlen( msg_buf ) + 1 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_reg_message():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        globus_l_hbmlm_free_cl_fields( cl_fields );
        globus_l_hbmlm_free_cldc_fields( cldc_fields );

        return GLOBUS_HBM_OUT_OF_MEMORY;
    }
    strcpy( cldc_fields->CL_procNameRpt_str, msg_buf );
    msg_buf += ( strlen( msg_buf ) + 1 );

/*
**  extract RegCDChbInterval
*/
    UnpackUInt32( msg_buf, cldc_fields->CL_rptInterval );
    msg_buf += NUM_PACKED_BYTES;

/*
**  extract RegCDCaddr
*/
    memcpy(             (void *) &cldc_fields->CL_DCaddrRpt,
                        (void *) msg_buf,
                        sizeof( struct sockaddr_in ));
    msg_buf += sizeof( struct sockaddr_in );

/*
**  extract RegCDCmsg if there is one
*/
    if( msg_buf != msg_end )
    {
        if( strcasecmp( msg_buf, "<None>" ))
        {
            if( ( cldc_fields->CL_DCMsgString =
                            globus_malloc( strlen( msg_buf ) + 1 )) ==
                                GLOBUS_NULL )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_reg_message():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

                globus_l_hbmlm_free_cl_fields( cl_fields );
                globus_l_hbmlm_free_cldc_fields( cldc_fields );

                return GLOBUS_HBM_OUT_OF_MEMORY;
            }
            strcpy( cldc_fields->CL_DCMsgString, msg_buf );
        }
    }

/*
**  validate and fill in the remainder of the structures
*/

/*
**  Fill in CL_DCHostIPNum_str.
**  Copy value returned by inet_ntoa,
**      it is static data and can be overwriten in subsequent calls.
*/
    ip_addr_str = inet_ntoa( cldc_fields->CL_DCaddrRpt.sin_addr );
    if( ( cldc_fields->CL_DCHostIPNum_str =
                            globus_malloc( strlen( ip_addr_str ) + 1 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_reg_message():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );
        globus_l_hbmlm_free_cl_fields( cl_fields );
        globus_l_hbmlm_free_cldc_fields( cldc_fields );

        return GLOBUS_HBM_OUT_OF_MEMORY;
    }
    strcpy( cldc_fields->CL_DCHostIPNum_str, ip_addr_str );

    cl_fields->CL_procStatus      = GLOBUS_HBM_PROCSTATUS_ACTIVE;
    cl_fields->CL_blockedTime     = 0;
    cl_fields->CL_cpuSecs         = 0;
    cl_fields->CL_updated         = GLOBUS_FALSE;

    cldc_fields->CL_shutdownTime  = 0;
    cldc_fields->CL_regTime       = time_curr.tv_sec;
    cldc_fields->CL_DCMsgNum      = 1;
    cldc_fields->CL_shutdownMsgCt = 0;
    cldc_fields->CL_shutdownType  = 0;
    cldc_fields->CL_rptNumber     = 0;
    cldc_fields->CL_rptTimeLast   = 0;
    cldc_fields->CL_rptTimeNext   = time_curr.tv_sec;

    globus_l_hbmlm_add_cl_entry(    &shadow_cl_list,
                                    cl_fields,
                                    cldc_fields,
                                    GLOBUS_NULL );

    return GLOBUS_SUCCESS;
}


static int
globus_l_hbmlm_extract_and_process_cl_unreg_message(
                char*                   msg_buf,
                int                     unreg_code )
{
    int         rc;
    char*       cl_procname             = GLOBUS_NULL;
    int         cl_pid;
    hbmlm_cl_fields_t*
                cl_fields               = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields             = GLOBUS_NULL;

/*
**  No need to get time of day now --
**  we did that when we got the request for the connection.
*/
    rc = gettimeofday( &time_curr, GLOBUS_NULL );
    if( rc == -1 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_unreg_message():\n"
                        "        gettimeofday() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }

/*
**  UnregCprocessPID
*/
    UnpackUInt32( msg_buf, cl_pid );
    msg_buf += sizeof( unsigned int );

/*
**  UnregCprocessName
*/
    if( ( cl_procname = globus_malloc( strlen( msg_buf ) + 1 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_unreg_message():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    strcpy( cl_procname, msg_buf );

    cl_fields = globus_i_hbm_find_cl_fields_entry( &hbmlm_cl_list, cl_pid );
    if( cl_fields == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_unreg_message():\n"
                        "        Unregistration message received from unknown "
                        "process\n"
                        "        Process Name:  %s  PID:  %hu\n\n",
                        hbmlm_proc_name_str,
                        cl_procname,
                        cl_pid );
        globus_free( cl_procname );

        return GLOBUS_FAILURE;
    }

    if( strcmp( cl_fields->CL_procName_str,
                     cl_procname ) != 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_extract_and_process_"
                        "cl_unreg_message():\n"
                        "        Unregistration message received with "
                        "wrong process name\n"
                        "            PID:  %hu\n"
                        "            Process Name:  %s\n"
                        "            Rcvd Process Name:  %s\n\n",
                        hbmlm_proc_name_str,
                        cl_pid,
                        cl_fields->CL_procName_str,
                        cl_procname );
        globus_free( cl_procname );

        return GLOBUS_FAILURE;
    }

    if( unreg_code == GLOBUS_HBM_MSGTYPE_SHUTDOWN_NORMAL )
    {
        cl_fields->CL_procStatus =
            GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL;
    }
    else
    {
        cl_fields->CL_procStatus =
            GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL;
    }
    cl_fields->CL_blockedTime = 0;

/*
**  Mark all of the data collectors for this PID.
*/
    for( cldc_fields = cl_fields->CL_dclist.head;
         cldc_fields;
         cldc_fields = cldc_fields->next )
    {
        cldc_fields->CL_shutdownType = cl_fields->CL_procStatus;
        cldc_fields->CL_shutdownMsgCt = 0;
        cldc_fields->CL_shutdownTime = time_curr.tv_sec;
        globus_l_hbmlm_format_send_rptmsg( cl_fields, cldc_fields );
    }

    globus_free( cl_procname );

    return GLOBUS_SUCCESS;
}


static int
globus_l_hbmlm_extract_cl_fields_from_ckpt_string(
                char*                   string,
                hbmlm_cl_fields_t*      cl_fields )
{
    char*       token                   = GLOBUS_NULL;
    int         wk_cpu_days;
    int         wk_cpu_hours;
    int         wk_cpu_mins;
    int         wk_cpu_secs;
    int         got_val;

/*
**  Client PID.
*/
    token = strtok( string, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_extract_cl_fields_from_"
                        "ckpt_string():\n"
                        "        Client PID not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    cl_fields->CL_procPID = (unsigned long )atol( token );

/*
** TODO:  fix this ifdef
*/
#   if defined(PID_MAX)
    {
        if( cl_fields->CL_procPID > PID_MAX )
        {
             globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_extract_cl_fields_from_"
                        "ckpt_string():\n"
                        "        Client PID not valid.\n\n",
                        hbmlm_proc_name_str );

            return GLOBUS_FAILURE;
        }
    }
#   endif

/*
**  Client process name.
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_extract_cl_fields_from_"
                        "ckpt_string():\n"
                        "        Client process name not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    if( ( cl_fields->CL_procName_str = globus_malloc( strlen( token ) + 1 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_extract_cl_fields_from_"
                        "ckpt_string():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    strcpy( cl_fields->CL_procName_str, token );

/*
**  Client Process status.
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_extract_cl_fields_from_"
                        "ckpt_string():\n"
                        "        Client process status not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    cl_fields->CL_procStatus = (unsigned int) atoi( token );
    if( cl_fields->CL_procStatus > GLOBUS_HBM_PROCSTATUS_MAXVAL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmlm_extract_cl_fields_from_"
                        "ckpt_string():\n"
                        "        Client process status invalid.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*
**  CL_blockedTime
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [07] in "
                        "globus_l_hbmlm_extract_cl_fields_from_"
                        "ckpt_string():\n"
                        "        Client blocked time not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    cl_fields->CL_blockedTime = globus_i_hbm_convert_string_to_UTCtime(
                                        token,
                                        time_conv_diff );
    if(   (   ( cl_fields->CL_procStatus == GLOBUS_HBM_PROCSTATUS_BLOCKED )
           && ( cl_fields->CL_blockedTime == 0 ))
       || (   ( cl_fields->CL_procStatus != GLOBUS_HBM_PROCSTATUS_BLOCKED )
           && ( cl_fields->CL_blockedTime != 0 )))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [08] in "
                        "globus_l_hbmlm_extract_cl_fields_from_"
                        "ckpt_string():\n"
                        "        Client blocked time invalid for status:\n"
                        "            Status:       %d.\n"
                        "            Blocked Time: %s [%d].\n\n",
                        hbmlm_proc_name_str,
                        cl_fields->CL_procStatus,
                        token,
                        (int) cl_fields->CL_blockedTime );

        return GLOBUS_FAILURE;
    }

/*
**  CL_cpuSecs
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( strchr( token, '-' ) != GLOBUS_NULL )
    {
        got_val = sscanf(token,
                         "%d-%d:%d:%d",
                         &wk_cpu_days,
                         &wk_cpu_hours,
                         &wk_cpu_mins,
                         &wk_cpu_secs );
    }
    else
    {
        wk_cpu_days = 0;
        if( globus_i_hbm_count_chars_in_string( token, ':' ) == 2 )
        {
            got_val = sscanf(token,
                             "%d:%d:%d\n",
                             &wk_cpu_hours,
                             &wk_cpu_mins,
                             &wk_cpu_secs );
            got_val++;
        }
        else
        {
            wk_cpu_hours = 0;
            got_val = sscanf(token,
                             "%d:%d\n",
                             &wk_cpu_mins,
                             &wk_cpu_secs );
            got_val += 2;
        }
    }
    if(got_val != 4 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [09] in "
                        "globus_l_hbmlm_extract_cl_fields_from_"
                        "ckpt_string():\n"
                        "        Client cpu seconds invalid.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    else
    {
        wk_cpu_secs += (wk_cpu_days  * GLOBUS_HBM_TIME_SECS_PER_DAY );
        wk_cpu_secs += (wk_cpu_hours * GLOBUS_HBM_TIME_SECS_PER_HOUR );
        wk_cpu_secs += (wk_cpu_mins  * GLOBUS_HBM_TIME_SECS_PER_MIN );
        cl_fields->CL_cpuSecs = wk_cpu_secs;
    }

    return GLOBUS_SUCCESS;
}


static int
globus_l_hbmlm_extract_cldc_fields_from_ckpt_string(
                char*                   string,
                hbmlm_cl_fields_t*      cl_fields,
                hbmlm_cldc_fields_t*    cldc_fields )
{
    char*       token                   = GLOBUS_NULL;

/*
**  Data Collector Reporting address -- CL_DChostIPNum.
*/
    token = strtok( string, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC address not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    if( ( cldc_fields->CL_DCHostIPNum_str =
                            globus_malloc( strlen( token ) + 1 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [07] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    strcpy( cldc_fields->CL_DCHostIPNum_str, token );


/*
**  Port number and build the CL_DCaddrRpt structure
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [08] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC port number not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    memset(             (void *) &cldc_fields->CL_DCaddrRpt,
                        0,
                        sizeof( struct sockaddr_in ));
    cldc_fields->CL_DCaddrRpt.sin_family = AF_INET;
    cldc_fields->CL_DCaddrRpt.sin_addr.s_addr =
                                inet_addr( cldc_fields->CL_DCHostIPNum_str );
    cldc_fields->CL_DCaddrRpt.sin_port = htons( (u_short) atoi( token ));
    if(   ( ntohl( cldc_fields->CL_DCaddrRpt.sin_addr.s_addr ) <= 0 )
       || ( ntohs( cldc_fields->CL_DCaddrRpt.sin_port ) == 0 ))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [09] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC address/port number invalid.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*
**  CL_procNameRpt_str
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC process name to report "
                        "not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    if( ( cldc_fields->CL_procNameRpt_str =
                            globus_malloc( strlen( token ) + 1 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    strcpy( cldc_fields->CL_procNameRpt_str, token );

/*
**  Client registration time.
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [16] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC registration time not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    cldc_fields->CL_regTime = globus_i_hbm_convert_string_to_UTCtime(
            token,
            time_conv_diff );

/*
**  CL_DCMsgNum
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [10] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC message number not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    cldc_fields->CL_DCMsgNum = (unsigned int) atoi( token );

/*
**  CL_DCMsgString
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [11] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC message string not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    if( strcasecmp( token, "<None>" ) == 0 )
    {
        cldc_fields->CL_DCMsgString = GLOBUS_NULL;
    }
    else
    {
        if( ( cldc_fields->CL_DCMsgString =
                            globus_malloc( strlen( token ) + 1 )) ==
                                GLOBUS_NULL )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [12] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

            return GLOBUS_FAILURE;
        }
        strcpy( cldc_fields->CL_DCMsgString, token );
    }

/*
**  Client report (message ) number.
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [17] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC report number not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    cldc_fields->CL_rptNumber = (unsigned int) atoi( token );

/*
**  Client last report time.
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [18] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC last report time not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    cldc_fields->CL_rptTimeLast = globus_i_hbm_convert_string_to_UTCtime(
                                token,
                                time_conv_diff );

    if( cldc_fields->CL_rptTimeLast < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [19] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC last report time invalid.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*
**  Client report interval (seconds ).
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [20] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC report interval not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    cldc_fields->CL_rptInterval = (unsigned int) atoi( token );
    if( cldc_fields->CL_rptInterval <= 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [21] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC report interval invalid.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*
**  Client next report time.
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [22] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC next report time not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    cldc_fields->CL_rptTimeNext =
                globus_i_hbm_convert_string_to_UTCtime(
                        token,
                        time_conv_diff );

    cldc_fields->CL_rptTimeNext = time_curr.tv_sec;

/*
**  CL_shutdownType
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [14] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC shutdown type not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    cldc_fields->CL_shutdownType = (unsigned int) atoi( token );
    if(   (   ( cl_fields->CL_procStatus ==
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
           || ( cl_fields->CL_procStatus ==
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL )
           || ( cl_fields->CL_procStatus ==
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED ))
       && ( cldc_fields->CL_shutdownType !=
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
       && ( cldc_fields->CL_shutdownType !=
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL )
       && ( cldc_fields->CL_shutdownType !=
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED ))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [15] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC shutdown type not valid for "
                        "shutdown process.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*
**  Client shutdown time.
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC shutdown time not found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    cldc_fields->CL_shutdownTime =
        globus_i_hbm_convert_string_to_UTCtime( token, time_conv_diff );
    if(   (   ( cl_fields->CL_procStatus ==
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
           || ( cl_fields->CL_procStatus ==
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL )
           || ( cl_fields->CL_procStatus ==
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED ))
       && ( cldc_fields->CL_shutdownTime == 0 ))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC shutdown time invalid.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*
**  Client shutdown message count.
*/
    token = strtok( GLOBUS_NULL, ";" );
    if( token == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [13] in "
                        "globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string():\n"
                        "        Client DC shutdown message count not "
                        "found.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    cldc_fields->CL_shutdownMsgCt = (unsigned int) atoi( token );

    return GLOBUS_SUCCESS;
}


static hbmlm_cldc_fields_t*
globus_l_hbmlm_find_cldc_fields_entry(
                hbmlm_cl_fields_t*      cl_fields,
                struct sockaddr_in      dc_addr )
{
    hbmlm_cldc_fields_t*
                cldc_fields_ptr         = GLOBUS_NULL;

    for ( cldc_fields_ptr = cl_fields->CL_dclist.head;
          cldc_fields_ptr;
          cldc_fields_ptr = cldc_fields_ptr->next )
    {
        if(   ( bcmp(   &(cldc_fields_ptr->CL_DCaddrRpt.sin_port ),
                        &(dc_addr.sin_port ),
                        sizeof( dc_addr.sin_port )) == 0 )
           && ( bcmp(   &(cldc_fields_ptr->CL_DCaddrRpt.sin_addr ),
                        &(dc_addr.sin_addr ),
                        sizeof( dc_addr.sin_addr )) == 0 ))
        {
            return cldc_fields_ptr;
        }
    }

    return GLOBUS_NULL;
}


/*
************************************************************************
**  format_send_rptmsg                                                **
**      Parameters: cl_entry_ptr        pointer to the client info    **
**                                      table entry for the client    **
**                                      to be reported on.            **
**      Returns:    (Nothing )                                         **
**                                                                    **
**      This routine formats a report message for the client and      **
**      sends it to the appropriate data collector.                   **
************************************************************************
*/
static void
globus_l_hbmlm_format_send_rptmsg(
                hbmlm_cl_fields_t*      cl_fields,
                hbmlm_cldc_fields_t*    cldc_fields )
{
    char*       send_buf                = GLOBUS_NULL;
    char*       buf_ptr                 = GLOBUS_NULL;
    int         nleft;
    int         nsent;
    int         msg_len                 = 0;

    /* Update Datacollector fields */
    cldc_fields->CL_rptTimeLast = time_curr.tv_sec;
    cldc_fields->CL_rptNumber++;
    cldc_fields->CL_rptTimeNext = time_curr.tv_sec
                                + cldc_fields->CL_rptInterval;

#ifdef DEBUG_MSGS
    globus_libc_fprintf(
                        stdout,
                        "time:  %d    interval:  %d    rptnext:  %d\n",
                        time_curr.tv_sec,
                        cldc_fields->CL_rptInterval,
                        cldc_fields->CL_rptTimeNext );
#endif /* DEBUG_MSGS */

    if( cldc_fields->CL_shutdownType )
    {
        cldc_fields->CL_shutdownMsgCt++;
        if( cldc_fields->CL_shutdownMsgCt > 5 )
        {
            return;
        }
    }

    /* calculate the length of the message */
    msg_len += NUM_PACKED_BYTES;        /* Version */
    msg_len += NUM_PACKED_BYTES;        /* RptLMmsgLength */
    msg_len += NUM_PACKED_BYTES;        /* RptLMhostIPNum */
    msg_len += NUM_PACKED_BYTES;        /* RptLMportNum */
    msg_len += NUM_PACKED_BYTES;        /* RptCprocessPID */
    msg_len += strlen( cldc_fields->CL_procNameRpt_str ) + 1;/*RptCprocessName*/
    msg_len += NUM_PACKED_BYTES;        /* RptCstatus */
    msg_len += NUM_PACKED_BYTES;        /* RptCregistrationTime */
    msg_len += NUM_PACKED_BYTES;        /* RptLMrptInterval */
    msg_len += NUM_PACKED_BYTES;        /* RptLMrptNum */
    msg_len += NUM_PACKED_BYTES;        /* RptCblockedTime */
    msg_len += NUM_PACKED_BYTES;        /* RptCcpuTime */
    msg_len += NUM_PACKED_BYTES;        /* RptCunregisterTime */
    msg_len += NUM_PACKED_BYTES;        /* RptCnumUnregisterMsg */
    msg_len += NUM_PACKED_BYTES;        /* RptCDCmsgNum */

    if( cldc_fields->CL_DCMsgString != GLOBUS_NULL )
    {              /* RptCDCmsg */
        msg_len += strlen( cldc_fields->CL_DCMsgString ) + 1;
    }
    else
    {
        msg_len++;              /* for empty string */
    }

    if( ( send_buf = globus_malloc( msg_len )) == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_format_send_rptmsg():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return;
    }

    memset(             (void *) send_buf,
                        0,
                        msg_len );

    /* build the message */
    buf_ptr = send_buf;

    /* Version */
    PackUInt32( buf_ptr, GLOBUS_HBM_VERSION );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptLMmsgLength */
    PackUInt32( buf_ptr, msg_len );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptLMhostIPNum */
    memcpy(             (void *) buf_ptr,
                        (void *) &( hbmlm_lm_info.LM_hostIPNum ),
                        NUM_PACKED_BYTES );
    buf_ptr += NUM_PACKED_BYTES;
        
    /* RptLMportNum */
    PackUInt32( buf_ptr, hbmlm_lm_info.LM_portNumRpt ); 
    buf_ptr += NUM_PACKED_BYTES;

    /* RptCprocessPID */
    PackUInt32( buf_ptr, cl_fields->CL_procPID );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptCprocessName */
    strcpy( buf_ptr, cldc_fields->CL_procNameRpt_str );
    buf_ptr += strlen( cldc_fields->CL_procNameRpt_str ) + 1;

    /* RptCstatus */
    PackUInt32( buf_ptr, cl_fields->CL_procStatus );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptCregistrationTime */
    PackUInt32( buf_ptr, cldc_fields->CL_regTime );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptLMrptInterval */
    PackUInt32( buf_ptr, cldc_fields->CL_rptInterval );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptLMrptNum */
    PackUInt32( buf_ptr, cldc_fields->CL_rptNumber );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptCblockedTime */
    PackUInt32( buf_ptr, cl_fields->CL_blockedTime );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptCcpuTime */
    PackUInt32( buf_ptr, cl_fields->CL_cpuSecs );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptCUnregisterTime */
    PackUInt32( buf_ptr, cldc_fields->CL_shutdownTime );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptCnumUnregisterMsg */
    PackUInt32( buf_ptr, cldc_fields->CL_shutdownMsgCt );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptCDCmsgNum */
    PackUInt32( buf_ptr, cldc_fields->CL_DCMsgNum );
    buf_ptr += NUM_PACKED_BYTES;

    /* RptCDCmsg */
    if( cldc_fields->CL_DCMsgString != GLOBUS_NULL )
    {
        strcpy( buf_ptr, cldc_fields->CL_DCMsgString );
    }
    else
    {
        *buf_ptr = '\0';
    }

    /* send the message */

    buf_ptr = send_buf;
    nleft = msg_len;
    while( nleft )
    {
        nsent = sendto( sock_fd_rpt,
                        buf_ptr,
                        nleft,
                        0,
                        (struct sockaddr *) &( cldc_fields->CL_DCaddrRpt ),
                        sizeof( struct sockaddr ));
        /* printf("HBM Message sent\n" ); */
        if( nsent <= 0 )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_format_send_rptmsg():\n"
                        "        Error sending report:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

            nleft = 0;
        }
        else
        {
            buf_ptr += nsent;
            nleft -= nsent;
        }
    }

    globus_free( send_buf );

    return;
}


static void
globus_l_hbmlm_free_cl_fields(
                hbmlm_cl_fields_t*      cl_fields )
{
    hbmlm_cldc_fields_t*    cldc_fields = GLOBUS_NULL;

    if( cl_fields != GLOBUS_NULL )
    {
        globus_hbm_free( cl_fields->CL_procName_str );

        while( cl_fields->CL_dclist.head != GLOBUS_NULL )
        {
            cldc_fields = cl_fields->CL_dclist.head;
            ListRemove( &( cl_fields->CL_dclist ), cldc_fields );
            globus_l_hbmlm_free_cldc_fields( cldc_fields );
        }

        globus_free( cl_fields );
    }

    return;
}


static void
globus_l_hbmlm_free_cldc_fields(
                hbmlm_cldc_fields_t*    cldc_fields )
{
    if( cldc_fields != GLOBUS_NULL )
    {
        globus_hbm_free( cldc_fields->CL_procNameRpt_str );
        globus_hbm_free( cldc_fields->CL_DCHostIPNum_str );
        globus_hbm_free( cldc_fields->CL_DCMsgString );

        globus_free( cldc_fields );
    }

    return;
}


static int
globus_l_hbmlm_get_arguments(
                globus_list_t*          args_options_found_list )
{
    globus_list_t*                      args_options_found_list_wk;
    globus_args_option_instance_t*      arg_option;

/*
**  First get all the options from the list.
**  Don't bother validating now,
**    because we will use the last value for each option,
**    so that is the only one that will need to be validated.
*/
    for( args_options_found_list_wk = args_options_found_list;
         !globus_list_empty( args_options_found_list_wk );
         args_options_found_list_wk =
                        globus_list_rest( args_options_found_list_wk ))
    {
        arg_option = globus_list_first( args_options_found_list_wk );
        if( arg_option->id_number == ARGS_ID_CHKPT )
        {
            arg_got_option_chkpt = GLOBUS_TRUE;
            if( arg_option_fname_chkpt_str != GLOBUS_NULL )
            {
                globus_free( arg_option_fname_chkpt_str );
            }
            if( ( arg_option_fname_chkpt_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 )) ==
                                GLOBUS_NULL )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

                return GLOBUS_FAILURE;
            }

            strcpy(     arg_option_fname_chkpt_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_LOG )
        {
            arg_got_option_log = GLOBUS_TRUE;
            if( arg_option_fname_log_str != GLOBUS_NULL )
            {
                globus_free( arg_option_fname_log_str );
            }
            if( ( arg_option_fname_log_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 )) ==
                                GLOBUS_NULL )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

                return GLOBUS_FAILURE;
            }
            strcpy(     arg_option_fname_log_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_PS )
        {
            arg_got_option_ps = GLOBUS_TRUE;
            if( arg_option_ps_program_name_str != GLOBUS_NULL )
            {
                globus_free( arg_option_ps_program_name_str );
            }
            if( ( arg_option_ps_program_name_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 )) ==
                                GLOBUS_NULL )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

                return GLOBUS_FAILURE;
            }
            strcpy(     arg_option_ps_program_name_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_RPTINTERVAL )
        {
            arg_got_option_rptinterval = GLOBUS_TRUE;
            arg_option_rptinterval = atoi( arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_STATUS )
        {
            arg_got_option_status = GLOBUS_TRUE;
            if( arg_option_fname_status_str != GLOBUS_NULL )
            {
                globus_free( arg_option_fname_status_str );
            }
            if( ( arg_option_fname_status_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 )) ==
                                GLOBUS_NULL )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

                return GLOBUS_FAILURE;
            }
            strcpy(     arg_option_fname_status_str,
                        arg_option->values[0] );
        }
        else
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        Parameter id number [%d] not recognized.\n"
                        "usage:\n"
                        "%s",
                        hbmlm_proc_name_str,
                        arg_option->id_number,
                        usage_short );

            return GLOBUS_FAILURE;
        }
    }
/*
**  Now validate the parameters.
**
**  First verify that all required parameters are present.
*/
    if(   ( arg_got_option_chkpt  != GLOBUS_TRUE )
       || ( arg_got_option_log    != GLOBUS_TRUE )
       || ( arg_got_option_status != GLOBUS_TRUE ))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        Required argument(s) missing\n"
                        "usage:\n"
                        "%s",
                        hbmlm_proc_name_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

/*
**  Now verify the parameters in order.
**
**  Verify the checkpoint file name,
**             log file name,
**             ps program (file) name, and
**             status file name.
*/
    hbmlm_lm_info.LM_ckptFileName_str =
                globus_i_hbm_validate_path_get_filename(
                        arg_option_fname_chkpt_str,
                        HBMLM_DEFAULT_FNAME_CHKPT );
    if( hbmlm_lm_info.LM_ckptFileName_str == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [07] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        Argument \"-chkpt %s\"invalid\n"
                        "usage:\n"
                        "%s",
                        hbmlm_proc_name_str,
                        arg_option_fname_chkpt_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }
    if( ( hbmlm_lm_info.LM_ckptFileNameWk_str =
                globus_malloc(
                    strlen( hbmlm_lm_info.LM_ckptFileName_str )
                            + 4 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [08] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    globus_libc_sprintf(
                        hbmlm_lm_info.LM_ckptFileNameWk_str,
                        "%s_wk",
                        arg_option_fname_chkpt_str );

    hbmlm_log_fname_str =
                globus_i_hbm_validate_path_get_filename(
                        arg_option_fname_log_str,
                        HBMLM_DEFAULT_FNAME_LOG );
    if( hbmlm_log_fname_str == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [09] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        Argument \"-log %s\"invalid\n"
                        "usage:\n"
                        "%s",
                        hbmlm_proc_name_str,
                        arg_option_fname_log_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    if( arg_got_option_ps != GLOBUS_TRUE )
    {
        if( ( arg_option_ps_program_name_str =
                    globus_malloc(
                        strlen( GLOBUS_HBM_DEFAULT_PATH_PS ) + 1 )) ==
                                GLOBUS_NULL )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [10] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

            return GLOBUS_FAILURE;
        }
        strcpy(         arg_option_ps_program_name_str,
                        GLOBUS_HBM_DEFAULT_PATH_PS );
    }
    if( !strcmp(        arg_option_ps_program_name_str,
                        GLOBUS_HBM_DEFAULT_FNAME_PS ))
    {
        hbmlm_ps_program_str = arg_option_ps_program_name_str;
    }
    else
    {
        hbmlm_ps_program_str =
                globus_i_hbm_validate_path_get_filename(
                        arg_option_ps_program_name_str,
                        GLOBUS_HBM_DEFAULT_FNAME_PS );
        if( hbmlm_ps_program_str == GLOBUS_NULL )
        {
            if( arg_got_option_ps != GLOBUS_TRUE )
            {
                if( ( hbmlm_ps_program_str =
                        globus_malloc(
                            strlen( GLOBUS_HBM_DEFAULT_FNAME_PS ) + 1 )) ==
                                    GLOBUS_NULL )
                {
                    globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [11] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

                    return GLOBUS_FAILURE;
                }
                strcpy( arg_option_ps_program_name_str,
                        GLOBUS_HBM_DEFAULT_PATH_PS );
            }
            else
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [12] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        Argument \"-ps %s\"invalid\n"
                        "usage:\n"
                        "%s",
                        hbmlm_proc_name_str,
                        arg_option_ps_program_name_str,
                        usage_short );

                return GLOBUS_FAILURE;
            }
        }
    }
    if( globus_i_hbm_build_ps_command_string(
                        hbmlm_ps_program_str,
                        &hbmlm_ps_command_str ) == GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [13] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "globus_l_hbmlm_get_arguments:  unable to build "
                        "ps command string\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    hbmlm_lm_info.LM_defaultReportInterval =
                            arg_option_rptinterval;

    hbmlm_status_fname_str =
                globus_i_hbm_validate_path_get_filename(
                        arg_option_fname_status_str,
                        HBMLM_DEFAULT_FNAME_STATUS );
    if( hbmlm_status_fname_str == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [14] in "
                        "globus_l_hbmlm_get_arguments():\n"
                        "        Argument \"-status %s\"invalid\n"
                        "usage:\n"
                        "%s",
                        hbmlm_proc_name_str,
                        arg_option_fname_status_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    return GLOBUS_SUCCESS;
}


static UTCtime
globus_l_hbmlm_get_boot_time()
{
    FILE*       boottime_fd             = GLOBUS_NULL;
    char        boottime_command[]      = "who -b";
    struct tm   boottime_tm;
    UTCtime     boottime;
    char*       boottime_ptr            = GLOBUS_NULL;
    char*       boottime_decode_ptr     = GLOBUS_NULL;
    char        boottime_decode_am_str[] =
                                          "%b %d %I:%M %p";
    char        boottime_decode_24_str[] =
                                          "%b %d %H:%M";
    globus_bool_t
                boottime_format_bad     = GLOBUS_FALSE;

    struct tm       time_curr_tm;
    struct tm*      time_curr_tm_ptr    = &time_curr_tm;

    char            read_buf[GLOBUS_HBM_BUFF_SIZE_CKPT];
    char*           buf_ptr             = GLOBUS_NULL;
    char*           token_ptr           = GLOBUS_NULL;

    int             rc;

/*
**  First get the current time and
**            the system boot time info, such as it is.
*/
    rc = gettimeofday(  &time_curr,
                        GLOBUS_NULL );
#ifdef HAVE_GMTIME_R
    gmtime_r( &(time_curr.tv_sec ), time_curr_tm_ptr );
#else
    time_curr_tm_ptr = gmtime( &(time_curr.tv_sec ));
#endif
    memset(             (void *) read_buf,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_CKPT );

    boottime_fd = popen( boottime_command, "r" );
    if( boottime_fd == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_get_boot_time():\n"
                        "        Command to get boottime (\"%s\" ) failed.\n\n",
                        hbmlm_proc_name_str,
                        boottime_command );

        return (UTCtime ) 0;
    }

    fgets( read_buf,
           GLOBUS_HBM_BUFF_SIZE_CKPT,
           boottime_fd );
    if(   ( ferror( boottime_fd ))
       || ( feof( boottime_fd )))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_get_boot_time():\n"
                        "        Read to get boottime output for \"%s\""
                        "failed.\n\n",
                        hbmlm_proc_name_str,
                        boottime_command );
        pclose( boottime_fd );

        return (UTCtime ) 0;
    }

    pclose( boottime_fd );

    buf_ptr = read_buf;
    while ( !isalpha( *buf_ptr ))
    {
        buf_ptr++;
    }
    if( strncasecmp( buf_ptr, "system boot", 11 ))
    {
        boottime_format_bad = GLOBUS_TRUE;
    }
    else
    {
        buf_ptr += 11;
        while ( isspace( *buf_ptr ))
        {
            buf_ptr++;
        }
        token_ptr = buf_ptr;
        while (   ( *buf_ptr != ':' )
               && ( *buf_ptr != '\0' ))
        {
            buf_ptr++;
        }
        while (   ( !isspace( *buf_ptr ))
               && ( *buf_ptr != '\0' ))
        {
            buf_ptr++;
        }
        if( *buf_ptr == '\0' )
        {
            boottime_format_bad = GLOBUS_TRUE;
        }
        else
        {
            while (   ( *buf_ptr != 'm' )
                   && ( *buf_ptr != 'M' )
                   && ( *buf_ptr != '\0' ))
            {
                buf_ptr++;
            }
            if( buf_ptr == '\0' )
            {
                boottime_decode_ptr = boottime_decode_am_str;
            }
            else
            {
                boottime_decode_ptr = boottime_decode_24_str;
            }
#ifdef HAVE_STRPTIME
            boottime_ptr = (char *) strptime(
                                        token_ptr,
                                        boottime_decode_ptr,
                                        &boottime_tm );
#else
            boottime_ptr = (char *) globus_strptime(
                                        token_ptr,
                                        boottime_decode_ptr,
                                        &boottime_tm );
#endif /* ifdef HAVE_STRPTIME */
            if( boottime_ptr == GLOBUS_NULL )
            {
                boottime_format_bad = GLOBUS_TRUE;
            }
            else
            {
                boottime_tm.tm_year = time_curr_tm_ptr->tm_year;
                if(   ( boottime_tm.tm_mon > time_curr_tm_ptr->tm_mon )
                   || (   ( boottime_tm.tm_mon == time_curr_tm_ptr->tm_mon )
                       && ( boottime_tm.tm_mday > time_curr_tm_ptr->tm_mday )))
                {
                    boottime_tm.tm_year--;
                }
#ifdef HAVE_MKTIME
                boottime_tm.tm_isdst = 0;
                boottime = mktime( &boottime_tm );
#else
                boottime = timegm( &boottime_tm );
                if( boottime > 0 )
                {
                    boottime += boottime_tm.tm_gmtoff;
                }
#endif
                boottime = (UTCtime ) ((long ) boottime - time_conv_diff );
            }
        }
    }
    if( boottime_format_bad == GLOBUS_TRUE )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_get_boot_time():\n"
                        "        Boottime output for \"%s\" in unexpected format\n"
                        "            Output was:  \"%s\".\n\n",
                        hbmlm_proc_name_str,
                        boottime_command,
                        read_buf );

        return (UTCtime ) 0;
    }

    return boottime;
}


/*
**************************************************************************
**  get_LM_machine_info                                                 **
**      Parameters: (None )                                             **
**                                                                      **
**      Returns:    integer return code -- 0 if ok, error otherwise     **
**                                                                      **
**      This routine sets the HBMLM host information fields that are    **
**      dependent on the address and configuration of the host.         **
**      Those fields are:                                               **
**              hostIPNum               IP Number (network format)      **
**              hostIPNum_str           IP Number (string in '.'        **
**                                      format)                         **
**              hostNameLen             length of the host name (int)   **
**              hostName_str            host name (string)              **
**                                                                      **
**************************************************************************
*/
static int
globus_l_hbmlm_get_LM_machine_info()
{
    char*       str_ptr1                = GLOBUS_NULL;

/*
**  Get/Set machine-specific information:
*/
/*
**  Get host name (and name length ) and IP Number.
*/
    if( ( hbmlm_lm_info.LM_hostName_str =
                            globus_malloc( MAXHOSTNAMELEN )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_get_LM_machine_info():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    memset(             (void *) hbmlm_lm_info.LM_hostName_str,
                        0,
                        MAXHOSTNAMELEN );
    if( globus_libc_gethostname(
                        hbmlm_lm_info.LM_hostName_str,
                        MAXHOSTNAMELEN ) < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_get_LM_machine_info():\n"
                        "        globus_libc_gethostname() failed.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

    if( globus_i_hbm_get_inaddr_from_hostname(
                    hbmlm_lm_info.LM_hostName_str,
                    &hbmlm_lm_info.LM_hostIPNum ) == GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_get_LM_machine_info():\n"
                        "        globus_i_hbm_get_inaddr_from_hostname()"
                        "failed.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*  LM_hostIPNum field, hbmlm_lm_info                        */
    if( hbmlm_lm_info.LM_hostIPNum.s_addr == 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_get_LM_machine_info():\n"
                        "        LM_hostIPNum.s_addr == 0.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }

/*  LM_hostIPNum_str field, hbmlm_lm_info                    */
    str_ptr1 = inet_ntoa(hbmlm_lm_info.LM_hostIPNum );
    if(( hbmlm_lm_info.LM_hostIPNum_str =
                        globus_malloc( strlen( str_ptr1 ) + 1 )) ==
                                GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_get_LM_machine_info():\n"
                        "        globus_malloc() failure.\n\n",
                        hbmlm_proc_name_str );

        return GLOBUS_FAILURE;
    }
    strcpy( hbmlm_lm_info.LM_hostIPNum_str, str_ptr1 );

/*  LM_hostNameLen field, hbmlm_lm_info                      */
    hbmlm_lm_info.LM_hostNameLen =
                            strlen( hbmlm_lm_info.LM_hostName_str );

    return GLOBUS_SUCCESS;
}


/*
************************************************************************
**  process_reg_unreg_msg                                             **
**      Parameters: sock_fd_reg             the socket fd to get the  **
**                                          message from              **
**      Returns:    (Nothing )                                         **
**                                                                    **
**      This routine reads in a message from the specified socket,    **
**      verifies that it is a register/unregister message, and        **
**      processes the message.                                        **
**                                                                    **
**      A report is sent to the appropriate DC if the message is      **
**      valid.                                                        **
************************************************************************
*/
/* TODO: only write ckpointfile if state of cl & cldc data structures
** change  (commit and unregister )
*/
static void
globus_l_hbmlm_process_reg_unreg_msg(
                int                     sock_fd_reg )
{
    hbmlm_cl_fields_t*
                cl_entry_curr_ptr       = GLOBUS_NULL;
    hbmlm_cl_fields_t*
                cl_entry_new_ptr        = GLOBUS_NULL;

    struct sockaddr_in
                from_addr;

    globus_netlen_t   from_addr_len;

    unsigned int
                hbmcl_msg_len;
    int         n_bytes_read;
    char*       msg_ptr                 = GLOBUS_NULL;
    char        read_buf[ GLOBUS_HBM_BUFF_SIZE_MSG ];
    int         read_fd;
    int         rc;
    int         reg_code;
    int         msg_version;

/*
**  Command port was selected,                                        **
**  so get the message and process the request.                       **
*/
    cl_entry_curr_ptr = cl_entry_new_ptr = GLOBUS_NULL;

    memset(             (void *) &from_addr,
                        0,
                        sizeof(from_addr ));
    from_addr.sin_family = AF_INET;
    from_addr_len = sizeof(from_addr );

    read_fd = accept(   sock_fd_reg,
                        (struct sockaddr *) &from_addr,
                        &from_addr_len );
    if( read_fd == -1 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_process_reg_unreg_msg():\n"
                        "        accept() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

        close( read_fd );
        return;
    }

    /* TODO: verify that the sender is on this host also */

    memset(             (void *) read_buf,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_MSG );

    while( ( n_bytes_read = read(   read_fd,
                                    read_buf,
                                    GLOBUS_HBM_BUFF_SIZE_MSG )) != 0 )
    {
        if( n_bytes_read == -1 )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_process_reg_unreg_msg():\n"
                        "        read() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

            close( read_fd );
            return;
        }

        msg_ptr = read_buf;

        UnpackUInt32( msg_ptr, msg_version );
        msg_ptr += NUM_PACKED_BYTES;
        if( msg_version != GLOBUS_HBM_VERSION )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_process_reg_unreg_msg():\n"
                        "        Incompatible Message Format:\n"
                        "            Invalid Version(%d)\n"
                        "            Current version(%d).\n\n",
                        hbmlm_proc_name_str,
                        msg_version,
                        GLOBUS_HBM_VERSION );
            UnpackUInt32(
                        msg_ptr,
                        hbmcl_msg_len );
            msg_ptr += NUM_PACKED_BYTES;
            globus_l_hbmlm_send_ack(
                        read_fd,
                        GLOBUS_HBM_INCOMPATIBLE_VERSION );
 
            close( read_fd );
            return;
        }
        
        UnpackUInt32( msg_ptr, hbmcl_msg_len );
        msg_ptr += NUM_PACKED_BYTES;
        if( ((unsigned int) n_bytes_read ) != hbmcl_msg_len )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "        Registration/Unregistration message wrong "
                        "length.\n"
                        "            Rcvd length:      %d\n"
                        "            Expected length:  %d\n"
                        "        Reading more.\n\n",
                        hbmlm_proc_name_str,
                        n_bytes_read,
                        hbmcl_msg_len );

            /*
            ** TODO: set the buffer to non-blocking and verify rcvd.
            */
            while( ((unsigned int) n_bytes_read ) < hbmcl_msg_len )
            {
                n_bytes_read += read(   read_fd,
                                        read_buf + n_bytes_read,
                                        GLOBUS_HBM_BUFF_SIZE_MSG );
            }
        }
        
        UnpackUInt32( msg_ptr, reg_code );
        msg_ptr += NUM_PACKED_BYTES;
        if( reg_code == GLOBUS_HBM_MSGTYPE_REGISTER )
        {
            rc = globus_l_hbmlm_extract_and_process_cl_reg_message(
                        msg_ptr,
                        reg_code,
                        read_buf + hbmcl_msg_len );

            if( rc != GLOBUS_SUCCESS )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_process_reg_unreg_msg():\n"
                        "        globus_l_hbmlm_extract_and_process_"
                        "cl_reg_message() failed.\n\n",
                        hbmlm_proc_name_str );
            }

            globus_l_hbmlm_send_ack(
                        read_fd,
                        rc );
        }
        else if( reg_code == GLOBUS_HBM_MSGTYPE_REGISTER_CANCEL )
        {
            globus_l_hbmlm_delete_and_free_cl_list( &shadow_cl_list );
            close( read_fd );

            return;
        }
        else if( reg_code == GLOBUS_HBM_MSGTYPE_REGISTER_COMMIT )
        {
            rc = globus_i_hbm_call_ps_update_client_data(
                        hbmlm_ps_command_str,
                        shadow_cl_list,
                        GLOBUS_HBM_CALL_PS_CHECK_PROCNAME,
                        &time_curr,
                        hbmlm_log_file_fptr );

            if( rc == GLOBUS_SUCCESS )
            {
                rc = globus_l_hbmlm_commit_shadow_cl_list();
            }
            /* TODO: write receive_ack on client side,
            ** so the user will know that their commit and
            ** registration failed
            */
            else /*  => ( rc != GLOBUS_SUCCESS )  */
            {
                globus_l_hbmlm_delete_and_free_cl_list(
                        &shadow_cl_list );
                /* TODO: write this routine  */
                /* hbmlm_send_ack( read_fd, */
                /*    HBMLM_REGISTRATION_COMMIT_FAILURE ); */
            }
            globus_l_hbmlm_write_ckpointfile();
            close( read_fd );

            return;
        }
        else if(   ( reg_code == GLOBUS_HBM_MSGTYPE_SHUTDOWN_NORMAL )
                || ( reg_code == GLOBUS_HBM_MSGTYPE_SHUTDOWN_ABNORMAL )) /* unregister */
        {
            rc = globus_l_hbmlm_extract_and_process_cl_unreg_message(
                        msg_ptr,
                        reg_code );
            if( rc != GLOBUS_SUCCESS )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmlm_process_reg_unreg_msg():\n"
                        "        globus_l_hbmlm_extract_and_process_"
                        "cl_unreg_message() failed.\n\n",
                        hbmlm_proc_name_str );

                globus_l_hbmlm_send_ack(
                        read_fd,
                        GLOBUS_FAILURE );
            }
            else
            {
                globus_l_hbmlm_send_ack(
                        read_fd,
                        GLOBUS_SUCCESS );
            }
            close( read_fd );

            globus_l_hbmlm_delete_and_free_cl_list(
                        &shadow_cl_list );

            globus_l_hbmlm_write_ckpointfile();

            return;
        }

        memset(         (void *) read_buf,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_MSG );
    }

    close( read_fd );
    globus_l_hbmlm_write_ckpointfile();

    return;
}


/*
************************************************************************
**  process_rpt_msg                                                   **
**      Parameters: sock_fd_rpt             the socket to get the     **
**                                          message from              **
**      Returns:    (Nothing )                                         **
**                                                                    **
**      This placeholder routine reads and discards a message         **
**      received on the UDP port used for sending reports to DCs.     **
************************************************************************
*/
static void
globus_l_hbmlm_process_rpt_msg(
                int                     sock_fd_rpt )
{
    int         msg_len;
    struct sockaddr
                from_addr;
    char        read_buf[GLOBUS_HBM_BUFF_SIZE_MSG];

#if  (   (defined TARGET_ARCH_CRAYT3E) \
      || (defined TARGET_ARCH_HPUX) \
      || (defined TARGET_ARCH_IRIX) \
      || (defined TARGET_ARCH_SOLARIS))
    int         from_addr_len;
#else
    size_t      from_addr_len;
#endif

    memset(             (void *) &from_addr,
                        0,
                        sizeof( from_addr ));
    from_addr.sa_family = AF_INET;
    from_addr_len = sizeof( from_addr );
    memset(             (void *) read_buf,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_MSG );
    msg_len = recvfrom( sock_fd_rpt,
                        read_buf,
                        GLOBUS_HBM_BUFF_SIZE_MSG,
                        0,
                        (struct sockaddr *) &from_addr,
                        &from_addr_len );
    /* TODO: get length, if don't have all of the message, read
    ** until we get it all
    */

    if( errno == EAGAIN )
    {
        return;
    }
}


/*
************************************************************************
**  read_hbmlm_ckpointfile                                            **
**      Parameters: <None>                                            **
**      Returns:    <None>                                            **
**                                                                    **
**      This routine validates the LM host IP Number and loads the    **
**      checkpointed data collector and client data.                  **
**      Entries are separated by newline characters ('\n' ).          **
**      Fields within entries are delimited by semi-colons (';' ).    **
**      If an entry is found to be invalid then that entry is         **
**      skipped.                                                      **
************************************************************************
*/
static void
globus_l_hbmlm_read_ckpointfile()
{
    char        read_buf[GLOBUS_HBM_BUFF_SIZE_CKPT];
    char*       buf_ptr                 = GLOBUS_NULL;

    FILE*       ckpt_fd                 = GLOBUS_NULL;

    int         cl_entry_ct;

    hbmlm_cl_fields_t*
                cl_fields               = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields             = GLOBUS_NULL;

    globus_bool_t
                need_datacollector;
    int         rc;

    if(( ckpt_fd = fopen( hbmlm_lm_info.LM_ckptFileName_str, "r" )) ==
                                GLOBUS_NULL )
    {
/*
**      There is no table file -- initialize table to be empty.
*/
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        HBMLM Checkpoint file \"%s\" not found.\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );

        return;
    }

/*
**  The first line has a literal "HBMLM CHECKPOINT" for file
**  identification.
**  We don't need/use that, just verify that it is an HBMLM checkpoint
**  file and throw it away.
*/
    if( fgets(  read_buf,
                GLOBUS_HBM_BUFF_SIZE_CKPT,
                ckpt_fd ) == GLOBUS_NULL )
    {
/*
**      The file is empty -- close it, clean up, and return.
*/
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        HBMLM Checkpoint file \"%s\" empty.\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
        fclose( ckpt_fd );

        return;
    }
    if( strncasecmp( read_buf, "HBMLM CHECKPOINT:", 17 ))
    {
        /*
        ** The file is not an HBMLM Checkpoint file --
        ** close it, clean up, and return.
        */
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        HBMLM Checkpoint file \"%s\" invalid.\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
        fclose( ckpt_fd );

        return;
    }
    buf_ptr = read_buf + strlen( "HBMLM CHECKPOINT:" );
    if( globus_l_hbmlm_ckpt_restore_validate_hdr_fields_from_ckpt_string(
                        buf_ptr ) == GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        globus_l_hbmlm_ckpt_restore_validate_"
                        "hdr_fields_from_ckpt_string() failed.\n"
                        "        HBMLM Checkpoint file \"%s\".\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
        fclose( ckpt_fd );

        return;
    }

    /*
    ** Read the next line and extract the hbmlm host information.
    */
    memset(             (void *) read_buf,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_CKPT );
    if( fgets( read_buf, GLOBUS_HBM_BUFF_SIZE_CKPT, ckpt_fd ) == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        HBMLM Checkpoint file \"%s\" data "
                        "incomplete.\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
        fclose( ckpt_fd );

        return;
    }

    /*
    ** Verify that the first line is an LM data record.
    */
    if( strncasecmp( read_buf, "LM Data:", 8 ))
    {
        /*
        ** The second record does not have LM data --
        ** close it, clean up, and return.
        */
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        HBMLM Checkpoint file \"%s\" invalid.\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
        fclose( ckpt_fd );

        return;
    }

    buf_ptr = read_buf + strlen( "LM Data:" );

    if( globus_l_hbmlm_ckpt_restore_extract_lm_fields_from_ckpt_string(
                        buf_ptr,
                        &cl_entry_ct ) == GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [07] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        globus_l_hbmlm_ckpt_restore_extract_"
                        "lm_fields_from_ckpt_string() failed.\n"
                        "        HBMLM Checkpoint file \"%s\".\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
        fclose( ckpt_fd );

        return;
    }

    /*
    ** Read client data one line at a time (up to GLOBUS_HBM_BUFF_SIZE_CKPT
    ** characters ) and extract the data for the table entry.
    */
    need_datacollector = GLOBUS_FALSE;
    memset(             (void *) read_buf,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_CKPT );
    while (fgets(read_buf, GLOBUS_HBM_BUFF_SIZE_CKPT, ckpt_fd ) != GLOBUS_NULL )
    {
        /*
        ** Verify that this is a Client data record.
        */
        if(strncasecmp(read_buf, "CL Data:", 8 ) == 0 )
        {
            if( need_datacollector == GLOBUS_TRUE )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [08] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        HBMLM Checkpoint file \"%s\" corrupt.\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
                globus_l_hbmlm_delete_and_free_cl_list( &shadow_cl_list );

                fclose( ckpt_fd );
                return;
            }
            buf_ptr = read_buf + strlen( "CL Data:" );

            /*
            ** Validate and load this client record data
            */
            if(( cl_fields = (hbmlm_cl_fields_t *)
                            globus_malloc( sizeof( hbmlm_cl_fields_t ))) ==
                                GLOBUS_NULL )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [09] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        globus_malloc() failure.\n"
                        "        HBMLM Checkpoint file \"%s\".\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
                globus_l_hbmlm_delete_and_free_cl_list( &shadow_cl_list );

                fclose( ckpt_fd );
                return;
            }
            cl_fields->next = cl_fields->prev = GLOBUS_NULL;
            cl_fields->CL_procName_str = GLOBUS_NULL;
            cl_fields->CL_dclist.head = cl_fields->CL_dclist.tail = GLOBUS_NULL;

            if( globus_l_hbmlm_extract_cl_fields_from_ckpt_string(
                        buf_ptr, cl_fields ) == GLOBUS_FAILURE )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [10] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        globus_l_hbmlm_extract_cl_fields_"
                        "from_ckpt_string() failed.\n"
                        "        HBMLM Checkpoint file \"%s\".\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
                globus_l_hbmlm_free_cl_fields( cl_fields );
                globus_l_hbmlm_delete_and_free_cl_list( &shadow_cl_list );

                fclose( ckpt_fd );
                return;
            }
            /* insert cl_fields into the table */
            cl_fields =
                    globus_l_hbmlm_add_cl_entry(
                        &shadow_cl_list,
                        cl_fields,
                        GLOBUS_NULL,
                        GLOBUS_NULL );
            need_datacollector = GLOBUS_TRUE;
        }
        else if( strncasecmp( read_buf, "DC Data:", 8 ) == 0 )
        {
            if( cl_fields == GLOBUS_NULL )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [11] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        HBMLM Checkpoint file \"%s\" corrupt.\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
                globus_l_hbmlm_delete_and_free_cl_list( &shadow_cl_list );

                fclose( ckpt_fd );
                return;
            }
            need_datacollector = GLOBUS_FALSE;
            buf_ptr = read_buf + strlen( "DC Data:" );

            /*
             ** Validate and load this datacollectors data
             */
            if( ( cldc_fields =
                            globus_malloc( sizeof( hbmlm_cldc_fields_t ))) ==
                                GLOBUS_NULL )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [12] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        globus_malloc() failure.\n"
                        "        HBMLM Checkpoint file \"%s\".\n"
                        "        Restore skipped.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
                if(cl_fields->CL_dclist.head == GLOBUS_NULL )
                {
                    ListRemove( &hbmlm_cl_list, cl_fields );
                    globus_l_hbmlm_free_cl_fields( cl_fields );
                }
                break;
            }
            cldc_fields->next = GLOBUS_NULL;
            cldc_fields->prev = GLOBUS_NULL;
            cldc_fields->CL_DCHostIPNum_str = GLOBUS_NULL;
            cldc_fields->CL_DCMsgString = GLOBUS_NULL;
            cldc_fields->next = cldc_fields->prev = GLOBUS_NULL;

            rc = globus_l_hbmlm_extract_cldc_fields_from_ckpt_string(
                        buf_ptr,
                        cl_fields,
                        cldc_fields );
            if( rc == GLOBUS_FAILURE )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [13] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        globus_l_hbmlm_extract_cldc_fields_from_"
                        "ckpt_string() failed.\n"
                        "        HBMLM Checkpoint file \"%s\".\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );
                globus_l_hbmlm_free_cldc_fields( cldc_fields );
                if  (cl_fields->CL_dclist.head == GLOBUS_NULL )
                {
                    ListRemove( &shadow_cl_list, cl_fields );
                    hbmlm_lm_info.LM_ClientsCt--;
                    globus_l_hbmlm_free_cl_fields( cl_fields );
                }
                break;
            }

            /* insert cldc_fields into the table */
            cl_fields =
                    globus_l_hbmlm_add_cl_entry(
                        &shadow_cl_list,
                        cl_fields,
                        cldc_fields,
                        GLOBUS_NULL );
        }
        else
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [14] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        HBMLM Checkpoint file \"%s\" corrupt.\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_lm_info.LM_ckptFileName_str );

            break;
        }
        memset(         (void *) read_buf,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_CKPT );
    }

    /*
    ** Have reached end of file (or non-fatal error ) --
    **   close file,
    **   copy to list from shadow_cl_list to hbmlm_cl_list,
    **   update the client status, and
    **   return.
    */
    fclose( ckpt_fd );

    globus_l_hbmlm_commit_shadow_cl_list();

    if( hbmlm_lm_info.LM_ClientsCt != cl_entry_ct )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [15] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        Warning -- client records missing or invalid.\n"
                        "        Will use available data.\n\n",
                        hbmlm_proc_name_str );
    }

    /*
    ** update the state of the localmonitor w.r.t the
    ** information we just read in
    */
    rc = gettimeofday( &time_curr, NULL );
    if( rc == -1 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [16] in "
                        "globus_l_hbmlm_read_ckpointfile():\n"
                        "        gettimeofday() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }

    globus_i_hbm_call_ps_update_client_data(
                        hbmlm_ps_command_str,
                        hbmlm_cl_list,
                        GLOBUS_HBM_CALL_PS_GET_STATUS,
                        &time_curr,
                        hbmlm_log_file_fptr );

    return;
}


static int
globus_l_hbmlm_send_ack(
                int                     client_fd,
                int                     ret_code )
{
    char        send_buf[HBMLM_ACK_MSG_LEN];
    int         n_bytes_sent;

    memset(             (void *) send_buf,
                        0,
                        HBMLM_ACK_MSG_LEN );

    /* Pack the buffer */
    PackUInt32( send_buf, ret_code );

    n_bytes_sent = write( client_fd, send_buf, HBMLM_ACK_MSG_LEN );
    if( n_bytes_sent != HBMLM_ACK_MSG_LEN )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_send_ack():\n"
                        "        write() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }

    return n_bytes_sent;
}


static void
globus_l_hbmlm_send_rpt_msgs()
{
    hbmlm_cl_fields_t*
                cl_entry_ptr            = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields             = GLOBUS_NULL;

    for( cl_entry_ptr = hbmlm_cl_list.head;
         cl_entry_ptr;
         cl_entry_ptr = cl_entry_ptr->next )
    {
        for( cldc_fields = cl_entry_ptr->CL_dclist.head;
             cldc_fields;
             cldc_fields = cldc_fields->next )
        {
            if(   (cldc_fields->CL_rptTimeNext <= time_curr.tv_sec )
               || (quit_now ))
            {
                globus_l_hbmlm_format_send_rptmsg(
                        cl_entry_ptr,
                        cldc_fields );
            }
        }
    }
    globus_l_hbmlm_delete_cl_entries_shutdown_procs();

    return;
}


/*
************************************************************************
**  set_up_tcp_port                                                   **
**      Parameters: pointer to fd field to use for the socket         **
**                  pointer to the (already filled in ) port address  **
**      Returns:    integer value: 0 - socket successfully set up     **
**                                <0 - error setting up the port      **
**                                                                    **
**      This routine sets up a tdp port for receiving registration    **
**      and unregistration messages from clients, or for              **
**      sending report messages to data collectors.                   **
************************************************************************
*/
static int
globus_l_hbmlm_set_up_tcp_port(
                int*                    sock_fd_tcp_ptr,
                struct sockaddr_in*     tcpaddr_in_ptr )
{
    int         rc;
    int         one                     = 1;
    
    globus_netlen_t         tcpaddr_in_len;

    tcpaddr_in_len  = sizeof( struct sockaddr_in );

    if(( *sock_fd_tcp_ptr = socket( AF_INET, SOCK_STREAM, 0 )) < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_set_up_tcp_port():\n"
                        "        socket() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

        *sock_fd_tcp_ptr = -1;

        return GLOBUS_FAILURE;
    }

#ifdef  TARGET_ARCH_HPUX
    if( *sock_fd_tcp_ptr >= ( 8 * sizeof( int )))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_set_up_tcp_port():\n"
                        "        Funky HP-UX error giving fd number "
                        "too large to process.\n\n",
                        hbmlm_proc_name_str );
        *sock_fd_tcp_ptr = -1;

        return GLOBUS_FAILURE;
    }
#endif  /*  TARGET_ARCH_HPUX defined  */

    rc = setsockopt(    *sock_fd_tcp_ptr,
                        SOL_SOCKET,
                        SO_REUSEADDR,
                        (char *) &one,
                        sizeof( one ));
    if( rc < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_set_up_tcp_port():\n"
                        "        setsockopt(SO_REUSEADDR) failed:  "
                        "errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }

    if( bind(   *sock_fd_tcp_ptr,
                (struct sockaddr *) tcpaddr_in_ptr,
                sizeof( *tcpaddr_in_ptr )) < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_set_up_tcp_port():\n"
                        "        bind() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

        close( *sock_fd_tcp_ptr );
        *sock_fd_tcp_ptr = -1;

        return GLOBUS_FAILURE;
    }

    if( getsockname(    *sock_fd_tcp_ptr,
                        (struct sockaddr *) tcpaddr_in_ptr,
                        &tcpaddr_in_len ) < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_set_up_tcp_port():\n"
                        "        getsockname() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

        close( *sock_fd_tcp_ptr );
        *sock_fd_tcp_ptr = -1;

        return GLOBUS_FAILURE;
    }

    if( listen( *sock_fd_tcp_ptr, SOMAXCONN ) < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmlm_set_up_tcp_port():\n"
                        "        listen() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

        close( *sock_fd_tcp_ptr );
        *sock_fd_tcp_ptr = -1;

        return GLOBUS_FAILURE;
    }

    return GLOBUS_SUCCESS;
}


/*
************************************************************************
**  set_up_udp_port                                                   **
**      Parameters: pointer to fd field to use for the socket         **
**                  pointer to the (already filled in ) port address   **
**      Returns:    integer value: 0 - socket successfully set up     **
**                                <0 - error setting up the port      **
**                                                                    **
**      This routine sets up a udp port for receiving registration    **
**      and unregistration messages from clients, or for              **
**      sending report messages to data collectors.                   **
************************************************************************
*/
static int
globus_l_hbmlm_set_up_udp_port(
                int*                    sock_fd_udp_ptr,
                struct sockaddr_in*     udpaddr_in_ptr )
{
    int         rc;
    int         one                     = 1;

    globus_netlen_t      udpaddr_in_len;

    udpaddr_in_len = sizeof( struct sockaddr_in );

    if(( *sock_fd_udp_ptr = socket( PF_INET,
                                    SOCK_DGRAM,
                                    IPPROTO_UDP )) < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_set_up_udp_port():\n"
                        "        socket() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

        *sock_fd_udp_ptr = -1;

        return GLOBUS_FAILURE;
    }
/*
**  Comment this out to see if really needed.
#ifdef  TARGET_ARCH_HPUX
    if( *sock_fd_udp_ptr >= ( 8 * sizeof( int )))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_set_up_udp_port():\n"
                        "        Funky HP-UX error giving fd number "
                        "too large to process.\n\n",
                        hbmlm_proc_name_str );
        *sock_fd_udp_ptr = -1;

        return GLOBUS_FAILURE;
    }
#endif  **  TARGET_ARCH_HPUX defined  **
**
*/

    rc = setsockopt(    *sock_fd_udp_ptr,
                        SOL_SOCKET,
                        SO_REUSEADDR,
                        (char *) &one,
                        sizeof( one ));
    if( rc < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_set_up_udp_port():\n"
                        "        setsockopt(SO_REUSEADDR) failed:  "
                        "errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }

    if( bind(   *sock_fd_udp_ptr,
                (struct sockaddr *) udpaddr_in_ptr,
                sizeof( *udpaddr_in_ptr )) < 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_set_up_udp_port():\n"
                        "        bind() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

        close( *sock_fd_udp_ptr );
        *sock_fd_udp_ptr = -1;

        return GLOBUS_FAILURE;
    }

    if( udpaddr_in_ptr->sin_port == htons( INADDR_ANY ))
    {
        if( getsockname(
                        *sock_fd_udp_ptr,
                        (struct sockaddr *) udpaddr_in_ptr,
                        &udpaddr_in_len ) < 0 )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_set_up_tcp_port():\n"
                        "        getsockname() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

            close( *sock_fd_udp_ptr );
            *sock_fd_udp_ptr = -1;

            return GLOBUS_FAILURE;
        }
    }

    return GLOBUS_SUCCESS;
}


RETSIGTYPE
globus_l_hbmlm_shutdown_quit(
                int                     dummy )
{
    quit_now = 1;
}


/*
************************************************************************
**  write_hbmlm_ckpointfile                                           **
**      Parameters: (None )                                            **
**      Returns:    (Nothing )                                         **
**                                                                    **
**      This routine writes the checkpoint file based on the data     **
**      in the hbmlm_lm_info structure and the hbmlm_cl_info table.   **
**      The fields are separated by semi-colons (';' ),                **
**      and entries are separated by new-lines ('\n' ).                **
************************************************************************
*/
static void
globus_l_hbmlm_write_ckpointfile()
{
    FILE*       ckpt_fd                 = GLOBUS_NULL;

    hbmlm_cl_fields_t*
                cl_fields               = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields             = GLOBUS_NULL;

    char        date_time_str[GLOBUS_HBM_DATE_TIME_LEN];

    int         retcd;
    unsigned int
                secs_left;

/*
**  First get the current time.
*/

    retcd = gettimeofday(&time_curr, GLOBUS_NULL );
    if( retcd == -1 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_write_ckpointfile():\n"
                        "        gettimeofday() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }

/*
**  Now open and write the checkpoint file.
*/
    if((ckpt_fd = fopen(
                        hbmlm_lm_info.LM_ckptFileNameWk_str,
                        "w" )) == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_write_ckpointfile():\n"
                        "        fopen() failed:  errno [%d]: %s.\n\n"
                        "        Error opening HBMLM checkpoint work file "
                        "\"%s\"\n"
                        "        Continuing without writing file.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ),
                        hbmlm_lm_info.LM_ckptFileNameWk_str );

        return;
    }

/*
**  First output
**      the header literal and
**      the current date and time
**  for identification and "version control".
*/
    globus_i_hbm_convert_UTC_to_str(
                        time_curr.tv_sec,
                        date_time_str );
    globus_libc_fprintf(
                        ckpt_fd,
                        "HBMLM CHECKPOINT:%s\n",
                        date_time_str );

/*
**  Now write the local monitor data to the file.
*/
    globus_libc_fprintf(
                        ckpt_fd,
                        "LM Data:%s;%s;%hu;%hu;%u;%d;\n",
                        hbmlm_lm_info.LM_hostIPNum_str,
                        hbmlm_lm_info.LM_hostName_str,
                        hbmlm_lm_info.LM_portNumReg,
                        hbmlm_lm_info.LM_portNumRpt,
                        hbmlm_lm_info.LM_defaultReportInterval,
                        hbmlm_lm_info.LM_ClientsCt );
/*
**  Now write the client data (for each client process ).
*/
    for( cl_fields = hbmlm_cl_list.head;
         cl_fields;
         cl_fields = cl_fields->next )
    {
/*
**      CL_procPID,
**      CL_procName_str,
**      CL_procStatus
*/
        globus_libc_fprintf(
                        ckpt_fd,
                        "CL Data:%u;%s;%d;",
                        cl_fields->CL_procPID,
                        cl_fields->CL_procName_str,
                        cl_fields->CL_procStatus );

/*
**      CL_blockedTime
*/
        globus_i_hbm_convert_UTC_to_str(
                        cl_fields->CL_blockedTime,
                        date_time_str );
        globus_libc_fprintf(
                        ckpt_fd,
                        "%s;",
                        date_time_str );

/*
**      CL_cpuSecs
*/
        secs_left = cl_fields->CL_cpuSecs;
        if( secs_left >= GLOBUS_HBM_TIME_SECS_PER_DAY )
        {
            globus_libc_fprintf(
                        ckpt_fd,
                        "%u-",
                        secs_left / GLOBUS_HBM_TIME_SECS_PER_DAY );
            secs_left %= GLOBUS_HBM_TIME_SECS_PER_DAY;
            globus_libc_fprintf(
                        ckpt_fd,
                        "%2.2u:",
                        secs_left / GLOBUS_HBM_TIME_SECS_PER_HOUR );
            secs_left %= GLOBUS_HBM_TIME_SECS_PER_HOUR;
        }
        else if( secs_left >= GLOBUS_HBM_TIME_SECS_PER_HOUR )
        {
            globus_libc_fprintf(
                        ckpt_fd,
                        "%2.2u:",
                        secs_left / GLOBUS_HBM_TIME_SECS_PER_HOUR );
            secs_left %= GLOBUS_HBM_TIME_SECS_PER_HOUR;
        }
        globus_libc_fprintf(
                        ckpt_fd,
                        "%2.2u:%2.2u;\n",
                        secs_left / GLOBUS_HBM_TIME_SECS_PER_MIN,
                        secs_left % GLOBUS_HBM_TIME_SECS_PER_MIN );

/*
**      Now write the data collector(s) information for this client.
*/
        for( cldc_fields = cl_fields->CL_dclist.head;
             cldc_fields;
             cldc_fields = cldc_fields->next )
        {
            globus_libc_fprintf(
                        ckpt_fd,
                        "DC Data:" );

/*
**          CL_DCHostIPNum_str
**          DC port, taken from CL_DCaddrRpt
**          CL_procNameRpt_str
*/
            globus_libc_fprintf(
                        ckpt_fd,
                        "%s;%hu;%s;",
                        cldc_fields->CL_DCHostIPNum_str,
                        ntohs( cldc_fields->CL_DCaddrRpt.sin_port ),
                        cldc_fields->CL_procNameRpt_str );

/*
**          CL_regTime
**          CL_DCMsgNum
*/
            globus_i_hbm_convert_UTC_to_str(
                        cldc_fields->CL_regTime,
                        date_time_str );
            globus_libc_fprintf(
                        ckpt_fd,
                        "%s;%u;",
                        date_time_str,
                        cldc_fields->CL_DCMsgNum );

/*
**          CL_DCMsgString (If null or empty use "<None>"
*/
            if( cldc_fields->CL_DCMsgString != GLOBUS_NULL )
            {
            	if( *(cldc_fields->CL_DCMsgString) != '\0' )
            	{
            	    globus_libc_fprintf(
                        ckpt_fd,
                        "%s;",
                        cldc_fields->CL_DCMsgString );
            	}
            	else
            	{
                    globus_libc_fprintf(
                        ckpt_fd,
                        "<None>;" );
            	}
            }
            else
            {
                globus_libc_fprintf(
                        ckpt_fd,
                        "<None>;" );
            }

/*
**          CL_rptNumber
*/
            globus_libc_fprintf(
                        ckpt_fd,
                        "%u;",
                        cldc_fields->CL_rptNumber );

/*
**          CL_rptTimeLast
**          CL_rptInterval
*/
            globus_i_hbm_convert_UTC_to_str(
                        cldc_fields->CL_rptTimeLast,
                        date_time_str );
            globus_libc_fprintf(
                        ckpt_fd,
                        "%s;%u;",
                        date_time_str,
                        cl_fields->CL_dclist.head->CL_rptInterval );

/*
**          CL_rptTimeNext
**          CL_shutdownType
*/
            globus_i_hbm_convert_UTC_to_str(
                        cl_fields->CL_dclist.head->CL_rptTimeNext,
                        date_time_str );
            globus_libc_fprintf(
                        ckpt_fd,
                        "%s;%u;",
                        date_time_str,
                        cldc_fields->CL_shutdownType );

/*
**          CL_shutdownTime
**          CL_shutdownMsgCt
*/
            globus_i_hbm_convert_UTC_to_str(
                        cldc_fields->CL_shutdownTime,
                        date_time_str );
            globus_libc_fprintf(
                        ckpt_fd,
                        "%s;%u;\n",
                        date_time_str,
                        cldc_fields->CL_shutdownMsgCt );
        }
    }

/*
**  Close the file and return.
*/
    fclose( ckpt_fd );

    unlink( hbmlm_lm_info.LM_ckptFileName_str );
    link(   hbmlm_lm_info.LM_ckptFileNameWk_str,
            hbmlm_lm_info.LM_ckptFileName_str );
    unlink( hbmlm_lm_info.LM_ckptFileNameWk_str );

    return;
}


static globus_bool_t
globus_l_hbmlm_poll(
                globus_abstime_t*       time_stop,
                void*                   user_args )
{

#if  (defined TARGET_ARCH_HPUX)
    size_t      max_fdp1;
#else
    int         max_fdp1;
#endif  /*  TARGET_ARCH_HPUX  */

    fd_set      fd_readset;
    struct timeval
                time_left;

    hbmlm_cl_fields_t*
                cl_entry_ptr            = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields             = GLOBUS_NULL;
    int         retcd;

    max_fdp1 = sock_fd_reg;
    if( sock_fd_rpt > max_fdp1 )
    {
        max_fdp1 = sock_fd_rpt;
    }
    max_fdp1++;

    FD_ZERO( &fd_readset );
    FD_SET( sock_fd_reg, &fd_readset );
    FD_SET( sock_fd_rpt, &fd_readset );

    if( gettimeofday(   &time_curr,
                        GLOBUS_NULL ) == -1 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmlm_poll():\n"
                        "        gettimeofday() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }

    if( !hbmlm_cl_list.head )
    {
        if( !quit_now )
        {
            if(globus_time_abstime_is_infinity(time_stop))
            {
                /* TODO:  why isn't this blocking? */
                retcd = select(
                        max_fdp1,
                        HBM_FD_SET_CAST &fd_readset,
                        GLOBUS_NULL,
                        GLOBUS_NULL,
                        GLOBUS_NULL );
            }
            else
            {
                globus_callback_get_timeout(&time_left);
                retcd = select(
                        max_fdp1,
                        HBM_FD_SET_CAST &fd_readset,
                        GLOBUS_NULL,
                        GLOBUS_NULL,
                        &time_left );
            }
        }
    }
    else
    {
        time_to_monitor_write.tv_sec = time_curr.tv_sec
                                     + HBMLM_LIM_REPORT_INTERVAL_MAX;
        time_to_monitor_write.tv_usec = 0;

        for (cl_entry_ptr = hbmlm_cl_list.head;
             cl_entry_ptr;
             cl_entry_ptr = cl_entry_ptr->next )
        {
            for( cldc_fields = cl_entry_ptr->CL_dclist.head;
                 cldc_fields;
                 cldc_fields = cldc_fields->next )
            {
                if( time_to_monitor_write.tv_sec >
                                cldc_fields->CL_rptTimeNext )
                {
                    time_to_monitor_write.tv_sec =
                                cldc_fields->CL_rptTimeNext;
                }
            }
        }

        if(globus_time_abstime_is_infinity(time_stop))
        {
            if( time_to_monitor_write.tv_sec <= time_curr.tv_sec )
            {
                time_left.tv_sec = time_left.tv_usec = 0;
            }
            else
            {
                time_left.tv_sec =
                    time_to_monitor_write.tv_sec - time_curr.tv_sec;
                if  (time_curr.tv_usec )
                {
                    time_left.tv_sec--;
                    time_left.tv_usec = 1000000 - time_curr.tv_usec;
                }
            }
        }
        else /* No blocking allowed */
        {
            globus_callback_get_timeout(&time_left);
        }

        if  (!quit_now )
        {
          /* printf("time_left:%d.%d (%d )\n", time_left.tv_sec, time_left.tv_usec,
                 can_i_block ); */
            retcd = select(
                        max_fdp1,
                        HBM_FD_SET_CAST &fd_readset,
                        GLOBUS_NULL,
                        GLOBUS_NULL,
                        &time_left );
            DEBUGF1( "timed select completed\n" );
        }
        else
        {
            retcd = 0;
        }
    }
    if( quit_now )
    {
        return GLOBUS_TRUE;
    }

    if(   ( retcd < 0 )
       && ( errno != EINTR ))
    {
        if(   ( errno == EBADF )
           || ( errno == EFAULT ))
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmlm_poll():\n"
                        "        select() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
        }

        return GLOBUS_FALSE;
    }

    if( gettimeofday(   &time_curr,
                        GLOBUS_NULL ) == -1 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmlm_poll():\n"
                        "        gettimeofday() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }

    if( retcd > 0 )
    {
        if( FD_ISSET( sock_fd_reg, &fd_readset ))
        {
            globus_l_hbmlm_process_reg_unreg_msg( sock_fd_reg );
        }
        if( FD_ISSET( sock_fd_rpt, &fd_readset ))
        {
            globus_l_hbmlm_process_rpt_msg( sock_fd_rpt );
        }
    }

    if( gettimeofday(   &time_curr,
                        GLOBUS_NULL ) == -1 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmlm_poll():\n"
                        "        gettimeofday() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }

    if(   ( time_curr.tv_sec >= time_to_monitor_write.tv_sec )
       || ( quit_now ))
    {
        if( hbmlm_lm_info.LM_ClientsCt )
        {
            globus_i_hbm_call_ps_update_client_data(
                        hbmlm_ps_command_str,
                        hbmlm_cl_list,
                        GLOBUS_HBM_CALL_PS_GET_STATUS,
                        &time_curr,
                        hbmlm_log_file_fptr );
        }

        if( gettimeofday(
                        &time_curr,
                        GLOBUS_NULL ) == -1 )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmlm_poll():\n"
                        "        gettimeofday() failed:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
        }

        globus_l_hbmlm_send_rpt_msgs();
        globus_l_hbmlm_write_ckpointfile();
    }

    return GLOBUS_TRUE;
}


int
main(           int                     argc,
                char                    *argv[] )
{
    hbmlm_cl_fields_t   lm_as_client =
                      { GLOBUS_NULL,                    /* prev */
                        GLOBUS_NULL,                    /* next */
                        0,                              /* CL_procPID */
                        GLOBUS_NULL,                    /* CL_procName_str */
                        GLOBUS_HBM_PROCSTATUS_UNKNOWN,  /* CL_procStatus */
                        0,                              /* CL_blockedTime */
                        0,                              /* CL_cpuSecs */
                        GLOBUS_FALSE,                   /* CL_updated */
                        { GLOBUS_NULL, GLOBUS_NULL }    /* CL_dclist */  };

    hbmlm_cl_list_head_t
                        lm_as_client_list;

    int         error;
    int         retcd,
                retcd_poll_activate,
                retcd_poll_add;

    globus_abstime_t
                globus_abstime_infinity_wk;

    hbmlm_cl_fields_t*
                cl_entry_ptr            = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields             = GLOBUS_NULL;

    hbmlm_proc_name_str = argv[0];
    hbmlm_log_file_fptr = stderr;

    memset(             (void *) &hbmlm_lm_info,
                        0,
                        sizeof( hbmlm_lm_fields_t ));

    hbmlm_cl_list.head = GLOBUS_NULL;
    hbmlm_cl_list.tail = GLOBUS_NULL;
    shadow_cl_list.head = GLOBUS_NULL;
    shadow_cl_list.tail = GLOBUS_NULL;

    lm_as_client_list.head = lm_as_client_list.tail =
                                          &lm_as_client;

/*
**  Activate modules we're going to use.
*/
    retcd_poll_activate = globus_module_activate( GLOBUS_POLL_MODULE );
    if( retcd_poll_activate == GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [10] in "
                        "main():\n"
                        "        globus_module_activate() failed for "
                        "GLOBUS_POLL_MODULE.\n\n",
                        hbmlm_proc_name_str );
        retcd_poll_add = GLOBUS_FAILURE;
        return GLOBUS_FAILURE;
    }

    globus_i_list_pre_activate();

    if(( retcd = globus_args_scan(
                        &argc,
                        &argv,
                        ARGS_ID_MAX,
                        args_option_list,
                        usage_short,
                        usage_long,
                        &args_options_found_list,
                        GLOBUS_NULL )) <= 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "main():\n"
                        "        Argument error.\n"
                        "usage:\n"
                        "%s",
                        hbmlm_proc_name_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    if(( retcd = globus_l_hbmlm_get_arguments(
                        args_options_found_list )) !=
                                        GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "main():\n"
                        "        globus_l_hbmlm_get_arguments() failed.\n"
                        "usage:\n"
                        "%s",
                        hbmlm_proc_name_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    globus_args_option_instance_list_free(
                        &args_options_found_list );

    time_conv_diff = globus_i_hbm_get_time_conv_diff();

/*
**  Open the log file.
*/
#ifndef DEBUG_STDERR
    if(( hbmlm_log_file_fptr = fopen( hbmlm_log_fname_str, "a" )) ==
                                GLOBUS_NULL )
    {
        hbmlm_log_file_fptr = stderr;

        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "main():\n"
                        "        fopen() failed for log file \"%s\".\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_log_fname_str );

        return GLOBUS_FAILURE;
    }
#endif

/*
**  Get the HBMLM host IP Number (same as for any clients ).
*/
    if(( error = globus_l_hbmlm_get_LM_machine_info()) != 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "main():\n"
                        "        globus_l_hbmlm_get_LM_machine_info() failed.\n\n",
                        hbmlm_proc_name_str );

        if( hbmlm_log_file_fptr != stderr )
        {
            fclose( hbmlm_log_file_fptr );
        }

        return GLOBUS_FAILURE;
    }

    DEBUGF1( "Arguments obtained.\n" );
    DEBUGF2( "Local Monitor hostname:  %s\n", hbmlm_lm_info.LM_hostName_str );
    DEBUGF2( "Local Monitor IP addr:   %s\n\n", hbmlm_lm_info.LM_hostIPNum_str );

/*
**  Now get the process name (for the status file).
*/
    if(( lm_as_client.CL_procPID = getpid()) <= 0 )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "main():\n"
                        "        getpid() failed.\n\n",
                        hbmlm_proc_name_str );

        if( hbmlm_log_file_fptr != stderr )
        {
            fclose( hbmlm_log_file_fptr );
        }

        return GLOBUS_FAILURE;
    }

    globus_i_hbm_call_ps_update_client_data(
                        hbmlm_ps_command_str,
                        lm_as_client_list,
                        GLOBUS_HBM_CALL_PS_GET_PROCNAME,
                        &time_curr,
                        hbmlm_log_file_fptr );

    if( lm_as_client.CL_procName_str == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "main():\n"
                        "        globus_i_hbm_call_ps_update_client_data() "
                        "failed.\n\n",
                        hbmlm_proc_name_str );
        if( hbmlm_log_file_fptr != stderr )
        {
            fclose( hbmlm_log_file_fptr );
        }

        return GLOBUS_FAILURE;
    }
        
    hbmlm_lm_info.LM_procNameFromPS_str = lm_as_client.CL_procName_str;
    lm_as_client.CL_procName_str        = GLOBUS_NULL;

/*
**  Check for and read the checkpoint file for previous run
**  information, including information about client processes
**  requiring continuing (resumed) monitoring.
**
**  Then set the next time to report (rptTimeNext) for all clients.
**
**  Then set up the sockets.
**  (If we found a valid checkpoint file then we will try to use
**  the same (UDP) socket as before for sending heartbeats.
**
**  After all that we will save the updated data to a new checkpoint file.
*/
    hbmlm_lm_info.LM_portNumReg = INADDR_ANY;
    hbmlm_lm_info.LM_portNumRpt = INADDR_ANY;
    hbmlm_lm_info.LM_ClientsCt = 0;
    retcd = gettimeofday( &time_curr, GLOBUS_NULL );
    if( gettimeofday( &time_curr, GLOBUS_NULL ))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [07] in "
                        "main():\n"
                        "        gettimeofday() failed:  errno [%d]: %s.\n"
                        "    Continuing.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

        time_curr.tv_sec = time_curr.tv_usec = 0;
    }

    globus_l_hbmlm_read_ckpointfile();

    retcd = gettimeofday( &time_curr, GLOBUS_NULL );
    if( hbmlm_lm_info.LM_ClientsCt == 0 )
    {
        time_to_monitor_write.tv_sec = 0;
        time_to_monitor_write.tv_usec = 0;
    }
    else
    {
/*
**      Set rptTimeNext to now for all data collectors for each client.
*/
        for( cl_entry_ptr = hbmlm_cl_list.head;
             cl_entry_ptr;
             cl_entry_ptr = cl_entry_ptr->next )
        {
            for( cldc_fields = cl_entry_ptr->CL_dclist.head;
                 cldc_fields;
                 cldc_fields = cldc_fields->next )
            {
                cldc_fields->CL_rptTimeNext = time_curr.tv_sec;
            }
        }
        time_to_monitor_write.tv_sec = time_curr.tv_sec;
        time_to_monitor_write.tv_usec = 0;
    }

/*
**
**  Now we have all the parameters,
**  The IP Number,
**  the registration socket number (from the checkpoint file),
**  the heartbeat socket number (also from the checkpoint file), and
**  the LM process name.
**
**  Next we set up the sockets.
**
*/
    hbmlm_lm_info.LM_addr_reg.sin_family = AF_INET;
    hbmlm_lm_info.LM_addr_reg.sin_addr.s_addr =
                        htonl( INADDR_ANY );
    hbmlm_lm_info.LM_addr_reg.sin_port =
                        htons( hbmlm_lm_info.LM_portNumReg );

    retcd = globus_l_hbmlm_set_up_tcp_port(
                        &sock_fd_reg,
                        &( hbmlm_lm_info.LM_addr_reg ));
    if(   ( retcd != GLOBUS_SUCCESS )
       && ( hbmlm_lm_info.LM_portNumReg != INADDR_ANY ))
    {
        hbmlm_lm_info.LM_addr_reg.sin_port = htons( INADDR_ANY );
        retcd = globus_l_hbmlm_set_up_tcp_port(
                        &sock_fd_reg,
                        &( hbmlm_lm_info.LM_addr_reg ));
    }
    if( retcd != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [08] in "
                        "main():\n"
                        "        globus_l_hbmlm_set_up_tcp_port() failed"
                        " for reg/unreg port.\n\n",
                        hbmlm_proc_name_str );
        if( hbmlm_log_file_fptr != stderr )
        {
            fclose( hbmlm_log_file_fptr );
        }

        return GLOBUS_FAILURE;
    }

    hbmlm_lm_info.LM_portNumReg =
                        ntohs( hbmlm_lm_info.LM_addr_reg.sin_port );

    DEBUGF2(    "Listening for regs/unregs on port %d\n",
                hbmlm_lm_info.LM_portNumReg );

    hbmlm_lm_info.LM_addr_rpt.sin_family = AF_INET;
    hbmlm_lm_info.LM_addr_rpt.sin_addr.s_addr = htonl( INADDR_ANY );
    hbmlm_lm_info.LM_addr_rpt.sin_port =
                        htons( hbmlm_lm_info.LM_portNumRpt );

    retcd = globus_l_hbmlm_set_up_udp_port(
                                &sock_fd_rpt,
                                &( hbmlm_lm_info.LM_addr_rpt ));
    if(   ( retcd != GLOBUS_SUCCESS )
       && ( hbmlm_lm_info.LM_portNumRpt != INADDR_ANY ))
    {
        hbmlm_lm_info.LM_addr_reg.sin_port = htons( INADDR_ANY );
        retcd = globus_l_hbmlm_set_up_udp_port(
                        &sock_fd_reg,
                        &( hbmlm_lm_info.LM_addr_rpt ));
    }
    if( retcd != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [09] in "
                        "main():\n"
                        "        globus_l_hbmlm_set_up_udp_port() failed"
                        " for heartbeat port.\n\n",
                        hbmlm_proc_name_str );
        close( sock_fd_reg );

        if( hbmlm_log_file_fptr != stderr )
        {
            fclose( hbmlm_log_file_fptr );
        }

        return GLOBUS_FAILURE;
    }

    hbmlm_lm_info.LM_portNumRpt =
                        ntohs( hbmlm_lm_info.LM_addr_rpt.sin_port );
    DEBUGF2(    "Sending reports on port %d.\n\n",
                hbmlm_lm_info.LM_portNumRpt );

    {
        struct sigaction act;

        memset(         (void *) &act,
                        0,
                        sizeof( struct sigaction ));
        act.sa_handler = globus_l_hbmlm_shutdown_quit;
        sigemptyset( &( act.sa_mask ));
        act.sa_flags = 0;

        sigaction(      SIGINT,
                        &act,
                        GLOBUS_NULL );
        sigaction(      SIGQUIT,
                        &act,
                        GLOBUS_NULL );
        sigaction(      SIGTERM,
                        &act,
                        GLOBUS_NULL );
    }

/*
**  Save the updated data to a new checkpoint file.
*/
    globus_l_hbmlm_write_ckpointfile();

    {
        globus_reltime_t     delay_time;

        GlobusTimeReltimeSet(delay_time, 0, 0);
        retcd_poll_add =
                    globus_callback_register_periodic(
                        &globus_hbmlm_callback_handle,
                        &delay_time,
                        &delay_time,
                        globus_l_hbmlm_poll,
                        GLOBUS_NULL,
                        GLOBUS_NULL,
                        GLOBUS_NULL );
    }
/*
**  Now write the status file.
*/
    if( globus_l_hbmlm_create_write_status_file(
                        hbmlm_status_fname_str ) ==
                                GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [11] in "
                        "main():\n"
                        "        globus_l_hbmlm_create_write_status_file() "
                        "failed.\n"
                        "            Status file name \"%s\".\n\n",
                        hbmlm_proc_name_str,
                        hbmlm_status_fname_str );

        if( retcd_poll_add == GLOBUS_SUCCESS )
        {
            if( globus_callback_unregister(
                        globus_hbmlm_callback_handle ) ==
                                        GLOBUS_FAILURE )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [12] in "
                        "main():\n"
                        "        globus_callback_unregister() "
                        "failed.\n\n",
                        hbmlm_proc_name_str );
            }
        }

        if( retcd_poll_activate == GLOBUS_SUCCESS )
        {
            if( globus_module_deactivate( GLOBUS_POLL_MODULE ) ==
                                        GLOBUS_FAILURE )
            {
                globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [13] in "
                        "main():\n"
                        "        globus_module_deactivate() failed for "
                        "GLOBUS_POLL_MODULE.\n\n",
                        hbmlm_proc_name_str );
            }
        }

        globus_free( hbmlm_lm_info.LM_hostName_str );
        globus_free( hbmlm_lm_info.LM_procNameFromPS_str );
        close( sock_fd_reg );
        close( sock_fd_rpt );
        if( hbmlm_log_file_fptr != stderr )
        {
            fclose( hbmlm_log_file_fptr );
        }

        return GLOBUS_FAILURE;
    }

#ifdef HBM_EXPERIMENT
/*
**  Code to get begin time and cpu usage.
*/
    retcd_time = gettimeofday( &time_monitoring_start, NULL );
    if( getrusage(      RUSAGE_SELF,
                        &rusage_hbm_before_processing ))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [14] in "
                        "main():\n"
                        "        HBM Experiment:  getrusage() failed at "
                        "beginning:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));

        rusage_hbm_before_processing.ru_utime.tv_sec =
                rusage_hbm_before_processing.ru_utime.tv_usec =
                rusage_hbm_before_processing.ru_stime.tv_sec =
                rusage_hbm_before_processing.ru_stime.tv_usec = 0;
    }
#endif

/**********************/
/*  Main Poll Loop    */
/**********************/

    if( retcd_poll_add == GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [15] in "
                        "main():\n"
                        "    Globus Poll problem -- Polling manually.\n\n",
                        hbmlm_proc_name_str );
        while( ! quit_now )
        {
            globus_abstime_infinity_wk.tv_sec =
                                        globus_i_abstime_infinity.tv_sec;
            globus_abstime_infinity_wk.tv_nsec =
                                        globus_i_abstime_infinity.tv_nsec;

            globus_l_hbmlm_poll(
                        &globus_abstime_infinity_wk,
                        GLOBUS_NULL );
        }
    }
    else
    {
        while( ! quit_now )
        {
            globus_poll_blocking();
        }
    }

/*
**  Clean up -- first remove the status file so noone else tries to register.
*/
    unlink( hbmlm_status_fname_str );

/*
**  Exiting --
**      Send out a last heartbeat for each client,
**      write one last checkpoint file.
*/
    globus_l_hbmlm_send_rpt_msgs();
    globus_l_hbmlm_write_ckpointfile();

#ifdef HBM_EXPERIMENT
/*
**  Code to get end time and cpu usage,
**  then calculate duration and usage.
*/
    retcd_time = gettimeofday(&time_monitoring_stop, NULL );
    if(getrusage_retcd = getrusage(RUSAGE_SELF,
                                     &rusage_hbm_after_processing ))
    {
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [16] in "
                        "main():\n"
                        "        HBM Experiment:  getrusage() failed at "
                        "end:  errno [%d]: %s.\n\n",
                        hbmlm_proc_name_str,
                        errno,
                        strerror( errno ));
    }
    else
    {
        rusage_hbm_used.ru_utime.tv_sec =
                                rusage_hbm_after_processing.ru_utime.tv_sec
                              - rusage_hbm_before_processing.ru_utime.tv_sec;
        if( rusage_hbm_after_processing.ru_utime.tv_usec >=
                                rusage_hbm_before_processing.ru_utime.tv_usec )
        {
            rusage_hbm_used.ru_utime.tv_usec =
                                rusage_hbm_after_processing.ru_utime.tv_usec
                              - rusage_hbm_before_processing.ru_utime.tv_usec;
        }
        else
        {
            rusage_hbm_used.ru_utime.tv_sec--;
            rusage_hbm_used.ru_utime.tv_usec =
                                1000000
                              + rusage_hbm_after_processing.ru_utime.tv_usec
                              - rusage_hbm_before_processing.ru_utime.tv_usec;
        }

        rusage_hbm_used.ru_stime.tv_sec =
                                rusage_hbm_after_processing.ru_stime.tv_sec
                              - rusage_hbm_before_processing.ru_stime.tv_sec;
        if( rusage_hbm_after_processing.ru_stime.tv_usec >=
                                rusage_hbm_before_processing.ru_stime.tv_usec )
        {
            rusage_hbm_used.ru_stime.tv_usec =
                                rusage_hbm_after_processing.ru_stime.tv_usec
                              - rusage_hbm_before_processing.ru_stime.tv_usec;
        }
        else
        {
            rusage_hbm_used.ru_stime.tv_sec--;
            rusage_hbm_used.ru_stime.tv_usec =
                                1000000
                              + rusage_hbm_after_processing.ru_stime.tv_usec
                              - rusage_hbm_before_processing.ru_stime.tv_usec;
        }

        time_monitoring_duration.tv_sec =
                                time_monitoring_stop.tv_sec
                              - time_monitoring_start.tv_sec;
        if( time_monitoring_stop.tv_usec >= time_monitoring_start.tv_usec )
        {
            time_monitoring_duration.tv_usec =
                                time_monitoring_stop.tv_usec
                              - time_monitoring_start.tv_usec;
        }
        else
        {
            time_monitoring_duration.tv_sec--;
            time_monitoring_duration.tv_usec =
                                1000000
                              + time_monitoring_stop.tv_usec
                              - time_monitoring_start.tv_usec;
        }

        time_hbm_before_total.tv_sec =
                                rusage_hbm_before_processing.ru_utime.tv_sec
                              + rusage_hbm_before_processing.ru_stime.tv_sec;
        time_hbm_before_total.tv_usec =
                                rusage_hbm_before_processing.ru_utime.tv_usec
                              + rusage_hbm_before_processing.ru_stime.tv_usec;
        if( time_hbm_before_total.tv_usec >= 1000000 )
        {
            time_hbm_before_total.tv_sec++;
            time_hbm_before_total.tv_usec -= 1000000;
        }

        time_hbm_after_total.tv_sec =
                                rusage_hbm_after_processing.ru_utime.tv_sec
                              + rusage_hbm_after_processing.ru_stime.tv_sec;
        time_hbm_after_total.tv_usec =
                                rusage_hbm_after_processing.ru_utime.tv_usec
                              + rusage_hbm_after_processing.ru_stime.tv_usec;
        if( time_hbm_after_total.tv_usec >= 1000000 )
        {
            time_hbm_after_total.tv_sec++;
            time_hbm_after_total.tv_usec -= 1000000;
        }

        time_hbm_used_total.tv_sec =
                                rusage_hbm_used.ru_utime.tv_sec
                              + rusage_hbm_used.ru_stime.tv_sec;
        time_hbm_used_total.tv_usec =
                                rusage_hbm_used.ru_utime.tv_usec
                              + rusage_hbm_used.ru_stime.tv_usec;
        if( time_hbm_used_total.tv_usec >= 1000000 )
        {
            time_hbm_used_total.tv_sec++;
            time_hbm_used_total.tv_usec -= 1000000;
        }

        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "HBM Experiment:  Time statistics for Local Monitor:\n"
                        "\n\n\n" );

        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "Real time:     after:  %12d.%6.6d\n\n",
                        time_monitoring_stop.tv_sec,
                        time_monitoring_stop.tv_usec );
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "Real time:    before:  %12d.%6.6d\n\n",
                        time_monitoring_start.tv_sec,
                        time_monitoring_start.tv_usec );
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "Real time:   elapsed:  %12d.%6.6d\n\n\n\n",
                        time_monitoring_duration.tv_sec,
                        time_monitoring_duration.tv_usec );


        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "After  processing values:\n"
                        "    user:    %12d.%6.6d"
                        "    system:  %12d.%6.6d"
                        "    total:   %12d.%6.6d\n\n",
                        rusage_hbm_after_processing.ru_utime.tv_sec,
                        rusage_hbm_after_processing.ru_utime.tv_usec,
                        rusage_hbm_after_processing.ru_stime.tv_sec,
                        rusage_hbm_after_processing.ru_stime.tv_usec,
                        time_hbm_after_total.tv_sec,
                        time_hbm_after_total.tv_usec );
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "Before processing values:\n"
                        "    user:    %12d.%6.6d"
                        "    system:  %12d.%6.6d"
                        "    total:   %12d.%6.6d\n\n",
                        rusage_hbm_before_processing.ru_utime.tv_sec,
                        rusage_hbm_before_processing.ru_utime.tv_usec,
                        rusage_hbm_before_processing.ru_stime.tv_sec,
                        rusage_hbm_before_processing.ru_stime.tv_usec,
                        time_hbm_before_total.tv_sec,
                        time_hbm_before_total.tv_usec );
        globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "Usage  processing values:\n"
                        "    user:    %12d.%6.6d"
                        "    system:  %12d.%6.6d"
                        "    total:   %12d.%6.6d\n\n",
                        rusage_hbm_used.ru_utime.tv_sec,
                        rusage_hbm_used.ru_utime.tv_usec,
                        rusage_hbm_used.ru_stime.tv_sec,
                        rusage_hbm_used.ru_stime.tv_usec,
                        time_hbm_used_total.tv_sec,
                        time_hbm_used_total.tv_usec );

    }
#endif

    if( retcd_poll_add == GLOBUS_SUCCESS )
    {
        if( globus_callback_unregister( globus_hbmlm_callback_handle ) ==
                                        GLOBUS_FAILURE )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [17] in "
                        "main():\n"
                        "        globus_callback_unregister() failed.\n\n",
                        hbmlm_proc_name_str );
        }
    }

    if( retcd_poll_activate == GLOBUS_SUCCESS )
    {
        if( globus_module_deactivate( GLOBUS_POLL_MODULE ) ==
                                        GLOBUS_FAILURE )
        {
            globus_libc_fprintf(
                        hbmlm_log_file_fptr,
                        "%s:\n"
                        "    Error [18] in "
                        "main():\n"
                        "        globus_module_deactivate() failed for"
                        "GLOBUS_POLL_MODULE.\n\n",
                        hbmlm_proc_name_str );
        }
    }

    globus_free( hbmlm_lm_info.LM_hostName_str );
    close( sock_fd_reg );
    close( sock_fd_rpt );
    if( hbmlm_log_file_fptr != stderr )
    {
        fclose( hbmlm_log_file_fptr );
    }

    return ( GLOBUS_SUCCESS );
}
