/*
************************************************************************
**                                                                    **
**  GUSTO Heartbeat Monitor Data Collector (HBM_DC) program           **
**                                                                    **
************************************************************************
*/
#define HBM_DC_C

#ifndef GLOBUS_HBM_VERSION
#define GLOBUS_HBM_VERSION 1
#endif

#include "globus_args.h"
#include "globus_hbm_codes.h"
#include "globus_hbm_datacollector.h"
#include "globus_hbm_defaults.h"

#include "hbm.h"


#ifdef DEBUG_MSGS
#define DEBUGF1(x)      globus_libc_fprintf( log_file,(x))
#define DEBUGF2(x,y)    globus_libc_fprintf( log_file,(x),(y))
#define DEBUGF3(x,y,z)  globus_libc_fprintf( log_file,(x),(y),(z))
#else
#define DEBUGF1(x)
#define DEBUGF2(x,y)
#define DEBUGF3(x,y,z)
#endif /* DEBUG_MSGS */

/*
** #define HBM_EXPERIMENT  is included in hbm_common.h
*/

/*
**  Argument fields.
*/
const char* usage_short =
    "globus_hbm_datacollector -chkpt <path> -cmdport # -hbport # "
    "-log <path> [ -reviewinterval # ] \n";

const char* usage_long =
    "\n"
    "globus_hbm_localmonitor options\n"
    "    Where options are:\n"
    "        -chkpt <path>    :  <path/filename> base for checkpoint file\n"
    "                            if a directory, then the filename used "
    "will be\n"
    "                               globus_hbm_datacollector.checkpoint\n"
    "        -cmdport #       :  the port number for the tcp port\n"
    "                            to listen to for checkpoint data requests\n"
    "        -hbport #        :  the port number for the udp port\n"
    "                            for receiving heartbeat messages\n"
    "        -log   <path>    :  <path/filename> base for (error) log file\n"
    "                            if a directory, then the filename used "
    "will be\n"
    "                               globus_hbm_datacollector.log\n"
    "      [ -reviewinterval # ] : \n"
    "                            the interval in seconds at which the status\n"
    "                            of client processes are to be reviewed\n";

#define ARGS_ID_CHKPT           1
#define ARGS_ID_CMDPORT         2
#define ARGS_ID_HBPORT          3
#define ARGS_ID_LOG             4
#define ARGS_ID_REVIEWINTERVAL  5
#define ARGS_ID_MAX             5

static char*    args_names_chkpt[]      = { "-chkpt",
                                            GLOBUS_NULL };
static char*    args_names_cmdport[]    = { "-cmdport",
                                            GLOBUS_NULL };
static char*    args_names_hbport[]     = { "-hbport",
                                            GLOBUS_NULL };
static char*    args_names_log[]        = { "-log",
                                            GLOBUS_NULL };
static char*    args_names_reviewinterval[] =
                                          { "-reviewinterval",
                                            GLOBUS_NULL };

static globus_args_valid_predicate_t
                args_validate_function_cmdport[] =
                                          { globus_validate_int };
static globus_args_valid_predicate_t
                args_validate_function_hbport[] =
                                          { globus_validate_int };
static globus_args_valid_predicate_t
                args_validate_function_reviewinterval[] =
                                          { globus_validate_int };

static globus_validate_int_parms_t
                args_validate_int_ge1_parms =
                                          { GLOBUS_VALIDATE_INT_MIN,
                                            1,
                                            0 };

static void*    args_validate_parms_cmdport[] =
                          { (void *) &args_validate_int_ge1_parms };
static void*    args_validate_parms_hbport[] =
                          { (void *) &args_validate_int_ge1_parms };
static void*    args_validate_parms_reviewinterval[] =
                          { (void *) &args_validate_int_ge1_parms };

static globus_args_option_descriptor_t
                args_option_list[] =      { { ARGS_ID_CHKPT,
                                              args_names_chkpt,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_CMDPORT,
                                              args_names_cmdport,
                                              1,
                                              args_validate_function_cmdport,
                                              args_validate_parms_cmdport },
                                            { ARGS_ID_HBPORT,
                                              args_names_hbport,
                                              1,
                                              args_validate_function_hbport,
                                              args_validate_parms_hbport },
                                            { ARGS_ID_LOG,
                                              args_names_log,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_REVIEWINTERVAL,
                                              args_names_reviewinterval,
                                              1,
                                              args_validate_function_reviewinterval,
                                              args_validate_parms_reviewinterval } };

globus_list_t*  args_options_found_list;
globus_args_option_instance_t*
                args_option;

globus_bool_t   arg_got_option_chkpt    = GLOBUS_FALSE;
globus_bool_t   arg_got_option_cmdport  = GLOBUS_FALSE;
globus_bool_t   arg_got_option_hbport   = GLOBUS_FALSE;
globus_bool_t   arg_got_option_log      = GLOBUS_FALSE;
globus_bool_t   arg_got_option_reviewinterval =
                                          GLOBUS_FALSE;

char*       arg_option_fname_chkpt_str  = GLOBUS_NULL;
char*       arg_option_fname_log_str    = GLOBUS_NULL;

short int   arg_option_cmdport          = INADDR_ANY;
short int   arg_option_hbport           = INADDR_ANY;
int         arg_option_reviewinterval   = HBMDC_DEFAULT_EVAL_INTERVAL_SECS;


#define GLOBUS_HBMDC_FNAME_CHKPT        "globus-hbm-datacollector.checkpoint"
#define GLOBUS_HBMDC_FNAME_LOG          "globus-hbm-datacollector.log"

char*       hbmdc_chkpt_fname_str       = GLOBUS_NULL;

char*       hbmdc_log_fname_str         = GLOBUS_NULL;
static FILE*    hbmdc_log_file_fptr     = GLOBUS_NULL;

u_short     hbmdc_command_port          = 0;
u_short     hbmdc_heartbeat_port        = 0;

int         hbmdc_review_interval       = 0;

char*       log_file_name_str           = GLOBUS_NULL;

char*       hbmdc_proc_name_str         = GLOBUS_NULL;

int     quit_now = 0;

#ifdef HBM_EXPERIMENT
    int                 retcd_time;

    struct timeval      time_monitoring_duration,
                        time_monitoring_start,
                        time_monitoring_stop;

    struct rusage       rusage_children,
                        rusage_hbm_before_processing,
                        rusage_hbm_after_processing,
                        rusage_hbm_used;

    struct timeval      time_hbm_before_total,
                        time_hbm_after_total,
                        time_hbm_used_total;

    int                 getrusage_retcd;
#endif /* defined HBM_EXPERIMENT */

/* xxx local function declarations */
static void
globus_l_hbmdc_callback(
                globus_hbm_datacollector_handle_t
                                        dc_handle,
                globus_hbm_client_callbackdata_t*
                                        client_callbackdata_ptr,
                globus_hbm_datacollector_client_entry_t*
                                        client_ptr,
                void*                   userdata_ptr );

static void
globus_l_hbmdc_callback_mail_msg(
                char*                   mail_address_str,
                char*                   mail_message_str );

static void
globus_l_hbmdc_callback_reg(
                globus_hbm_datacollector_handle_t
                                        dc_handle,
                globus_hbm_client_callbackdata_t*
                                        client_callbackdata_ptr,
                globus_hbm_datacollector_client_entry_t*
                                        client_ptr );

static void
globus_l_hbmdc_callback_shutdown(
                globus_hbm_datacollector_handle_t
                                        dc_handle,
                globus_hbm_client_callbackdata_t*
                                        client_callbackdata_ptr,
                globus_hbm_datacollector_client_entry_t*
                                        client_ptr );

static int
globus_l_hbmdc_get_arguments(
                globus_list_t* );

static void
globus_l_hbmdc_handle_tcp_request(
                int                     sock_fd_tcp,
                globus_hbm_datacollector_handle_t
                                        dc_handle );

static int
globus_l_hbmdc_set_up_tcp_port(
                int*                    sock_fd_tcp_ptr,
                struct sockaddr_in*     tcpaddr_in_ptr );

#ifdef  TARGET_ARCH_HPUX

static void
globus_l_hbmdc_shutdown_quit();

#else

static void
globus_l_hbmdc_shutdown_quit(
                int                     dummy );

#endif /* TARGET_ARCH_HPUX */


/*
************************************************************************
**  globus_l_hbmdc_callback                                           **
**      Parameters: handle for the data collector instance            **
**                  pointer to struct with the callback data          **
**                  pointer to struct with the client data            **
**                  pointer to user data                              **
**                                                                    **
**      Returns:    Nothing                                           **
**                                                                    **
************************************************************************
*/
static void
globus_l_hbmdc_callback(
                globus_hbm_datacollector_handle_t
                                        dc_handle,
                globus_hbm_client_callbackdata_t*
                                        client_callbackdata_ptr,
                globus_hbm_datacollector_client_entry_t*
                                        client_ptr,
                void*                   userdata_ptr )
{
    if(   ( client_callbackdata_ptr->cl_eventmask &
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_HEARTBEAT_RECEIVED )
       && ( client_ptr->CL_eventmask &
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_HEARTBEAT_RECEIVED ))
    {
        globus_l_hbmdc_callback_reg(
                        dc_handle,
                        client_callbackdata_ptr,
                        client_ptr );

        return;
    }

    if( client_callbackdata_ptr->cl_eventmask &
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_REGISTRATION )
    {
        globus_l_hbmdc_callback_reg(
                        dc_handle,
                        client_callbackdata_ptr,
                        client_ptr );
    }
    if( client_callbackdata_ptr->cl_eventmask &
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_SHUTDOWN_ABNORMAL )
    {
        globus_l_hbmdc_callback_shutdown(
                        dc_handle,
                        client_callbackdata_ptr,
                        client_ptr );
    }
}


/*
************************************************************************
**  globus_l_hbmdc_callback_mail_msg                                  **
**      Parameters: mail destination address                          **
**                  mail message                                      **
**                                                                    **
**      Returns:    Nothing                                           **
**                                                                    **
************************************************************************
*/
static void
globus_l_hbmdc_callback_mail_msg(
                char*                   mail_address_str,
                char*                   mail_message_str )
{
    FILE*   mail_fd                   = GLOBUS_NULL;
    char    command_str[512];

    if( mail_address_str != GLOBUS_NULL )
    {
        globus_libc_sprintf(
                        command_str,
                        "rmail %s",
                        mail_address_str );

        if(( mail_fd = popen( command_str, "w" )) == GLOBUS_NULL )
        {
            globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmdc_callback_mail_msg():\n"
                        "        popen() failed:  calling rmail to send "
                        "notification message\n"
                        "            Command string \"%s\"\n"
                        "            Message (re: situation recognized by "
                        "HBM data collector):\n\n"
                        "%s\n\n",
                        hbmdc_proc_name_str,
                        command_str,
                        mail_message_str );
        }
        else
        {
            globus_libc_fprintf(
                        mail_fd,
                        mail_message_str );

            pclose( mail_fd );
        }
    }

}


/*
************************************************************************
**  globus_l_hbmdc_callback_reg                                       **
**      Parameters: handle for the data collector instance            **
**                  pointer to struct with the callback data          **
**                  pointer to struct with the client data            **
**                                                                    **
**      Returns:    Nothing                                           **
**                                                                    **
************************************************************************
*/
static void
globus_l_hbmdc_callback_reg(
                globus_hbm_datacollector_handle_t
                                        dc_handle,
                globus_hbm_client_callbackdata_t*
                                        client_callbackdata_ptr,
                globus_hbm_datacollector_client_entry_t*
                                        client_ptr )
{
    char    msg_str[512];
    u_int   cl_eventmask_save;

    cl_eventmask_save = client_callbackdata_ptr->cl_eventmask;

    if(   ( client_callbackdata_ptr->cl_eventmask &
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_HEARTBEAT_RECEIVED )
       && ( client_ptr->CL_eventmask &
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_HEARTBEAT_RECEIVED ))
    {
        /*
        **  This heartbeat is for a client previously reported
        **      as SHUTDOWN_NO_RPT.
        **
        **  Let's send a status update report and get back to normal.
        */

        globus_libc_sprintf(
                        msg_str,
                        "Subject: GLOBUS/GUSTO HBM Client Status Report\n"
                        "Heartbeat recieved for process previously reported "
                        "with\n"
                        "with status SHUTDOWN\n"
                        "New Client Information:\n"
                        "    Host:           %s\n"
                        "    PID:            %d\n"
                        "    Process name:   %s\n",
                        client_ptr->CL_hostIPNum,
                        client_ptr->CL_procPID,
                        client_ptr->CL_procName );

        switch( client_ptr->CL_procStatus )
        {
            case GLOBUS_HBM_PROCSTATUS_ACTIVE:
                strcat( msg_str,
                        "    Process status:  ACTIVE\n" );
                break;

            case GLOBUS_HBM_PROCSTATUS_BLOCKED:
                strcat( msg_str,
                        "    Process status:  BLOCKED\n" );
                break;

            case GLOBUS_HBM_PROCSTATUS_OVERDUE:
                strcat( msg_str,
                        "    Process status:  OVERDUE\n" );
                break;

            case GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL:
                strcat( msg_str,
                        "    Process status:  SHUTDOWN - NORMAL\n" );
                break;

            case GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL:
                strcat( msg_str,
                        "    Process status:  SHUTDOWN - ABNORMAL\n" );
                break;

            case GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED:
                strcat( msg_str,
                        "    Process status:  SHUTDOWN - DIED\n" );
                break;

            default:
                strcat( msg_str,
                        "    Process status:  INVALID/UNKNOWN\n" );
        }

        globus_l_hbmdc_callback_mail_msg(
                        client_ptr->CL_Msg,
                        msg_str );

        if(   ( client_ptr->CL_procStatus ==
                        GLOBUS_HBM_PROCSTATUS_ACTIVE )
           || ( client_ptr->CL_procStatus ==
                        GLOBUS_HBM_PROCSTATUS_BLOCKED )
           || ( client_ptr->CL_procStatus ==
                        GLOBUS_HBM_PROCSTATUS_OVERDUE ))
        {
            client_callbackdata_ptr->cl_eventmask =
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_SHUTDOWN_ABNORMAL;
            globus_hbm_datacollector_set_clientevent_callback(
                        dc_handle,
                        client_callbackdata_ptr,
                        0,
                        0,
                        GLOBUS_NULL,
                        globus_l_hbmdc_callback );

            client_callbackdata_ptr->cl_eventmask = cl_eventmask_save;

            return;
        }
        client_callbackdata_ptr->cl_eventmask =
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_NONE;
        globus_hbm_datacollector_set_clientevent_callback(
                        dc_handle,
                        client_callbackdata_ptr,
                        0,
                        0,
                        GLOBUS_NULL,
                        globus_l_hbmdc_callback );

        client_callbackdata_ptr->cl_eventmask = cl_eventmask_save;

        return;
    }

    if( client_ptr->CL_Msg != GLOBUS_NULL )
    {
        if(( *( client_ptr->CL_Msg )) != '\0' )
        {
            if( client_ptr->CL_procStatus ==
                                    GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NO_RPT )
            {
/*
**              This callback could only have been generated by
**              a restore from checkpoint file.
*/
                cl_eventmask_save = client_callbackdata_ptr->cl_eventmask;
                client_callbackdata_ptr->cl_eventmask =
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_HEARTBEAT_RECEIVED;
                globus_hbm_datacollector_set_clientevent_callback(
                        dc_handle,
                        client_callbackdata_ptr,
                        0,
                        0,
                        GLOBUS_NULL,
                        globus_l_hbmdc_callback );

                client_callbackdata_ptr->cl_eventmask = cl_eventmask_save;
            }
            else
            {
/*
**              This callback could have been generated by
**              a restore from checkpoint file or
**              a heartbeat for a new client or
**              a heartbeat for a client with a new message.
**
**              It doesn't matter which case it is,
**              they all get treated the same.
*/
                client_callbackdata_ptr->cl_eventmask =
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_SHUTDOWN_ABNORMAL;
                globus_hbm_datacollector_set_clientevent_callback(
                        dc_handle,
                        client_callbackdata_ptr,
                        0,
                        0,
                        GLOBUS_NULL,
                        globus_l_hbmdc_callback );

                client_callbackdata_ptr->cl_eventmask = cl_eventmask_save;

                return;
            }
        }
    }

/*
**  There is no client message,
**  so we have no reason to set a callback.
*/
    client_callbackdata_ptr->cl_eventmask =
                        GLOBUS_HBM_DATACOLLECTOR_EVENT_NONE;

    globus_hbm_datacollector_set_clientevent_callback(
                        dc_handle,
                        client_callbackdata_ptr,
                        0,
                        0,
                        GLOBUS_NULL,
                        globus_l_hbmdc_callback );

    client_callbackdata_ptr->cl_eventmask = cl_eventmask_save;
}


/*
************************************************************************
**  globus_l_hbmdc_callback_shutdown                                  **
**      Parameters: handle for the data collector instance            **
**                  pointer to struct with the callback data          **
**                  pointer to struct with the client data            **
**                                                                    **
**      Returns:    Nothing                                           **
**                                                                    **
************************************************************************
*/
static void
globus_l_hbmdc_callback_shutdown(
                globus_hbm_datacollector_handle_t
                                        dc_handle,
                globus_hbm_client_callbackdata_t*
                                        client_callbackdata_ptr,
                globus_hbm_datacollector_client_entry_t*
                                        client_ptr )
{
    char    msg_str[512];
    u_int   cl_eventmask_save;


/*  (client_callbackdata_ptr->cl_eventmask ==
                    GLOBUS_HBM_DATACOLLECTOR_EVENT_SHUTDOWN_ABNORMAL  */

    globus_libc_sprintf(
                        msg_str,
                        "Subject: GLOBUS/GUSTO HBM Client Status Report\n"
                        "Abnormal shutdown recognized for client process.\n"
                        "    Host:           %s\n"
                        "    PID:            %d\n"
                        "    Process name:   %s\n",
                        client_ptr->CL_hostIPNum,
                        client_ptr->CL_procPID,
                        client_ptr->CL_procName );

    switch( client_ptr->CL_procStatus )
    {
        case GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL:
            strcat(     msg_str,
                        "    Process status:  SHUTDOWN - ABNORMAL\n" );
            break;

        case GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED:
            strcat(     msg_str,
                        "    Process status:  SHUTDOWN - DIED\n" );
            break;

        case GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NO_RPT:
            strcat(     msg_str,
                        "    Process status:  SHUTDOWN - NO HEARTBEAT "
                        "REPORTS\n" );
            break;

        default:
            strcat(     msg_str,
                        "    Process status:  INVALID/UNKNOWN\n" );
    }

    globus_l_hbmdc_callback_mail_msg(
                        client_ptr->CL_Msg,
                        msg_str );

    if( client_ptr->CL_procStatus == GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NO_RPT )
    {
        cl_eventmask_save = client_callbackdata_ptr->cl_eventmask;
        client_callbackdata_ptr->cl_eventmask =
                                GLOBUS_HBM_DATACOLLECTOR_EVENT_HEARTBEAT_RECEIVED;
        globus_hbm_datacollector_set_clientevent_callback(
                        dc_handle,
                        client_callbackdata_ptr,
                        0,
                        0,
                        GLOBUS_NULL,
                        globus_l_hbmdc_callback );

        client_callbackdata_ptr->cl_eventmask = cl_eventmask_save;
    }
}


static int
globus_l_hbmdc_get_arguments(
                globus_list_t*          args_options_found_list )
{
    globus_list_t*                      args_options_found_list_wk;
    globus_args_option_instance_t*      arg_option;

/*
**  First get all the options from the list.
**  Don't bother validating now,
**    because we will use the last value for each option,
**    so that is the only one that will need to be validated.
*/
    for( args_options_found_list_wk = args_options_found_list;
         !globus_list_empty( args_options_found_list_wk );
         args_options_found_list_wk =
                        globus_list_rest( args_options_found_list_wk ))
    {
        arg_option = globus_list_first( args_options_found_list_wk );
        if( arg_option->id_number == ARGS_ID_CHKPT )
        {
            arg_got_option_chkpt = GLOBUS_TRUE;
            if( arg_option_fname_chkpt_str != GLOBUS_NULL )
            {
                globus_free( arg_option_fname_chkpt_str );
            }
            arg_option_fname_chkpt_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 );
            strcpy(     arg_option_fname_chkpt_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_CMDPORT )
        {
            arg_got_option_cmdport = GLOBUS_TRUE;
            arg_option_cmdport = (short int) atoi( arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_HBPORT )
        {
            arg_got_option_hbport = GLOBUS_TRUE;
            arg_option_hbport = (short int) atoi( arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_LOG )
        {
            arg_got_option_log = GLOBUS_TRUE;
            if( arg_option_fname_log_str != GLOBUS_NULL )
            {
                globus_free( arg_option_fname_log_str );
            }
            arg_option_fname_log_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 );
            strcpy( arg_option_fname_log_str, arg_option->values[0]);
        }
        else if( arg_option->id_number == ARGS_ID_REVIEWINTERVAL )
        {
            arg_got_option_reviewinterval = GLOBUS_TRUE;
            arg_option_reviewinterval = atoi( arg_option->values[0] );
        }
        else
        {
            globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmdc_get_arguments():\n"
                        "        Parameter id number [%d] not recognized.\n"
                        "usage:\n"
                        "%s",
                        hbmdc_proc_name_str,
                        arg_option->id_number,
                        usage_short );

            return GLOBUS_FAILURE;
        }
    }
/*
**  Now validate the parameters.
**
**  First verify that all required parameters are present.
*/
    if(   ( arg_got_option_chkpt     != GLOBUS_TRUE )
       || ( arg_got_option_cmdport   != GLOBUS_TRUE )
       || ( arg_got_option_hbport    != GLOBUS_TRUE )
       || ( arg_got_option_log       != GLOBUS_TRUE ))
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmdc_get_arguments():\n"
                        "        Required argument(s) missing\n"
                        "usage:\n"
                        "%s",
                        hbmdc_proc_name_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

/*
**  Now verify the parameters in order.
**
**  Verify the checkpoint file name,
**             log file name, and
**             pid file name.
*/
    hbmdc_chkpt_fname_str =
                    globus_i_hbm_validate_path_get_filename(
                        arg_option_fname_chkpt_str,
                        GLOBUS_HBMDC_FNAME_CHKPT );
    if( hbmdc_chkpt_fname_str == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmdc_get_arguments():\n"
                        "        Argument \"-chkpt %s\"invalid\n"
                        "usage:\n"
                        "%s",
                        hbmdc_proc_name_str,
                        arg_option_fname_chkpt_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    hbmdc_command_port = arg_option_cmdport;

    hbmdc_heartbeat_port = arg_option_hbport;

    if( hbmdc_command_port == hbmdc_heartbeat_port )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmdc_get_arguments():\n"
                        "        Command/Checkpoint port [%d] and "
                        "Heartbeat port [%d] must differ.\n"
                        "%s\n\n",
                        hbmdc_proc_name_str,
                        hbmdc_command_port,
                        hbmdc_heartbeat_port,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    hbmdc_log_fname_str =
                    globus_i_hbm_validate_path_get_filename(
                        arg_option_fname_log_str,
                        GLOBUS_HBMDC_FNAME_LOG );
    if( hbmdc_log_fname_str == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmdc_get_arguments():\n"
                        "        Argument \"-log %s\"invalid\n"
                        "usage:\n"
                        "%s",
                        hbmdc_proc_name_str,
                        arg_option_fname_log_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    hbmdc_review_interval = arg_option_reviewinterval;

    return GLOBUS_SUCCESS;
}


/*
************************************************************************
**  globus_l_hbmdc_handle_tcp_request                                 **
**      Parameters: sock_fd_tcp         tcp fd of request             **
**                  dc_handle           handle for dc instance        **
**                                                                    **
**      Returns:    integer: GLOBUS_SUCCESS or GLOBUS_FAILURE         **
**                                 as appropriate                     **
**                                                                    **
**      This routine sets up a tcp port for receiving registration    **
**      and unregistration messages from clients, or for              **
**      sending report messages to data collectors.                   **
************************************************************************
*/
static void
globus_l_hbmdc_handle_tcp_request(
                int                     sock_fd_tcpreq,
                globus_hbm_datacollector_handle_t
                                        dc_handle )
{
    struct sockaddr_in  from_addr;

#if  (   (defined TARGET_ARCH_CRAYT3E) \
      || (defined TARGET_ARCH_HPUX) \
      || (defined TARGET_ARCH_IRIX) \
      || (defined TARGET_ARCH_SOLARIS))
    int                 from_addr_len;
#else
    size_t              from_addr_len;
#endif

    int                 n_bytes_read;

    int                 sizeof_u_int;
    char                read_buff[1024];
    int                 readwrite_fd;

    sizeof_u_int = sizeof( unsigned int );
/*
**  TCP report request message was recieved,                            **
**  so get the request and generate the report.                         **
*/

    from_addr_len = sizeof( from_addr );
    memset(             (void *) &from_addr,
                        0,
                        from_addr_len );
    from_addr.sin_family = AF_INET;

    readwrite_fd = accept( sock_fd_tcpreq,
                           (struct sockaddr*) &from_addr,
                           &from_addr_len );
    if( readwrite_fd == -1 )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmdc_handle_tcp_request():\n"
                        "        accept() failed:  errno [%d]: %s.\n\n",
                        hbmdc_proc_name_str,
                        errno,
                        strerror( errno ) );

        return;
    }

    n_bytes_read = read(
                        readwrite_fd,
                        read_buff,
                        sizeof( read_buff ));

    if( n_bytes_read == -1 )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmdc_handle_tcp_request():\n"
                        "        read() failed:  errno[%d]: %s.\n\n",
                        hbmdc_proc_name_str,
                        errno,
                        strerror( errno ));

        close( readwrite_fd );
        return;
    }

    read_buff[n_bytes_read] = '\0';

#ifdef DEBUG_BUFFERS
    globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    globus_l_hbmdc_handle_tcp_request():\n"
                        "        Received request:\n"
                        "            Length:  %d bytes\n"
                        "            Value:  \"%s\"\n\n",
                        hbmdc_proc_name_str,
                        n_bytes_read,
                        read_buff );
#endif /* DEBUG_BUFFERS */

    if(strncasecmp(     read_buff,
                        "Checkpoint data",
                        15 ))
    {
        close( readwrite_fd );
        if(strncasecmp( read_buff,
                        "Clear Shutdown Processes",
                        24 ))
        {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmdc_handle_tcp_request():\n"
                        "        Invalid request:\n"
                        "            Length:  %d bytes\n"
                        "            Value:  \"%s\"\n\n",
                        hbmdc_proc_name_str,
                        n_bytes_read,
                        read_buff );

            return;
        }
        globus_hbm_datacollector_clear_unregistered_clients( dc_handle );

        return;
    }

    write(  readwrite_fd,
            "<BEGIN CHECKPOINT>\n",
            19 );

    globus_hbm_datacollector_user_checkpoint(
                        dc_handle,
                        readwrite_fd );

    write(  readwrite_fd,
            "<END>\n",
            6 );

    close( readwrite_fd );

    return;
}


/*
************************************************************************
**  globus_l_hbm_set_up_tcp_port                                      **
**      Parameters: pointer to fd field to use for the socket         **
**                  pointer to the (already filled in) port address   **
**                                                                    **
**      Returns:    integer: GLOBUS_SUCCESS or GLOBUS_FAILURE         **
**                                 as appropriate                     **
**                                                                    **
**      This routine sets up a tcp port for receiving registration    **
**      and unregistration messages from clients, or for              **
**      sending report messages to data collectors.                   **
************************************************************************
*/
static int
globus_l_hbmdc_set_up_tcp_port(
                int*                    sock_fd_tcp_ptr,
                struct sockaddr_in*     tcpaddr_in_ptr )
{
    int             one = 1;
    struct linger   linger;

    if(( *sock_fd_tcp_ptr = socket( PF_INET, SOCK_STREAM, 0 )) < 0 )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmdc_set_up_tcp_port():\n"
                        "        socket() failed:  errno [%d]: %s.\n\n",
                        hbmdc_proc_name_str,
                        errno,
                        strerror( errno ));

        return GLOBUS_FAILURE;
    }

    if(setsockopt(      *sock_fd_tcp_ptr,
                        SOL_SOCKET,
                        SO_REUSEADDR,
                        (char*) &one,
                        sizeof( one )) < 0 )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmdc_set_up_tcp_port():\n"
                        "        setsockopt(SO_REUSEADDR) failed:  "
                        "errno [%d]: %s.\n\n",
                        hbmdc_proc_name_str,
                        errno,
                        strerror( errno ));
    }

    linger.l_onoff = 0;
    linger.l_linger = 0;
    if( setsockopt(     *sock_fd_tcp_ptr,
                        SOL_SOCKET,
                        SO_LINGER,
                        (char *) &linger,
                        sizeof( linger )) < 0 )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmdc_set_up_tcp_port():\n"
                        "        setsockopt(SO_LINGER) failed:  "
                        "errno [%d]: %s.\n\n",
                        hbmdc_proc_name_str,
                        errno,
                        strerror( errno ));
    }

    if( bind(   *sock_fd_tcp_ptr,
                (struct sockaddr *) tcpaddr_in_ptr,
                sizeof( *tcpaddr_in_ptr )) < 0 )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmdc_set_up_tcp_port():\n"
                        "        bind() failed:  errno [%d]: %s.\n\n",
                        hbmdc_proc_name_str,
                        errno,
                        strerror( errno ));

        return GLOBUS_FAILURE;
    }

    if( listen( *sock_fd_tcp_ptr, 5 ) < 0 )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmdc_set_up_tcp_port():\n"
                        "        listen() failed:  errno [%d]: %s.\n\n",
                        hbmdc_proc_name_str,
                        errno,
                        strerror( errno ));

        return GLOBUS_FAILURE;
    }

    return GLOBUS_SUCCESS;
}


/*
************************************************************************
**  globus_l_hbmdc_shutdown_quit                                      **
**      Parameters: dummy       integer parameter that is ignored.    **
**                                                                    **
**      Returns:    (Nothing)                                         **
**                                                                    **
**      This routine sets the "quit_now" flag to 1.                   **
************************************************************************
*/
#ifdef  TARGET_ARCH_HPUX

static void
globus_l_hbmdc_shutdown_quit()

#else

static void
globus_l_hbmdc_shutdown_quit(
                int                     dummy )

#endif /* TARGET_ARCH_HPUX */
{
    quit_now = 1;
}


/*
************************************************************************
**  main                                                              **
**      Parameters: program parameters (argc, argv)                   **
**      Returns:    integer return code                               **
**                                                                    **
************************************************************************
*/
int
main(           int                     argc,
                char                    *argv[] )
{
    globus_hbm_datacollector_handle_t
                dc_handle;
    globus_hbm_datacollector_handle_t*
                dc_handle_ptr = &dc_handle;

    int         retcd;

    int         sock_fd_tcpreq;
    struct sockaddr_in
                addr_tcpreq;

    int         num_live_clients;

    struct sigaction
                act;

    hbmdc_proc_name_str = argv[0];
    hbmdc_log_file_fptr = stderr;

/*
**  Activate the modules that we will be using.
*/
    if( globus_module_activate( GLOBUS_COMMON_MODULE ) !=
                                GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [05] in "
                        "main():\n"
                        "        globus_module_activate(GLOBUS_COMMON_MODULE) "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        return GLOBUS_FAILURE;
    }

    if( globus_module_activate( GLOBUS_HBM_DATACOLLECTOR_MODULE ) !=
                                GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [06] in "
                        "main():\n"
                        "        globus_module_activate"
                        "(GLOBUS_HBM_DATACOLLECTOR_MODULE) "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        return GLOBUS_FAILURE;
    }

    if( globus_module_activate( GLOBUS_POLL_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [07] in "
                        "main():\n"
                        "        globus_module_activate(GLOBUS_POLL_MODULE) "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        globus_module_deactivate( GLOBUS_HBM_DATACOLLECTOR_MODULE );
        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        return GLOBUS_FAILURE;
    }

    if(( retcd = globus_args_scan(
                        &argc,
                        &argv,
                        ARGS_ID_MAX,
                        args_option_list,
                        usage_short,
                        usage_long,
                        &args_options_found_list,
                        GLOBUS_NULL )) <= 0)
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [01] in "
                        "main():\n"
                        "        Argument error.\n"
                        "usage:\n"
                        "%s",
                        hbmdc_proc_name_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    if(( retcd = globus_l_hbmdc_get_arguments(
                        args_options_found_list )) !=
                                        GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [02] in "
                        "main():\n"
                        "        globus_l_hbmdc_get_arguments() failed.\n"
                        "usage:\n"
                        "%s",
                        hbmdc_proc_name_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    globus_args_option_instance_list_free(
                        &args_options_found_list );

/*
**  Open the log file.
*/
#ifndef DEBUG_STDERR
    if(( hbmdc_log_file_fptr = fopen( hbmdc_log_fname_str, "a" )) ==
                                GLOBUS_NULL )
    {
        hbmdc_log_file_fptr = stderr;

        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [03] in "
                        "main():\n"
                        "        fopen() failed for log file \"%s\".\n\n",
                        hbmdc_proc_name_str,
                        hbmdc_log_fname_str );

        return GLOBUS_FAILURE;
    }
#endif

/*
**  Open the command port
*/
    memset(             (void *) &addr_tcpreq,
                        0,
                        sizeof( addr_tcpreq ));
    addr_tcpreq.sin_family = AF_INET;
    addr_tcpreq.sin_addr.s_addr = htonl( INADDR_ANY );
    addr_tcpreq.sin_port = htons( hbmdc_command_port );
    if( globus_l_hbmdc_set_up_tcp_port(
                        &sock_fd_tcpreq,
                        &addr_tcpreq ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [04] in "
                        "main():\n"
                        "        globus_l_hbmdc_set_up_tcp_port() failed.\n\n",
                        hbmdc_proc_name_str );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        globus_hbm_free( hbmdc_chkpt_fname_str );
        globus_hbm_free( hbmdc_log_fname_str );

        return GLOBUS_FAILURE;
    }

/*
**  Create the data collector.
*/
    if( globus_hbm_datacollector_create(
                        &hbmdc_heartbeat_port,
                        hbmdc_review_interval,
                        0,
                        0,
                        0,
                        hbmdc_chkpt_fname_str,
                        hbmdc_chkpt_fname_str,
                        hbmdc_log_file_fptr,
                        globus_l_hbmdc_callback,
                        NULL,
                        dc_handle_ptr ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [08] in "
                        "main():\n"
                        "        globus_hbm_datacollector_create() "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        globus_module_deactivate( GLOBUS_HBM_DATACOLLECTOR_MODULE );
        globus_module_deactivate( GLOBUS_POLL_MODULE );
        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        globus_hbm_free( hbmdc_chkpt_fname_str );
        globus_hbm_free( hbmdc_log_fname_str );

        return GLOBUS_FAILURE;
    }

    quit_now = 0;
    memset(             (void *) &act,
                        0,
                        sizeof( struct sigaction ));
    act.sa_handler = globus_l_hbmdc_shutdown_quit;
    sigemptyset( &( act.sa_mask ));
    act.sa_flags = 0;

    sigaction( SIGINT,  &act, GLOBUS_NULL );
    sigaction( SIGQUIT, &act, GLOBUS_NULL );
    sigaction( SIGTERM, &act, GLOBUS_NULL );

/*
**  Register the tcp port (v1.0 hack).
*/
    if( globus_hbm_datacollector_reg_tcpfd(
                        sock_fd_tcpreq,
                        &globus_l_hbmdc_handle_tcp_request,
                        dc_handle ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [09] in "
                        "main():\n"
                        "        globus_hbm_datacollector_reg_tcpfd() "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        globus_module_deactivate( GLOBUS_HBM_DATACOLLECTOR_MODULE );
        globus_module_deactivate( GLOBUS_POLL_MODULE );
        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        globus_hbm_free( hbmdc_chkpt_fname_str );
        globus_hbm_free( hbmdc_log_fname_str );

        return GLOBUS_FAILURE;
    }

#ifdef HBM_EXPERIMENT
/*
**  Code to get begin time and cpu usage.
*/
    retcd_time = gettimeofday( &time_monitoring_start, NULL );
    if( getrusage( RUSAGE_SELF, &rusage_hbm_before_processing ))
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [10] in "
                        "main():\n"
                        "        HBM Experiment:  getrusage() failed at "
                        "beginning:  errno [%d]: %s.\n\n",
                        hbmdc_proc_name_str,
                        errno,
                        strerror( errno ));

        rusage_hbm_before_processing.ru_utime.tv_sec =
                rusage_hbm_before_processing.ru_utime.tv_usec =
                rusage_hbm_before_processing.ru_stime.tv_sec =
                rusage_hbm_before_processing.ru_stime.tv_usec = 0;
    }
#endif /* HBM_EXPERIMENT */

/*
**  Main Poll Loop.
**  (Don't need to add because the API did that.)
*/
    while( ! quit_now )
    {
        globus_poll_blocking();
    }

/*
**  Clean up.
*/

#ifdef HBM_EXPERIMENT
/*
**  Code to get end time and cpu usage,
**  then calculate duration and usage.
*/
    retcd_time = gettimeofday( &time_monitoring_stop, NULL );
    if( getrusage_retcd = getrusage(    RUSAGE_SELF,
                                        &rusage_hbm_after_processing ))
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [11] in "
                        "main():\n"
                        "        HBM Experiment:  getrusage() failed at "
                        "end:  errno [%d]: %s.\n\n",
                        hbmdc_proc_name_str,
                        errno,
                        strerror( errno ));
    }
    else
    {
        rusage_hbm_used.ru_utime.tv_sec =
                                rusage_hbm_after_processing.ru_utime.tv_sec
                              - rusage_hbm_before_processing.ru_utime.tv_sec;
        if( rusage_hbm_after_processing.ru_utime.tv_usec >=
                                rusage_hbm_before_processing.ru_utime.tv_usec )
        {
            rusage_hbm_used.ru_utime.tv_usec =
                                rusage_hbm_after_processing.ru_utime.tv_usec
                              - rusage_hbm_before_processing.ru_utime.tv_usec;
        }
        else
        {
            rusage_hbm_used.ru_utime.tv_sec--;
            rusage_hbm_used.ru_utime.tv_usec =
                                1000000
                              + rusage_hbm_after_processing.ru_utime.tv_usec
                              - rusage_hbm_before_processing.ru_utime.tv_usec;
        }

        rusage_hbm_used.ru_stime.tv_sec =
                                rusage_hbm_after_processing.ru_stime.tv_sec
                              - rusage_hbm_before_processing.ru_stime.tv_sec;
        if( rusage_hbm_after_processing.ru_stime.tv_usec >=
                                rusage_hbm_before_processing.ru_stime.tv_usec )
        {
            rusage_hbm_used.ru_stime.tv_usec =
                                rusage_hbm_after_processing.ru_stime.tv_usec
                              - rusage_hbm_before_processing.ru_stime.tv_usec;
        }
        else
        {
            rusage_hbm_used.ru_stime.tv_sec--;
            rusage_hbm_used.ru_stime.tv_usec =
                                1000000
                              + rusage_hbm_after_processing.ru_stime.tv_usec
                              - rusage_hbm_before_processing.ru_stime.tv_usec;
        }

        time_monitoring_duration.tv_sec =
                                time_monitoring_stop.tv_sec
                              - time_monitoring_start.tv_sec;
        if( time_monitoring_stop.tv_usec >= time_monitoring_start.tv_usec )
        {
            time_monitoring_duration.tv_usec =
                                time_monitoring_stop.tv_usec
                              - time_monitoring_start.tv_usec;
        }
        else
        {
            time_monitoring_duration.tv_sec--;
            time_monitoring_duration.tv_usec =
                                1000000
                              + time_monitoring_stop.tv_usec
                              - time_monitoring_start.tv_usec;
        }

        time_hbm_before_total.tv_sec =
                                rusage_hbm_before_processing.ru_utime.tv_sec
                              + rusage_hbm_before_processing.ru_stime.tv_sec;
        time_hbm_before_total.tv_usec =
                                rusage_hbm_before_processing.ru_utime.tv_usec
                              + rusage_hbm_before_processing.ru_stime.tv_usec;
        if( time_hbm_before_total.tv_usec >= 1000000 )
        {
            time_hbm_before_total.tv_sec++;
            time_hbm_before_total.tv_usec -= 1000000;
        }

        time_hbm_after_total.tv_sec =
                                rusage_hbm_after_processing.ru_utime.tv_sec
                              + rusage_hbm_after_processing.ru_stime.tv_sec;
        time_hbm_after_total.tv_usec =
                                rusage_hbm_after_processing.ru_utime.tv_usec
                              + rusage_hbm_after_processing.ru_stime.tv_usec;
        if( time_hbm_after_total.tv_usec >= 1000000 )
        {
            time_hbm_after_total.tv_sec++;
            time_hbm_after_total.tv_usec -= 1000000;
        }

        time_hbm_used_total.tv_sec =
                                rusage_hbm_used.ru_utime.tv_sec
                              + rusage_hbm_used.ru_stime.tv_sec;
        time_hbm_used_total.tv_usec =
                                rusage_hbm_used.ru_utime.tv_usec
                              + rusage_hbm_used.ru_stime.tv_usec;
        if( time_hbm_used_total.tv_usec >= 1000000 )
        {
            time_hbm_used_total.tv_sec++;
            time_hbm_used_total.tv_usec -= 1000000;
        }

        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "HBM Experiment:  Time statistics for Data Collector:\n"
                        "\n\n\n" );

        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "    Real time:     after:  %12d.%6.6d\n\n",
                        time_monitoring_stop.tv_sec,
                        time_monitoring_stop.tv_usec );
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "    Real time:    before:  %12d.%6.6d\n\n",
                        time_monitoring_start.tv_sec,
                        time_monitoring_start.tv_usec );
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "    Real time:   elapsed:  %12d.%6.6d\n\n\n\n",
                        time_monitoring_duration.tv_sec,
                        time_monitoring_duration.tv_usec );


        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "After  processing values:\n"
                        "    user:    %12d.%6.6d"
                        "    system:  %12d.%6.6d"
                        "    total:   %12d.%6.6d\n\n",
                        rusage_hbm_after_processing.ru_utime.tv_sec,
                        rusage_hbm_after_processing.ru_utime.tv_usec,
                        rusage_hbm_after_processing.ru_stime.tv_sec,
                        rusage_hbm_after_processing.ru_stime.tv_usec,
                        time_hbm_after_total.tv_sec,
                        time_hbm_after_total.tv_usec );
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "Before processing values:\n"
                        "    user:    %12d.%6.6d"
                        "    system:  %12d.%6.6d"
                        "    total:   %12d.%6.6d\n\n",
                        rusage_hbm_before_processing.ru_utime.tv_sec,
                        rusage_hbm_before_processing.ru_utime.tv_usec,
                        rusage_hbm_before_processing.ru_stime.tv_sec,
                        rusage_hbm_before_processing.ru_stime.tv_usec,
                        time_hbm_before_total.tv_sec,
                        time_hbm_before_total.tv_usec );
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "Usage  processing values:\n"
                        "    user:    %12d.%6.6d"
                        "    system:  %12d.%6.6d"
                        "    total:   %12d.%6.6d\n\n",
                        rusage_hbm_used.ru_utime.tv_sec,
                        rusage_hbm_used.ru_utime.tv_usec,
                        rusage_hbm_used.ru_stime.tv_sec,
                        rusage_hbm_used.ru_stime.tv_usec,
                        time_hbm_used_total.tv_sec,
                        time_hbm_used_total.tv_usec );

    }
#endif /* HBM_EXPERIMENT */

    globus_hbm_datacollector_user_checkpoint(
                        dc_handle,
                        -1 );

    if( globus_hbm_datacollector_unreg_tcpfd(
                        sock_fd_tcpreq,
                        dc_handle ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [12] in "
                        "main():\n"
                        "        globus_hbm_datacollector_unreg_tcpfd() "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        globus_module_deactivate( GLOBUS_HBM_DATACOLLECTOR_MODULE );
        globus_module_deactivate( GLOBUS_POLL_MODULE );
        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        globus_hbm_free( hbmdc_chkpt_fname_str );
        globus_hbm_free( hbmdc_log_fname_str );

        return GLOBUS_FAILURE;
    }
    close( sock_fd_tcpreq );

    if( globus_hbm_datacollector_destroy(
                        dc_handle,
                        GLOBUS_HBM_DATACOLLECTOR_FORCE_DESTROY_YES,
                        &num_live_clients ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [13] in "
                        "main():\n"
                        "        globus_hbm_datacollector_destroy() "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        globus_module_deactivate( GLOBUS_HBM_DATACOLLECTOR_MODULE );
        globus_module_deactivate( GLOBUS_POLL_MODULE );
        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        globus_hbm_free( hbmdc_chkpt_fname_str );
        globus_hbm_free( hbmdc_log_fname_str );

        return GLOBUS_FAILURE;
    }
    if( num_live_clients )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [14] in "
                        "main():\n"
                        "        globus_hbm_datacollector_destroy():\n"
                        "            %d live clients remained at shutdown.\n\n",
                        hbmdc_proc_name_str,
                        num_live_clients );
    }

    if( globus_module_deactivate( GLOBUS_HBM_DATACOLLECTOR_MODULE ) !=
                                GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [15] in "
                        "main():\n"
                        "        globus_hbm_datacollector_destroy"
                        "(GLOBUS_HBM_DATACOLLECTOR_MODULE) "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        globus_module_deactivate( GLOBUS_POLL_MODULE );
        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        globus_hbm_free( hbmdc_chkpt_fname_str );
        globus_hbm_free( hbmdc_log_fname_str );

        return GLOBUS_FAILURE;
    }
    if( globus_module_deactivate( GLOBUS_POLL_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [16] in "
                        "main():\n"
                        "        globus_hbm_datacollector_destroy"
                        "(GLOBUS_POLL_MODULE) "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        globus_hbm_free( hbmdc_chkpt_fname_str );
        globus_hbm_free( hbmdc_log_fname_str );

        return GLOBUS_FAILURE;
    }
    if( globus_module_deactivate( GLOBUS_COMMON_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        hbmdc_log_file_fptr,
                        "%s:\n"
                        "    Error [17] in "
                        "main():\n"
                        "        globus_hbm_datacollector_destroy"
                        "(GLOBUS_COMMON_MODULE) "
                        "failed.\n\n",
                        hbmdc_proc_name_str );

        if( hbmdc_log_file_fptr != stderr )
        {
            fclose( hbmdc_log_file_fptr );
        }

        globus_hbm_free( hbmdc_chkpt_fname_str );
        globus_hbm_free( hbmdc_log_fname_str );

        return GLOBUS_FAILURE;
    }

    if( hbmdc_log_file_fptr != stderr )
    {
        fclose( hbmdc_log_file_fptr );
    }

    globus_hbm_free( hbmdc_chkpt_fname_str );
    globus_hbm_free( hbmdc_log_fname_str );

    return GLOBUS_SUCCESS;
}
