/*
************************************************************************
**                                                                    **
**  hbm_monitor_proc.c - source file for external                     **
**                       heartbeat client routines                    **
**                                                                    **
************************************************************************
*/
#define HBM_MONITOR_PROC_C

#include "globus_args.h"

#include "globus_hbm_client.h"
#include "globus_hbm_defaults.h"

#include "hbm.h"


/*
**  Argument fields.
*/
const char* usage_short =
    "globus_hbm_client_register "
        "{ -register | -reg | -unregabnormal | -ua | "
            " -unregnormal | un | -u } "
        "-pid # "
        "{ -dcspec <rsl dc specification string> | "
            "{ { -dchost <hostname> | -dchost <ipaddress> } "
              "-dcport # "
              "[ -rptname <string> ] "
              "[ -rptinterval # ] "
              "[ -msg <string> ] } } "
        "[ -requireall | -reqall ] "
        " { -lmstatus <path> | -lmstat <path } "
        " [ -lmvalidatesecs # ]\n\n";

const char* usage_long =
    "\n"
    "globus_hbm_client_register options\n\n"
    "    Where options are:\n\n"
    "        registration/unregistration code, one of the following:\n"
    "            -register | -reg | -r   :\n"
    "                    register the designated client\n"
    "            -unregabnormal | -ua    :\n"
    "                    unregister the designated client due to\n"
    "                    termination from (trapped) fault\n"
    "            -unregnormal | -un | -u :\n"
    "                    unregister the designated client due to\n"
    "                    normal termination\n\n"
    "        -pid #:     pid of the client process to register/unregister\n\n"
    "                    data collector specification, using either:\n"
    "            -dcspec <rsl dc specification string> :\n"
    "                    data collector specification(s) as an rsl "
    "string\n"
    "                    e.g.,\n"
    "                    &(Host=999.999.999.999)(Portnum=99999)"
    "(Rptname=xxxx)\n"
    "                     (Interval=9999)(Message=xxxx)\n"
    "        or the following:\n"
    "            -dchost <hostname> | -dchost <ipnum> :\n"
    "                    data collector host address as fully qualified "
    "host name\n"
    "                    or as IP number\n"
    "            -dcport #:\n"
    "                    data collector port number to which heartbeats "
    "are to be sent\n"
    "          [ -rptname <name>] :\n"
    "                    name <name> to be used by Local Monitor when\n"
    "                    sending heartbeats to the designated DC\n"
    "          [ -rptinterval # ] :\n"
    "                    interval in seconds at which heartbeats for "
    "this\n"
    "                    client are to be sent to the designated DC\n"
    "          [ -msg <string> ] :\n"
    "                    message string to send with each heartbeat "
    "for\n"
    "                    this client that is sent to the designated "
    "DC\n"
    "                    If the client process is being registered\n"
    "                    with the Globus system Data Collector, then\n"
    "                    <string> should be the e-mail address of\n"
    "                    the responsible person at the site to whom\n"
    "                    messages should be sent if the process\n"
    "                    terminates abnormally\n\n"
    "      [ -requireall | -reqall ] :\n"
    "                    require all registrations to be valid and\n"
    "                    successful, otherwise do none of them\n"
    "      { -lmstatus <path> | -lmstat <path } :\n"
    "                    local monitor status file to use to get\n"
    "                    information about the local monitor for this host,\n"
    "                    including the local monitor pid, process name, and\n"
    "                    the port number for "
    "registrations/unregistrations\n"
    "      [ -lmvalidatesecs # ]:\n"
    "                    the length of the time period in seconds\n"
    "                    over which the validation of the Local Monitor\n"
    "                    process with which the client process is to be\n"
    "                    registered/unregistered should be retried before\n"
    "                    declaring failure.\n"
    "                    (Default 180, -1 means keep trying forever).\n\n";

#define ARGS_ID_REGISTER                1
#define ARGS_ID_UNREGISTER_ABNORMAL     2
#define ARGS_ID_UNREGISTER_NORMAL       3
#define ARGS_ID_PID                     4
#define ARGS_ID_DCSPEC                  5
#define ARGS_ID_DCHOST                  6
#define ARGS_ID_DCPORT                  7
#define ARGS_ID_RPTNAME                 8
#define ARGS_ID_RPTINTERVAL             9
#define ARGS_ID_MSG                    10
#define ARGS_ID_REQUIREALL             11
#define ARGS_ID_LMSTATUS               12
#define ARGS_ID_LMVALIDATESECS         13
#define ARGS_ID_MAX                    13

static char*    args_names_register[]   = { "-register",
                                            "-reg",
                                            "-r",
                                            GLOBUS_NULL };
static char*    args_names_unregister_abnormal[] =
                                          { "-unregisterabnormal",
                                            "-ua",
                                            GLOBUS_NULL };
static char*    args_names_unregister_normal[] =
                                          { "-unregisternormal",
                                            "-un",
                                            "-u",
                                            GLOBUS_NULL };
static char*    args_names_pid[]        = { "-pid",
                                            GLOBUS_NULL };
static char*    args_names_dcspec[]     = { "-dcspec",
                                            GLOBUS_NULL };
static char*    args_names_dchost[]     = { "-dchost",
                                            GLOBUS_NULL };
static char*    args_names_dcport[]     = { "-dcport",
                                            GLOBUS_NULL };
static char*    args_names_rptname[]    = { "-rptname",
                                            GLOBUS_NULL };
static char*    args_names_rptinterval[] =
                                          { "-rptinterval",
                                            GLOBUS_NULL };
static char*    args_names_msg[]        = { "-msg",
                                            GLOBUS_NULL };
static char*    args_names_requireall[] = { "-requireall",
                                            "-reqall",
                                            GLOBUS_NULL };
static char*    args_names_lmstatus[]   = { "-lmstatus",
                                            "-lmstat",
                                            GLOBUS_NULL };
static char*    args_names_lmvalidatesecs[] =
                                          { "-lmvalidatesecs",
                                            GLOBUS_NULL };

static globus_args_valid_predicate_t
                args_validate_function_pid[] =
                                          { globus_validate_int };
static globus_args_valid_predicate_t
                args_validate_function_dcport[] =
                                          { globus_validate_int };
static globus_args_valid_predicate_t
                args_validate_function_rptinterval[] =
                                          { globus_validate_int };
static globus_args_valid_predicate_t
                args_validate_function_lmvalidatesecs[] =
                                          { globus_validate_int };

static globus_validate_int_parms_t
                args_validate_int_gem1_parms =
                                          { GLOBUS_VALIDATE_INT_MIN,
                                            -1,
                                            0 };
static globus_validate_int_parms_t
                args_validate_int_ge0_parms =
                                          { GLOBUS_VALIDATE_INT_MIN,
                                            0,
                                            0 };
static globus_validate_int_parms_t
                args_validate_int_ge1_parms =
                                          { GLOBUS_VALIDATE_INT_MIN,
                                            1,
                                            0 };

static void*    args_validate_parms_pid[] =
                          { (void *) &args_validate_int_ge0_parms };
static void*    args_validate_parms_dcport[] =
                          { (void *) &args_validate_int_ge1_parms };
static void*    args_validate_parms_rptinterval[] =
                          { (void *) &args_validate_int_ge0_parms };
static void*    args_validate_parms_lmvalidatesecs[] =
                          { (void *) &args_validate_int_gem1_parms };

static globus_args_option_descriptor_t
                args_option_list[]      = { { ARGS_ID_REGISTER,
                                              args_names_register,
                                              0,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_UNREGISTER_ABNORMAL,
                                              args_names_unregister_abnormal,
                                              0,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_UNREGISTER_NORMAL,
                                              args_names_unregister_normal,
                                              0,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_PID,
                                              args_names_pid,
                                              1,
                                              args_validate_function_pid,
                                              args_validate_parms_pid },
                                            { ARGS_ID_DCSPEC,
                                              args_names_dcspec,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_DCHOST,
                                              args_names_dchost,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_DCPORT,
                                              args_names_dcport,
                                              1,
                                              args_validate_function_dcport,
                                              args_validate_parms_dcport },
                                            { ARGS_ID_RPTNAME,
                                              args_names_rptname,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_RPTINTERVAL,
                                              args_names_rptinterval,
                                              1,
                                              args_validate_function_rptinterval,
                                              args_validate_parms_rptinterval },
                                            { ARGS_ID_MSG,
                                              args_names_msg,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_REQUIREALL,
                                              args_names_requireall,
                                              0,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_LMSTATUS,
                                              args_names_lmstatus,
                                              1,
                                              GLOBUS_NULL,
                                              GLOBUS_NULL },
                                            { ARGS_ID_LMVALIDATESECS,
                                              args_names_lmvalidatesecs,
                                              1,
                                              args_validate_function_lmvalidatesecs,
                                              args_validate_parms_lmvalidatesecs } };

globus_list_t*  args_options_found_list;
globus_args_option_instance_t*
                args_option;

globus_bool_t   arg_got_option_register = GLOBUS_FALSE;
globus_bool_t   arg_got_option_unregister_abnormal =
                                          GLOBUS_FALSE;
globus_bool_t   arg_got_option_unregister_normal =
                                          GLOBUS_FALSE;
globus_bool_t   arg_got_option_pid      = GLOBUS_FALSE;
globus_bool_t   arg_got_option_dcspec   = GLOBUS_FALSE;
globus_bool_t   arg_got_option_dchost   = GLOBUS_FALSE;
globus_bool_t   arg_got_option_dcport   = GLOBUS_FALSE;
globus_bool_t   arg_got_option_rptname  = GLOBUS_FALSE;
globus_bool_t   arg_got_option_rptinterval =
                                          GLOBUS_FALSE;
globus_bool_t   arg_got_option_msg      = GLOBUS_FALSE;
globus_bool_t   arg_got_option_requireall =
                                          GLOBUS_FALSE;
globus_bool_t   arg_got_option_lmstatus = GLOBUS_FALSE;
globus_bool_t   arg_got_option_lmvalidatesecs =
                                          GLOBUS_FALSE;

char*           arg_option_dcspec_str   = GLOBUS_NULL;
char*           arg_option_dchost_str   = GLOBUS_NULL;
char*           arg_option_dcport_str   = GLOBUS_NULL;
char*           arg_option_rptname_str  = GLOBUS_NULL;
char*           arg_option_rptinterval_str =
                                          GLOBUS_NULL;
char*           arg_option_msg_str      = GLOBUS_NULL;
char*           arg_option_fname_lmstatus_str =
                                          GLOBUS_NULL;

int             arg_option_register_unregister =
                                          0;
int             arg_option_pid          = 0;
int             arg_option_lmvalidatesecs =
                                          HBMCL_DEFAULT_LM_VALIDATE_SECONDS;

globus_bool_t   arg_option_requireall   = GLOBUS_FALSE;


#if !defined(MAXPATHLEN)
#define MAXPATHLEN PATH_MAX
#endif

char*           lm_status_fname_str     = GLOBUS_NULL;

char*           hbmcl_proc_name_str     = GLOBUS_NULL;


static int
globus_l_hbmcl_get_arguments(
                globus_list_t* );


static int
globus_l_hbmcl_get_arguments(
                globus_list_t*          args_options_found_list )
{
    globus_list_t*                      args_options_found_list_wk;
    globus_args_option_instance_t*      arg_option;

/*
**  First get all the options from the list.
**  Don't bother validating now,
**    because we will use the last value for each option,
**    so that is the only one that will need to be validated.
*/
    for( args_options_found_list_wk = args_options_found_list;
         !globus_list_empty( args_options_found_list_wk );
         args_options_found_list_wk =
                        globus_list_rest( args_options_found_list_wk ))
    {
        arg_option = globus_list_first( args_options_found_list_wk );
        if( arg_option->id_number == ARGS_ID_REGISTER )
        {
            arg_got_option_register = GLOBUS_TRUE;
            arg_option_register_unregister = GLOBUS_HBM_MSGTYPE_REGISTER;
        }
        else if( arg_option->id_number == ARGS_ID_UNREGISTER_ABNORMAL )
        {
            arg_got_option_unregister_abnormal = GLOBUS_TRUE;
            arg_option_register_unregister = GLOBUS_HBM_MSGTYPE_SHUTDOWN_ABNORMAL;
        }
        else if( arg_option->id_number == ARGS_ID_UNREGISTER_NORMAL )
        {
            arg_got_option_unregister_normal = GLOBUS_TRUE;
            arg_option_register_unregister = GLOBUS_HBM_MSGTYPE_SHUTDOWN_NORMAL;
        }
        else if( arg_option->id_number == ARGS_ID_PID )
        {
            arg_got_option_pid = GLOBUS_TRUE;
            arg_option_pid = atoi( arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_DCSPEC )
        {
            arg_got_option_dcspec = GLOBUS_TRUE;
            if( arg_option_dcspec_str != GLOBUS_NULL )
            {
                globus_free( arg_option_dcspec_str );
            }
            arg_option_dcspec_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 );
            strcpy(     arg_option_dcspec_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_DCHOST )
        {
            arg_got_option_dchost = GLOBUS_TRUE;
            if( arg_option_dchost_str != GLOBUS_NULL )
            {
                globus_free( arg_option_dchost_str );
            }
            arg_option_dchost_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 );
            strcpy(     arg_option_dchost_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_DCPORT )
        {
            arg_got_option_dcport = GLOBUS_TRUE;
            arg_option_dcport_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 );
            strcpy(     arg_option_dcport_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_RPTNAME )
        {
            arg_got_option_rptname = GLOBUS_TRUE;
            if( arg_option_rptname_str != GLOBUS_NULL )
            {
                globus_free( arg_option_rptname_str );
            }
            arg_option_rptname_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 );
            strcpy( arg_option_rptname_str, arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_RPTINTERVAL )
        {
            arg_got_option_rptinterval = GLOBUS_TRUE;
            arg_option_rptinterval_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 );
            strcpy(     arg_option_rptinterval_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_MSG )
        {
            arg_got_option_msg = GLOBUS_TRUE;
            arg_option_msg_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 );
            strcpy(     arg_option_msg_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_REQUIREALL )
        {
            arg_got_option_requireall = GLOBUS_TRUE;
            arg_option_requireall = GLOBUS_TRUE;
        }
        else if( arg_option->id_number == ARGS_ID_LMSTATUS )
        {
            arg_got_option_lmstatus = GLOBUS_TRUE;
            if( arg_option_fname_lmstatus_str != GLOBUS_NULL )
            {
                globus_free( arg_option_fname_lmstatus_str );
            }
            arg_option_fname_lmstatus_str =
                        globus_malloc( strlen( arg_option->values[0] ) + 1 );
            strcpy(     arg_option_fname_lmstatus_str,
                        arg_option->values[0] );
        }
        else if( arg_option->id_number == ARGS_ID_LMVALIDATESECS )
        {
            arg_got_option_lmvalidatesecs = GLOBUS_TRUE;
            arg_option_lmvalidatesecs = atoi( arg_option->values[0] );
        }
        else
        {
            globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [01] in "
                        "globus_l_hbmcl_get_arguments():\n"
                        "    Parameter id number [%d] not recognized.\n"
                        "usage:\n"
                        "%s",
                        hbmcl_proc_name_str,
                        arg_option->id_number,
                        usage_short );

            return GLOBUS_FAILURE;
        }
    }
/*
**  Now validate the parameters.
**
**  First verify that all required parameters are present,
**  and that the combination of parameters is valid.
*/
    if(   (   ( arg_got_option_register             != GLOBUS_TRUE )
           && ( arg_got_option_unregister_abnormal  != GLOBUS_TRUE )
           && ( arg_got_option_unregister_normal    != GLOBUS_TRUE ))
       || ( arg_got_option_pid                      != GLOBUS_TRUE )
       || ( arg_got_option_lmstatus                 != GLOBUS_TRUE ))
    {
        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [02] in "
                        "globus_l_hbmcl_get_arguments():\n"
                        "    Required parameter(s) missing.\n"
                        "usage:\n"
                        "%s",
                        hbmcl_proc_name_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    if(   (   ( arg_got_option_register             == GLOBUS_TRUE )
           && ( arg_got_option_unregister_abnormal  == GLOBUS_TRUE ))
       || (   ( arg_got_option_register             == GLOBUS_TRUE )
           && ( arg_got_option_unregister_normal    == GLOBUS_TRUE ))
       || (   ( arg_got_option_unregister_abnormal  == GLOBUS_TRUE )
           && ( arg_got_option_unregister_normal    == GLOBUS_TRUE )))
    {
        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [03] in "
                        "globus_l_hbmcl_get_arguments():\n"
                        "        Cannot register and unregister at the "
                        "same time.\n"
                        "usage:\n"
                        "%s",
                        hbmcl_proc_name_str,
                        usage_short );

        return GLOBUS_FAILURE;
    }

    if( arg_got_option_register == GLOBUS_TRUE )
    {
        if(   ( arg_got_option_dcspec               != GLOBUS_TRUE )
           && (   ( arg_got_option_dchost           != GLOBUS_TRUE )
               || ( arg_got_option_dcport           != GLOBUS_TRUE )))
        {
            globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [04] in "
                        "globus_l_hbmcl_get_arguments():\n"
                        "        -dcspec or (-dchost and -dcport) required "
                        "for resgistration.\n"
                        "usage:\n"
                        "%s",
                        hbmcl_proc_name_str,
                        usage_short );

            return GLOBUS_FAILURE;
        }
        if(   ( arg_got_option_dcspec               == GLOBUS_TRUE )
           && (   ( arg_got_option_dchost           == GLOBUS_TRUE )
               || ( arg_got_option_dcport           == GLOBUS_TRUE )
               || ( arg_got_option_rptinterval      == GLOBUS_TRUE )
               || ( arg_got_option_rptname          == GLOBUS_TRUE )
               || ( arg_got_option_msg              == GLOBUS_TRUE )))
        {
            globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [05] in "
                        "globus_l_hbmcl_get_arguments():\n"
                        "        Use of -dcspec precludes use of "
                        "-dchost, -dcport, -rptname, -rptinterval, -msg.\n"
                        "usage:\n"
                        "%s",
                        hbmcl_proc_name_str,
                        usage_short );

            return GLOBUS_FAILURE;
        }
    }
    else  /*  unregistration  */
    {
/*
**      (   ( arg_got_option_unregister_abnormal    == GLOBUS_TRUE )
**       || ( arg_got_option_unregister_normal      != GLOBUS_TRUE ))
*/
        if(   ( arg_got_option_dcspec               == GLOBUS_TRUE )
           || ( arg_got_option_dchost               == GLOBUS_TRUE )
           || ( arg_got_option_dcport               == GLOBUS_TRUE ))
        {
            globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [06] in "
                        "globus_l_hbmcl_get_arguments():\n"
                        "        -dcspec or (-dchost and -dcport) not allowed "
                        "for unresgistration.\n"
                        "usage:\n"
                        "%s",
                        hbmcl_proc_name_str,
                        usage_short );

            return GLOBUS_FAILURE;
        }
    }


/*
**  Now verify the parameters in order.
**
**  Verify the lmstatus file name.
*/
    lm_status_fname_str =
                    globus_i_hbm_validate_path_get_filename(
                            arg_option_fname_lmstatus_str,
                            HBMLM_DEFAULT_FNAME_STATUS );
    if( lm_status_fname_str == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [07] in "
                        "globus_l_hbmcl_get_arguments():\n"
                        "        -lmstatus specifies invalid path name.\n\n",
                        hbmcl_proc_name_str );

        return GLOBUS_FAILURE;
    }

    return GLOBUS_SUCCESS;
}


int
main(           int                     argc,
                char                    *argv[] )
{
    char    rsl_buff[GLOBUS_HBM_BUFF_SIZE_MSG];
    char   *tmp;

    int     retcd;

    globus_hbm_client_regerr_t  regerr;

    globus_bool_t  require_all = GLOBUS_FALSE;

    hbmcl_proc_name_str = argv[0];

/*
**  Activate the modules that we will be using.
*/
    if( globus_module_activate( GLOBUS_COMMON_MODULE ) !=
                                GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [05] in "
                        "main():\n"
                        "        globus_module_activate(GLOBUS_COMMON_MODULE) "
                        "failed.\n\n",
                        hbmcl_proc_name_str );

        return GLOBUS_FAILURE;
    }

    if( globus_module_activate( GLOBUS_HBM_CLIENT_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [03] in "
                        "main():\n"
                        "        globus_module_activate"
                        "(GLOBUS_HBM_CLIENT_MODULE) failed.\n\n",
                        hbmcl_proc_name_str );

        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        return GLOBUS_FAILURE;
    }

    if( globus_module_activate( GLOBUS_POLL_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [07] in "
                        "main():\n"
                        "        globus_module_activate(GLOBUS_POLL_MODULE) "
                        "failed.\n\n",
                        hbmcl_proc_name_str );

        globus_module_deactivate( GLOBUS_HBM_CLIENT_MODULE );
        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        return GLOBUS_FAILURE;
    }

    if( globus_args_scan(
                        &argc,
                        &argv,
                        ARGS_ID_MAX,
                        args_option_list,
                        usage_short,
                        usage_long,
                        &args_options_found_list,
                        GLOBUS_NULL ) <= 0 )
    {
        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [01] in "
                        "main():\n"
                        "        globus_args_scan() failed.\n\n",
                        hbmcl_proc_name_str );

        return GLOBUS_FAILURE;
    }

    if( globus_l_hbmcl_get_arguments(
                        args_options_found_list ) !=
                                        GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [02] in "
                        "main():\n"
                        "        globus_l_hbmcl_get_arguments() failed.\n\n",
                        hbmcl_proc_name_str );

        return GLOBUS_FAILURE;
    }

    if( arg_option_register_unregister ==
                                GLOBUS_HBM_MSGTYPE_SHUTDOWN_ABNORMAL )
    {
        if( globus_hbm_client_unregister_all(
                        arg_option_pid,
                        GLOBUS_HBM_SHUTDOWN_ABNORMAL,
                        lm_status_fname_str,
                        arg_option_lmvalidatesecs ) ==
                                GLOBUS_FAILURE )
        {
            globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [04] in "
                        "main():\n"
                        "        globus_hbm_client_unregister_all() "
                        "failed:\n"
                        "            Abnormal Unregistration failed:  "
                        "PID [%d].\n\n",
                        hbmcl_proc_name_str,
                        arg_option_pid );

            return GLOBUS_FAILURE;
        }
        else
        {
            globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Abnormal Unregistration Succeeded:  "
                        "PID [%d].\n\n",
                        hbmcl_proc_name_str,
                        arg_option_pid );

            return GLOBUS_SUCCESS;
        }
    }
    else if( arg_option_register_unregister ==
                                GLOBUS_HBM_MSGTYPE_SHUTDOWN_NORMAL )
    {
        if( globus_hbm_client_unregister_all(
                        arg_option_pid,
                        GLOBUS_HBM_SHUTDOWN_NORMAL,
                        lm_status_fname_str,
                        arg_option_lmvalidatesecs ) ==
                                GLOBUS_FAILURE )
        {
            globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [05] in "
                        "main():\n"
                        "        globus_hbm_client_unregister_all() "
                        "failed:\n"
                        "            Normal Unregistration failed:  "
                        "PID [%d].\n\n",
                        hbmcl_proc_name_str,
                        arg_option_pid );

            return GLOBUS_FAILURE;
        }
        else
        {
            globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Normal Unregistration Succeeded:  "
                        "PID [%d].\n\n",
                        hbmcl_proc_name_str,
                        arg_option_pid );

            return GLOBUS_SUCCESS;
        }
    }
    else /* registration message */
    {
        if( arg_got_option_dcspec == GLOBUS_TRUE )
        {
            retcd = globus_hbm_client_register(
                        arg_option_pid,
                        arg_option_dcspec_str,
                        arg_option_requireall,
                        lm_status_fname_str,
                        arg_option_lmvalidatesecs,
                        &regerr );
        }
        else
        {
/*
**          Need to build the rsl string.
*/
            tmp = rsl_buff;
            *tmp = '&';
            tmp++;

/*
**          host
*/
            memcpy(     (void *) tmp,
                        (void *) "(Host=",
                        strlen( "(Host=" ));
            tmp += strlen( "(Host=" );
            memcpy(     (void *) tmp,
                        (void *) arg_option_dchost_str,
                        strlen( arg_option_dchost_str ));
            tmp += strlen( arg_option_dchost_str );
            *tmp = ')'; tmp++;

/*
**          port
*/
            memcpy(     (void *) tmp,
                        (void *) "(Portnum=",
                        strlen( "(Portnum=" ));
            tmp += strlen( "(Portnum=" );
            memcpy(     (void *) tmp,
                        (void *) arg_option_dcport_str,
                        strlen( arg_option_dcport_str ));
            tmp += strlen( arg_option_dcport_str );
            *tmp = ')'; tmp++;

/*
**          rptname
*/
            if( arg_got_option_rptname == GLOBUS_TRUE )
            {
                memcpy( (void *) tmp,
                        (void *) "(Rptname=\"",
                        strlen( "(Rptname=\"" ));
                tmp += strlen( "(Rptname=\"" );
                memcpy( (void *) tmp,
                        (void *) arg_option_rptname_str,
                        strlen( arg_option_rptname_str ));
                tmp += strlen( arg_option_rptname_str );
                *tmp = '\"'; tmp++;
                *tmp = ')'; tmp++;
            }

/*
**          interval
*/
            if( arg_got_option_rptinterval == GLOBUS_TRUE )
            {
                memcpy( (void *) tmp,
                        (void *) "(Interval=",
                        strlen( "(Interval=" ));
                tmp += strlen( "(Interval=" );
                memcpy( (void *) tmp,
                        (void *) arg_option_rptinterval_str,
                        strlen( arg_option_rptinterval_str ));
                tmp += strlen( arg_option_rptinterval_str );
                *tmp = ')'; tmp++;
            }

            /* message */
            if( arg_option_msg_str != GLOBUS_NULL )
            {
                memcpy( (void *) tmp,
                        (void *) "(Message=\"",
                        strlen( "(Message=\"" ));
                tmp += strlen( "(Message=\"" );
                memcpy( (void *) tmp,
                        (void *) arg_option_msg_str,
                        strlen( arg_option_msg_str ));
                tmp += strlen( arg_option_msg_str );
                *tmp = '\"'; tmp++;
                *tmp = ')'; tmp++;
            }

            *tmp = '\0';

            /* printf( "RSL String:%s\n", rsl_buff ); */

            retcd = globus_hbm_client_register(
                        arg_option_pid,
                        rsl_buff,
                        GLOBUS_FALSE,
                        lm_status_fname_str,
                        arg_option_lmvalidatesecs,
                        &regerr );
        }

        if( retcd == GLOBUS_FAILURE )
        {
            globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [06] in "
                        "main():\n"
                        "        globus_hbm_client_register() "
                        "failed:\n"
                        "            Registration failed:  "
                        "PID [%d].\n\n",
                        hbmcl_proc_name_str,
                        arg_option_pid );

            return GLOBUS_FAILURE;
        }

        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Registration Succeeded:  PID [%d].\n",
                        hbmcl_proc_name_str,
                        arg_option_pid );

        if( arg_got_option_dcspec )
        {
            globus_libc_fprintf(
                        stderr,
                        "    Using specification:\n"
                        "        %s\n\n",
                        arg_option_dcspec_str );
        }
        else if( arg_got_option_rptname == GLOBUS_TRUE )
        {
            globus_libc_fprintf(
                        stderr,
                        "    Using report name:  %s\n\n",
                        arg_option_rptname_str );
        }
        else
        {
            globus_libc_fprintf(
                        stderr,
                        "\n" );
        }
    }

    if( globus_module_deactivate( GLOBUS_HBM_CLIENT_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "%s:\n"
                        "    Error [07] in "
                        "main():\n"
                        "        globus_module_deactivate"
                        "(GLOBUS_HBM_CLIENT_MODULE) failed.\n\n",
                        hbmcl_proc_name_str );

        return GLOBUS_FAILURE;
    }

    return GLOBUS_SUCCESS;
}

/*
************************************************************************
**                                                                    **
**  hbm_monitor_proc.c - end                                          **
**                                                                    **
************************************************************************
*/
