/* TODO:
**   - pass regerr_t through the convert and validate routine, and the
**     build_reg_message routine.
**   - add a check to make sure that the DC_Message is <= GLOBUS_HBM_DC_MSG_SIZE_MAX
*/

/*
********************************************************************************
**                                                                            **
**  hbmcl.c - source file for heartbeat client library                        **
**                                                                            **
********************************************************************************
*/
#define HBM_CL_C

/*
#define BUFFER_DEBUG 1
*/

#include "globus_hbm_client.h"
#include "globus_hbm_defaults.h"

#include "hbm.h"


#if !defined(MAXPATHLEN )
#   define MAXPATHLEN PATH_MAX
#endif



struct dc_vals_s {
    char*   hostname;
    char*   ipnum;
    char*   interval;
    char*   port;
    char*   msg;
    char*   rptstring;
};
typedef struct dc_vals_s dc_vals_t;

/* local function declarations */

static char*
globus_l_hbmcl_build_reg_msg(
                dc_vals_t*              vals,
                struct sockaddr_in*     dc_addr,
                char*                   procname,
                int                     heartbeat_interval,
                pid_t                   cl_pid,
                int*                    msg_len );

static char*
globus_l_hbmcl_build_unreg_msg(
                char*                   procname,
                pid_t                   cl_pid,
                unsigned int            shutdown_mode,
                int*                    msg_len );

static int
globus_l_hbmcl_client_activate(
                void );

static int
globus_l_hbmcl_client_deactivate(
                void );

static int
globus_l_hbmcl_convert_and_validate_dc_vals(
                dc_vals_t*              vals,
                struct sockaddr_in*     dc_addr,
                int*                    interval );

static void
globus_l_hbmcl_free_dc_vals(
                dc_vals_t*              vals );

static int
globus_l_hbmcl_get_dc_specs_from_rsl_list(
                globus_list_t*          list,
                dc_vals_t*              vals );

static int
globus_l_hbmcl_get_hbmlm_status_vals(
                char*                   lm_status_file_str,
                int*                    lm_portnum_ptr,
                char**                  ps_command_str_ptr,
                int                     lm_validation_time_limit_secs );

static int
globus_l_hbmcl_send_msg(
                int                     lm_fd,
                char*                   reg_msg,
                int                     msg_len );

static int
globus_l_hbmcl_send_short_reg_msg(
                int                     lm_fd,
                int                     reg_code );


/*
 ****************************************************************************
 *
 *
 *                        Module activation structure
 *
 *
 ****************************************************************************
 */

globus_module_descriptor_t              globus_i_hbm_client_module =
{
    "globus_hbm_client",
    globus_l_hbmcl_client_activate,
    globus_l_hbmcl_client_deactivate,
    GLOBUS_NULL,
    GLOBUS_NULL
};


/*
** Global Variables
*/

static char globus_l_hbm_client_hostname[MAXHOSTNAMELEN];


/*
********************************************************************************
**                                                                            **
**  globus_hbm_client_register                                                **
**                                                                            **
********************************************************************************
*/
int
globus_hbm_client_register(
                pid_t                   cl_pid,
                char*                   rsl_string,
                globus_bool_t           require_all,
                char*                   lm_status_fname_str,
                int                     lm_status_validation_time_limit_secs,
                globus_hbm_client_regerr_t
                                        *regerr )
{
    int                 rc;

    int                 dc_specs_first_bad = 0;
    int                 dc_specs_num_good = 0;

    int                 hbmlm_port;

    char*               ps_command_str  = GLOBUS_NULL;

    char                my_hostname[MAXHOSTNAMELEN];
    struct hostent*     my_hostent      = GLOBUS_NULL;

    globus_fifo_t       reg_msg_fifo;

    typedef struct msg_info_s {
        char*           msg_str;
        int             len;
    } msg_info_t;
    msg_info_t*         reg_msg_ptr;

    char                read_buf[ GLOBUS_HBM_BUFF_SIZE_MSG ];

    fd_set              readfds;
    int                 lm_fd;

    struct timeval      dummy_timeval   = { 0, 0 };
    struct timeval      timeout;

    struct sockaddr_in  dc_addr;
    struct sockaddr_in  lm_addr;

    int                 heartbeat_interval;
    dc_vals_t           vals;

    hbmlm_cl_fields_t   client          =
                      { GLOBUS_NULL,                    /* prev */
                        GLOBUS_NULL,                    /* next */
                        0,                              /* CL_procPID */
                        GLOBUS_NULL,                    /* CL_procName_str */
                        GLOBUS_HBM_PROCSTATUS_ACTIVE,   /* CL_procStatus */
                        0,                              /* CL_blockedTime */
                        0,                              /* CL_cpuSecs */
                        GLOBUS_FALSE,                   /* CL_updated */
                        { GLOBUS_NULL, GLOBUS_NULL }    /* CL_dclist */  };

    hbmlm_cl_list_head_t
                        client_list;

    globus_rsl_t*       root            = GLOBUS_NULL;
    globus_rsl_t*       rsl_ptr         = GLOBUS_NULL;
    globus_list_t*      rsl_list        = GLOBUS_NULL;
    globus_list_t*      tmp_list        = GLOBUS_NULL;


    vals.hostname                       = GLOBUS_NULL;
    vals.ipnum                          = GLOBUS_NULL;
    vals.interval                       = GLOBUS_NULL;
    vals.port                           = GLOBUS_NULL;
    vals.msg                            = GLOBUS_NULL;
    vals.rptstring                      = GLOBUS_NULL;

    client_list.head = client_list.tail = &client;

    regerr->num_reg_ok                  = 0;
    regerr->first_bad_dc                = 0;
    regerr->dc_error                    = 0;

/*
**  Get the local monitor port-num and the ps command name from the
**  LM status file.
*/
    if( globus_l_hbmcl_get_hbmlm_status_vals(
                        lm_status_fname_str,
                        &hbmlm_port,
                        &ps_command_str,
                        lm_status_validation_time_limit_secs ) ==
                                    GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [01] in "
                        "globus_hbm_client_register():\n"
                        "        globus_l_hbmcl_get_hbmlm_status_vals() "
                        "failed.\n"
                        "            Unable to get LocalMonitor status file "
                        "values.\n\n" );

        return GLOBUS_FAILURE;
    }

/*
**  If cl_pid is -1,
**      then use the pid of the local (this) process.
*/
    if( cl_pid == (pid_t) -1 )
    {
        cl_pid = getpid();
    }
    client.CL_procPID = cl_pid;

/*
**  Get the process name for the pid using ps.
*/
    globus_i_hbm_call_ps_update_client_data(
                        ps_command_str,
                        client_list,
                        GLOBUS_HBM_CALL_PS_GET_PROCNAME,
                        &dummy_timeval,
                        stderr );

/*
**  Done with the ps_command_str.
*/
    globus_free( ps_command_str );
    ps_command_str = GLOBUS_NULL;

    if( client.CL_procName_str == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [02] in "
                        "globus_hbm_client_register():\n"
                        "        globus_i_hbm_call_ps_update_client_data() "
                        "failed:\n"
                        "            Unable to get process information "
                        " for pid[%d].\n\n",
                        (int) cl_pid );

        return GLOBUS_FAILURE;
    }

/*
**  Check now for an rsl string.
**  We'll validate it later.
*/
    if( rsl_string == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [03] in "
                        "globus_hbm_client_register():\n"
                        "        rsl_string is GLOBUS_NULL.\n\n" );

        globus_free( client.CL_procName_str );

        return GLOBUS_FAILURE;
    }

/*
**  Get host information for communication with the local monitor.
*/
    rc = globus_libc_gethostname(
                        (char *) &my_hostname,
                        MAXHOSTNAMELEN );
    if( rc == -1 )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [04] in "
                        "globus_hbm_client_register():\n"
                        "        globus_libc_gethostname() "
                        "failed:  errno [%d]: %s.\n\n",
                        errno,
                        strerror( errno ));

        globus_free( client.CL_procName_str );

        return GLOBUS_FAILURE;
    }

    my_hostent = gethostbyname(
                        my_hostname );
    if( my_hostent == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [05] in "
                        "globus_hbm_client_register():\n"
                        "        gethostbyname() "
                        "failed:  errno [%d]: %s.\n\n",
                        errno,
                        strerror( errno ));

        globus_free( client.CL_procName_str );

        return GLOBUS_FAILURE;
    }

    memset(             &lm_addr,
                        0,
                        sizeof( struct sockaddr_in ));
    lm_addr.sin_family      = AF_INET;
    lm_addr.sin_port        = htons( hbmlm_port );
    memcpy(             (void *) &lm_addr.sin_addr,
                        (void *) my_hostent->h_addr_list[0],
                        sizeof( struct in_addr ));

/*
**  Now validate the rsl string, and parse the rsl expressions.
**  For each expression in the string,
**      build a registration message and put it in the fifo.
*/
    root = globus_rsl_parse( rsl_string );
    if( root == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [06] in "
                        "globus_hbm_client_register():\n"
                        "        globus_rsl_parse() failed:  Invalid rsl "
                        "string.\n\n" );

        globus_free( client.CL_procName_str );

        return GLOBUS_FAILURE;
    }

    globus_fifo_init( &reg_msg_fifo );

    rsl_ptr = root;
    dc_specs_first_bad = 0;
    dc_specs_num_good = 0;

    while( rsl_ptr )
    {
        if( globus_rsl_is_boolean_multi( rsl_ptr ))
        {
            rsl_list = (globus_list_t *) rsl_ptr->req.boolean.operand_list;
            rsl_ptr = (globus_rsl_t *) globus_list_first( rsl_list );
            continue;
        }
        else if( globus_rsl_is_boolean_and( rsl_ptr ))
        {
            tmp_list = (globus_list_t *) rsl_ptr->req.boolean.operand_list;
            if( globus_l_hbmcl_get_dc_specs_from_rsl_list(
                        tmp_list,
                        &vals ) ==  GLOBUS_FAILURE )
            {
                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [07] in "
                        "globus_hbm_client_register():\n"
                        "        globus_l_hbmcl_get_dc_specs_from_rsl_list() "
                        "failed:\n"
                        "            Invalid dc specs in rsl string.\n\n" );

                globus_l_hbmcl_free_dc_vals( &vals );
                if( dc_specs_first_bad == 0 )
                {
                    dc_specs_first_bad = dc_specs_num_good + 1;
                }
                rsl_ptr = GLOBUS_NULL;
                continue;
            }

            if( globus_l_hbmcl_convert_and_validate_dc_vals(
                        &vals,
                        &dc_addr,
                        &heartbeat_interval ) == GLOBUS_FAILURE )
            {
                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [08] in "
                        "globus_hbm_client_register():\n"
                        "        globus_l_hbmcl_convert_and_validate_dc_vals() "
                        "failed:\n"
                        "            Invalid dc vals in dc spec rsl "
                        "string.\n\n" );

                globus_l_hbmcl_free_dc_vals( &vals );
                if( dc_specs_first_bad == 0 )
                {
                    dc_specs_first_bad = dc_specs_num_good + 1;
                }
                continue;
            }

            reg_msg_ptr = globus_malloc( sizeof( msg_info_t ));
            if( reg_msg_ptr == GLOBUS_NULL )
            {
                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [09] in "
                        "globus_hbm_client_register():\n"
                        "        globus_malloc() failed:  out of memory.\n\n" );

                globus_l_hbmcl_free_dc_vals( &vals );
                if( dc_specs_first_bad == 0 )
                {
                    dc_specs_first_bad = dc_specs_num_good + 1;
                }
                continue;
            }

            reg_msg_ptr->msg_str = globus_l_hbmcl_build_reg_msg(
                        &vals,
                        &dc_addr,
                        client.CL_procName_str,
                        heartbeat_interval,
                        cl_pid,
                        &( reg_msg_ptr->len ));

	    if( reg_msg_ptr->msg_str == GLOBUS_NULL )
	    {
                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [10] in "
                        "globus_hbm_client_register():\n"
                        "        globus_l_hbmcl_build_reg_msg() "
                        "failed.\n\n" );

                globus_free( reg_msg_ptr );
                reg_msg_ptr = GLOBUS_NULL;
                globus_l_hbmcl_free_dc_vals( &vals );
                if( dc_specs_first_bad == 0 )
                {
                    dc_specs_first_bad = dc_specs_num_good + 1;
                }
                continue;
            }
            else
            {
                globus_fifo_enqueue(
                        &reg_msg_fifo,
                        reg_msg_ptr );
                reg_msg_ptr = GLOBUS_NULL;
            }

            globus_l_hbmcl_free_dc_vals( &vals );

            if( globus_list_empty( rsl_list ))
            {
                rsl_ptr = GLOBUS_NULL;
            }
            else
            {
                rsl_list = globus_list_rest( rsl_list );
                rsl_ptr = (globus_rsl_t *) globus_list_first( rsl_list );
            }
        }
        else
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [11] in "
                        "globus_hbm_client_register():\n"
                        "        Invalid rsl dc specification.\n\n" );

            if( dc_specs_first_bad == 0 )
            {
                dc_specs_first_bad = dc_specs_num_good + 1;
            }
            rsl_ptr = GLOBUS_NULL;
            continue;
        }
/*
**      Made it! This dc spec is good.
*/
        dc_specs_num_good++;
    }

/*
**  Done with the root and client.CL_procName_str.
*/
    globus_rsl_free_recursive( root );
    globus_free( client.CL_procName_str );
    client.CL_procName_str = GLOBUS_NULL;

/*
**  Make sure that registration is supposed to go on,
**  then finish the job.
*/
    if( dc_specs_num_good == 0 )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [12] in "
                        "globus_hbm_client_register():\n"
                        "        No valid rsl dc specification, "
                        "registration aborted.\n\n" );

        regerr->num_reg_ok                  = 0;
        regerr->first_bad_dc                = 1;
        regerr->dc_error                    = 1;

        return GLOBUS_FAILURE;
    }

    if(   ( dc_specs_first_bad > 0 )
       && ( require_all == GLOBUS_TRUE ))
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [13] in "
                        "globus_hbm_client_register():\n"
                        "        Invalid rsl dc specification %d.\n"
                        "        Require_all specified, "
                        "registration aborted.\n\n",
                        dc_specs_first_bad );

        globus_fifo_destroy(
                        &reg_msg_fifo );
        regerr->num_reg_ok                  = 0;
        regerr->first_bad_dc                = dc_specs_first_bad;
        regerr->dc_error                    = 2;

        return GLOBUS_FAILURE;
    }

/*
**  There are good registration messages for us to process.
**  Get ready to send them, and do it.
**
**  First establish a tcp connection for communication with the local monitor.
*/
    lm_fd = socket( AF_INET,
                    SOCK_STREAM,
                    0 );
    if( lm_fd == -1 )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [14] in "
                        "globus_hbm_client_register():\n"
                        "        socket() failed:  errno [%d]: %s.\n\n",
                        errno,
                        strerror( errno ));

        return GLOBUS_FAILURE;
    }

    rc = connect(   lm_fd,
                    (struct sockaddr *) &lm_addr,
                    sizeof( struct sockaddr_in ));
    if( rc == -1 )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [15] in "
                        "globus_hbm_client_register():\n"
                        "        connect() failed:  errno [%d]: %s.\n"
                        "            Unable to establish TCP connection to "
                        "LM.\n\n",
                        errno,
                        strerror( errno ));

        close( lm_fd );

        return GLOBUS_FAILURE;
    }

/*
**  Now remove the registration messages from the fifo,
**  send them one by one to the local monitor, and
**  wait for an ack for each.
*/
    while( !globus_fifo_empty( &reg_msg_fifo ))
    {
        reg_msg_ptr = globus_fifo_dequeue(
                        &reg_msg_fifo );

        rc = globus_l_hbmcl_send_msg(
                        lm_fd,
                        reg_msg_ptr->msg_str,
                        reg_msg_ptr->len );

        if( rc == GLOBUS_FAILURE )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [16] in "
                        "globus_hbm_client_register():\n"
                        "        globus_l_hbmcl_send_msg() failed, "
                        "registration aborted.\n\n" );

            if( regerr->first_bad_dc == 0 )
            {
                regerr->first_bad_dc = regerr->num_reg_ok + 1;
            }

            globus_l_hbmcl_send_short_reg_msg(
                        lm_fd,
                        GLOBUS_HBM_MSGTYPE_REGISTER_CANCEL );
            close( lm_fd );

            globus_free( reg_msg_ptr->msg_str );
            globus_free( reg_msg_ptr );
            while( !globus_fifo_empty( &reg_msg_fifo ))
            {
                reg_msg_ptr = globus_fifo_dequeue(
                        &reg_msg_fifo );
                globus_free(
                        reg_msg_ptr->msg_str );
                globus_free(
                        reg_msg_ptr );
            }
            regerr->dc_error = 3;

            return GLOBUS_FAILURE;
        }

        globus_free(    reg_msg_ptr->msg_str );
        globus_free(    reg_msg_ptr );
        memset(         read_buf,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_MSG );

/*
**      Get ack, timeout if too long.
*/
        FD_ZERO( &readfds );
        FD_SET( lm_fd, &readfds );
        timeout.tv_sec = GLOBUS_HBM_ACK_TIMEOUT;
        timeout.tv_usec = 0;
        rc = select(    lm_fd + 1,
                        HBM_FD_SET_CAST &readfds,
                        GLOBUS_NULL,
                        GLOBUS_NULL,
                        &timeout );

        if( !FD_ISSET( lm_fd, &readfds ))
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [17] in "
                        "globus_hbm_client_register():\n"
                        "        select() failed.\n"
                        "            No ack received for registration "
                        "message, registration aborted.\n\n" );

            globus_l_hbmcl_send_short_reg_msg(
                        lm_fd,
                        GLOBUS_HBM_MSGTYPE_REGISTER_CANCEL );
            close( lm_fd );

            while( !globus_fifo_empty( &reg_msg_fifo ))
            {
                reg_msg_ptr = globus_fifo_dequeue(
                            &reg_msg_fifo );
                globus_free( reg_msg_ptr->msg_str );
                globus_free( reg_msg_ptr );
            }
            if( regerr->first_bad_dc == 0 )
            {
                regerr->first_bad_dc = regerr->num_reg_ok + 1;
            }
            if(   ( dc_specs_first_bad != 0 )
               && ( dc_specs_first_bad < regerr->first_bad_dc ))
            {
                regerr->first_bad_dc = dc_specs_first_bad;
            }
            regerr->num_reg_ok = 0;
            regerr->dc_error = GLOBUS_HBM_ACK_TIMEOUT;

            return GLOBUS_FAILURE;
        }

        rc = read(      lm_fd,
                        &read_buf,
                        GLOBUS_HBM_BUFF_SIZE_MSG );
        if( rc == -1 )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [18] in "
                        "globus_hbm_client_register():\n"
                        "        read() failed.\n"
                        "            No ack received for registration "
                        "message, registration aborted.\n\n" );

            globus_l_hbmcl_send_short_reg_msg(
                        lm_fd,
                        GLOBUS_HBM_MSGTYPE_REGISTER_CANCEL );
            close( lm_fd );

            while( !globus_fifo_empty( &reg_msg_fifo ))
            {
                reg_msg_ptr = globus_fifo_dequeue(
                            &reg_msg_fifo );
                globus_free( reg_msg_ptr->msg_str );
                globus_free( reg_msg_ptr );
            }
            if( regerr->first_bad_dc == 0 )
            {
                regerr->first_bad_dc = regerr->num_reg_ok + 1;
            }
            if(   ( dc_specs_first_bad != 0 )
               && ( dc_specs_first_bad < regerr->first_bad_dc ))
            {
                regerr->first_bad_dc = dc_specs_first_bad;
            }
            regerr->num_reg_ok = 0;
            regerr->dc_error = GLOBUS_HBM_ACK_TIMEOUT;

            return GLOBUS_FAILURE;
        }

        UnpackUInt32( read_buf, rc );
        if( rc != GLOBUS_SUCCESS )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [19] in "
                        "globus_hbm_client_register():\n"
                        "        Registration failed, error recieved "
                        "from LM.\n" );

            if( regerr->first_bad_dc == 0 )
            {
                regerr->first_bad_dc = regerr->num_reg_ok + 1;
            }
            if( require_all == GLOBUS_TRUE )
            {
                globus_libc_fprintf(
                        stderr,
                        "    require_all == TRUE:  Registration "
                        "aborted.\n\n" );

                globus_l_hbmcl_send_short_reg_msg(
                        lm_fd,
                        GLOBUS_HBM_MSGTYPE_REGISTER_CANCEL );
                close( lm_fd );

                while( !globus_fifo_empty( &reg_msg_fifo ))
                {
                    reg_msg_ptr = globus_fifo_dequeue(
                            &reg_msg_fifo );
                    globus_free( reg_msg_ptr->msg_str );
                    globus_free( reg_msg_ptr );
                }
                if( regerr->first_bad_dc == 0 )
                {
                    regerr->first_bad_dc = regerr->num_reg_ok + 1;
                }
                if(   ( dc_specs_first_bad != 0 )
                   && ( dc_specs_first_bad < regerr->first_bad_dc ))
                {
                    regerr->first_bad_dc = dc_specs_first_bad;
                }
                regerr->num_reg_ok = 0;
                regerr->dc_error = rc;

                return GLOBUS_FAILURE;
            }
            else
            {
                globus_libc_fprintf(
                        stderr,
                        "    require_all == FALSE:  Registration "
                        "continuing.\n\n" );
            }
        }
        else
        {
            regerr->num_reg_ok++;
        }
        /* cleanup */
        FD_ZERO( &readfds );
    }
/*
**  Tell the local monitor that we're done, close the connection.
*/
    rc = globus_l_hbmcl_send_short_reg_msg(
                        lm_fd,
                        GLOBUS_HBM_MSGTYPE_REGISTER_COMMIT );

    close( lm_fd );

    if(   ( dc_specs_first_bad != 0 )
       && ( dc_specs_first_bad < regerr->first_bad_dc ))
    {
        regerr->first_bad_dc = dc_specs_first_bad;
    }

    return GLOBUS_SUCCESS;
}/* globus_hbm_client_register */


/*
********************************************************************************
**                                                                            **
**  globus_hbm_client_unregister_all                                          **
**                                                                            **
********************************************************************************
*/
int
globus_hbm_client_unregister_all(
                pid_t                   cl_pid,
                unsigned int            cl_shutdown_mode,
                char*                   lm_status_fname_str,
                int                     lm_status_validation_time_limit_secs )
{
    int                 rc;

    int                 hbmlm_port;

    char*               ps_command_str  = GLOBUS_NULL;

    char                my_hostname[MAXHOSTNAMELEN];
    struct hostent*     my_hostent      = GLOBUS_NULL;

    char*               unreg_msg       = GLOBUS_NULL;
    int                 msg_len         = 0;
    char                read_buf[GLOBUS_HBM_BUFF_SIZE_MSG];

    fd_set              readfds;
    int                 lm_fd;

    struct timeval      dummy_timeval   = { 0, 0 };
    struct timeval      timeout;

    struct sockaddr_in  lm_addr;

    hbmlm_cl_fields_t   client          =
                      { GLOBUS_NULL,                    /* prev */
                        GLOBUS_NULL,                    /* next */
                        0,                              /* CL_procPID */
                        GLOBUS_NULL,                    /* CL_procName_str */
                        GLOBUS_HBM_PROCSTATUS_ACTIVE,   /* CL_procStatus */
                        0,                              /* CL_blockedTime */
                        0,                              /* CL_cpuSecs */
                        GLOBUS_FALSE,                   /* CL_updated */
                        { GLOBUS_NULL, GLOBUS_NULL }    /* CL_dclist */ };

    hbmlm_cl_list_head_t
                        client_list;


    client_list.head = client_list.tail = &client;

/*
**  Get the local monitor port-num and the ps command name
**  from the LM status file.
*/
    if( globus_l_hbmcl_get_hbmlm_status_vals(
                        lm_status_fname_str,
                        &hbmlm_port,
                        &ps_command_str,
                        lm_status_validation_time_limit_secs ) == GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [01] in "
                        "globus_hbm_client_unregister_all():\n"
                        "        globus_l_hbmcl_get_hbmlm_status_vals() "
                        "failed.\n"
                        "            Unable to get LocalMonitor status file "
                        "values.\n\n" );

        return GLOBUS_FAILURE;
    }

/*
**  If cl_pid is -1,
**      then use the pid of the local (this) process.
*/
    if( cl_pid == (pid_t) -1 )
    {
        cl_pid = getpid();
    }
    client.CL_procPID = cl_pid;

/*
**  Get the process name for the pid using ps.
*/
    globus_i_hbm_call_ps_update_client_data(
                        ps_command_str,
                        client_list,
                        GLOBUS_HBM_CALL_PS_GET_PROCNAME,
                        &dummy_timeval,
                        stderr );

/*
**  Done with the ps_command_str.
*/
    globus_free( ps_command_str );
    ps_command_str = GLOBUS_NULL;

    if( client.CL_procName_str == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [02] in "
                        "globus_hbm_client_unregister_all():\n"
                        "        globus_i_hbm_call_ps_update_client_data() "
                        "failed:\n"
                        "            Unable to get process information "
                        " for pid[%d].\n\n",
                        (int) cl_pid );

        return GLOBUS_FAILURE;
    }

/*
**  Get host information for communication with the local monitor.
*/
    rc = globus_libc_gethostname(
                        (char *) &my_hostname,
                        MAXHOSTNAMELEN );
    if( rc == -1 )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [03] in "
                        "globus_hbm_client_unregister_all():\n"
                        "        globus_libc_gethostname() "
                        "failed:  errno [%d]: %s.\n\n",
                        errno,
                        strerror( errno ));

        globus_free( client.CL_procName_str );

        return GLOBUS_FAILURE;
    }

    my_hostent = gethostbyname(
                        my_hostname );
    if( my_hostent == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [04] in "
                        "globus_hbm_client_unregister_all():\n"
                        "        gethostbyname() "
                        "failed:  errno [%d]: %s.\n\n",
                        errno,
                        strerror( errno ));

        globus_free( client.CL_procName_str );

        return GLOBUS_FAILURE;
    }

    memset(             &lm_addr,
                        0,
                        sizeof( struct sockaddr_in ));
    lm_addr.sin_family      = AF_INET;
    lm_addr.sin_port        = htons( hbmlm_port );
    memcpy(             (void *) &lm_addr.sin_addr,
                        (void *) my_hostent->h_addr_list[0],
                        sizeof( struct in_addr ));

/*
**  Now format the unregister message.
*/
    if(   ( cl_shutdown_mode != GLOBUS_HBM_SHUTDOWN_NORMAL )
       && ( cl_shutdown_mode != GLOBUS_HBM_SHUTDOWN_ABNORMAL ))
    {
        cl_shutdown_mode = GLOBUS_HBM_SHUTDOWN_ABNORMAL;
    }

    unreg_msg = globus_l_hbmcl_build_unreg_msg(
                        client.CL_procName_str,
                        cl_pid,
                        cl_shutdown_mode,
                        &msg_len );

/*
**  Done with the client.CL_procName_str.
*/
    globus_free( client.CL_procName_str );
    client.CL_procName_str = GLOBUS_NULL;

    if( unreg_msg == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [05] in "
                        "globus_hbm_client_register():\n"
                        "        globus_l_hbmcl_build_unreg_msg() "
                        "failed.\n\n" );

        return GLOBUS_FAILURE;
    }

/*
**  Now establish a tcp connection for communication with the local monitor.
*/
    lm_fd = socket( AF_INET,
                    SOCK_STREAM,
                    0 );
    if( lm_fd == -1 )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [06] in "
                        "globus_hbm_client_register():\n"
                        "        socket() failed:  errno [%d]: %s.\n\n",
                        errno,
                        strerror( errno ));

        globus_free( client.CL_procName_str );

        return GLOBUS_FAILURE;
    }

    rc = connect(   lm_fd,
                    (struct sockaddr *) &lm_addr,
                    sizeof( struct sockaddr_in ));
    if( rc == -1 )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [07] in "
                        "globus_hbm_client_register():\n"
                        "        connect() failed:  errno [%d]: %s.\n"
                        "            Unable to establish TCP connection to "
                        "LM.\n\n",
                        errno,
                        strerror( errno ));

        close( lm_fd );
        globus_free( client.CL_procName_str );

        return GLOBUS_FAILURE;
    }

/*
**  Got the connection.
**  Now send the unregister message and get the response.
*/
    rc = globus_l_hbmcl_send_msg(
                        lm_fd,
                        unreg_msg,
                        msg_len );

    if( rc == GLOBUS_FAILURE )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [08] in "
                        "globus_hbm_client_register():\n"
                        "        globus_l_hbmcl_send_msg() failed, "
                        "unregistration aborted.\n\n" );

        globus_free( unreg_msg );
        close( lm_fd );

        return GLOBUS_FAILURE;
    }

    globus_free( unreg_msg );

    memset(             (char *) &read_buf,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_MSG );

/*
**  Get ack, timeout if too long.
*/
    FD_ZERO( &readfds );
    FD_SET( lm_fd, &readfds );
    timeout.tv_sec = GLOBUS_HBM_ACK_TIMEOUT;
    timeout.tv_usec = 0;
    rc = select(        lm_fd + 1,
                        HBM_FD_SET_CAST &readfds,
                        GLOBUS_NULL,
                        GLOBUS_NULL,
                        &timeout );
    if( !FD_ISSET( lm_fd, &readfds ))
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [09] in "
                        "globus_hbm_client_register():\n"
                        "        select() failed.\n"
                        "            No ack received for unregistration "
                        "message.\n\n" );

        close( lm_fd );

        return GLOBUS_FAILURE;
    }

    rc = read(  lm_fd,
                &read_buf,
                GLOBUS_HBM_BUFF_SIZE_MSG );
    if( rc == -1 )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [10] in "
                        "globus_hbm_client_register():\n"
                        "        read() failed.\n"
                        "            No ack received for unregistration "
                        "message.\n\n" );

        close( lm_fd );

        return GLOBUS_FAILURE;
    }

    UnpackUInt32( read_buf, rc );

    if( rc != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [11] in "
                        "globus_hbm_client_register():\n"
                        "        Unregistration for pid [%d] failed [%d], "
                        "error recieved "
                        "from LM.\n\n",
                        cl_pid,
                        rc );

        close( lm_fd );

        return GLOBUS_FAILURE;
    }

    close( lm_fd );

    return GLOBUS_SUCCESS;
}


static char*
globus_l_hbmcl_build_reg_msg(
                dc_vals_t*              vals,
                struct sockaddr_in*     dc_addr,
                char*                   procname,
                int                     heartbeat_interval,
                pid_t                   cl_pid,
                int*                    msg_len )
{
    char*               msg_buf       = GLOBUS_NULL;
    char*               tmp           = GLOBUS_NULL;
    unsigned int        length = 0;

#ifdef BUFFER_DEBUG
    int                 i;
#endif

    /* calculate length of message buffer */
    length += NUM_PACKED_BYTES; /* Version */
    length += NUM_PACKED_BYTES; /* RegCmsgLength */
    length += NUM_PACKED_BYTES; /* RegCregCode */
    length += NUM_PACKED_BYTES; /* RegCprocessPID */
    length += strlen( procname ) + 1; /* RegCprocessName */
    if( vals->rptstring != GLOBUS_NULL ) /* RegCreportName */
    {
        length += strlen( vals->rptstring ) + 1;
    }
    else /* use procname as default */
    {
        length += strlen( procname ) + 1;
    }
    length += NUM_PACKED_BYTES; /* RegCDChbInterval */
    length += sizeof( struct sockaddr_in ); /* RegCDCaddr */
    if( vals->msg != GLOBUS_NULL ) /* RegCDCmsg */
    {
        length += strlen( vals->msg ) + 1;
    }
    else
    {
        length += 7; /* 7 = len( "<none>\0" ) */
    }

    /* allocate start packing the message buffer */
    msg_buf = globus_malloc( length );
    if( msg_buf == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [01] in "
                        "globus_l_hbmcl_build_reg_msg():\n"
                        "        globus_malloc() failed:  out of memory.\n\n" );

        return GLOBUS_NULL;
    }

    tmp = msg_buf;

    PackUInt32( tmp, GLOBUS_HBM_VERSION ); /* Version */
    tmp += NUM_PACKED_BYTES;
    PackUInt32( tmp, length );  /* RegCmsgLength */
    tmp += NUM_PACKED_BYTES;
    PackUInt32( tmp, GLOBUS_HBM_MSGTYPE_REGISTER );/* RegCregCode */
    tmp += NUM_PACKED_BYTES;
    PackUInt32( tmp, cl_pid );  /* RegCprocessPID */
    tmp += NUM_PACKED_BYTES;
    strcpy( tmp, procname );    /* RegCprocessName */
    tmp += strlen( procname ) + 1;
    if( vals->rptstring != GLOBUS_NULL ) /* RegCreportName */
    {
        strcpy( tmp, vals->rptstring );
        tmp += strlen( vals->rptstring ) + 1;
    }
    else
    {
        strcpy( tmp, procname );
        tmp += strlen( procname ) + 1;
    }
    PackUInt32( tmp, heartbeat_interval ); /* RegCDChbInterval */
    tmp += NUM_PACKED_BYTES;
    /* RegCDCaddr */
    memcpy(             (void *) tmp,
                        (void *) dc_addr,
                        sizeof( struct sockaddr_in ));
    tmp += sizeof( struct sockaddr_in );
    if( vals->msg != GLOBUS_NULL ) /* RegCDCmsg */
    {
        strcpy(         tmp,
                        vals->msg );
    }
    else
    {
        strcpy(         tmp,
                        "<none>" );
    }

#ifdef BUFFER_DEBUG
    globus_libc_fprintf(
                        stderr,
                        "buffer[" );
    for( i = 0; i < length; i++ )
    {
        globus_libc_fprintf(
                        stderr,
                        "%c",
                        msg_buf[i] );
    }
    globus_libc_fprintf(
                        stderr,
                        "]\n\n" );
#endif /* BUFFER_DEBUG */

    *msg_len = length;
    return msg_buf;
}


static char*
globus_l_hbmcl_build_unreg_msg(
                char*                   procname,
                pid_t                   cl_pid,
                unsigned int            shutdown_mode,
                int*                    msg_len )
{
    char*               msg_buf       = GLOBUS_NULL;
    char*               tmp           = GLOBUS_NULL;
    unsigned int        length = 0;

#ifdef BUFFER_DEBUG
    int                 i;
#endif

    /* calculate length of message buffer */
    length += NUM_PACKED_BYTES; /* Version */
    length += NUM_PACKED_BYTES; /* UnregCmsgLength */
    length += NUM_PACKED_BYTES; /* UnregCregCode */
    length += NUM_PACKED_BYTES; /* UnregCprocessPID */
    length += strlen( procname ) + 1; /* UnregCprocessName */

    /* allocate start packing the message buffer */
    msg_buf = globus_malloc( length );
    if( msg_buf == GLOBUS_NULL )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [01] in "
                        "globus_l_hbmcl_build_unreg_msg():\n"
                        "        globus_malloc() failed:  out of memory.\n\n" );

        return GLOBUS_NULL;
    }

    tmp = msg_buf;
    PackUInt32( tmp, GLOBUS_HBM_VERSION ); /* Version */
    tmp += NUM_PACKED_BYTES;
    PackUInt32( tmp, length );  /* UnregCmsgLength */
    tmp += NUM_PACKED_BYTES;
    if( shutdown_mode == GLOBUS_HBM_SHUTDOWN_NORMAL )
    {
        PackUInt32( tmp, GLOBUS_HBM_MSGTYPE_SHUTDOWN_NORMAL );
    }
    else
    {
        PackUInt32( tmp, GLOBUS_HBM_MSGTYPE_SHUTDOWN_ABNORMAL );
    }
    tmp += NUM_PACKED_BYTES;           /* UnregCregCode */
    PackUInt32( tmp, cl_pid );  /* UnregCprocessPID */
    tmp += NUM_PACKED_BYTES;
    strcpy( tmp, procname );    /* UnregCprocessName */
    tmp += strlen( procname ) + 1;

#ifdef BUFFER_DEBUG
    globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "buffer[" );
    for(    i = 0;
            i < length;
            i++ )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "%c",
                        msg_buf[i] );
    }
    globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "]\n\n" );
#endif /* BUFFER_DEBUG */

    *msg_len = length;
    return msg_buf;
}


/*
 ****************************************************************************
 *
 *                 globus_hbm_client module activation functions
 *
 ****************************************************************************
 */

static int
globus_l_hbmcl_client_activate(
                void )
{
    if( globus_libc_gethostname(
                        (char *) &globus_l_hbm_client_hostname,
                        MAXHOSTNAMELEN ))
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [01] in "
                        "globus_l_hbmcl_client_activate():\n"
                        "        globus_libc_gethostname() failed.\n\n" );

        return GLOBUS_FAILURE;
    }

    if( globus_module_activate( GLOBUS_COMMON_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [02] in "
                        "globus_l_hbmcl_client_activate():\n"
                        "        globus_module_activate(GLOBUS_COMMON_MODULE) "
                        "failed.\n\n" );

        return GLOBUS_FAILURE;
    }

#ifdef GLOBUS_RSL_MODULE
    if( globus_module_activate( GLOBUS_RSL_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [03] in "
                        "globus_l_hbmcl_client_activate():\n"
                        "        globus_module_activate(GLOBUS_RSL_MODULE) "
                        "failed.\n\n" );

        globus_module_deactivate( GLOBUS_COMMON_MODULE );

        return GLOBUS_FAILURE;
    }
#endif

    return GLOBUS_SUCCESS;
}


static int
globus_l_hbmcl_client_deactivate(
                void )
{
    int  rc = GLOBUS_SUCCESS;

    if( globus_module_deactivate( GLOBUS_COMMON_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [01] in "
                        "globus_l_hbmcl_client_deactivate():\n"
                        "        globus_module_activate(GLOBUS_COMMON_MODULE) "
                        "failed.\n\n" );

        rc = GLOBUS_FAILURE;
    }

#ifdef GLOBUS_RSL_MODULE
    if( globus_module_deactivate( GLOBUS_RSL_MODULE ) != GLOBUS_SUCCESS )
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [02] in "
                        "globus_l_hbmcl_client_deactivate():\n"
                        "        globus_module_activate(GLOBUS_RSL_MODULE) "
                        "failed.\n\n" );

        rc = GLOBUS_FAILURE;
    }
#endif

    return rc;
}


static int
globus_l_hbmcl_convert_and_validate_dc_vals(
                dc_vals_t*              vals,
                struct sockaddr_in*     dc_addr,
                int*                    interval )
{
    /* validation */
    if(   ( vals->port == GLOBUS_NULL )
       || (   ( vals->hostname == GLOBUS_NULL )
           && ( vals->ipnum == GLOBUS_NULL )))
    {
        globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [01] in "
                        "globus_l_hbmcl_convert_and_validate_dc_vals():\n"
                        "        Invalid expression.\n\n" );

        return GLOBUS_NULL;
    }

    if( vals->interval != GLOBUS_NULL )
    {
        *interval = atoi( vals->interval );

        /* make sure interval is within range, if it is
        ** below min reset it to min, if it is above max
        ** set it to max
        */
        if( *interval < HBMDC_LIM_EVAL_INTERVAL_SECS_MIN )
            *interval =  HBMDC_LIM_EVAL_INTERVAL_SECS_MIN;
        if( *interval > HBMDC_LIM_EVAL_INTERVAL_SECS_MAX )
            *interval = HBMDC_LIM_EVAL_INTERVAL_SECS_MAX;
    }
    else
    {
        /* if no interval specified in rsl string,
        ** set default interval
        */
        *interval = HBMDC_DEFAULT_EVAL_INTERVAL_SECS;
    }

    dc_addr->sin_family = AF_INET;
    dc_addr->sin_addr.s_addr = 0;
    dc_addr->sin_port = htons( atoi( vals->port ));

    if( vals->ipnum != GLOBUS_NULL )
    {
        dc_addr->sin_addr.s_addr = inet_addr( vals->ipnum );
        if( dc_addr->sin_addr.s_addr == (u_long) -1 )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [02] in "
                        "globus_l_hbmcl_convert_and_validate_dc_vals():\n"
                        "        inet_addr() failed.\n\n" );

            return GLOBUS_FAILURE;
        }
    }

    /* TODO: return error codes for the following errors so they
    ** can be propagated through to the registration regerr_t
    */
    if( vals->hostname != GLOBUS_NULL )
    {
        if( globus_i_hbm_get_inaddr_from_hostname(
                        vals->hostname,
                        &dc_addr->sin_addr ) == GLOBUS_FAILURE )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [03] in "
                        "globus_l_hbmcl_convert_and_validate_dc_vals():\n"
                        "        globus_i_hbm_get_inaddr_from_hostname() "
                        "failed.\n\n" );

            return GLOBUS_FAILURE;
        }
    }

    return GLOBUS_SUCCESS;

}/* int globus_hbm_i_convert_and_validate_expression() */


static void
globus_l_hbmcl_free_dc_vals(
                dc_vals_t*              vals )
{
    if( vals->hostname )
    {
        globus_free( vals->hostname );
        vals->hostname = GLOBUS_NULL;
    }
    if( vals->ipnum )
    {
        globus_free( vals->ipnum );
        vals->ipnum = GLOBUS_NULL;
    }
    if( vals->interval )
    {
        globus_free( vals->interval );
        vals->interval = GLOBUS_NULL;
    }
    if( vals->port )
    {
        globus_free( vals->port );
        vals->port = GLOBUS_NULL;
    }
    if( vals->msg )
    {
        globus_free( vals->msg );
        vals->msg = GLOBUS_NULL;
    }
    if( vals->rptstring )
    {
        globus_free( vals->rptstring );
        vals->rptstring = GLOBUS_NULL;
    }

    return;
}


/* Grab the strings and put them into the dc_vals structure, validation
 * will be next in the build_msg routing
 */
static int
globus_l_hbmcl_get_dc_specs_from_rsl_list(
                globus_list_t*          list,
                dc_vals_t*              vals )
{
    globus_rsl_t*       rsl_ptr       = GLOBUS_NULL;
    globus_list_t*      val_list      = GLOBUS_NULL;
    globus_rsl_value_t* val_ptr       = GLOBUS_NULL;
    char*               attribute_name = GLOBUS_NULL;
    int                 operator;
    char*               literal_string = GLOBUS_NULL;

    rsl_ptr = (globus_rsl_t *) globus_list_first( list );

    while( rsl_ptr != GLOBUS_NULL )
    {
        attribute_name = globus_rsl_relation_get_attribute( rsl_ptr );

        operator = globus_rsl_relation_get_operator( rsl_ptr );
        if( operator != GLOBUS_RSL_EQ )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [01] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Invalid operator in RSL string.\n\n" );

            return GLOBUS_FAILURE;
        }

        val_ptr = globus_rsl_relation_get_value_sequence( rsl_ptr );

        if( ! globus_rsl_value_is_sequence( val_ptr ))
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [02] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Invalid RSL string.\n\n" );

            return GLOBUS_FAILURE;
        }

        val_list = globus_rsl_value_sequence_get_value_list( val_ptr );
        if( val_list == GLOBUS_NULL )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [03] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Invalid RSL string.\n\n" );

            return GLOBUS_FAILURE;
        }

        val_ptr = (globus_rsl_value_t* )globus_list_first( val_list );
        if( ! globus_rsl_value_is_literal( val_ptr ))
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [04] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Invalid RSL string.\n\n" );

            return GLOBUS_FAILURE;
        }

        literal_string = globus_rsl_value_literal_get_string( val_ptr );
        if( literal_string == GLOBUS_NULL )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [05] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Invalid RSL string:  null literal "
                        "string.\n\n" );

            return GLOBUS_FAILURE;
        }

        /* now that I have the string, put it in the structure */
        if( strncasecmp( attribute_name, "Host", 8 ) == 0 )
        {
            if(   ( vals->hostname != NULL )
               || ( vals->ipnum != NULL ))
            {
                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [06] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Duplicate Variable in RSL String.\n\n" );

                globus_l_hbmcl_free_dc_vals( vals );

                return GLOBUS_FAILURE;
            }
            else
            {
                if( isdigit( (int) ( *literal_string )))
                {
                    vals->ipnum = globus_malloc( strlen( literal_string ) + 1 );
                    if( vals->ipnum == NULL )
                    {
                        globus_libc_fprintf(
                            stderr,
                            "Globus HBM Client library:\n"
                            "    Error [07] in "
                            "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                            "        globus_malloc() failed:  out of "
                            "memory.\n\n" );

                        globus_l_hbmcl_free_dc_vals( vals );

                        return GLOBUS_FAILURE;
                    }
                    strcpy(
                        vals->ipnum,
                        literal_string );
                }
                else
                {
                    vals->hostname = globus_malloc( strlen( literal_string ) + 1 );
                    if( vals->hostname == NULL )
                    {
                        globus_libc_fprintf(
                            stderr,
                            "Globus HBM Client library:\n"
                            "    Error [08] in "
                            "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                            "        globus_malloc() failed:  out of "
                            "memory.\n\n" );

                        globus_l_hbmcl_free_dc_vals( vals );

                        return GLOBUS_FAILURE;
                    }
                    strcpy(
                        vals->hostname,
                        literal_string );
                }
            }
        }
        else if( strncasecmp( attribute_name, "Portnum", 7 ) == 0 )
        {
            if( vals->port != NULL )
            {
                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [09] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Duplicate Variable in RSL String.\n\n" );

                globus_l_hbmcl_free_dc_vals( vals );
                return GLOBUS_FAILURE;
            }
            else
            {
                vals->port = globus_malloc(strlen(literal_string ) + 1 );
                if( vals->port == NULL )
                {
                    globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [10] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        globus_malloc() failed:  out of "
                        "memory.\n\n" );

                    globus_l_hbmcl_free_dc_vals( vals );

                    return GLOBUS_FAILURE;
                }
                strcpy( vals->port, literal_string );
            }
        }
        else if( strncasecmp( attribute_name, "Rptname", 7 ) == 0 )
        {
            if( vals->rptstring != NULL )
            {
                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [11] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Duplicate Variable in RSL String.\n\n" );

                globus_l_hbmcl_free_dc_vals( vals );

                return GLOBUS_FAILURE;
            }
            else
            {
                vals->rptstring = globus_malloc( strlen( literal_string ) + 1 );
                if( vals->rptstring == NULL )
                {
                    globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [12] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        globus_malloc() failed:  out of "
                        "memory.\n\n" );

                    globus_l_hbmcl_free_dc_vals( vals );

                    return GLOBUS_FAILURE;
                }
                strcpy( vals->rptstring, literal_string );
            }
        }
        else if( strncasecmp( attribute_name, "Interval", 8 ) == 0 )
        {
            if( vals->interval != NULL )
            {
                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [13] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Duplicate Variable in RSL String.\n\n" );

                globus_l_hbmcl_free_dc_vals( vals );

                return GLOBUS_FAILURE;
            }
            else
            {
                vals->interval = globus_malloc(strlen(literal_string ) + 1 );
                if( vals->interval == NULL )
                {
                    globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [14] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        globus_malloc() failed:  out of "
                        "memory.\n\n" );

                    globus_l_hbmcl_free_dc_vals( vals );

                    return GLOBUS_FAILURE;
                }
                strcpy( vals->interval, literal_string );
            }
        }
        else if( strncasecmp( attribute_name, "Message", 7 ) == 0 )
        {
            if( vals->msg != NULL )
            {
                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [15] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        Duplicate Variable in RSL String.\n\n" );

                globus_l_hbmcl_free_dc_vals( vals );

                return GLOBUS_FAILURE;
            }
            else
            {
                vals->msg = globus_malloc(strlen(literal_string ) + 1 );
                if( vals->msg == NULL )
                {
                    globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [16] in "
                        "globus_l_hbmcl_get_dc_specs_from_rsl_list():\n"
                        "        globus_malloc() failed:  out of "
                        "memory.\n\n" );

                    globus_l_hbmcl_free_dc_vals( vals );
                    return GLOBUS_FAILURE;
                }
                strcpy( vals->msg, literal_string );
            }
        }

        list = globus_list_rest( list );
        if( globus_list_empty( list ))
        {
            rsl_ptr = GLOBUS_NULL;
        }
        else
        {
            rsl_ptr = (globus_rsl_t* )globus_list_first( list );
        }
    }
    return GLOBUS_SUCCESS;
}


static int
globus_l_hbmcl_get_hbmlm_status_vals(
                char*                   lm_status_fname_str,
                int*                    lm_portnum_ptr,
                char**                  ps_command_str_ptr,
                int                     lm_validation_time_limit_secs )
{
    int                 validation_attempts;
    int                 lm_validation_time_waited_secs;

    FILE*               lm_status_fp    = GLOBUS_NULL;

    int                 statusval_str_len;

    char*               lm_status_fname_dflt1_str =
                                          "../var/"
                                          HBMLM_DEFAULT_FNAME_STATUS;
    char*               lm_status_fname_dflt2_str =
                                          "./"
                                          HBMLM_DEFAULT_FNAME_STATUS;

    char                read_buf[GLOBUS_HBM_BUFF_SIZE_STATUS];
    char*               buf_ptr;

    globus_bool_t       got_lm_pid      = GLOBUS_FALSE;
    globus_bool_t       got_lm_procname = GLOBUS_FALSE;
    globus_bool_t       got_lm_portnum  = GLOBUS_FALSE;
    globus_bool_t       got_ps_command  = GLOBUS_FALSE;

    hbmlm_cl_fields_t   lm_as_client    =
                      { GLOBUS_NULL,                    /* prev */
                        GLOBUS_NULL,                    /* next */
                        0,                              /* CL_procPID */
                        GLOBUS_NULL,                    /* CL_procName_str */
                        GLOBUS_HBM_PROCSTATUS_ACTIVE,   /* CL_procStatus */
                        0,                              /* CL_blockedTime */
                        0,                              /* CL_cpuSecs */
                        GLOBUS_FALSE,                   /* CL_updated */
                        { GLOBUS_NULL, GLOBUS_NULL }    /* CL_dclist */  };

    hbmlm_cl_list_head_t
                        lm_as_client_list;

    struct timeval      dummy_timeval   = { 0, 0 };


    lm_as_client_list.head = lm_as_client_list.tail = &lm_as_client;

    validation_attempts = 0;
    if( lm_validation_time_limit_secs >= 0 )
    {
        lm_validation_time_waited_secs = 0;
    }
    else
    {
        lm_validation_time_waited_secs = lm_validation_time_limit_secs - 1;
    }
    while( 1 )
    {
        validation_attempts++;

        if(   ( lm_status_fname_str != GLOBUS_NULL )
           && ( *lm_status_fname_str != '\0' ))
        {
            lm_status_fp = fopen( lm_status_fname_str, "r" );
        }
        if( lm_status_fp == GLOBUS_NULL )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [01] in "
                        "globus_l_hbmcl_get_hbmlm_status_vals():\n"
                        "        fopen() failed:  errno [%d]: %s\n"
                        "            Error opening LM status file [%s]\n"
                        "            on LM validation attempt %d:\n"
                        "            Trying default names.\n\n",
                        errno,
                        strerror( errno ),
                        lm_status_fname_str,
                        validation_attempts );

            if(( lm_status_fp = fopen( lm_status_fname_dflt1_str, "r" )) ==
                                GLOBUS_NULL )
            {
                if(( lm_status_fp = fopen( lm_status_fname_dflt2_str, "r" )) ==
                                GLOBUS_NULL )
                {
                    globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [02] in "
                        "globus_l_hbmcl_get_hbmlm_status_vals():\n"
                        "        fopen() failed:  errno [%d]: %s\n"
                        "            Error opening default LM status files:\n"
                        "                [%s]\n"
                        "                [%s]\n"
                        "            on LM validation attempt %d:\n\n",
                        errno,
                        strerror( errno ),
                        lm_status_fname_dflt1_str,
                        lm_status_fname_dflt2_str,
                        validation_attempts );
                }
            }
        }

        if( lm_status_fp != GLOBUS_NULL )
        {
            got_lm_pid      = GLOBUS_FALSE;
            got_lm_procname = GLOBUS_FALSE;
            got_lm_portnum  = GLOBUS_FALSE;
            got_ps_command  = GLOBUS_FALSE;

            while( ( buf_ptr = fgets(
                        read_buf,
                        GLOBUS_HBM_BUFF_SIZE_STATUS,
                        lm_status_fp )) != GLOBUS_NULL )
            {
                buf_ptr = read_buf;
                while( isspace( *buf_ptr ))
                    buf_ptr++;

                if( strncasecmp(
                        buf_ptr,
                        "LocalMonitorPID",
                        15 ) == 0 )
                {
                    if( got_lm_pid == GLOBUS_FALSE )
                    {
                        got_lm_pid = GLOBUS_TRUE;
                        buf_ptr += 15;
                        if( *buf_ptr == ':' )
                            buf_ptr++;
                        lm_as_client.CL_procPID =
                                atoi( buf_ptr );  /* atoi ignores whitespace */
                    }
                }
                else if( strncasecmp(
                        buf_ptr,
                        "LocalMonitorProcName",
                        20 ) == 0 )
                {
                    if( got_lm_procname == GLOBUS_FALSE )
                    {
                        got_lm_procname = GLOBUS_TRUE;
                        buf_ptr += 20;
                        if( *buf_ptr == ':' )
                            buf_ptr++;
                        while( isspace( *buf_ptr ))
                            buf_ptr++;
                        statusval_str_len = strlen( buf_ptr );
                        if( *( buf_ptr + statusval_str_len - 1 ) == '\n' )
                        {
                            *( buf_ptr + statusval_str_len - 1 ) = '\0';
                        }
                        else
                        {
                            statusval_str_len++;
                        }
                        lm_as_client.CL_procName_str =
                            (char *) globus_malloc( statusval_str_len );
                        strcpy(
                            lm_as_client.CL_procName_str,
                            buf_ptr );
                    }
                }
                else if( strncasecmp(
                        buf_ptr,
                        "LocalMonitorPortNumReg",
                        22 ) == 0 )
                {
                    if( got_lm_portnum == GLOBUS_FALSE )
                    {
                        got_lm_portnum = GLOBUS_TRUE;
                        buf_ptr += 22;
                        if( *buf_ptr == ':' )
                            buf_ptr++;
                        *lm_portnum_ptr =
                                atoi( buf_ptr );  /* atoi ignores whitespace */
                    }
                }
                else if( strncasecmp(
                        buf_ptr,
                        "LocalMonitorPSCommand",
                        21 ) == 0 )
                {
                    if( got_ps_command == GLOBUS_FALSE )
                    {
                        got_ps_command = GLOBUS_TRUE;
                        buf_ptr += 21;
                        if( *buf_ptr == ':' )
                            buf_ptr++;
                        while( isspace( *buf_ptr ))
                            buf_ptr++;
                        statusval_str_len = strlen( buf_ptr );
                        if( *( buf_ptr + statusval_str_len - 1 ) == '\n' )
                        {
                            *( buf_ptr + statusval_str_len - 1) = '\0';
                        }
                        else
                        {
                            statusval_str_len++;
                        }
                        *ps_command_str_ptr =
                            (char *) globus_malloc( statusval_str_len );
                        strcpy(
                            *ps_command_str_ptr,
                            buf_ptr );
                    }
                }
                if(   ( got_lm_pid      == GLOBUS_TRUE )
                   && ( got_lm_procname == GLOBUS_TRUE )
                   && ( got_lm_portnum  == GLOBUS_TRUE )
                   && ( got_ps_command  == GLOBUS_TRUE ))
                {
                    break;
                }
            }

            fclose( lm_status_fp );

            if(   ( got_lm_pid      != GLOBUS_TRUE )
               || ( got_lm_procname != GLOBUS_TRUE )
               || ( got_lm_portnum  != GLOBUS_TRUE )
               || ( got_ps_command  != GLOBUS_TRUE ))
            {
                if( lm_as_client.CL_procName_str != GLOBUS_NULL )
                {
                    globus_free( lm_as_client.CL_procName_str );
                    lm_as_client.CL_procName_str = GLOBUS_NULL;
                }
                *lm_portnum_ptr = 0;
                if( *ps_command_str_ptr != GLOBUS_NULL )
                {
                    globus_free( *ps_command_str_ptr );
                    *ps_command_str_ptr = GLOBUS_NULL;
                }

                globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [03] in "
                        "globus_l_hbmcl_get_hbmlm_status_vals():\n"
                        "        Incomplete data found in LM status file"
                        " on attempt %d.\n\n",
                        validation_attempts );
            }
            else
            {
/*
**              Got all the necessary fields from the LM status file.
**              Now validate the Local Monitor using a call to ps.
*/
/*              lm_as_client.CL_procPID      was read from the status file.  */
/*              lm_as_client.CL_procName_str was read from the status file.  */
                lm_as_client.CL_procStatus   = GLOBUS_HBM_PROCSTATUS_ACTIVE;
                lm_as_client.CL_blockedTime  = 0;
                lm_as_client.CL_cpuSecs      = 0;
                lm_as_client.CL_updated      = GLOBUS_FALSE;

                globus_i_hbm_call_ps_update_client_data(
                        *ps_command_str_ptr,
                        lm_as_client_list,
                        GLOBUS_HBM_CALL_PS_GET_STATUS,
                        &dummy_timeval,
                        stderr );

/*
**              Don't need process name to validate the results.
*/
                globus_free( lm_as_client.CL_procName_str );
                lm_as_client.CL_procName_str = GLOBUS_NULL;

                if( lm_as_client.CL_updated == GLOBUS_FALSE )
                {
                    globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [04] in "
                        "globus_l_hbmcl_get_hbmlm_status_vals():\n"
                        "        globus_i_hbm_call_ps_update_client_data() "
                        "failed:\n"
                        "            LocalMonitor process not found using "
                        "ps on attempt %d.\n\n",
                        validation_attempts );
                }
                else if(   ( lm_as_client.CL_procStatus !=
                                GLOBUS_HBM_PROCSTATUS_ACTIVE )
                        && ( lm_as_client.CL_procStatus !=
                                GLOBUS_HBM_PROCSTATUS_BLOCKED ))
                {
                    globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [05] in "
                        "globus_l_hbmcl_get_hbmlm_status_vals():\n"
                        "        globus_i_hbm_call_ps_update_client_data() "
                        "failed:\n"
                        "            Zombie LocalMonitor process found using "
                        "ps on attempt %d.\n\n",
                        validation_attempts );
                }
                else  /*  Client was updated and is alive => we found it!  */
                {
                    return GLOBUS_SUCCESS;
                }
            }
        }

        if( lm_as_client.CL_procName_str != GLOBUS_NULL )
        {
            globus_free( lm_as_client.CL_procName_str );
            lm_as_client.CL_procName_str = GLOBUS_NULL;
        }
        if( *ps_command_str_ptr != GLOBUS_NULL )
        {
            globus_free( *ps_command_str_ptr );
            *ps_command_str_ptr = GLOBUS_NULL;
        }

/*
**      Was unable to validate the local monitor.
**      If have tried the max number of times, exit as failure.
**      Otherwise, wait 5 seconds and try again.
*/
        if( lm_validation_time_waited_secs >= lm_validation_time_limit_secs )
        {
            globus_libc_fprintf(
                        stderr,
                        "Globus HBM Client library:\n"
                        "    Error [06] in "
                        "globus_l_hbmcl_get_hbmlm_status_vals():\n"
                        "        Unable to validate Local Monitor and get "
                        "LM status values\n"
                        "        after %d LM validation attempts "
                        "over a period of %d seconds.\n\n",
                        validation_attempts,
                        lm_validation_time_waited_secs );

            break;
        }

        sleep( 5 );
        if( lm_validation_time_limit_secs >= 0 )
        {
            lm_validation_time_waited_secs += 5;
        }
    }

    return GLOBUS_FAILURE;
}


static int
globus_l_hbmcl_send_msg(
                int                     lm_fd,
                char*                   reg_msg,
                int                     msg_len )
{
    int rc;

    rc = write( lm_fd, reg_msg, msg_len );
    if( rc == -1 || rc != msg_len )
    {
        return GLOBUS_FAILURE;
    }

    return GLOBUS_SUCCESS;
}

static int
globus_l_hbmcl_send_short_reg_msg(
                int                     lm_fd,
                int                     reg_code )
{
    char*   msg                       = GLOBUS_NULL;
    char*   msg_ptr                   = GLOBUS_NULL;
    int     msg_len;
    int     rc;

    msg_len = NUM_PACKED_BYTES * 3; /* version, len, type */

    msg = globus_malloc( msg_len );
    if( msg == GLOBUS_NULL )
    {
        return GLOBUS_FAILURE;
    }
    msg_ptr = msg;

    PackUInt32( msg_ptr, GLOBUS_HBM_VERSION ); /* version */
    msg_ptr += NUM_PACKED_BYTES;
    PackUInt32( msg_ptr, msg_len ); /* msg length */
    msg_ptr += NUM_PACKED_BYTES;
    PackUInt32( msg_ptr, reg_code ); /* type of msg */

    rc = write( lm_fd, msg, msg_len );
    globus_free( msg );
    if( rc != msg_len )
    {
    /*  rc shows error or incomplete write.  */
        return GLOBUS_FAILURE;
    }

    return GLOBUS_SUCCESS;
}


/*
********************************************************************************
**                                                                            **
**  hbm_cl.c - end                                                            **
**                                                                            **
********************************************************************************
*/
