/*
************************************************************************
**                                                                    **
**  hbm_common.c - HBM CL, LM, and DC common routines                 **
**                                                                    **
************************************************************************
*/

/*
** Include files
*/

#include "globus_common.h"
#include "globus_config.h"
#include "globus_libc.h"
#include "globus_strptime.h"

#include "globus_hbm_codes.h"
#include "globus_hbm_defaults.h"

#include "hbm.h"

/* **
#define DEBUGGING_COMMON 1
** */


int
globus_i_hbm_build_ps_command_string(
                char*                   ps_program_name_str,
                char**                  ps_call_str_ptr )
{
#if   (defined TARGET_ARCH_AIX)
#define  HBM_PS_OPTIONS_STR             "-o \"pid status time comm\""
#elif (defined TARGET_ARCH_BSDI)
#define  HBM_PS_OPTIONS_STR             "-x -o \"pid state time command\""
#elif (defined TARGET_ARCH_CRAYT3E)
#define  HBM_PS_OPTIONS_STR             "-o \"pid state time command\""
#elif (defined TARGET_ARCH_FREEBSD)
#define  HBM_PS_OPTIONS_STR             "-xf -o \"pid state time command\""
#elif (defined TARGET_ARCH_HPUX)
#define  HBM_PS_OPTIONS_STR             "-fl "
#elif (defined TARGET_ARCH_IRIX)
#define  HBM_PS_OPTIONS_STR             "-o \"pid state time comm\""
#elif (defined TARGET_ARCH_LINUX)
#define  HBM_PS_OPTIONS_STR             "o \"pid state time command\""
#elif (defined TARGET_ARCH_OSF1)
#define  HBM_PS_OPTIONS_STR             "-o \"pid state time command\""
#elif (defined TARGET_ARCH_SOLARIS)
#define  HBM_PS_OPTIONS_STR             "-o \"pid s time comm\""
#elif (defined TARGET_ARCH_SUNOS41)
#define  HBM_PS_OPTIONS_STR             "-gx"
#elif (defined TARGET_ARCH_BSD)
/*
**      BSD defaults.
*/
#define  HBM_PS_OPTIONS_STR             "-x -o \"pid state time command\""
#else
/*
**      System V, other non-BSD defaults.
*/
#define  HBM_PS_OPTIONS_STR             "-o \"pid state time comm\" -p"
#endif

    *ps_call_str_ptr =
                (char *) globus_malloc( strlen( ps_program_name_str )
                                      + strlen( HBM_PS_OPTIONS_STR )
                                      + 2 );
    if( *ps_call_str_ptr == GLOBUS_NULL )
    {
        return GLOBUS_FAILURE;
    }

    globus_libc_sprintf(
                        *ps_call_str_ptr,
                        "%s %s",
                        ps_program_name_str,
                        HBM_PS_OPTIONS_STR );

    return GLOBUS_SUCCESS;
}


int
globus_i_hbm_call_ps_update_client_data(
                char*                   ps_command_options_str,
                hbmlm_cl_list_head_t    client_list,
                const int               call_ps_reason_code,
                struct timeval*         time_curr_ptr,
                FILE*                   error_file_ptr )
{
    FILE*       fd                      = GLOBUS_NULL;
    char        ps_command[ GLOBUS_HBM_BUFF_SIZE_PS ];

#if  (defined TARGET_ARCH_HPUX)
    int         dummy;
    char        dummys[64];
#endif

    hbmlm_cl_fields_t*
                cl_fields               = GLOBUS_NULL;
    hbmlm_cl_fields_t*
                cl_fields_for_ps_ptr    = GLOBUS_NULL;
    hbmlm_cldc_fields_t*
                cldc_fields             = GLOBUS_NULL;

    char        pid_buff[32];

    char        ps_buff[ GLOBUS_HBM_BUFF_SIZE_PS ];
    char*       ps_buff_ptr_wk1         = GLOBUS_NULL;

#if (defined TARGET_ARCH_SUNOS41)
    char*       ps_buff_ptr_wk2         = GLOBUS_NULL;
#endif

    int         max_pids_per_ps         = 1;

    int         ps_output_pid;
    char        ps_output_status;

#if (   (defined TARGET_ARCH_BSD) \
     || (defined TARGET_ARCH_BSDI) \
     || (defined TARGET_ARCH_FREEBSD))
    char        ps_output_status_str[10];
#endif

    unsigned int
                ps_output_cpu_days      = 0;
    unsigned int
                ps_output_cpu_hours     = 0;
    unsigned int
                ps_output_cpu_secs      = 0;
    unsigned int
                ps_output_cpu_mins      = 0;

    char        ps_output_procname_array[MAXPATHLEN];

    int         new_procStatus;

    int         i;

    if( client_list.head == GLOBUS_NULL)
    {
        return GLOBUS_FAILURE;
    }

    for( cl_fields = client_list.head;
         cl_fields;
         cl_fields = cl_fields->next )
    {
        cl_fields->CL_updated = GLOBUS_FALSE;
    }

#if   (defined TARGET_ARCH_AIX)
    max_pids_per_ps = 1000;
#elif (defined TARGET_ARCH_BSD)
    max_pids_per_ps = 1000;
#elif (defined TARGET_ARCH_BSDI)
    max_pids_per_ps = 1000;
#elif (defined TARGET_ARCH_CRAYT3E)
    max_pids_per_ps = 1000;
#elif (defined TARGET_ARCH_FREEBSD)
    max_pids_per_ps = 1;
#elif (defined TARGET_ARCH_HPUX)
    max_pids_per_ps = 30;
#elif (defined TARGET_ARCH_IRIX)
    max_pids_per_ps = 20;
#elif (defined TARGET_ARCH_LINUX)
    max_pids_per_ps = 1000;
#elif (defined TARGET_ARCH_OSF1)
    max_pids_per_ps = 1000;
#elif (defined TARGET_ARCH_SOLARIS)
    max_pids_per_ps = 1000;
#elif (defined TARGET_ARCH_SUNOS41)
    max_pids_per_ps = 1000;
#else
    max_pids_per_ps = 1000;
#endif

    cl_fields_for_ps_ptr = client_list.head;
    while( cl_fields_for_ps_ptr != GLOBUS_NULL )
    {
/*
**      build the ps command string
*/
        memset(         (void *) ps_command,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_PS );

        strcpy(         ps_command,
                        ps_command_options_str );

        for( i = 1;
             (   ( i <= max_pids_per_ps )
              && ( cl_fields_for_ps_ptr != GLOBUS_NULL )
              && ( GLOBUS_HBM_BUFF_SIZE_PS - strlen( ps_command ) > 10 ));
             i++ )
        {
#if (defined TARGET_ARCH_LINUX)
            globus_libc_sprintf(
                        pid_buff,
                        " p %d",
                        cl_fields_for_ps_ptr->CL_procPID );
#else
            globus_libc_sprintf(
                        pid_buff,
                        " -p %d",
                        cl_fields_for_ps_ptr->CL_procPID );
#endif
            strcat(     ps_command,
                        pid_buff );
            cl_fields_for_ps_ptr = cl_fields_for_ps_ptr->next;
        }

        fd = popen(     ps_command,
                        "r" );
        if( fd == GLOBUS_NULL )
        {
            globus_libc_fprintf(
                        error_file_ptr,
                        "ps command failed\n\n" );

            return GLOBUS_FAILURE;
        }

/*
**      get beyond the first line of the ps output
*/
        fgets(          ps_buff,
                        GLOBUS_HBM_BUFF_SIZE_PS,
                        fd );
        memset(         (void *) ps_buff,
                        0,
                        GLOBUS_HBM_BUFF_SIZE_PS );

/*
**      get the first status line
*/
        if( fgets(  ps_buff,
                    GLOBUS_HBM_BUFF_SIZE_PS,
                    fd ) == GLOBUS_NULL )
        {
            *ps_buff = '\0';
        }
        if(   ( ferror( fd ))
           || ( feof( fd )))
        {
            *ps_buff = '\0';
        }

        while( *ps_buff != '\0' )
        {
/*
**          Parse the status line:
**              PID, ps_oputput_status_ptr, cpu seconds, minutes,
**              ps_output_procname
*/
            ps_buff_ptr_wk1 = ps_buff;

#if  (defined TARGET_ARCH_AIX)

            ps_buff_ptr_wk1 = index( ps_buff, ':' );
            if( *( ps_buff_ptr_wk1 - 3 ) == '-' )
            {
                sscanf( ps_buff,
                        "%d %c %d-%d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_days,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            else
            {
                ps_output_cpu_days = 0;
                if( *( ps_buff_ptr_wk1 + 3 ) == ':' )
                {
                    sscanf(
                        ps_buff,
                        "%d %c %d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
                }
                else
                {
                    ps_output_cpu_hours = 0;
                    sscanf(
                        ps_buff,
                        "%d %c %d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
                }
            }

            if(   ( ps_output_status == 'A' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'I' )
                    || ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' )
                    || ( ps_output_status == 'W' )
                    || ( ps_output_status == 'X' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if(   ( ps_output_status == '0' )
                    || ( ps_output_status == 'O' )
                    || ( ps_output_status == 'Z' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_BSDI) /* default */

            ps_buff_ptr_wk1 = index( ps_buff, ':' );
            if( *( ps_buff_ptr_wk1 - 3 ) == '-' )
            {
                sscanf( ps_buff,
                        "%d %s %d-%d:%d.%d %[ -z]\n",
                        &ps_output_pid,
                        ps_output_status_str,
                        &ps_output_cpu_days,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            else
            {
                ps_output_cpu_days = 0;
                sscanf( ps_buff,
                        "%d %s %d:%d.%d %[ -z]\n",
                        &ps_output_pid,
                        ps_output_status_str,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            ps_output_status = *ps_output_status_str;

            if(   ( ps_output_status == 'O' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'I' )
                    || ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' )
                    || ( ps_output_status == 'W' )
                    || ( ps_output_status == 'X' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_CRAYT3E)

            ps_buff_ptr_wk1 = ps_buff;
            ps_output_cpu_days = 0;
            sscanf(     ps_buff_ptr_wk1,
                        "%d %c %d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );

            if(   ( ps_output_status == 'O' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'I' )
                    || ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' )
                    || ( ps_output_status == 'W' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_FREEBSD)

            ps_buff_ptr_wk1 = index( ps_buff, ':' );
            if( *( ps_buff_ptr_wk1 - 3 ) == '-' )
            {
                sscanf( ps_buff,
                        "%d %s %d-%d:%d.%d %[ -z]\n",
                        &ps_output_pid,
                        ps_output_status_str,
                        &ps_output_cpu_days,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            else
            {
                ps_output_cpu_days = 0;
                sscanf( ps_buff,
                        "%d %s %d:%d.%d %[ -z]\n",
                        &ps_output_pid,
                        ps_output_status_str,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            ps_output_status = *ps_output_status_str;

            if(   ( ps_output_status == 'O' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'I' )
                    || ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' )
                    || ( ps_output_status == 'W' )
                    || ( ps_output_status == 'X' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_HPUX)

            ps_buff_ptr_wk1 = ps_buff;
/*
**          Skip past leading spaces.
*/
            while( isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
/*
**          Skip past flags (numeric) and spaces.
*/
            while( !isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
            while( isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
/*
**          Get the status.
*/
            ps_output_status = *ps_buff_ptr_wk1;
            ps_buff_ptr_wk1++;
/*
**          Skip past spaces.
*/
            while( isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
/*
**          Skip past user id (alphanumeric) and spaces.
*/
            while( !isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
            while( isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
/*
**          Get the pid.
*/
            sscanf(     ps_buff_ptr_wk1,
                        "%d",
                        &ps_output_pid );
/*
**          Skip past spaces.
*/
            while( isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
/*
**          Skip past 7 fields and their following (intervening) spaces.
**              (fields are: ppid, cpu, intpri, nice, addr, sz, wchan)
*/
            for( i = 0; i < 7; i++ )
            {
                while( !isspace( *ps_buff_ptr_wk1 ))
                    ps_buff_ptr_wk1++;
                while( isspace( *ps_buff_ptr_wk1 ))
                    ps_buff_ptr_wk1++;
            }

/*
**          Skip past stime and following spaces.
**          Format could be "month day" or "hh:mm:ss" or "mm:ss".
**          Either way we don't need the contents, just get past it.
*/
            while(   ( !isspace( *ps_buff_ptr_wk1 ))
                  && ( *ps_buff_ptr_wk1 != ':' ))
                ps_buff_ptr_wk1++;
/*
**          If this character is a space,
**              then format must be "month day",
**                   so skip spaces to get to the next (last)
**                   part of the stime field.
**          Otherwise we are in the middle of the "hh:mm:ss" or "mm:ss" format,
**              and we just need to get to the end of it.
*/
            while( isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
            while( !isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
            while( isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;

/*
**          Skip past tty and following spaces.
*/
            while( !isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;
            while( isspace( *ps_buff_ptr_wk1 ))
                ps_buff_ptr_wk1++;

/*
**          Get the time ("mmm:ss") and the command.
*/

            ps_output_cpu_days      = 0;
            ps_output_cpu_hours     = 0;
            sscanf(     ps_buff_ptr_wk1,
                        "%d:%d %[ -z]\n",
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );

/*
**          Set the new status value.
*/
            if( ps_output_status == 'R' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'I' )
                    || ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' )
                    || ( ps_output_status == 'W' )
                    || ( ps_output_status == 'X' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if(   ( ps_output_status == '0' )
                    || ( ps_output_status == 'Z' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_IRIX)

            ps_buff_ptr_wk1 = index( ps_buff, ':' );
            if( *( ps_buff_ptr_wk1 - 3 ) == '-' )
            {
                sscanf( ps_buff,
                        "%d %c %d-%d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_days,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            else
            {
                ps_output_cpu_days = 0;
                if( *( ps_buff_ptr_wk1 + 3 ) == ':' )
                {
                    sscanf(
                        ps_buff,
                        "%d %c %d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
                }
                else
                {
                    ps_output_cpu_hours = 0;
                    sscanf(
                        ps_buff,
                        "%d %c %d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
                }
            }

            if(   ( ps_output_status == 'O' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'I' )
                    || ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' )
                    || ( ps_output_status == 'X' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_LINUX)

            ps_output_cpu_days = 0;
            sscanf(     ps_buff,
                        "%d %c %d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );

            if( ps_output_status == 'R' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'D' )
                    || ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_OSF1)

            ps_buff_ptr_wk1 = index( ps_buff, ':' );

            sscanf(     ps_buff,
                        "%d %c",
                        &ps_output_pid,
                        &ps_output_status );

            if( *( ps_buff_ptr_wk1 - 3 ) == '-' )
            {
                ps_buff_ptr_wk1 -= 5;
                sscanf( ps_buff_ptr_wk1,
                        "%d-%d:%d.%d %[ -z]\n",
                        &ps_output_cpu_days,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            else
            {
                ps_output_cpu_days = 0;
                ps_buff_ptr_wk1 -= 3;
                sscanf( ps_buff_ptr_wk1,
                        "%d:%d.%d %[ -z]\n",
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }

            if(   ( ps_output_status == 'O' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'S' )
                    || ( ps_output_status == 'U' )
                    || ( ps_output_status == 'T' )
                    || ( ps_output_status == 'I' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_SOLARIS)

            ps_buff_ptr_wk1 = index( ps_buff, ':' );
            if( *( ps_buff_ptr_wk1 - 3 ) == '-' )
            {
                sscanf( ps_buff,
                        "%d %c %d-%d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_days,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            else
            {
                ps_output_cpu_days = 0;
                if( *( ps_buff_ptr_wk1 + 3 ) == ':' )
                {
                    sscanf(
                        ps_buff,
                        "%d %c %d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
                }
                else
                {
                    ps_output_cpu_hours = 0;
                    sscanf(
                        ps_buff,
                        "%d %c %d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
                }
            }

            if(   ( ps_output_status == 'O' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_SUNOS41)

/*
**          Get the PID, and skip the pointer past it.
*/
            while(   ( isspace( *ps_buff_ptr_wk1 )
                  && ( *ps_buff_ptr_wk1 != '\n' ))
            {
                ps_buff_ptr_wk1++;
            }
            sscanf(     ps_buff_ptr_wk1,
                        "%d",
                        &ps_output_pid );
            /* xx added ! to isspace */
            /* past the pid */
            while(   ( !isspace( *ps_buff_ptr_wk1 ))
                  && ( *ps_buff_ptr_wk1 != '\n' ))
            {
                ps_buff_ptr_wk1++;
            }
            /* past the whitespace */
            while(   ( isspace( *ps_buff_ptr_wk1 ))
                  && ( *ps_buff_ptr_wk1 != '\n' ))
            {
                ps_buff_ptr_wk1++;
            }

/*
**          Skip the pointer past the terminal.
**          (The pointed to character is a space,
**           terminal id field begins with next (may be blank ),
**           then we have spaces until first status character. )
*/
            ps_buff_ptr_wk1++;
            while(   ( !isspace( *ps_buff_ptr_wk1 ))
                  && ( *ps_buff_ptr_wk1 != '\n' ))
            {
                ps_buff_ptr_wk1++;
            }
            while(   ( isspace( *ps_buff_ptr_wk1 ))
                  && ( *ps_buff_ptr_wk1 != '\n' ))
            {
                ps_buff_ptr_wk1++;
            }
/*
**          The pointer is on the first status character
**          (the only one we're interested in ).
*/
            if( isalpha( *ps_buff_ptr_wk1 ))
            {
                ps_output_status = *ps_buff_ptr_wk1;
            }
/*
**          Skip the pointer to the CPU time.
*/
            ps_buff_ptr_wk1++;
            while(   ( *ps_buff_ptr_wk1 != '\n' )
                  && ( !isdigit( *ps_buff_ptr_wk1 )))
            {
                ps_buff_ptr_wk1++;
            }
            ps_buff_ptr_wk2 = index( ps_buff_ptr_wk1, ':' );
            if( ps_buff_ptr_wk2 )
            {
                if( *( ps_buff_ptr_wk2 - 3 ) == '-' )
                {
                    sscanf(
                        ps_buff_ptr_wk1,
                        "%d-%d:%d:%d %[ -z]\n",
                        &ps_output_cpu_days,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
                }
                else
                {
                    ps_output_cpu_days = 0;
                    if( *( ps_buff_ptr_wk2 + 3 ) == ':' )
                    {
                        sscanf(
                            ps_buff_ptr_wk1,
                            "%d:%d:%d %[ -z]\n",
                            &ps_output_cpu_hours,
                            &ps_output_cpu_mins,
                            &ps_output_cpu_secs,
                            ps_output_procname_array );
                    }
                    else
                    {
                        ps_output_cpu_hours = 0;
                        sscanf(
                            ps_buff_ptr_wk1,
                            "%d:%d %[ -z]\n",
                            &ps_output_cpu_mins,
                            &ps_output_cpu_secs,
                            ps_output_procname_array );
                    }
                }
            }
            if(   ( ps_output_status == 'O' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#elif (defined TARGET_ARCH_BSD)

/*
**          BSD defaults.
*/
            ps_buff_ptr_wk1 = index( ps_buff, ':' );
            if( *( ps_buff_ptr_wk1 - 3 ) == '-' )
            {
                sscanf( ps_buff,
                        "%d %s %d-%d:%d.%d %[ -z]\n",
                        &ps_output_pid,
                        ps_output_status_str,
                        &ps_output_cpu_days,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            else
            {
                ps_output_cpu_days = 0;
                sscanf( ps_buff,
                        "%d %s %d:%d.%d %[ -z]\n",
                        &ps_output_pid,
                        ps_output_status_str,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            ps_output_status = *ps_output_status_str;

            if(   ( ps_output_status == 'O' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'I' )
                    || ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' )
                    || ( ps_output_status == 'W' )
                    || ( ps_output_status == 'X' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#else

/*
**          System V, other non-BSD defaults.
*/
            ps_buff_ptr_wk1 = index( ps_buff, ':' );
            if( *( ps_buff_ptr_wk1 - 3 ) == '-' )
            {
                sscanf( ps_buff,
                        "%d %c %d-%d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_days,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
            }
            else
            {
                ps_output_cpu_days = 0;
                if( *( ps_buff_ptr_wk1 + 3 ) == ':' )
                {
                    sscanf(
                        ps_buff,
                        "%d %c %d:%d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_hours,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
                }
                else
                {
                    ps_output_cpu_hours = 0;
                    sscanf(
                        ps_buff,
                        "%d %c %d:%d %[ -z]\n",
                        &ps_output_pid,
                        &ps_output_status,
                        &ps_output_cpu_mins,
                        &ps_output_cpu_secs,
                        ps_output_procname_array );
                }
            }

            if(   ( ps_output_status == 'O' )
               || ( ps_output_status == 'R' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_ACTIVE;
            }
            else if(   ( ps_output_status == 'S' )
                    || ( ps_output_status == 'T' ))
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_BLOCKED;
            }
            else if( ps_output_status == 'Z' )
            {
                new_procStatus = GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
            }

#endif  /*  defined TARGET_ARCH_xxxx  */

            ps_output_cpu_secs +=
                                ( ps_output_cpu_days
                                * GLOBUS_HBM_TIME_SECS_PER_DAY );
            ps_output_cpu_secs +=
                                ( ps_output_cpu_hours
                                * GLOBUS_HBM_TIME_SECS_PER_HOUR );
            ps_output_cpu_secs +=
                                ( ps_output_cpu_mins
                                * GLOBUS_HBM_TIME_SECS_PER_MIN );

/*
**          Done parsing the status line.
**
**          Now have all the fields,
**              get the client entry (if any ) and update it.
*/
            cl_fields = globus_i_hbm_find_cl_fields_entry(
                                    &client_list,
                                    ps_output_pid );

            if( cl_fields != GLOBUS_NULL )
            {
/*
**              Behavior here depends on call_ps_reason_code.
**
**              If call was made to get the process name,
**                  then ignore the old name (if any )
**                       and set the process name to what we got.
**              If call was made to check the process name,
**                  then compare the old name to the name we got.
**                       if they are different
**                           then write out error messages
**                           and set the name to what we got.
**              If call was made to get the client status,
**                  then make sure the names are the same before updating the status.
*/
                if( call_ps_reason_code == GLOBUS_HBM_CALL_PS_GET_PROCNAME )
                {
                    cl_fields->CL_updated = GLOBUS_TRUE;

                    if( cl_fields->CL_procName_str != GLOBUS_NULL )
                    {
                        globus_free( cl_fields->CL_procName_str );
                        cl_fields->CL_procName_str = GLOBUS_NULL;
                    }
                    cl_fields->CL_procName_str =
                            (char *) globus_malloc(
                                        strlen( ps_output_procname_array )
                                      + 1 );
                    strcpy(
                        cl_fields->CL_procName_str,
                        ps_output_procname_array );
                    cl_fields->CL_procStatus = new_procStatus;
                }
                else if(   ( call_ps_reason_code ==
                                    GLOBUS_HBM_CALL_PS_CHECK_PROCNAME )
                        && ( cl_fields->CL_procName_str == GLOBUS_NULL ))
                {
/*
**                  There is no name so just use the one we got.
*/
                    cl_fields->CL_updated = GLOBUS_TRUE;

                    cl_fields->CL_procName_str =
                            (char *) globus_malloc(
                                        strlen( ps_output_procname_array )
                                      + 1 );
                    strcpy( cl_fields->CL_procName_str,
                            ps_output_procname_array );
                }
                else if( globus_i_hbm_compare_ps_name_to_client(
                            ps_output_procname_array,
                            cl_fields ))
		{
                    if( call_ps_reason_code ==
                                    GLOBUS_HBM_CALL_PS_CHECK_PROCNAME )
                    {
                        cl_fields->CL_updated = GLOBUS_TRUE;

                        globus_libc_fprintf(
                            error_file_ptr,
                            "Process name from ps doesn't "
                            "match process name from registration info for pid %d\n"
                            "            Reg name: %s  PS name: %s\n"
                            "Using name from ps\n\n",
                            ps_output_pid,
                            cl_fields->CL_procName_str,
                            ps_output_procname_array );

                        globus_free( cl_fields->CL_procName_str );
                        cl_fields->CL_procName_str =
                            (char *) globus_malloc(
                                        strlen( ps_output_procname_array )
                                      + 1 );
                        strcpy( cl_fields->CL_procName_str,
                            ps_output_procname_array );
                    }
                    else if( call_ps_reason_code ==
                                    GLOBUS_HBM_CALL_PS_GET_STATUS )
                    {
/*
**                      The name is different than the one obtained at
**                          registration.
**                      Treat as though this is a different process --
**                          if registered process is not unregistered/died,
**                              then write error messages.
*/
                        if(   ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
                           && ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL )
                           && ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED ))
                        {
                            globus_libc_fprintf(
                                error_file_ptr,
                                "Process name from ps doesn't match "
                                "client registration info for pid %d\n"
                                "        Reg name: %s\n"
                                "        PS name: %s\n"
                                "Assuming that registered process died.\n\n",
                                ps_output_pid,
                                cl_fields->CL_procName_str,
                                ps_output_procname_array );
                        }

                        cl_fields = GLOBUS_NULL;
                    }
		}
                else
                {
/*
**                  The process name from ps matches the client.
*/
                    if( call_ps_reason_code ==
                                    GLOBUS_HBM_CALL_PS_CHECK_PROCNAME )
                    {
                        cl_fields->CL_updated = GLOBUS_TRUE;
                    }
                    else if( call_ps_reason_code ==
                                    GLOBUS_HBM_CALL_PS_GET_STATUS )
                    {
/*
**                      Update the status information for this client.
*/
                        cl_fields->CL_updated = GLOBUS_TRUE;

                        if( new_procStatus == GLOBUS_HBM_PROCSTATUS_ACTIVE )
                        {
                            if(   ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
                               && ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL ))
                            /* && ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED )) */
                            {
                                cl_fields->CL_procStatus =
                                        GLOBUS_HBM_PROCSTATUS_ACTIVE;
                                cl_fields->CL_blockedTime = 0;
                                cl_fields->CL_cpuSecs =
                                        ps_output_cpu_secs;
                            }
                        }
                        else if( new_procStatus == GLOBUS_HBM_PROCSTATUS_BLOCKED )
                        {
                            if( ps_output_cpu_secs > cl_fields->CL_cpuSecs )
                            {
                                if(   ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
                                   && ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL ))
                                /* && ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED )) */
                                {
                                    cl_fields->CL_procStatus =
                                        GLOBUS_HBM_PROCSTATUS_ACTIVE;
                                    cl_fields->CL_blockedTime = 0;
                                    cl_fields->CL_cpuSecs =
                                        ps_output_cpu_secs;
                                }
                            }
                            else
                            {
                                if( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_BLOCKED )
                                {
                                    if(   ( cl_fields->CL_procStatus !=
                                            GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
                                       && ( cl_fields->CL_procStatus !=
                                            GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL ))
                                    /* && ( cl_fields->CL_procStatus !=
                                            GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED )) */
                                    {
                                        cl_fields->CL_procStatus =
                                            GLOBUS_HBM_PROCSTATUS_BLOCKED;
                                        cl_fields->CL_blockedTime =
                                            time_curr_ptr->tv_sec;
                                    }
                                }
                            }
                        }
                        else if( new_procStatus ==
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED )
                        {
                            if(   ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
                               && ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL )
                               && ( cl_fields->CL_procStatus !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED ))
                            {
                                cl_fields->CL_procStatus =
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
                                cl_fields->CL_blockedTime = 0;
                                cl_fields->CL_cpuSecs =
                                        ps_output_cpu_secs;
                                for( cldc_fields = cl_fields->CL_dclist.head;
                                     cldc_fields;
                                     cldc_fields = cldc_fields->next )
                                {
                                    if(   ( cldc_fields->CL_shutdownType !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
                                       && ( cldc_fields->CL_shutdownType !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL )
                                       && ( cldc_fields->CL_shutdownType !=
                                        GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED ))
                                    {
                                        cldc_fields->CL_shutdownType =
                                            cl_fields->CL_procStatus;
                                        cldc_fields->CL_shutdownTime =
                                            time_curr_ptr->tv_sec;
                                        cldc_fields->CL_shutdownMsgCt = 0;
                                    }
                                }
                            }
                        }
                    }
                }
            }
/*
**          If the command line was long enough that the status line is
**              longer than the buffer,
**          then we didn't get it all.
**          If we didn't get it all,
**          then the last character we got back was not a '\n'.
**          In that case we read until we get the newline.
*/
            do
            {
                for( ps_buff_ptr_wk1 = ps_buff;
                     *ps_buff_ptr_wk1 != '\0';
                     ps_buff_ptr_wk1++ );
                ps_buff_ptr_wk1--;
                if( *ps_buff_ptr_wk1 == '\n' )
                {
                    break;
                }

                if( fgets(  ps_buff,
                            GLOBUS_HBM_BUFF_SIZE_PS,
                            fd ) ==
                                        GLOBUS_NULL )
                {
                    break;
                }
                if(   ( feof( fd ))
                   || ( ferror( fd )))
                {
                    break;
                }
            } while ( 1 );
            if(   ( feof( fd ))
               || ( ferror( fd )))
            {
                break;
            }

            /* get the next status line */
            if( fgets(  ps_buff,
                        GLOBUS_HBM_BUFF_SIZE_PS,
                        fd ) == GLOBUS_NULL )
            {
                break;
            }
            if(   ( feof( fd ))
               || ( ferror( fd )))
            {
                break;
            }
        }  /* end while */

        pclose( fd );
    }

/*
**  Check status of all clients.
**  If they did not get updated during ps,
**      then the process is considered dead:
**      Set the status, shutdowntime, and msgcnt appropriately.
*/
    for ( cl_fields = client_list.head;
          cl_fields;
          cl_fields = cl_fields->next )
    {
        if( cl_fields->CL_updated == GLOBUS_FALSE )
        {
            if(   ( call_ps_reason_code == GLOBUS_HBM_CALL_PS_GET_PROCNAME )
               || ( call_ps_reason_code == GLOBUS_HBM_CALL_PS_CHECK_PROCNAME ))
            {
                return GLOBUS_FAILURE;
            }
            if(   ( cl_fields->CL_procStatus !=
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
               && ( cl_fields->CL_procStatus !=
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL )
               && ( cl_fields->CL_procStatus !=
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED ))
            {
                cl_fields->CL_procStatus =
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED;
                cl_fields->CL_blockedTime = 0;

                for( cldc_fields = cl_fields->CL_dclist.head;
                     cldc_fields;
                     cldc_fields = cldc_fields->next )
                {
                    if(   ( cldc_fields->CL_shutdownType !=
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_NORMAL )
                       && ( cldc_fields->CL_shutdownType !=
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_ABNORMAL )
                       && ( cldc_fields->CL_shutdownType !=
                                GLOBUS_HBM_PROCSTATUS_SHUTDOWN_DIED ))
                    {
                        cldc_fields->CL_shutdownType =
                                        cl_fields->CL_procStatus;
                        cldc_fields->CL_shutdownTime =
                                        time_curr_ptr->tv_sec;
                        cldc_fields->CL_shutdownMsgCt = 0;
                    }
                }
            }
        }
    }

    return GLOBUS_SUCCESS;
}


int
globus_i_hbm_compare_ps_name_to_client(
                char*                   ps_procname,
                hbmlm_cl_fields_t*      cl_fields )
{
    char*       cl_procname;

    int         ps_procname_len;
    int         cl_procname_len;

    cl_procname = cl_fields->CL_procName_str;

    if( strcmp(             ps_procname,
                            cl_procname ) == 0)
    {
        return 0;
    }
    ps_procname_len = strlen( ps_procname );
    cl_procname_len = strlen( cl_procname );

    if(   (   (   ( ps_procname[ 0 ]                        == '[' )
               && ( ps_procname[ ps_procname_len - 1 ]      == ']' ))
           && (   ( cl_procname[ 0 ]                        != '[' )
               || ( cl_procname[ cl_procname_len - 1 ]      != ']' )))
       || (   (   ( ps_procname[ 0 ]                        == '[' )
               && ( ps_procname[ ps_procname_len - 1 ]      == ']' )
               && ( ps_procname[ 1 ]                        == '[' )
               && ( ps_procname[ ps_procname_len - 2 ]      == ']' ))
           && (   ( cl_procname[ 0 ]                        == '[' )
               && ( cl_procname[ cl_procname_len - 1 ]      == ']' )
               && (   ( cl_procname[ 1 ]                    != '[' )
                   || ( cl_procname[ cl_procname_len - 2 ]  != ']' )))))
    {
/*
**      The ps_procname may be an abbreviation of the cl_procname.
*/
        if( (ps_procname_len - 2) > cl_procname_len )
        {
            return 1;
        }

        return strncmp(     ps_procname,
                            cl_procname,
                            ( ps_procname_len - 2 ) );
    }

    if(   (   (   ( cl_procname[ 0 ]                        == '[' )
               && ( cl_procname[ cl_procname_len - 1 ]      == ']' ))
           && (   ( ps_procname[ 0 ]                        != '[' )
               || ( ps_procname[ ps_procname_len - 1 ]      != ']' )))
       || (   (   ( cl_procname[ 0 ]                        == '[' )
               && ( cl_procname[ cl_procname_len - 1 ]      == ']' )
               && ( cl_procname[ 1 ]                        == '[' )
               && ( cl_procname[ cl_procname_len - 2 ]      == ']' ))
           && (   ( ps_procname[ 0 ]                        == '[' )
               && ( ps_procname[ ps_procname_len - 1 ]      == ']' )
               && (   ( ps_procname[ 1 ]                    != '[' )
                   || ( ps_procname[ ps_procname_len - 2 ]  != ']' )))))
    {
/*
**      The cl_procname may be an abbreviation of the ps_procname.
*/
        if( (cl_procname_len - 2) > ps_procname_len )
        {
            return 1;
        }
        if( strncmp(        ps_procname,
                            cl_procname,
                            ( cl_procname_len - 2 ) ))
        {
/*
**          The names are different.
*/
            return 1;
        }
        else
	{
/*
**          The names are the same --
**              replace the (abbreviated) client name
**              with the    (longer)      ps     name
*/
            globus_free( cl_fields->CL_procName_str );
            cl_fields->CL_procName_str = globus_malloc( ps_procname_len + 1);
            strcpy(         cl_fields->CL_procName_str,
		            ps_procname );

	    return 0;
	}
    }

/*
**  The names are different, and
**  neither is an abbreviation of the other.
*/
    return 1;
}


UTCtime
globus_i_hbm_convert_string_to_UTCtime(
                char*                   time_str,
                long                    time_conv_diff )
{
    UTCtime     time;
    struct tm   tm;
    char*       ptr                     = GLOBUS_NULL;
    char*       date_decode_str         = "%Y/%m/%d %T";


#ifndef HAVE_STRPTIME
    ptr = (char *) strptime(
                        time_str,
                        date_decode_str,
                        &tm );
#else
    ptr = (char *) globus_strptime(
                        time_str,
                        date_decode_str,
                        &tm );
#endif

    if( ptr == NULL )
    {
        return( (UTCtime) -1 );
    }

#ifdef HAVE_MKTIME
    tm.tm_isdst = 0;
    time = mktime( &tm );
#else
    time = timegm( &tm );
    if( time > 0 )
    {
        time += tm.tm_gmtoff;
    }
#endif

    time = (UTCtime) ((long) time - time_conv_diff );

    return time;
}


void
globus_i_hbm_convert_UTC_to_str(
                UTCtime                 time,
                char*                   date_time_str )
{
    struct tm   tm;
    char        date_time_fmt_out_str[] = "%Y/%m/%d %T GMT";
    struct tm*  tm_ptr                  = GLOBUS_NULL;

    memset(             (void *) date_time_str,
                        0,
                        GLOBUS_HBM_DATE_TIME_LEN );

#ifdef HAVE_GMTIME_R
    gmtime_r(           &time,
                        &tm );
    tm_ptr = &tm;
#else
    tm_ptr = gmtime(    &time );
#endif

    strftime(           date_time_str,
                        GLOBUS_HBM_DATE_TIME_LEN,
                        date_time_fmt_out_str,
                        tm_ptr );

    return;
}


int
globus_i_hbm_count_chars_in_string(
                char*                   string,
                char                    c )
{
    char*       ptr                   = GLOBUS_NULL;
    int         count                 = 0;

    ptr = string;
    while( *ptr != '\0' )
    {
        if( *ptr == c )
            count++;
        ptr++;
    }
    return count;
}


hbmlm_cl_fields_t*
globus_i_hbm_find_cl_fields_entry(
                hbmlm_cl_list_head_t*   cl_list,
                unsigned long           cl_proc_PID )
{
    hbmlm_cl_fields_t*
                cl_entry_curr_ptr       = GLOBUS_NULL;

    for ( cl_entry_curr_ptr = cl_list->head;
          cl_entry_curr_ptr;
          cl_entry_curr_ptr = cl_entry_curr_ptr->next )
    {
        if( cl_entry_curr_ptr->CL_procPID == cl_proc_PID )
        {
            return cl_entry_curr_ptr;
        }
    }

    return GLOBUS_NULL;
}


int
globus_i_hbm_get_inaddr_from_hostname(
                char*                   hostname,
                struct in_addr          *addr )
{
    struct hostent*
                hp                      = GLOBUS_NULL;
    int         error;
    struct hostent
                hostent_result;
    char        buffer_ptr[ GLOBUS_HBM_BUFF_SIZE_HOSTENT ];

    memset(             (void *) &hostent_result,
                        0,
                        sizeof( hostent_result ));

    hp = globus_libc_gethostbyname_r(
                        hostname,
                        &hostent_result,
                        buffer_ptr,
                        GLOBUS_HBM_BUFF_SIZE_HOSTENT,
                        &error );

    if( hp != GLOBUS_NULL )
    {
        memcpy(         (void *) addr,
                        (void *) hp->h_addr,
                        hp->h_length );

        return GLOBUS_SUCCESS;
    }
    else
    {
        return GLOBUS_FAILURE;
    }
}


long
globus_i_hbm_get_time_conv_diff()
{
    struct timeval
                time;
    time_t      curr_time;
    time_t      curr_time_conv;
    long        time_conv_diff;
    struct tm   tm;
    struct tm*  tm_ptr                  = GLOBUS_NULL;
    int         rc;

    memset(             (void *) &tm,
                        0,
                        sizeof( tm ));

    rc = gettimeofday(  &time,
                        GLOBUS_NULL );

#ifdef DEBUGGING_COMMON
    globus_libc_fprintf(
                        stderr,
                        "time: %d\n",
                        time.tv_sec );
#endif

    curr_time = (time_t) time.tv_sec;
#ifdef HAVE_GMTIME_R
    gmtime_r(           &curr_time,
                        &tm );
    tm_ptr = &tm;
#else
    tm_ptr = gmtime(    &curr_time );
#endif

#ifdef DEBUGGING_COMMON
    globus_libc_fprintf(
                        stderr,
                        "curr_time: %d\n",
                        curr_time );
#endif

#ifdef HAVE_MKTIME
    curr_time_conv = mktime( tm_ptr );
#else
    curr_time_conv = timegm( tm_ptr );
    if( curr_time_conv > 0 )
    {
        curr_time_conv += tm.tm_gmtoff;
    }
#endif

#ifdef DEBUGGING_COMMON
    globus_libc_fprintf(
                        stderr,
                        "curr_time: %d curr_time_conv: %d\n",
                        curr_time,
                        curr_time_conv );
#endif

    time_conv_diff = (long) curr_time_conv - (long) curr_time;

#ifdef DEBUGGING_COMMON
    globus_libc_fprintf(
                        stderr,
                        "returning: %d\n\n",
                        time_conv_diff );
#endif

    return time_conv_diff;
}


char*
globus_i_hbm_gethostname_from_addr(
                struct in_addr          *addr )
{
    struct hostent*
                hostent_ptr             = GLOBUS_NULL;
    int         error;
    int         addr_len;
    struct hostent
                hostent_result;
    char*       buffer_ptr              = GLOBUS_NULL;
    char*       hostname                = GLOBUS_NULL;

    memset(             (void *) &hostent_result,
                        0,
                        sizeof( hostent_result ));
    addr_len = sizeof( struct in_addr );

    buffer_ptr = (char *) globus_malloc( GLOBUS_HBM_BUFF_SIZE_HOSTENT );
    if( buffer_ptr == GLOBUS_NULL )
    {
        return GLOBUS_NULL;
    }
    error = 0;
    hostent_ptr = globus_libc_gethostbyaddr_r(
                        (char *) addr,
                        addr_len,
                        AF_INET,
                        &hostent_result,
                        buffer_ptr,
                        GLOBUS_HBM_BUFF_SIZE_HOSTENT,
                        &error );

    if( hostent_ptr != GLOBUS_NULL )
    {
        hostname = globus_malloc( strlen( hostent_ptr->h_name ) + 1 );
        if( hostname == GLOBUS_NULL )
        {
            globus_free( buffer_ptr );

            return GLOBUS_NULL;
        }
        strcpy(         hostname,
                        hostent_ptr->h_name );

        globus_free( buffer_ptr );

        return hostname;
    }

    globus_free( buffer_ptr );

    return GLOBUS_NULL;
}


/*
************************************************************************
**  globus+hbm_read_line -- read fd until \n (up to (max-1));         **
**                   replace \n with \0;                              **
**                   return #char read                                **
**  This will work for now; we can do something better later.         **
************************************************************************
*/
int
globus_i_hbm_read_line(
                int                     fd,
                char*                   buf,
                int                     max )
{
    int         cnt                     = 0,
                nbytes;

    max--;

    while(( nbytes = read( fd, buf, 1 )) > 0 )
    {
        if( nbytes < 0 )
        {
            return -1;
        }
        cnt++;
        if( *buf == '\n' )
        {
            break;
        }
        if( cnt == max )
        {
            buf++;
            break;
        }
        buf++;
    }

    *buf = '\0';
    return cnt;
}


char*
globus_i_hbm_validate_path_get_filename(
                char*                   path_str,
                char*                   default_file_name )
{
    char*       path_copy_str;
    char*       path_copy_wk_ptr;
    char*       path_fname_str;

    int         path_len;

    int         ret_cd;

    struct stat path_status;

    path_len = strlen( path_str );
    if( ( path_copy_str = globus_malloc( path_len + 1 )) == GLOBUS_NULL )
    {
        return GLOBUS_NULL;
    }
    strcpy( path_copy_str, path_str );

    ret_cd = stat( path_copy_str, &path_status );
    if( ret_cd == 0 )
    {
        if( path_status.st_mode & S_IFDIR )
        {
            if( path_copy_str[ path_len - 1 ] == '/' )
            {
                path_copy_str[ path_len - 1 ] = '\0';
                path_len--;
            }
            if( ( path_fname_str = globus_malloc(
                                        path_len
                                      + strlen( default_file_name )
                                      + 2 )) == GLOBUS_NULL )
            {
                globus_free( path_copy_str );
                return GLOBUS_NULL;
            }
            globus_libc_sprintf(
                        path_fname_str,
                        "%s/%s",
                        path_copy_str,
                        default_file_name );
        }
        else if( path_status.st_mode & S_IFREG )
        {
            if( ( path_fname_str = globus_malloc(
                                        path_len
                                      + 1 )) == GLOBUS_NULL )
            {
                globus_free( path_copy_str );
                return GLOBUS_NULL;
            }
            strcpy( path_fname_str,
                    path_copy_str );
        }
        else
        {
            globus_free( path_copy_str );
            return GLOBUS_NULL;
        }
        globus_free( path_copy_str );
        return path_fname_str;
    }
/*
**  stat() returned an error.
*/
    if( errno != ENOENT )
    {
        globus_free( path_copy_str );
        return GLOBUS_NULL;
    }
/*
**  Full path name does not exist.
**  I.e., the file does not exist,
**      so now check if the directory exists.
*/
    path_copy_wk_ptr = path_copy_str + path_len;
    while(   ( path_copy_wk_ptr >= path_copy_str )
          && ( *path_copy_wk_ptr != '/' ))
    {
        path_copy_wk_ptr--;
    }
    if( path_copy_wk_ptr < path_copy_str )
    {
/*
**      It's a filename with no path specified.
*/
        return path_copy_str;
    }
    *path_copy_wk_ptr = '\0';

    ret_cd = stat( path_copy_str, &path_status );
    if( ret_cd == 0 )
    {
        if( path_status.st_mode & S_IFDIR )
        {
            *path_copy_wk_ptr = '/';
            return path_copy_str;
        }
    }
    globus_free( path_copy_str );
    return GLOBUS_NULL;
}


/*
************************************************************************
**                                                                    **
**  hbm_common.c - end                                                **
**                                                                    **
************************************************************************
*/
