/******************************************************************************
globus_gass_transfer_assist_put.c
 
Description:
    This module implements the put and append functions for the GASS transfer
    assist library
 
CVS Information:
 
    $Source: /home/globdev/CVS/globus-current/Globus/FileAccess/gass/libraries/transfer_assist/globus_gass_transfer_assist_put.c,v $
    $Date: 1999/09/23 21:05:08 $
    $Revision: 1.4 $
    $Author: bester $
******************************************************************************/

#include "globus_i_gass_transfer_assist.h"

#include <sys/stat.h>

static
void
globus_l_gass_transfer_assist_put_callback(
    void *				arg,
    globus_gass_transfer_request_t	request);


static
void
globus_l_gass_transfer_assist_read_callback(
    void *				arg,
    globus_io_handle_t *		handle,
    globus_result_t			result,
    globus_byte_t *			buf,
    globus_size_t			nbytes);

static
void
globus_l_gass_transfer_assist_send_callback(
    void *				user_arg,
    globus_gass_transfer_request_t	request,
    globus_byte_t *			bytes,
    globus_size_t			length,
    globus_bool_t			last_data);

/* API functions */

int
globus_i_gass_transfer_assist_file_to_url(
    globus_gass_transfer_request_t *		request,
    globus_gass_transfer_requestattr_t *	attr,
    char *					url,
    char *					file,
    globus_io_attr_t *				file_attr,
    globus_bool_t				follow_referrals,
    globus_i_gass_transfer_assist_op_t		op)
{
    globus_gass_transfer_assist_monitor_t *	monitor;
    int						rc;
    globus_bool_t				done = GLOBUS_FALSE;
    globus_gass_transfer_referral_t		referral;

    memset(&referral,
	   '\0',
	   sizeof(globus_gass_transfer_referral_t));


    monitor = (globus_gass_transfer_assist_monitor_t *)
	globus_malloc(sizeof(globus_gass_transfer_assist_monitor_t));

    if(monitor == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_MALLOC_FAILED;
    }

    globus_mutex_init(&monitor->mutex,
		      GLOBUS_NULL);
    globus_cond_init(&monitor->cond,
		     GLOBUS_NULL);

    globus_mutex_lock(&monitor->mutex);
    while(!done)
    {
	monitor->done = GLOBUS_FALSE;
	switch(op)
	{
	  case GLOBUS_L_GASS_TRANSFER_ASSIST_PUT:
	    rc = globus_gass_transfer_assist_register_put_file_to_url(
		request,
		attr,
		url,
		file,
		file_attr,
		globus_i_gass_transfer_assist_monitor_callback,
		monitor);
	    break;
	
	  case GLOBUS_L_GASS_TRANSFER_ASSIST_APPEND:
	    rc = globus_gass_transfer_assist_register_append_file_to_url(
		request,
		attr,
		url,
		file,
		file_attr,
		globus_i_gass_transfer_assist_monitor_callback,
		monitor);
	    break;
	    
	  default:
	    rc = GLOBUS_GASS_ERROR_INTERNAL_ERROR;
	    break;
	}

	if(rc != GLOBUS_SUCCESS)
	{
	    globus_mutex_unlock(&monitor->mutex);
	    goto finish;
	}
	
	while(!monitor->done)
	{
	    globus_cond_wait(&monitor->cond,
			     &monitor->mutex);
	}

	switch(globus_gass_transfer_request_get_status(*request))
	{
	case GLOBUS_GASS_TRANSFER_REQUEST_REFERRED:
	    if(globus_gass_transfer_referral_get_count(&referral) != 0)
	    {
		globus_gass_transfer_referral_destroy(&referral);
	    }
		
	    rc = globus_gass_transfer_request_get_referral(
		*request,
		&referral);
	    if(rc != GLOBUS_SUCCESS)
	    {
		done = GLOBUS_TRUE;
		break;
	    }
	    url = globus_gass_transfer_referral_get_url(&referral,
							0);
	    globus_gass_transfer_request_destroy(*request);
	    if(url == GLOBUS_NULL)
	    {
		done = GLOBUS_TRUE;
	    }
	    break;
	case GLOBUS_GASS_TRANSFER_REQUEST_FAILED:
	case GLOBUS_GASS_TRANSFER_REQUEST_DENIED:
	    rc = GLOBUS_GASS_ERROR_TRANSFER_FAILED;
	    done = GLOBUS_TRUE;
	    break;
	case GLOBUS_GASS_TRANSFER_REQUEST_DONE:
	    done = GLOBUS_TRUE;
	    break;
	case GLOBUS_GASS_TRANSFER_REQUEST_STARTING:
	case GLOBUS_GASS_TRANSFER_REQUEST_PENDING:
	    globus_assert(GLOBUS_FALSE);
	    done = GLOBUS_TRUE;
	    break;
	}
	globus_mutex_unlock(&monitor->mutex);
    }
  finish:
    if(globus_gass_transfer_referral_get_count(&referral) != 0)
    {
	globus_gass_transfer_referral_destroy(&referral);
    }
    globus_mutex_destroy(&monitor->mutex);
    globus_cond_destroy(&monitor->cond);
    return rc;
}

int
globus_i_gass_transfer_assist_register_file_to_url(
    globus_gass_transfer_request_t *		request,
    globus_gass_transfer_requestattr_t *	attr,
    char *					url,
    char *					file,
    globus_io_attr_t *				file_attr,
    globus_i_gass_transfer_assist_op_t		op,
    globus_gass_transfer_callback_t		callback,
    void *					user_arg)
{
    int						rc;
    globus_result_t				result;
    globus_gass_transfer_assist_callback_info_t *	callback_info;
    struct stat					s;


    if(request == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_NULL_POINTER;
    }
    if(url == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_NULL_POINTER;
    }
    if(file == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_NULL_POINTER;
    }
    if(callback == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_NULL_POINTER;
    }

    callback_info = (globus_gass_transfer_assist_callback_info_t *)
	globus_malloc(sizeof(globus_gass_transfer_assist_callback_info_t));

    if(callback_info == GLOBUS_NULL)
    {
	rc = GLOBUS_GASS_ERROR_MALLOC_FAILED;

	goto error_exit;
    }
    
    callback_info->buffer = (globus_byte_t *)
	globus_malloc(GLOBUS_GASS_DEFAULT_BUFFER_LENGTH);

    if(callback_info->buffer == GLOBUS_NULL)
    {
	rc = GLOBUS_GASS_ERROR_MALLOC_FAILED;

	goto callback_info_exit;
    }
    callback_info->callback = callback;
    callback_info->user_arg = user_arg;
    /* stat the file for it's length */
    rc = stat(file,
	      &s);
    if(rc < 0)
    {
	rc = GLOBUS_GASS_ERROR_OPEN_FAILED;
	
	goto callback_info_exit;
    }
    
    /* open the file with globus_io */
    result = globus_io_file_open(file,
				 GLOBUS_IO_FILE_RDONLY,
				 GLOBUS_IO_FILE_IRUSR,
				 file_attr,
				 &callback_info->handle);
    if(result != GLOBUS_SUCCESS)
    {
	rc = GLOBUS_GASS_ERROR_OPEN_FAILED;

	goto buffer_exit;
    }

    /* open the connection to the server */
    switch(op)
    {
      case GLOBUS_L_GASS_TRANSFER_ASSIST_PUT:
	rc = globus_gass_transfer_register_put(
	    request,
	    attr,
	    url,
	    s.st_size,
	    globus_l_gass_transfer_assist_put_callback,
	    callback_info);
	break;
	
      case GLOBUS_L_GASS_TRANSFER_ASSIST_APPEND:
	rc = globus_gass_transfer_register_append(
	    request,
	    attr,
	    url,
	    s.st_size,
	    globus_l_gass_transfer_assist_put_callback,
	    callback_info);
	break;

      default:
	rc = GLOBUS_GASS_ERROR_INTERNAL_ERROR;
	break;
    }
    
    if(rc != GLOBUS_SUCCESS)
    {
	goto file_exit;
    }

    return GLOBUS_SUCCESS;
    

  file_exit:
    globus_io_close(&callback_info->handle);

  buffer_exit:
    globus_free(callback_info->buffer);

  callback_info_exit:
    globus_free(callback_info);

  error_exit:
    return rc;
}
/* globus_gass_transfer_assist_register_put_file_to_url() */

static
void
globus_l_gass_transfer_assist_put_callback(
    void *				arg,
    globus_gass_transfer_request_t	request)
{
    globus_gass_transfer_assist_callback_info_t *	callback_info;
    globus_result_t				result;

    callback_info = (globus_gass_transfer_assist_callback_info_t *) arg;
    
    if(globus_gass_transfer_request_get_status(request) !=
       GLOBUS_GASS_TRANSFER_REQUEST_PENDING)
    {
	callback_info->callback(callback_info->user_arg,
				request);
	globus_free(callback_info->buffer);
	globus_free(callback_info);

	return;
    }

    callback_info->request = request;
    result = globus_io_register_read(&callback_info->handle,
				     callback_info->buffer,
				     GLOBUS_GASS_DEFAULT_BUFFER_LENGTH,
				     1,
				     globus_l_gass_transfer_assist_read_callback,
				     callback_info);
    if(result != GLOBUS_SUCCESS)
    {
	globus_gass_transfer_fail(request,
				  globus_i_gass_transfer_assist_fail_callback,
				  (void *) callback_info);
    }

    return;
}
/* globus_l_gass_transfer_assist_put_callback() */

static
void
globus_l_gass_transfer_assist_read_callback(
    void *				arg,
    globus_io_handle_t *		handle,
    globus_result_t			result,
    globus_byte_t *			buf,
    globus_size_t			nbytes)
{
    globus_gass_transfer_assist_callback_info_t *	callback_info;
    globus_object_t *				err;
    globus_bool_t				last_data = GLOBUS_FALSE;
    int						rc;

    callback_info = (globus_gass_transfer_assist_callback_info_t *) arg;
    if(result != GLOBUS_SUCCESS)
    {
	err = globus_error_get(result);
	
	if(!globus_io_eof(err))
	{
	    globus_gass_transfer_fail(callback_info->request,
				      globus_i_gass_transfer_assist_fail_callback,
				      (void *) callback_info);
	    globus_error_put(err);
	    return;
	}
	else
	{
	    globus_object_free(err);
	    last_data = GLOBUS_TRUE;
	}
    }
    rc = globus_gass_transfer_send_bytes(
	callback_info->request,
	buf,
	nbytes,
	last_data,
	globus_l_gass_transfer_assist_send_callback,
	callback_info);
    if(rc != GLOBUS_SUCCESS)
    {
	globus_gass_transfer_fail(callback_info->request,
				  globus_i_gass_transfer_assist_fail_callback,
				  (void *) callback_info);
    }
}
/* globus_l_gass_transfer_assist_read_callback() */

static
void
globus_l_gass_transfer_assist_send_callback(
    void *				user_arg,
    globus_gass_transfer_request_t	request,
    globus_byte_t *			bytes,
    globus_size_t			length,
    globus_bool_t			last_data)
{
    globus_result_t                     result;
    globus_gass_transfer_assist_callback_info_t *callback_info;

    callback_info = (globus_gass_transfer_assist_callback_info_t *) user_arg;

    if(last_data)
    {
	globus_io_register_close(&callback_info->handle,
				 globus_i_gass_transfer_assist_close_callback,
				 callback_info);

	return;
    }
  
    /*
    if(globus_gass_transfer_request_get_status(request) !=
       GLOBUS_GASS_TRANSFER_REQUEST_STATE_PENDING)
    {
	globus_gass_transfer_fail(request,
				  globus_i_gass_transfer_assist_fail_callback,
				  user_arg);
	return;
    }
    */

    result = globus_io_register_read(&callback_info->handle,
				     callback_info->buffer,
				     GLOBUS_GASS_DEFAULT_BUFFER_LENGTH,
				     1,
				     globus_l_gass_transfer_assist_read_callback,
				     callback_info);
    if(result != GLOBUS_SUCCESS)
    {
	globus_gass_transfer_fail(request,
				  globus_i_gass_transfer_assist_fail_callback,
				  (void *) callback_info);
    }
}
/* globus_l_gass_transfer_assist_send_callback() */

