/******************************************************************************
globus_gass_transfer_assist_get.c
 
Description:
    This module implements the get functions for the GASS transfer
    assist library
 
CVS Information:
 
    $Source: /home/globdev/CVS/globus-current/Globus/FileAccess/gass/libraries/transfer_assist/globus_gass_transfer_assist_get.c,v $
    $Date: 1999/09/23 21:05:07 $
    $Revision: 1.4 $
    $Author: bester $
******************************************************************************/

#include "globus_i_gass_transfer_assist.h"

static
void
globus_l_gass_transfer_assist_get_callback(
    void *				arg,
    globus_gass_transfer_request_t	request);

static
void
globus_l_gass_transfer_assist_recv_callback(
    void *					arg,
    globus_gass_transfer_request_t		request,
    globus_byte_t *				bytes,
    globus_size_t				length,
    globus_bool_t				last_data);

static
void
globus_l_gass_transfer_assist_write_callback(
    void *				arg, 
    globus_io_handle_t *		handle, 
    globus_result_t			result,
    globus_byte_t *			buf,
    globus_size_t			nbytes);

/* API functions */
int
globus_gass_transfer_assist_get_file_from_url(
    globus_gass_transfer_request_t *		request,
    globus_gass_transfer_requestattr_t *	attr,
    char *					url,
    char *					file,
    globus_io_attr_t *				file_attr,
    globus_bool_t				follow_referrals)
{
    globus_gass_transfer_assist_monitor_t *	monitor;
    int						rc;
    globus_bool_t				done = GLOBUS_FALSE;
    globus_gass_transfer_referral_t		referral;

    memset(&referral,
	   '\0',
	   sizeof(globus_gass_transfer_referral_t));

    monitor = (globus_gass_transfer_assist_monitor_t *)
	globus_malloc(sizeof(globus_gass_transfer_assist_monitor_t));

    if(monitor == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_MALLOC_FAILED;
    }

    globus_mutex_init(&monitor->mutex,
		      GLOBUS_NULL);
    globus_cond_init(&monitor->cond,
		     GLOBUS_NULL);

    globus_mutex_lock(&monitor->mutex);
    while(!done)
    {
	monitor->done = GLOBUS_FALSE;

	rc = globus_gass_transfer_assist_register_get_file_from_url(
	    request,
	    attr,
	    url,
	    file,
	    file_attr,
	    globus_i_gass_transfer_assist_monitor_callback,
	    monitor);

	if(rc != GLOBUS_SUCCESS)
	{
	    globus_mutex_unlock(&monitor->mutex);
	    goto finish;
	}

	while(!monitor->done)
	{
	    globus_cond_wait(&monitor->cond,
			     &monitor->mutex);
	}

	switch(globus_gass_transfer_request_get_status(*request))
	{
	case GLOBUS_GASS_TRANSFER_REQUEST_REFERRED:
	    if(globus_gass_transfer_referral_get_count(&referral) != 0)
	    {
		globus_gass_transfer_referral_destroy(&referral);
	    }
		
	    rc = globus_gass_transfer_request_get_referral(
		*request,
		&referral);
	    if(rc != GLOBUS_SUCCESS)
	    {
		done = GLOBUS_TRUE;
		break;
	    }
	    url = globus_gass_transfer_referral_get_url(&referral,
							0);
	    globus_gass_transfer_request_destroy(*request);
	    if(url == GLOBUS_NULL)
	    {
		done = GLOBUS_TRUE;
	    }
	    break;
	case GLOBUS_GASS_TRANSFER_REQUEST_FAILED:
	case GLOBUS_GASS_TRANSFER_REQUEST_DENIED:
	    rc = GLOBUS_GASS_ERROR_TRANSFER_FAILED;
	    done = GLOBUS_TRUE;
	    break;
	case GLOBUS_GASS_TRANSFER_REQUEST_DONE:
	    done = GLOBUS_TRUE;
	    break;
	case GLOBUS_GASS_TRANSFER_REQUEST_STARTING:
	case GLOBUS_GASS_TRANSFER_REQUEST_PENDING:
	    globus_assert(GLOBUS_FALSE);
	    done = GLOBUS_TRUE;
	    break;
	}
	globus_mutex_unlock(&monitor->mutex);
    }
  finish:
    if(globus_gass_transfer_referral_get_count(&referral) != 0)
    {
	globus_gass_transfer_referral_destroy(&referral);
    }
    globus_mutex_destroy(&monitor->mutex);
    globus_cond_destroy(&monitor->cond);

    return rc;
}

int
globus_gass_transfer_assist_register_get_file_from_url(
    globus_gass_transfer_request_t *		request,
    globus_gass_transfer_requestattr_t *	attr,
    char *					url,
    char *					file,
    globus_io_attr_t *				file_attr,
    globus_gass_transfer_callback_t		callback,
    void *					user_arg)
{
    int						rc;
    globus_result_t				result;
    globus_gass_transfer_assist_callback_info_t *
						callback_info;
    struct stat					s;

    if(request == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_NULL_POINTER;
    }
    if(url == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_NULL_POINTER;
    }
    if(file == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_NULL_POINTER;
    }
    if(callback == GLOBUS_NULL)
    {
	return GLOBUS_GASS_ERROR_NULL_POINTER;
    }

    callback_info = (globus_gass_transfer_assist_callback_info_t *)
	globus_malloc(sizeof(globus_gass_transfer_assist_callback_info_t));

    if(callback_info == GLOBUS_NULL)
    {
	rc = GLOBUS_GASS_ERROR_MALLOC_FAILED;

	goto error_exit;
    }
    
    callback_info->buffer = (globus_byte_t *)
	globus_malloc(GLOBUS_GASS_DEFAULT_BUFFER_LENGTH);

    if(callback_info->buffer == GLOBUS_NULL)
    {
	rc = GLOBUS_GASS_ERROR_MALLOC_FAILED;

	goto callback_info_exit;
    }
    callback_info->callback = callback;
    callback_info->user_arg = user_arg;
    callback_info->last_data = GLOBUS_FALSE;

    /* open the file with globus_io */
    result = globus_io_file_open(file,
				 GLOBUS_IO_FILE_WRONLY |
				 GLOBUS_IO_FILE_CREAT,
				 GLOBUS_IO_FILE_IRWXU,
				 file_attr,
				 &callback_info->handle);
    if(result != GLOBUS_SUCCESS)
    {
	rc = GLOBUS_GASS_ERROR_OPEN_FAILED;

	goto buffer_exit;
    }

    /* open the connection to the server */
    rc = globus_gass_transfer_register_get(
	request,
	attr,
	url,
	globus_l_gass_transfer_assist_get_callback,
	callback_info);
    
    if(rc != GLOBUS_SUCCESS)
    {
	goto file_exit;
    }

    return GLOBUS_SUCCESS;
    

  file_exit:
    globus_io_close(&callback_info->handle);

  buffer_exit:
    globus_free(callback_info->buffer);

  callback_info_exit:
    globus_free(callback_info);

  error_exit:
    return rc;
}
/* globus_gass_transfer_assist_register_put_file_to_url() */

static
void
globus_l_gass_transfer_assist_get_callback(
    void *				arg,
    globus_gass_transfer_request_t	request)
{
    globus_gass_transfer_assist_callback_info_t *	callback_info;
    int					rc;

    callback_info = (globus_gass_transfer_assist_callback_info_t *) arg;
    
    if(globus_gass_transfer_request_get_status(request) !=
       GLOBUS_GASS_TRANSFER_REQUEST_PENDING)
    {
	callback_info->callback(callback_info->user_arg,
				request);
	globus_free(callback_info->buffer);
	globus_free(callback_info);

	return;
    }

    callback_info->request = request;
    rc = globus_gass_transfer_receive_bytes(request,
					    callback_info->buffer,
					    GLOBUS_GASS_DEFAULT_BUFFER_LENGTH,
					    1,
					    globus_l_gass_transfer_assist_recv_callback,
					    callback_info);
    if(rc != GLOBUS_SUCCESS)
    {
	globus_gass_transfer_fail(request,
				  globus_i_gass_transfer_assist_fail_callback,
				  (void *) callback_info);
    }

    return;
}
/* globus_l_gass_transfer_assist_get_callback() */

static
void
globus_l_gass_transfer_assist_monitor_callback(
    void *				arg,
    globus_gass_transfer_request_t	request)
{
    globus_gass_transfer_assist_monitor_t *monitor;

    monitor = (globus_gass_transfer_assist_monitor_t *)
	arg;

    globus_mutex_lock(&monitor->mutex);
    monitor->done = GLOBUS_TRUE;
    globus_cond_signal(&monitor->cond);
    globus_mutex_unlock(&monitor->mutex);
}
/* globus_l_gass_transfer_assist_monitor_callback() */

static
void
globus_l_gass_transfer_assist_recv_callback(
    void *					arg,
    globus_gass_transfer_request_t		request,
    globus_byte_t *				bytes,
    globus_size_t				length,
    globus_bool_t				last_data)

{
    globus_result_t                     result;
    globus_gass_transfer_assist_callback_info_t *callback_info;

   callback_info = (globus_gass_transfer_assist_callback_info_t *) arg;
  /*
    if(globus_gass_transfer_request_get_status(request) !=
       GLOBUS_GASS_TRANSFER_REQUEST_STATE_PENDING)
    {
	globus_gass_transfer_fail(request,
				  globus_i_gass_transfer_assist_fail_callback,
				  user_arg);
	return;
    }
    */
   if(last_data)
   {
       callback_info->last_data = GLOBUS_TRUE;
   }

    result = globus_io_register_write(
	&callback_info->handle,
	callback_info->buffer,
	length,
	globus_l_gass_transfer_assist_write_callback,
	callback_info);

    if(result != GLOBUS_SUCCESS)
    {
	callback_info->last_data = GLOBUS_TRUE;
	globus_gass_transfer_fail(request,
				  globus_i_gass_transfer_assist_fail_callback,
				  (void *) callback_info);
    }
}
/* globus_l_gass_transfer_assist_recv_callback() */

static
void
globus_l_gass_transfer_assist_write_callback(
    void *				arg, 
    globus_io_handle_t *		handle, 
    globus_result_t			result,
    globus_byte_t *			buf,
    globus_size_t			nbytes)
{
    globus_gass_transfer_assist_callback_info_t *callback_info;
    int						rc;

    callback_info = (globus_gass_transfer_assist_callback_info_t *) arg;

    if(result == GLOBUS_SUCCESS && callback_info->last_data)
    {
	globus_io_register_close(handle,
				 globus_i_gass_transfer_assist_close_callback,
				 arg);
    }
    else if(result == GLOBUS_SUCCESS)
    {
	rc = globus_gass_transfer_receive_bytes(callback_info->request,
						callback_info->buffer,
						GLOBUS_GASS_DEFAULT_BUFFER_LENGTH,
						1,
						globus_l_gass_transfer_assist_recv_callback,
						callback_info);
	if(rc != GLOBUS_SUCCESS)
	{
	    callback_info->last_data = GLOBUS_TRUE;
	    globus_gass_transfer_fail(callback_info->request,
				      globus_i_gass_transfer_assist_fail_callback,
				      (void *) callback_info);
	}
    }
    else if(result != GLOBUS_SUCCESS &&
	    callback_info->last_data == GLOBUS_FALSE)
    {
       callback_info->last_data = GLOBUS_TRUE;
       globus_gass_transfer_fail(callback_info->request,
				 globus_i_gass_transfer_assist_fail_callback,
				 (void *) callback_info);
    }
}
/* globus_l_gass_transfer_assist_write_callback() */
