/******************************************************************************
globus_gass_transfer_file.c
 
Description:
    This module implements the file URL scheme for the GASS transfer library
 
CVS Information:
 
    $Source: /home/globdev/CVS/globus-current/Globus/FileAccess/gass/libraries/transfer/globus_gass_transfer_file.c,v $
    $Date: 1999/06/10 02:22:52 $
    $Revision: 1.1 $
    $Author: bester $
******************************************************************************/

#include "globus_i_gass_transfer.h"
#include "globus_io.h"

/******************************************************************************
			  Module specific Types
******************************************************************************/
typedef enum
{
    GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE,
    GLOBUS_GASS_TRANSFER_FILE_STATE_PENDING,
    GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING,
    GLOBUS_GASS_TRANSFER_FILE_STATE_DONE
} globus_gass_transfer_file_state_t;

typedef struct 
{
    globus_gass_transfer_proto_send_t		send_buffer;
    globus_gass_transfer_proto_receive_t	recv_buffer;

    globus_gass_transfer_proto_func_t		fail;

    globus_gass_transfer_proto_deny_func_t	deny;
    globus_gass_transfer_proto_func_t		refer;
    globus_gass_transfer_proto_func_t		authorize;

    globus_gass_transfer_proto_func_t		destroy;

    globus_io_handle_t				handle;
    globus_bool_t				last_data;
    volatile globus_gass_transfer_file_state_t	state;

    globus_gass_transfer_request_t		request;

    globus_bool_t				destroy_called;
} globus_gass_transfer_file_proto_t;

/******************************************************************************
			  Module specific Prototypes
******************************************************************************/
static
void
globus_l_gass_transfer_file_send(
    globus_gass_transfer_proto_t *		proto,
    globus_gass_transfer_request_t		request,
    globus_byte_t *				buffer,
    globus_size_t				buffer_length,
    globus_bool_t				last_data);

static
void
globus_l_gass_transfer_file_receive(
    globus_gass_transfer_proto_t *		proto,
    globus_gass_transfer_request_t		request,
    globus_byte_t *				buffer,
    globus_size_t				buffer_length,
    globus_size_t				wait_for_length);

static
void
globus_l_gass_transfer_file_write_callback(
    void *					callback_arg,
    globus_io_handle_t *			handle,
    globus_result_t				result,
    globus_byte_t *				buf,
    globus_size_t				nbytes);

static
globus_gass_transfer_file_proto_t *
globus_l_gass_tranfer_file_new_proto(
    globus_gass_transfer_request_t		request,
    globus_gass_transfer_requestattr_t *	attr,
    int						flags,
    int						mode);

static
void
globus_l_gass_transfer_file_read_callback(
    void *					callback_arg,
    globus_io_handle_t *			handle,
    globus_result_t				result,
    globus_byte_t *				buf,
    globus_size_t				nbytes);

static
void
globus_l_gass_transfer_file_fail(
    globus_gass_transfer_proto_t *		proto,
    globus_gass_transfer_request_t		request);

static
void
globus_l_gass_transfer_file_close_callback(
    void *					callback_arg,
    globus_io_handle_t *			handle,
    globus_result_t				result);

static
void
globus_l_gass_transfer_file_destroy(
    globus_gass_transfer_proto_t *		proto,
    globus_gass_transfer_request_t		request);

static
void
globus_l_gass_transfer_file_put_request(
    globus_gass_transfer_request_t		request,
    globus_gass_transfer_requestattr_t *	attr);

static
void
globus_l_gass_transfer_file_get_request(
    globus_gass_transfer_request_t		request,
    globus_gass_transfer_requestattr_t *	attr);

static
void
globus_l_gass_transfer_file_append_request(
    globus_gass_transfer_request_t		request,
    globus_gass_transfer_requestattr_t *	attr);

static
globus_object_t *
globus_l_gass_transfer_file_new_requestattr(
    char *                                      url_scheme);

/* Protocol Descriptor */
globus_gass_transfer_proto_descriptor_t
globus_gass_transfer_file_descriptor =
{
    "file",

    /* client-side support */
    globus_l_gass_transfer_file_new_requestattr,
    globus_l_gass_transfer_file_put_request,
    globus_l_gass_transfer_file_get_request,
    GLOBUS_NULL, /* POST--unimplemented */
    globus_l_gass_transfer_file_append_request,

    /* server-side support */
    GLOBUS_NULL /* new_listenerattr */,
    GLOBUS_NULL /* new_listener */,
    GLOBUS_NULL /* close_listener */,
    GLOBUS_NULL /* listen */,
    GLOBUS_NULL /* accept */
};

/*
 * Function: globus_l_gass_transfer_file_send()
 * 
 * Description: Send a byte array to a file handle
 * 
 * Parameters: 
 * 
 * Returns: 
 */
static
void
globus_l_gass_transfer_file_send(
    globus_gass_transfer_proto_t *		proto,
    globus_gass_transfer_request_t		request,
    globus_byte_t *				buffer,
    globus_size_t				buffer_length,
    globus_bool_t				last_data)
{
    globus_result_t				result;
    globus_gass_transfer_file_proto_t *		new_proto;

    globus_l_gass_transfer_file_lock();

    new_proto = (globus_gass_transfer_file_proto_t *) proto;

    /* We can only process a send if the proto is in the "idle" state */
    globus_assert(new_proto->state == GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE);

    /* state change to "pending" */
    new_proto->last_data = last_data;
    new_proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_PENDING;

    /* Register our output operation */
    result =
	globus_io_register_write(&new_proto->handle,
				 buffer,
				 buffer_length,
				 globus_l_gass_transfer_file_write_callback,
				 new_proto);

    if(result == GLOBUS_SUCCESS)
    {
	/*
	 * Registration succeeded. Let the callback happen when I/O
	 * completes.
	 */
	globus_l_gass_transfer_file_unlock();
	return;
    }

    /* Registration failed, signal failure to GASS */
    new_proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING;

    result = globus_io_register_close(&new_proto->handle,
				      globus_l_gass_transfer_file_close_callback,
				      new_proto);
    if(result != GLOBUS_SUCCESS)
    {
	new_proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_DONE;
    }
    globus_l_gass_transfer_file_unlock();

    globus_gass_transfer_proto_request_fail(request);

    return;
}
/* globus_l_gass_transfer_file_send() */


/*
 * Function: globus_l_gass_transfer_file_receive()
 * 
 * Description: Schedule the next block of data from the server (file)
 *              to end up in the provided byte array
 * 
 * Parameters: 
 * 
 * Returns: 
 */
static
void
globus_l_gass_transfer_file_receive(
    globus_gass_transfer_proto_t *		proto,
    globus_gass_transfer_request_t		request,
    globus_byte_t *				buffer,
    globus_size_t				buffer_length,
    globus_size_t				wait_for_length)
{
    globus_result_t				result;
    globus_gass_transfer_file_proto_t *		new_proto;

    globus_l_gass_transfer_file_lock();
    new_proto = (globus_gass_transfer_file_proto_t *) proto;

    /* We can only process a send if the proto is in the "idle" state */
    globus_assert(new_proto->state == GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE);

    /* state change to "pending" */
    new_proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_PENDING;

    /* Register our input operation */
    result =
	globus_io_register_read(&new_proto->handle,
				buffer,
				buffer_length,
				wait_for_length,
				globus_l_gass_transfer_file_read_callback,
				new_proto);

    if(result == GLOBUS_SUCCESS)
    {
	/*
	 * Registration succeeded. Let the callback happen when I/O
	 * completes.
	 */
	globus_l_gass_transfer_file_unlock();
	return;
    }

    /* Registration failed, signal failure to GASS */
    new_proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING;

    result = globus_io_register_close(&new_proto->handle,
				      globus_l_gass_transfer_file_close_callback,
				      new_proto);
    if(result != GLOBUS_SUCCESS)
    {
	new_proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_DONE;
    }
    globus_l_gass_transfer_file_unlock();

    globus_gass_transfer_proto_request_fail(request);
}
/* globus_l_gass_transfer_file_receive() */

/*
 * Function: globus_l_gass_transfer_file_fail()
 * 
 * Description: Cause the given request to fail for client-caused reasons
 * 
 * Parameters: 
 * 
 * Returns: 
 */
static
void
globus_l_gass_transfer_file_fail(
    globus_gass_transfer_proto_t *		proto,
    globus_gass_transfer_request_t		request)
{
    globus_gass_transfer_file_proto_t *		new_proto;
    globus_result_t				result;

    new_proto = (globus_gass_transfer_file_proto_t *) proto;

    globus_l_gass_transfer_file_lock();

    switch(new_proto->state)
    {
      case GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE:
      case GLOBUS_GASS_TRANSFER_FILE_STATE_PENDING:
	/* We will transition to the closing state, signalling the failure,
	 * and registering the close (which will transition us to the 
	 * done state).
	 */
	new_proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING;

	result = globus_io_register_close(&new_proto->handle,
					  globus_l_gass_transfer_file_close_callback,
					  new_proto);

	if(result != GLOBUS_SUCCESS)
	{
	    new_proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_DONE;
	}
	globus_l_gass_transfer_file_unlock();

	globus_gass_transfer_proto_request_fail(request);

	break;

      case GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING:
      case GLOBUS_GASS_TRANSFER_FILE_STATE_DONE:
	globus_l_gass_transfer_file_unlock();
	break;
    }
}
/* globus_l_gass_transfer_file_fail() */

/*
 * Function: globus_l_gass_transfer_file_write_callack()
 * 
 * Description: Callback when the write of the active buffer has completed
 *              or failed.
 * 
 * Parameters: 
 * 
 * Returns: 
 */
static
void
globus_l_gass_transfer_file_write_callback(
    void *					callback_arg,
    globus_io_handle_t *			handle,
    globus_result_t				result,
    globus_byte_t *				buf,
    globus_size_t				nbytes)
{
    globus_gass_transfer_file_proto_t *		proto;
    
    proto = (globus_gass_transfer_file_proto_t *) callback_arg;
    globus_l_gass_transfer_file_lock();

    switch(proto->state)
    {
      case GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING:
      case GLOBUS_GASS_TRANSFER_FILE_STATE_DONE:
	/* If we've already finished, there is nothing we can do here */
	globus_l_gass_transfer_file_unlock();
	return;

      case GLOBUS_GASS_TRANSFER_FILE_STATE_PENDING:
	if(result != GLOBUS_SUCCESS)
	{
	    /*
	     * I/O error occurred, go to failure state, and schedule
	     * transition to the DONE state
	     */
	    proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING;

	    result = globus_io_register_close(&proto->handle,
					      globus_l_gass_transfer_file_close_callback,
					      proto);
	    if(result != GLOBUS_SUCCESS)
	    {
		proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_DONE;
	    }
	    globus_l_gass_transfer_file_unlock();

	    globus_gass_transfer_proto_request_fail(proto->request);
	}
	else
	{
	    /*
	     * If this is the last data, schedule transition to "done".
	     */
	    if(proto->last_data)
	    {
		proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING;

		result =
		    globus_io_register_close(&proto->handle,
					     globus_l_gass_transfer_file_close_callback,
					     proto);
		if(result != GLOBUS_SUCCESS)
		{
		    proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_DONE;
		}
	    }
	    /*
	     * Data write for this buffer completed, let GASS know
	     */
	    globus_l_gass_transfer_file_unlock();
	    globus_gass_transfer_proto_send_complete(proto->request,
						     buf,
						     nbytes);
	}
	break;
      case GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE:
	globus_assert(proto->state != GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE);
	globus_l_gass_transfer_file_unlock();
	break;
    }
}
/* globus_l_gass_transfer_file_write_callback() */

/*
 * Function: globus_l_gass_transfer_file_read_callack()
 * 
 * Description: Callback when the read of from the file to thethe active
 *              buffer has completed or failed.
 * 
 * Parameters: 
 * 
 * Returns: 
 */
static
void
globus_l_gass_transfer_file_read_callback(
    void *					callback_arg,
    globus_io_handle_t *			handle,
    globus_result_t				result,
    globus_byte_t *				buf,
    globus_size_t				nbytes)
{
    globus_object_t *				err;
    globus_gass_transfer_file_proto_t *		proto;

    proto = (globus_gass_transfer_file_proto_t *) callback_arg;

    err = globus_error_get(result);

    globus_l_gass_transfer_file_lock();

    switch(proto->state)
    {
      case GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING:
      case GLOBUS_GASS_TRANSFER_FILE_STATE_DONE:
	globus_l_gass_transfer_file_unlock();
	break;
      case GLOBUS_GASS_TRANSFER_FILE_STATE_PENDING:
	if(result == GLOBUS_SUCCESS)
	{
	    proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE;
	    globus_l_gass_transfer_file_unlock();
	    globus_gass_transfer_proto_receive_complete(proto->request,
							buf,
							nbytes,
							GLOBUS_FALSE);
	}
	else if(globus_io_eof(err))
	{
	    /* Got the final amount of the data */
	    proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING;

	    result =
		globus_io_register_close(&proto->handle,
					 globus_l_gass_transfer_file_close_callback,
					 proto);
	    if(result != GLOBUS_SUCCESS)
	    {
		proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_DONE;

	    }

	    globus_l_gass_transfer_file_unlock();
	    globus_gass_transfer_proto_receive_complete(proto->request,
							buf,
							nbytes,
							GLOBUS_TRUE);
	}
	else
	{
	    /* I/O Error occurred */
	    proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING;

	    result =
		globus_io_register_close(&proto->handle,
					 globus_l_gass_transfer_file_close_callback,
					 proto);
	    if(result != GLOBUS_SUCCESS)
	    {
		proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_DONE;

	    }

	    globus_l_gass_transfer_file_unlock();

	    globus_gass_transfer_proto_request_fail(proto->request);
	}
	break;
      case GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE:
	globus_assert(proto->state != GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE);
	break;
    }
    globus_object_free(err);
}
/* globus_l_gass_transfer_file_read_callback() */

/*
 * Function: globus_l_gass_transfer_file_close_callback()
 *
 * Description: Called upon completion of close()ing the file handle,
 *              Will free the proto instance if the client has called
 *              the "done" function
 *
 *
 * Parameters:
 *
 * Returns:
 */
static
void
globus_l_gass_transfer_file_close_callback(
    void *					callback_arg,
    globus_io_handle_t *			handle,
    globus_result_t				result)
{
    globus_gass_transfer_file_proto_t *		proto;

    proto = (globus_gass_transfer_file_proto_t *) callback_arg;
    globus_l_io_gass_transfer_file_lock();
    proto->state = GLOBUS_GASS_TRANSFER_FILE_STATE_DONE;

    if(proto->destroy_called)
    {
	globus_free(proto);
    }
    globus_l_io_gass_transfer_file_unlock();
}
/* globus_l_gass_transfer_file_close_callback() */

/*
 * Function: globus_l_gass_transfer_file_proto_destroy()
 *
 * Description:
 *
 * Parameters:
 *
 * Returns:
 */
static
void
globus_l_gass_transfer_file_destroy(
    globus_gass_transfer_proto_t *		proto,
    globus_gass_transfer_request_t		request)
{
    globus_gass_transfer_file_proto_t *		new_proto;

    new_proto = (globus_gass_transfer_file_proto_t *) proto;

    globus_l_io_gass_transfer_file_lock();
    if(new_proto->state == GLOBUS_GASS_TRANSFER_FILE_STATE_DONE)
    {
        globus_free(new_proto);
    }
    else if(new_proto->state == GLOBUS_GASS_TRANSFER_FILE_STATE_CLOSING)
    {
	new_proto->destroy_called;
    }
    globus_l_io_gass_transfer_file_unlock();
}
/* globus_l_gass_transfer_file_proto_destroy() */

/*
 * Function: globus_l_gass_transfer_file_put_request()
 *
 * Description: Create a new protocol instance to handle this
 *              new "put" request.
 *
 * Parameters:
 *
 * Returns:
 */
static
void
globus_l_gass_transfer_file_put_request(
    globus_gass_transfer_request_t		request,
    globus_gass_transfer_requestattr_t *	attr)
{
    globus_gass_transfer_file_proto_t *		proto;
    int						flags;
    int						mode;

    flags = GLOBUS_IO_FILE_WRONLY|GLOBUS_IO_FILE_CREAT|GLOBUS_IO_FILE_TRUNC;
    mode = GLOBUS_IO_FILE_IRWXU;

    proto = globus_l_gass_tranfer_file_new_proto(request,
						 attr,
						 flags,
						 mode);

    if(proto)
    {
	globus_gass_transfer_proto_request_ready(
	    request,
	    (globus_gass_transfer_proto_t *) proto);
    }
    else
    {
	globus_gass_transfer_proto_request_denied(
	    request,
	    GLOBUS_FAILURE,
	    GLOBUS_NULL);
    }
}
/* globus_l_gass_transfer_file_put_request() */

/*
 * Function: globus_l_gass_transfer_file_put_request()
 *
 * Description: Create a new protocol instance to handle this
 *              new "get" request.
 *
 * Parameters:
 *
 * Returns:
 */
static
void
globus_l_gass_transfer_file_get_request(
    globus_gass_transfer_request_t		request,
    globus_gass_transfer_requestattr_t *	attr)
{
    globus_gass_transfer_file_proto_t *		proto;
    int						flags;
    int						mode;

    flags = GLOBUS_IO_FILE_RDONLY|GLOBUS_IO_FILE_CREAT;
    mode = GLOBUS_IO_FILE_IRWXU;

    proto = globus_l_gass_tranfer_file_new_proto(request,
						 attr,
						 flags,
						 mode);

    if(proto)
    {
	globus_gass_transfer_proto_request_ready(
	    request,
	    (globus_gass_transfer_proto_t *) proto);
    }
    else
    {
	globus_gass_transfer_proto_request_denied(
	    request,
	    GLOBUS_FAILURE,
	    GLOBUS_NULL);
    }
}
/* globus_l_gass_transfer_file_get_request() */

/*
 * Function: globus_l_gass_transfer_file_append_request()
 *
 * Description: Create a new protocol instance to handle this
 *              new "get" request.
 *
 * Parameters:
 *
 * Returns:
 */
static
void
globus_l_gass_transfer_file_append_request(
    globus_gass_transfer_request_t		request,
    globus_gass_transfer_requestattr_t *	attr)
{
    globus_gass_transfer_file_proto_t *		proto;
    int						flags;
    int						mode;

    flags = GLOBUS_IO_FILE_WRONLY|GLOBUS_IO_FILE_CREAT|GLOBUS_IO_FILE_APPEND;
    mode = GLOBUS_IO_FILE_IRWXU;

    proto = globus_l_gass_tranfer_file_new_proto(request,
						 attr,
						 flags,
						 mode);

    if(proto)
    {
	globus_gass_transfer_proto_request_ready(
	    request,
	    (globus_gass_transfer_proto_t *) proto);
    }
    else
    {
	globus_gass_transfer_proto_request_denied(
	    request,
	    GLOBUS_FAILURE,
	    GLOBUS_NULL);
    }
}
/* globus_l_gass_transfer_file_append_request() */

/*
 * Function: globus_l_gass_tranfer_file_new_proto(()
 *
 * Description: Create the protocol instantiation used
 *              for the above "new request" operations
 *
 * Parameters:
 *
 * Returns:
 */
static
globus_gass_transfer_file_proto_t *
globus_l_gass_tranfer_file_new_proto(
    globus_gass_transfer_request_t		request,
    globus_gass_transfer_requestattr_t *	attr,
    int						flags,
    int						mode)
{
    globus_url_t				url;
    int						rc;
    char *					proxy;
    globus_gass_transfer_file_mode_t		file_mode;
    globus_io_attr_t				io_attr;
    globus_result_t				result;
    globus_gass_transfer_file_proto_t *		proto;

    /* Check attributes we care about */
    globus_gass_transfer_requestattr_get_proxy_url(attr,
						   &proxy);
    /* File system can not proxy */
    if(proxy != GLOBUS_NULL)
    {
	return GLOBUS_NULL;
    }

    result = globus_io_fileattr_init(&io_attr);
    if(result != GLOBUS_SUCCESS)
    {
	return GLOBUS_NULL;
    }

    /* File mode is important on Windows */
    rc = globus_gass_transfer_requestattr_get_file_mode(attr,
							&file_mode);
    if(rc == GLOBUS_SUCCESS &&
       file_mode == GLOBUS_GASS_FILE_MODE_TEXT)
    {
	result = globus_io_attr_set_file_type(&io_attr,
					      GLOBUS_IO_FILE_TYPE_TEXT);
	if(result != GLOBUS_SUCCESS)
	{
	    goto error_exit;
	}
    }

    /* Verify URL */
    rc = globus_url_parse(globus_gass_transfer_request_get_url(request),
			  &url);
    if(rc != GLOBUS_SUCCESS)
    {
	goto error_exit;
    }
    if(url.scheme_type != GLOBUS_URL_SCHEME_FILE)
    {
	goto url_error;
    }

    /* Initialize the proto instance */
    proto = (globus_gass_transfer_file_proto_t *) 
	globus_malloc(sizeof(globus_gass_transfer_file_proto_t));

    if(proto == GLOBUS_NULL)
    {
	goto url_error;
    }

    proto->send_buffer	= globus_l_gass_transfer_file_send;
    proto->recv_buffer	= globus_l_gass_transfer_file_receive;
    proto->fail		= globus_l_gass_transfer_file_fail;
    proto->deny		= GLOBUS_NULL;
    proto->refer	= GLOBUS_NULL;
    proto->authorize	= GLOBUS_NULL;
    proto->destroy	= globus_l_gass_transfer_file_destroy;
    proto->last_data	= GLOBUS_FALSE;
    proto->state	= GLOBUS_GASS_TRANSFER_FILE_STATE_IDLE;
    proto->request	= request;

    /* Open the handle */
    result = globus_io_open(url.url_path,
			    flags,
			    mode,
			    &io_attr,
			    &proto->handle);

    if(result != GLOBUS_SUCCESS)
    {
	goto proto_error;
    }
    /* Success! */
    return proto;

  proto_error:
    globus_free(proto);
  url_error:
    globus_url_destroy(&url);
  error_exit:
    globus_io_attr_destroy(&io_attr);
    return GLOBUS_NULL;
}
/* globus_l_gass_tranfer_file_new_proto() */
/*
 * Function: globus_l_gass_transfer_file_new_requestattr()
 *
 * Description: Create a new request attribute structure,
 *              appropriate for the "file" url scheme
 *
 * Parameters:
 *
 * Returns:
 */
static
globus_object_t *
globus_l_gass_transfer_file_new_requestattr(
    char *                                      url_scheme)
{
    globus_object_t *				obj;

    globus_assert(strcasecmp(url_scheme, "file") == 0);
    
    globus_object_construct(GLOBUS_GASS_OBJECT_TYPE_REQUESTATTR);

    return globus_gass_transfer_requestattr_initialize(
	obj,
	GLOBUS_NULL,
	GLOBUS_GASS_TRANSFER_FILE_MODE_BINARY,
	GLOBUS_GASS_LENGTH_UNKNOWN,
	GLOBUS_FALSE);
}
/* globus_l_gass_transfer_file_new_requestattr() */
