/*
 *  this file will contain utilities for the server.
 *
 *  - parsing the programs arguements
 *  - printing debug messages
 *  - printing log messages
 *  - printing help messages
 */
#include <gftpd.h>
#include <string.h>
#include <stdio.h>

typedef globus_bool_t (*gftpd_arg_function_t)(char * argv[], int argc, int *ctr);

globus_bool_t
gftpd_set_debug_print_level(
    char *                                      argv[],
    int                                         argc,
    int *                                       ctr);

globus_bool_t
gftpd_set_port(
    char *                                      argv[], 
    int                                         argc,
    int *                                       ctr);

globus_bool_t
gftpd_set_fork(
    char *                                      argv[], 
    int                                         argc,
    int *                                       ctr);

globus_bool_t
gftpd_set_nofork(
    char *                                      argv[], 
    int                                         argc,
    int *                                       ctr);

char *
trim_spaces(
    char *                                      str);

globus_bool_t
gftpd_daemon(
    char *                                      argv[],
    int                                         argc,
    int *                                       ctr);

typedef struct gftpd_argument_s
{
    char *                                        arg;
    char *                                        description;
    gftpd_arg_function_t                          func;
} gftpd_argument_t;

/*
 *  the arguments table
 */
static gftpd_argument_t                           g_args[] =
{
    "-d", "set the debug print level.",  gftpd_set_debug_print_level,
    "--debug-print-level", "set the debug print level.",  gftpd_set_debug_print_level,
    "-D", "set the debug print file [stdout].",  gftpd_set_debug_print_file,
    "--debug-print-file", "set the debug print file [stdout].",  gftpd_set_debug_print_file,
    "-f", "tell the process to fork when a client connects", gftpd_set_fork,
    "-F", "tell the process to not fork when a client connects", gftpd_set_nofork,
    "-p", "specify the port number", gftpd_set_port,
    "-s", "run process as a daemon", gftpd_daemon,
};

#define ARGUMENT_TABLE_SIZE                    8

/*
 *  global variables
 */
static FILE *                                  g_debug_stream = GLOBUS_NULL;
static FILE *                                  g_logfile = GLOBUS_NULL;
static int                                     g_debug_level = 1;
char *                                         g_root_path = GLOBUS_NULL;
globus_bool_t                                  g_secure = GLOBUS_FALSE;
char                                           g_hostname[MAXHOSTNAMELEN];
char                                           g_version[SFTPD_STRING_SIZE];

unsigned short                                 g_port = 0;
globus_bool_t                                  g_fork = GLOBUS_TRUE;
globus_bool_t                                  g_daemon = GLOBUS_FALSE;
char *                                         g_rc_file_name = ".gftpdrc";

char *                                         g_backend = "./backend"; 
char **                                        g_backend_args;

globus_reltime_t                               g_timeout;

globus_bool_t
gftpd_daemon(
    char *                                      argv[], 
    int                                         argc,
    int *                                       ctr)
{
    g_daemon = GLOBUS_TRUE;
}

globus_bool_t
gftpd_set_port(
    char *                                      argv[], 
    int                                         argc,
    int *                                       ctr)
{
    if(*ctr + 1 < argc)
    {
        (*ctr)++;
        g_port = atoi(argv[*ctr]);

        return GLOBUS_TRUE;
    }

    return GLOBUS_FALSE;
}

globus_bool_t
gftpd_set_fork(
    char *                                      argv[], 
    int                                         argc,
    int *                                       ctr)
{
    g_fork = GLOBUS_TRUE;

    return GLOBUS_TRUE;
}

globus_bool_t
gftpd_set_nofork(
    char *                                      argv[], 
    int                                         argc,
    int *                                       ctr)
{
    g_fork = GLOBUS_FALSE;

    return GLOBUS_TRUE;
}

globus_bool_t
gftpd_set_debug_print_level(
    char *                                      argv[], 
    int                                         argc,
    int *                                       ctr)
{
    if(*ctr + 1 >= argc)
    {
        return GLOBUS_FALSE;
    }

    g_debug_level = atoi(argv[*ctr+1]);
    (*ctr)++;

    return GLOBUS_TRUE;
}

globus_bool_t
gftpd_set_debug_print_file(
    char *                                      argv[], 
    int                                         argc,
    int *                                       ctr)
{
    if(*ctr + 1 >= argc)
    {
        return GLOBUS_FALSE;
    }

    g_debug_stream = fopen(argv[*ctr+1], "w");
    if(g_debug_stream == NULL)
    {
        return GLOBUS_FALSE;
    }
    (*ctr)++;

    return GLOBUS_TRUE;
}

void
gftpd_print_help()
{
    int                                          ctr;

    printf("sftp\n");
    printf("====\n");
    printf("Globus FTP Striped Server\n");
    printf("-------------------------\n");
    printf("options:\n");
    for(ctr = 0; ctr < ARGUMENT_TABLE_SIZE; ctr++)
    {
        printf("%s: %s\n", g_args[ctr].arg, g_args[ctr].description);
    }
}

globus_bool_t
gftpd_parse_main_args(
    int                                          argc,
    char *                                       argv[])
{
    int                                          ctr;
    int                                          ctr2;
    globus_bool_t                                found = GLOBUS_FALSE;
    globus_bool_t                                done = GLOBUS_FALSE;

    globus_libc_gethostname(g_hostname, MAXHOSTNAMELEN);
    sprintf(g_version, "Globus Striped Server 0.1beta");

    g_backend_args = (char **)globus_malloc(sizeof(char *) * (argc + 1));
    GlobusTimeReltimeSet(g_timeout, 600, 0);
    for(ctr = 1; ctr < argc; ctr++)
    {
        g_backend_args[ctr] = argv[ctr];
        done = GLOBUS_FALSE;
        for(ctr2 = 0; ctr2 < ARGUMENT_TABLE_SIZE && !done; ctr2++)
        {
            if(strcmp(g_args[ctr2].arg, argv[ctr]) == 0)
            {
                found = g_args[ctr2].func(argv, argc, &ctr);

                done = GLOBUS_TRUE;
            }
        }
    }
    g_backend_args[0] = g_backend;
    g_backend_args[argc] = GLOBUS_NULL;

    if(!found)
    {
        gftpd_print_help();

        return GLOBUS_FALSE;
    }

    return GLOBUS_TRUE;
}

void
gftpd_debug_print(
    int                                       level,
    char *                                    fmt,
    ...)
{
    va_list                                   ap;
    pid_t                                     pid;

    pid = getpid();

    if(level <= g_debug_level)
    {
        va_start(ap, fmt);
        if(g_debug_stream)
        {
            fprintf(g_debug_stream, "p%d:", pid);
            vfprintf(g_debug_stream, fmt, ap);
        }
        else
        {
            fprintf(stdout, "p%d:", pid);
            vfprintf(stdout, fmt, ap);
        }
        va_end(ap);
    }
}

void
gftpd_log_message(
    int                                        type,
    char *                                     fmt,
    ...)
{
    va_list                                   ap;

    if(g_logfile != GLOBUS_NULL)
    {
        va_start(ap, fmt);
        vfprintf(g_logfile, fmt, ap);
        va_end(ap);
    }
}


/*
 *  this function will parse a string into substrings.
 *  it returns the number of substrings found.
 *
 *  a call to this should be followed by a call to destroy.
 */
int
gftpd_parse_into_substrings(
    char *                                      list,
    char *                                      arg[],
    int                                         argc)
{
    int                                       count = 0;
    char *                                    tmp_ptr;
    char *                                    tmp_ptr2;
    int                                       ctr;

    memset(arg, '\0', sizeof(char *) * argc);
    if(list == GLOBUS_NULL || strlen(list) < 1)
    {
        return 0;
    }

    tmp_ptr = trim_spaces(list);
    for(ctr = 0; ctr < argc; ctr++)
    {
        /* allocate max amount of memory that could be needed */
        arg[ctr] = globus_malloc(strlen(list));

        tmp_ptr2 = strstr(tmp_ptr, " ");
        tmp_ptr2 = trim_spaces(tmp_ptr2);
        if(tmp_ptr2 == GLOBUS_NULL)
        {
            if(strlen(tmp_ptr) > 0)
            {
                count++;
                strcpy(arg[ctr], trim_spaces(tmp_ptr));
            }
            return count;
        }
        else
        {
            count++;
            strncpy(arg[ctr],  tmp_ptr,  tmp_ptr2 -  tmp_ptr);
            arg[ctr][tmp_ptr2-tmp_ptr] = '\0';
        }
        /* trim off the end spaces */
        trim_spaces(arg[ctr]);
        tmp_ptr = tmp_ptr2;
    }

    return count;
}

void
gftpd_substrings_destroy(
    char *                                      arg[],
    int                                         argc)
{
    int                                         ctr;

    for(ctr = 0; ctr < argc; ctr++)
    {
        if(arg[ctr] != GLOBUS_NULL)
        {
            globus_free(arg[ctr]);
            arg[ctr] = GLOBUS_NULL;
        }
    }
}

/*
 *  This function removes spaces from the end of a string and
 *  returns a pointer to the first character that is not a space
 *  in the passed in string.
 */
char *
trim_spaces(
    char *                                      str)
{
    int                                         ctr;
    char *                                      tmp_ptr;
    char *                                      back_ptr;

    if(str == GLOBUS_NULL)
    {
        return GLOBUS_NULL;
    }

    tmp_ptr = str;
    ctr = 0;
    while(ctr < strlen(str) && isspace(*tmp_ptr))
    {
        tmp_ptr++;
        ctr++;
    }

    back_ptr = str + strlen(str) - 1;
    ctr = 0;
    while(ctr < strlen(str) && isspace(*back_ptr))
    {
        *back_ptr = '\0';
        back_ptr--;
        ctr++;
    }

    return tmp_ptr;
}


void
gftpd_monitor_init(
    gftpd_monitor_t *                            monitor)
{
    globus_mutex_init(&monitor->mutex, GLOBUS_NULL);
    globus_cond_init(&monitor->cond, GLOBUS_NULL);

    gftpd_monitor_reset(monitor);
}

void
gftpd_monitor_reset(
    gftpd_monitor_t *                            monitor)
{
    monitor->count = 0;
    monitor->done = GLOBUS_FALSE;
    monitor->res = GLOBUS_SUCCESS;
}

void
gftpd_monitor_destroy(
    gftpd_monitor_t *                            monitor)
{
    globus_mutex_destroy(&monitor->mutex);
    globus_cond_destroy(&monitor->cond);
}
