#include <gftpd.h>

/*
 *  internal data defintions and types
 */
static globus_result_t
gftpd_control_send_response(
    globus_ftp_control_handle_t *            handle,
    char *                                   msg);

void
gftpd_control_read_commands_callback(
    void *                                   callback_arg,
    struct globus_ftp_control_handle_s *     handle,
    globus_object_t *                        error,
    union globus_ftp_control_command_u *     command);

globus_result_t
gftpd_control_forward_command(
    gftpd_control_handle_t *                 gftpd_control_handle,
    char *                                   cmd);

void
gftpd_control_authenicate_callback(
    void *                                   callback_arg,
    struct globus_ftp_control_handle_s *     handle,
    globus_object_t *                        error,
    globus_ftp_control_auth_info_t *         auth_result);

gftpd_control_handle_t *
gftpd_control_create_handle();

globus_result_t
gftpd_control_destroy_handle(
    gftpd_control_handle_t *                  gftpd_control_handle);

globus_result_t
gftpd_control_connect_to_backend(
    gftpd_control_handle_t *                  gftpd_control_handle);
/*********************************************************************
 *                       functions
 ********************************************************************/

void
gftpd_control_main(
    int                                      fd)
{
    gftpd_control_handle_t *                 gftpd_control_handle;
    globus_result_t                          res;
    globus_io_attr_t                         attr;
    globus_io_handle_t                       io_handle;
    globus_size_t                            nbytes;
    char                                     msg_buf[SFTPD_STRING_SIZE];
    time_t                                   time_now;
    char *                                   time_ptr;
    char *                                   tmp_ptr;
    globus_ftp_control_auth_requirements_t   auth_requirements;

    globus_module_activate(GLOBUS_IO_MODULE);
    globus_module_activate(GLOBUS_FTP_CONTROL_MODULE);

    gftpd_control_handle = gftpd_control_create_handle();

    /*
     *  convert the fd into a globus_ftp_control_handle.
     */
    res = globus_ftp_control_handle_init(&gftpd_control_handle->client_handle);
    assert(res == GLOBUS_SUCCESS);
    gftpd_control_handle->client_handle.cc_handle.cc_state =
                                        GLOBUS_FTP_CONTROL_CONNECTED;
    res = globus_io_tcp_posix_convert(
              fd,
              GLOBUS_NULL,
              &gftpd_control_handle->client_handle.cc_handle.io_handle);
    assert(res == GLOBUS_SUCCESS);
    gftpd_control_handle->client_handle_connected = GLOBUS_TRUE;

    /*
     *  build and send 220 connection message
     */
    time_now = time(NULL);
    time_ptr = ctime(&time_now);
    tmp_ptr = strstr(time_ptr, "\n");
    if(tmp_ptr != NULL)
    { 
        *tmp_ptr = '\0';
    }

    sprintf(msg_buf, "220 %s Globus FTP server. (%s %s) ready.\r\n",
        g_hostname,
        g_version,
        time_ptr);
    res = gftpd_control_send_response(
              &gftpd_control_handle->client_handle,
              msg_buf);
    if(res != GLOBUS_SUCCESS)
    {
        goto exit;
    }

    /*
     *  authenticate the user
     */
    auth_requirements = GLOBUS_FTP_CONTROL_AUTH_REQ_GSSAPI |
                        GLOBUS_FTP_CONTROL_AUTH_REQ_USER |
                        GLOBUS_FTP_CONTROL_AUTH_REQ_PASS;
    res = globus_ftp_control_server_authenticate(
              &gftpd_control_handle->client_handle,
              auth_requirements,
              gftpd_control_authenicate_callback,
              (void *)gftpd_control_handle);
    if(res != GLOBUS_SUCCESS) 
    {
        goto exit;
    }

    /* 
     *  wait until we are finished with the connection
     */
    globus_mutex_lock(&gftpd_control_handle->mutex);
    {
        while(!gftpd_control_handle->done)
        {
            globus_cond_wait(&gftpd_control_handle->cond, 
                &gftpd_control_handle->mutex);
        }
    }
    globus_mutex_unlock(&gftpd_control_handle->mutex);
   
  exit:
    gftpd_control_destroy_handle(gftpd_control_handle);
 
    globus_module_deactivate(GLOBUS_FTP_CONTROL_MODULE);
}

gftpd_control_handle_t *
gftpd_control_create_handle()
{
    gftpd_control_handle_t *                  gftpd_control_handle;

    gftpd_control_handle = (gftpd_control_handle_t *)
               globus_malloc(sizeof(gftpd_control_handle_t));
    globus_mutex_init(&gftpd_control_handle->mutex, GLOBUS_NULL);
    globus_cond_init(&gftpd_control_handle->cond, GLOBUS_NULL);
    gftpd_control_handle->done = GLOBUS_FALSE;
    gftpd_control_handle->client_handle_connected = GLOBUS_FALSE;
    gftpd_control_handle->state = SFTPD_CONTROL_STATE_NONE;

    return gftpd_control_handle;
}

globus_result_t
gftpd_control_destroy_handle(
    gftpd_control_handle_t *                  gftpd_control_handle)
{

    globus_mutex_destroy(&gftpd_control_handle->mutex);
    globus_cond_destroy(&gftpd_control_handle->cond);
    globus_free(gftpd_control_handle);


    return GLOBUS_SUCCESS;
}

void
gftpd_control_handle_close_callback(
    void *                                      callback_arg,
    struct globus_ftp_control_handle_s *        handle,
    globus_object_t *                           error,
    globus_ftp_control_response_t *             ftp_response)
{
    globus_result_t                             res;
    gftpd_control_handle_t *                    gftpd_control_handle;

    gftpd_control_handle = (gftpd_control_handle_t *)callback_arg;

    globus_mutex_lock(&gftpd_control_handle->mutex);
    {
        gftpd_control_handle->done = GLOBUS_TRUE;
        globus_cond_signal(&gftpd_control_handle->cond);
    }
    globus_mutex_unlock(&gftpd_control_handle->mutex);
}

/*
 *  this function should be called locked 
 */
globus_result_t
gftpd_control_handle_close(
    gftpd_control_handle_t *                  gftpd_control_handle)
{
    globus_result_t                           res;

    if(gftpd_control_handle->state == SFTPD_CONTROL_STATE_CLOSING)
    {
        return globus_error_put(GLOBUS_ERROR_NO_INFO);
    }

    gftpd_control_handle->state = SFTPD_CONTROL_STATE_CLOSING;

    if(gftpd_control_handle->client_handle_connected)
    {
        gftpd_control_handle->client_handle_connected = GLOBUS_FALSE;
        res = globus_ftp_control_force_close(
                  &gftpd_control_handle->client_handle,
                  gftpd_control_handle_close_callback,
                  (void*)gftpd_control_handle);
        if(res == GLOBUS_SUCCESS)
        {
            return GLOBUS_SUCCESS;
        }
    }

    globus_mutex_lock(&gftpd_control_handle->mutex);
    {
        gftpd_control_handle->done = GLOBUS_TRUE;
        globus_cond_signal(&gftpd_control_handle->cond);
    }
    globus_mutex_unlock(&gftpd_control_handle->mutex);


    return GLOBUS_SUCCESS;
}


/*
 *  decide how to authenticate the user
 */
void
gftpd_control_authenicate_callback(
    void *                                   callback_arg,
    struct globus_ftp_control_handle_s *     handle,
    globus_object_t *                        error,
    globus_ftp_control_auth_info_t *         auth_result)
{
    globus_bool_t                            accepted = GLOBUS_FALSE;
    gftpd_control_handle_t *                 gftpd_control_handle;
    globus_result_t                          res;
    char *                                   username;
    struct passwd *                          pw_ent;
    char                                     msg[SFTPD_STRING_SIZE];

    gftpd_control_handle = (gftpd_control_handle_t *)callback_arg;

    /*
     *  if an error coccured the proces cannot continue
     */
    if(error != GLOBUS_NULL)
    {
        goto err;
    }

    /*
     *  decide if the user is acceptable, and create error message
     */
    sprintf(msg, "530 You are not authorized to use this server.\r\n");
    if(strcmp(auth_result->user, ":globus-mapping:") == 0 &&
       globus_gss_assist_gridmap(
           auth_result->auth_gssapi_subject, &username) == 0)
    {
        sprintf(msg, "530 Could not find password entry for %s.\r\n",
                username);
        pw_ent = getpwnam(username);
        if(pw_ent != GLOBUS_NULL)
        {
            sprintf(msg, "530 Could set user id for %s.\r\n",
                username);
            if(setuid(pw_ent->pw_uid) == 0)
            {
                chdir(pw_ent->pw_dir);
                sprintf(msg, "530 Could set home directory for %s.\r\n",
                   username);
                if(globus_libc_setenv(
                       "HOME", pw_ent->pw_dir, GLOBUS_TRUE) == 0) 
                {
                    sprintf(msg, 
                        "530 Could not connect to backend server: %s.\r\n",
                        globus_object_printable_to_string(
                                          globus_error_get(res)));
                    accepted = GLOBUS_TRUE;
                }
            }
        }
    }

    if(accepted)
    {
        sprintf(msg, "230 User %s loggen on.\r\n", username);
        res = gftpd_control_send_response(
                  handle,
                  msg);
        if(res != GLOBUS_SUCCESS)
        {
            goto err;
        }

        res = globus_ftp_control_read_commands(
                  handle, 
                  gftpd_command_read_commands_callback,
                  callback_arg);
        if(res != GLOBUS_SUCCESS)
        {
            goto err;
        }
    }
    else
    {
        res = gftpd_control_send_response(
                  handle,
                  msg);

        gftpd_control_handle_close(gftpd_control_handle);
    }

    return;

  err:
    globus_mutex_lock(&gftpd_control_handle->mutex);
    {
        gftpd_control_handle->done = GLOBUS_TRUE;
        globus_cond_signal(&gftpd_control_handle->cond);
    }
    globus_mutex_unlock(&gftpd_control_handle->mutex);
}

/*
 *  callback for send response
 */
static void
gftpd_control_send_response_callback(
    void *                                      callback_arg,
    struct globus_ftp_control_handle_s *        handle,
    globus_object_t *                           error)
{
    gftpd_monitor_t *                           monitor;

    monitor = (gftpd_monitor_t *)callback_arg;

    globus_mutex_lock(&monitor->mutex);
    {
        monitor->done = GLOBUS_TRUE;
        if(error != GLOBUS_NULL)
        {
            monitor->res = globus_error_put(globus_object_copy(error));
        }
        globus_cond_signal(&monitor->cond);
    }
    globus_mutex_unlock(&monitor->mutex);
}

/*
 *  send a response back to the client
 *
 *  this function should be called locked
 */
static globus_result_t
gftpd_control_send_response(
    globus_ftp_control_handle_t *               handle,
    char *                                      msg)
{
    globus_result_t                             res;
    gftpd_monitor_t                             monitor;

    gftpd_monitor_init(&monitor);

    res = globus_ftp_control_send_response(
              handle,
              msg,
              gftpd_control_send_response_callback,
              (void *)&monitor);
    if(res != GLOBUS_SUCCESS)
    {
        return res;
    }

    globus_mutex_lock(&monitor.mutex);
    {
        while(!monitor.done)
        {
            globus_cond_wait(&monitor.cond, &monitor.mutex);
        }
        res = monitor.res;
    }
    globus_mutex_unlock(&monitor.mutex);

    gftpd_monitor_destroy(&monitor);

    return res;
}
