#! /usr/bin/perl 
#
# Test to exercise the "put" functionality of the Globus FTP client library
# in extended block mode

use strict;
use POSIX;
use Test;

my $test_exec = '../extended_put';
my $test_file = '/bin/sh';

my @tests;
my @todo;

# Test #1-11. Basic functionality: Do a put of $test_file to
# a new unique file name on localhost, varying parallelism level.
# Compare the resulting file with the real file
# Success if program returns 0, files compare,
# and no core file is generated.
sub basic_func
{
    my ($parallelism) = (shift);
    my $tmpname = POSIX::tmpnam();
    my ($errors,$rc) = ("",0);

    unlink('core', $tmpname);

    $rc = system("$test_exec -d 'gsiftp://localhost/$tmpname' -P $parallelism < $test_file 2>/dev/null") / 256;
    if($rc != 0)
    {
        $errors .= "\n#Test exited with $rc. ";
    }
    if(-r 'core')
    {
        $errors .= "\n# Core file generated.";
    }

    my $diffs = `diff $test_file $tmpname 2>&1 | sed -e 's/^/# /'`;
	
    if($diffs ne "")
    {
	$errors .= "\n# Differences between $test_file and output.";
	$errors .= "$diffs";
    }

    if($errors eq "")
    {
        ok('success', 'success');
    }
    else
    {
        ok("\n#$test_exec -d 'gsiftp://localhost$tmpname' -P $parallelism < $test_file\n#$errors", 'success');
    }
    unlink($tmpname);
}
for(my $par = 0; $par <= 10; $par++)
{
    push(@tests, "basic_func($par);");
}

# Test #12: Bad URL: Do a simple put to a bad location on the ftp server.
# Success if program returns 1 and no core file is generated.
sub bad_url
{
    my $tmpname = POSIX::tmpnam();
    my ($errors,$rc) = ("",0);

    unlink('core', $tmpname);

    $rc = system("$test_exec -d 'gsiftp://localhost/no/such/file/here' < $test_file >/dev/null 2>/dev/null") / 256;
    if($rc != 1)
    {
        $errors .= "\n# Test exited with $rc.";
    }
    if(-r 'core')
    {
        $errors .= "\n# Core file generated.";
    }

    if($errors eq "")
    {
        ok('success', 'success');
    }
    else
    {
        ok($errors, 'success');
    }
    unlink($tmpname);
}
push(@tests, "bad_url");

# Test #13-53: Do a simple put of $test_file to localhost, aborting
# at each possible position. Note that not all aborts may be reached.
# Success if no core file is generated for all abort points. (we could use
# a stronger measure of success here)
sub abort_test
{
    my $tmpname = POSIX::tmpnam();
    my ($errors,$rc) = ("", 0);
    my ($abort_point) = shift;

    unlink('core', $tmpname);

    $rc = system("$test_exec -a $abort_point -d 'gsiftp://localhost/$tmpname' <$test_file >/dev/null 2>/dev/null") / 256;
    if(-r 'core')
    {
        $errors .= "\n# Core file generated.";
    }

    if($errors eq "")
    {
        ok('success', 'success');
    }
    else
    {
        ok($errors, 'success');
    }
    unlink($tmpname);
}
for(my $i = 1; $i <= 41; $i++)
{
    push(@tests, "abort_test($i);");
}

# Test #54-94. Restart functionality: Do a simple put of $test_file to
# localhost, restarting at each plugin-possible point.
# Compare the resulting file with the real file
# Success if program returns 0, files compare,
# and no core file is generated.
sub restart_test
{
    my $tmpname = POSIX::tmpnam();
    my ($errors,$rc) = ("",0);
    my ($restart_point) = shift;

    unlink('core', $tmpname);

    $rc = system("$test_exec -r $restart_point -d 'gsiftp://localhost/$tmpname' < $test_file >/dev/null 2>&1") / 256;
    if($rc != 0)
    {
        $errors .= "Test exited with $rc. ";
    }
    if(-r 'core')
    {
        $errors .= "\n# Core file generated.";
    }
    my $diffs = `sed -e 's/\\[restart plugin\\].*\$//' $tmpname | diff $test_file - 2>&1 | sed -e 's/^/#/'`;
    if($diffs ne "")
    {
        $errors .= "\n# Differences between $test_file and output.";
	$errors .= "$diffs"
    }

    if($errors eq "")
    {
        ok('success', 'success');
    }
    else
    {
        ok("\n# $test_exec -r $restart_point\n#$errors", 'success');
    }
    unlink($tmpname);
}
for(my $i = 1; $i <= 41; $i++)
{
    push(@tests, "restart_test($i);");
}
push(@todo, 79);

# Now that the tests are defined, set up the Test to deal with them.
plan tests => scalar(@tests), todo => \@todo;

# And run them all.
foreach (@tests)
{
    eval "&$_";
}
