#ifndef GLOBUS_DONT_DOCUMENT_INTERNAL
/**
 * @file globus_ftp_client_plugin.c Plugin Implementation
 *
 * $RCSfile: globus_ftp_client_plugin.c,v $
 * $Revision: 1.21 $
 * $Date: 2001/06/04 17:07:20 $
 */
#endif

#include "globus_i_ftp_client.h"

#include <string.h>

#ifndef GLOBUS_DONT_DOCUMENT_INTERNAL
#define PLUGIN_SUPPORTS_OP(op,plugin) \
	   (((op) == GLOBUS_FTP_CLIENT_GET && (plugin)->get_func) \
	    || ((op) == GLOBUS_FTP_CLIENT_PUT && (plugin)->put_func) \
	    || ((op) == GLOBUS_FTP_CLIENT_TRANSFER && \
                (plugin)->transfer_func) \
            || ((op) == GLOBUS_FTP_CLIENT_LIST && (plugin)->list_func) \
            || ((op) == GLOBUS_FTP_CLIENT_NLST && (plugin)->vlist_func) \
            || ((op) == GLOBUS_FTP_CLIENT_DELETE && (plugin)->delete_func) \
            || ((op) == GLOBUS_FTP_CLIENT_MKDIR && (plugin)->mkdir_func) \
            || ((op) == GLOBUS_FTP_CLIENT_RMDIR && (plugin)->rmdir_func) \
            || ((op) == GLOBUS_FTP_CLIENT_MOVE && (plugin)->move_func))
#endif


static
globus_result_t
globus_l_ftp_client_plugin_restart_operation(
    globus_i_ftp_client_handle_t *		handle,
    const char *				source_url,
    globus_ftp_client_operationattr_t *		source_attr,
    const char *				dest_url,
    globus_ftp_client_operationattr_t *		dest_attr,
    globus_ftp_client_restart_marker_t *	restart,
    globus_abstime_t *				when);



/**
 * Restart an existing list.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing transfer operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially a new URL and 
 * attributes. Any data buffers which are
 * currently queued will be cleared and reused once the connection is
 * re-established.
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in list events will
 * receive a list callback with the restart boolean set to GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the list.
 * @param source_url
 *        The destination URL of the transfer. This may be different than
 *        the original list's URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL.
 * @param source_attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original list's attribute set.
 * @param when
 *        Absolute time for when to restart the list. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted.
 */
globus_result_t
globus_ftp_client_plugin_restart_list(
    globus_ftp_client_handle_t *		handle,
    const char *				url,
    globus_ftp_client_operationattr_t *		attr,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char * myname = "globus_ftp_client_plugin_restart_list";

    if(url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }

    i_handle = *handle;

    return globus_l_ftp_client_plugin_restart_operation(i_handle,
							url,
							attr,
							GLOBUS_NULL,
							GLOBUS_NULL,
							GLOBUS_NULL,
							when);
}
/* globus_ftp_client_plugin_restart_list() */

/**
 * Restart an existing verbose list.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing transfer operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially a new URL and 
 * attributes. Any data buffers which are
 * currently queued will be cleared and reused once the connection is
 * re-established.
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in list events will
 * receive a list callback with the restart boolean set to GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the list.
 * @param source_url
 *        The destination URL of the transfer. This may be different than
 *        the original list's URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL.
 * @param source_attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original list's attribute set.
 * @param when
 *        Absolute time for when to restart the list. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted.
 */
globus_result_t
globus_ftp_client_plugin_restart_verbose_list(
    globus_ftp_client_handle_t *		handle,
    const char *				url,
    globus_ftp_client_operationattr_t *		attr,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char * myname = "globus_ftp_client_plugin_restart_list";

    if(url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }

    i_handle = *handle;

    return globus_l_ftp_client_plugin_restart_operation(i_handle,
							url,
							attr,
							GLOBUS_NULL,
							GLOBUS_NULL,
							GLOBUS_NULL,
							when);
}
/* globus_ftp_client_plugin_restart_verbose_list() */

/**
 * Restart an existing delete.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing delete operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially a new URL and 
 * attributes. Any data buffers which are
 * currently queued will be cleared and reused once the connection is
 * re-established.
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in delete events will
 * receive a delete callback with the restart boolean set to GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the delete.
 * @param source_url
 *        The destination URL of the transfer. This may be different than
 *        the original delete's URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL.
 * @param source_attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original delete's attribute set.
 * @param when
 *        Absolute time for when to restart the delete. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted.
 */
globus_result_t
globus_ftp_client_plugin_restart_delete(
    globus_ftp_client_handle_t *		handle,
    const char *				url,
    globus_ftp_client_operationattr_t *		attr,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char * myname = "globus_ftp_client_plugin_restart_delete";

    if(url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }
    
    i_handle = *handle;

    return globus_l_ftp_client_plugin_restart_operation(i_handle,
							url,
							attr,
							GLOBUS_NULL,
							GLOBUS_NULL,
							GLOBUS_NULL,
							when);
}
/* globus_ftp_client_plugin_restart_delete() */

/**
 * Restart an existing mkdir.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially a new URL and 
 * attributes. Any data buffers which are
 * currently queued will be cleared and reused once the connection is
 * re-established.
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in mkdir events will
 * receive a mkdir callback with the restart boolean set to GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the mkdir.
 * @param source_url
 *        The destination URL of the transfer. This may be different than
 *        the original mkdir's URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL.
 * @param source_attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original mkdir's attribute set.
 * @param when
 *        Absolute time for when to restart the mkdir. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted.
 */
globus_result_t
globus_ftp_client_plugin_restart_mkdir(
    globus_ftp_client_handle_t *		handle,
    const char *				url,
    globus_ftp_client_operationattr_t *		attr,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char * myname = "globus_ftp_client_plugin_restart_mkdir";

    if(url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }

    i_handle = *handle;

    return globus_l_ftp_client_plugin_restart_operation(i_handle,
							url,
							attr,
							GLOBUS_NULL,
							GLOBUS_NULL,
							GLOBUS_NULL,
							when);
}
/* globus_ftp_client_plugin_restart_mkdir() */

/**
 * Restart an existing rmdir.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially a new URL and 
 * attributes. Any data buffers which are
 * currently queued will be cleared and reused once the connection is
 * re-established.
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in rmdir events will
 * receive a rmdir callback with the restart boolean set to GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the rmdir.
 * @param source_url
 *        The destination URL of the transfer. This may be different than
 *        the original rmdir's URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL.
 * @param source_attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original rmdir's attribute set.
 * @param when
 *        Absolute time for when to restart the rmdir. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted.
 */
globus_result_t
globus_ftp_client_plugin_restart_rmdir(
    globus_ftp_client_handle_t *		handle,
    const char *				url,
    globus_ftp_client_operationattr_t *		attr,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char * myname = "globus_ftp_client_plugin_restart_rmdir";

    if(url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }
    i_handle = *handle;

    return globus_l_ftp_client_plugin_restart_operation(i_handle,
							url,
							attr,
							GLOBUS_NULL,
							GLOBUS_NULL,
							GLOBUS_NULL,
							when);
}
/* globus_ftp_client_plugin_restart_rmdir() */



/**
 * Restart an existing move.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing move operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially new URLs and 
 * attributes. 
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in get events will
 * receive a move callback with the restart boolean set to GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the move.
 * @param source_url
 *        The source URL of the move. This may be different than
 *        the original get's URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL.
 * @param dest_url
 *        The destination URL of the move. This may be different than
 *        the original get's URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL. Note that only the path
 *        component of this URL is used.
 * @param attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original move's attribute set. This
 *        may be useful when the plugin wishes to send restart markers 
 *        to the FTP server to prevent re-sending the data which has
 *        already been sent.
 * @param when
 *        Absolute time for when to restart the move. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted.
 */
globus_result_t
globus_ftp_client_plugin_restart_move(
    globus_ftp_client_handle_t *		handle,
    const char *				source_url,
    const char *				dest_url,
    globus_ftp_client_operationattr_t *		attr,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char * myname = "globus_ftp_client_plugin_restart_move";

    if(source_url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL source url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }

    if(dest_url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL destination url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }

    i_handle = *handle;

    return globus_l_ftp_client_plugin_restart_operation(i_handle,
							source_url,
							attr,
							dest_url,
							GLOBUS_NULL,
							GLOBUS_NULL,
							when);
}
/* globus_ftp_client_plugin_restart_move() */

/**
 * Restart an existing get.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing transfer operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially a new URL and 
 * attributes. Any data buffers which are
 * currently queued will be cleared and reused once the connection is
 * re-established.
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in get events will
 * receive a get callback with the restart boolean set to GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the get.
 * @param url
 *        The source URL of the transfer. This may be different than
 *        the original get's URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL.
 * @param attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original get's attribute set. This
 *        may be useful when the plugin wishes to send restart markers 
 *        to the FTP server to prevent re-sending the data which has
 *        already been sent.
 * @param when
 *        Absolute time for when to restart the get. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted.
 */
globus_result_t
globus_ftp_client_plugin_restart_get(
    globus_ftp_client_handle_t *		handle,
    const char *				url,
    globus_ftp_client_operationattr_t *		attr,
    globus_ftp_client_restart_marker_t *	restart_marker,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char * myname = "globus_ftp_client_plugin_restart_get";

    if(url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }

    i_handle = *handle;

    return globus_l_ftp_client_plugin_restart_operation(i_handle,
							url,
							attr,
							GLOBUS_NULL,
							GLOBUS_NULL,
							restart_marker,
							when);
}
/* globus_ftp_client_plugin_restart_get() */

/**
 * Restart an existing put.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing transfer operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially a new URL and 
 * attributes. Any data buffers which are
 * currently queued but not called back will be resent once the
 * connection is re-established.
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in get events will
 * receive a put callback with the restart boolean set to GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the put.
 * @param url
 *        The URL of the transfer. This may be different than
 *        the original put's URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL. If the put is restarted with a 
 *        different URL, the plugin must re-send any data which has
 *        already been acknowledged by it's callback.
 * @param attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original put's attribute set. This
 *        may be useful when the plugin wishes to send restart markers 
 *        to the FTP server to prevent re-sending the data which has
 *        already been sent.
 * @param when
 *        Absolute time for when to restart the put. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted. 
 */
globus_result_t
globus_ftp_client_plugin_restart_put(
    globus_ftp_client_handle_t *		handle,
    const char *				url,
    globus_ftp_client_operationattr_t *		attr,
    globus_ftp_client_restart_marker_t *	restart_marker,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char * myname = "globus_ftp_client_plugin_restart_put";

    if(url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }
    i_handle = *handle;

    return globus_l_ftp_client_plugin_restart_operation(i_handle,
							GLOBUS_NULL,
							GLOBUS_NULL,
							url,
							attr,
							restart_marker,
							when);
}
/* globus_ftp_client_plugin_restart_put() */

/**
 * Restart an existing third-party transfer.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing transfer operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially a new URLs and 
 * attributes.
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in third-party transfer
 * events will receive a transfer callback with the restart boolean set to
 * GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the transfer.
 * @param source_url
 *        The source URL of the transfer. This may be different than
 *        the original URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL
 * @param source_attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original transfer's attribute set. This
 *        may be useful when the plugin wishes to send restart markers 
 *        to the FTP server to prevent re-sending the data which has
 *        already been sent.
 * @param dest_url
 *        The destination URL of the transfer. This may be different than
 *        the original destination URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL.
 * @param dest_attr
 *        The attributes to use for the new transfer. This may be a
 *        modified version of the original transfer's attribute set. This
 *        may be useful when the plugin wishes to send restart markers 
 *        to the FTP server to prevent re-sending the data which has
 *        already been sent.
 * @param when
 *        Absolute time for when to restart the transfer. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted.
 */
globus_result_t
globus_ftp_client_plugin_restart_transfer(
    globus_ftp_client_handle_t *		handle,
    const char *				source_url,
    globus_ftp_client_operationattr_t *		source_attr,
    const char *				dest_url,
    globus_ftp_client_operationattr_t *		dest_attr,
    globus_ftp_client_restart_marker_t *	restart_marker,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char * myname = "globus_ftp_client_plugin_restart_transfer";
    
    if(source_url == GLOBUS_NULL ||
       dest_url == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] NULL url at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	return globus_error_put(err);
    }

    i_handle = *handle;

    return globus_l_ftp_client_plugin_restart_operation(i_handle,
							source_url,
							source_attr,
							dest_url,
							dest_attr,
							restart_marker,
							when);
}
/* globus_ftp_client_plugin_restart_transfer() */


/**
 * Restart an existing operation.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing operation 
 * to be restarted. When a restart happens, the operation will be
 * silently aborted, and then restarted with potentially a new URLs and 
 * attributes.
 *
 * The user will not receive any notification that a restart has
 * happened. Each plugin which is interested in events caused by the
 * operation will receive a callback with the restart boolean set to
 * GLOBUS_TRUE.
 *
 * @param handle
 *        The handle which is associated with the operation.
 * @param source_url
 *        The source URL of the operation. This may be different than
 *        the original URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL
 * @param source_attr
 *        The attributes to use for the new operation. This may be a
 *        modified version of the original operations's attribute set. This
 *        may be useful when the plugin wishes to send restart markers 
 *        to the FTP server to prevent re-sending the data which has
 *        already been sent.
 * @param dest_url
 *        The destination URL of the operation. This may be different than
 *        the original destination URL, if the plugin decides to redirect to 
 *        another FTP server due to performance or reliability
 *        problems with the original URL.
 * @param dest_attr
 *        The attributes to use for the new operation. This may be a
 *        modified version of the original operations's attribute set. This
 *        may be useful when the plugin wishes to send restart markers 
 *        to the FTP server to prevent re-sending the data which has
 *        already been sent.
 * @param when
 *        Absolute time for when to restart the operation. The current
 *        control and data connections will be stopped
 *        immediately. If this completes before <b>when</b>, then the 
 *	  restart will be delayed until that time. Otherwise, it will
 *        be immediately restarted.
 */
static
globus_result_t
globus_l_ftp_client_plugin_restart_operation(
    globus_i_ftp_client_handle_t *		handle,
    const char *				source_url,
    globus_ftp_client_operationattr_t *		source_attr,
    const char *				dest_url,
    globus_ftp_client_operationattr_t *		dest_attr,
    globus_ftp_client_restart_marker_t *	restart_marker,
    globus_abstime_t *				when)
{
    globus_object_t *				err;
    globus_result_t				result;
    globus_i_ftp_client_restart_t *		restart_info;
    static char * myname = "globus_i_ftp_client_plugin_restart_operation";

    if(handle == GLOBUS_NULL)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] Cannot restart a NULL handle at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	goto error_exit;
    }
    /* Check handle state */
    if(GLOBUS_I_FTP_CLIENT_BAD_MAGIC(handle->handle))
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] Cannot restart with an invalid handle at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	goto error_exit;
    }
    restart_info = (globus_i_ftp_client_restart_t *)
	globus_libc_malloc(sizeof(globus_i_ftp_client_restart_t));

    if(restart_info == GLOBUS_NULL)
    {
        err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] Could not allocate internal data structure at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	goto error_exit;
    }

    if(dest_url != GLOBUS_NULL)
    {
	restart_info->dest_url = globus_libc_strdup(dest_url);
	if(restart_info->dest_url == GLOBUS_NULL)
	{
	    err = globus_error_construct_string(
		GLOBUS_FTP_CLIENT_MODULE,
		GLOBUS_NULL,
		"[%s] Could not allocate internal data structure at %s\n",
		GLOBUS_FTP_CLIENT_MODULE->module_name,
		myname);
	    
	    goto free_restart_exit;
	}
	if(dest_attr)
	{
	    result = globus_ftp_client_operationattr_copy(
		&restart_info->dest_attr,
		dest_attr);
	    if(result)
	    {
		err = globus_error_get(result);
		
		goto free_dest_attr_exit;
	    }
	}
	else
	{
	    restart_info->dest_attr = GLOBUS_NULL;
	}
    }
    else
    {
	restart_info->dest_url = GLOBUS_NULL;
	restart_info->dest_attr = GLOBUS_NULL;
    }

    if(source_url != GLOBUS_NULL)
    {
	
	restart_info->source_url = globus_libc_strdup(source_url);
	if(restart_info->source_url == GLOBUS_NULL)
	{
	    err = globus_error_construct_string(
		GLOBUS_FTP_CLIENT_MODULE,
		GLOBUS_NULL,
		"[%s] Could not allocate internal data structure at %s\n",
		GLOBUS_FTP_CLIENT_MODULE->module_name,
		myname);
	    
	    goto destroy_dest_attr_exit;
	}
	
	if(source_attr)
	{
	    result = globus_ftp_client_operationattr_copy(
		&restart_info->source_attr,
		source_attr);
	    if(result)
	    {
		err = globus_error_get(result);
		
		goto free_source_attr_exit;
	    }
	}
	else
	{
	    restart_info->source_attr = GLOBUS_NULL;
	}
    }
    else
    {
	restart_info->source_url = GLOBUS_NULL;
	restart_info->source_attr = GLOBUS_NULL;
    }

    if(restart_marker)
    {
	globus_ftp_client_restart_marker_copy(&restart_info->marker,
					      restart_marker);
    }
    else
    {
	globus_ftp_client_restart_marker_copy(&restart_info->marker,
					      &handle->restart_marker);
    }

    if(when)
    {
	GlobusTimeAbstimeCopy(restart_info->when, *when);
    }
    else
    {
	GlobusTimeAbstimeSet(restart_info->when, 0L, 0L);
    }


    globus_i_ftp_client_handle_lock(handle);
    
    if(handle->op == GLOBUS_FTP_CLIENT_IDLE)
    {
        err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] Cannot restart transfer for a %s operation at %s\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    globus_i_ftp_op_to_string(handle->op),
	    myname);

	goto unlock_error;
    }

    err = globus_i_ftp_client_restart(handle,
				      restart_info);

    if(err != GLOBUS_SUCCESS)
    {
	goto unlock_error;
    }

    globus_i_ftp_client_handle_unlock(handle);
    return GLOBUS_SUCCESS;

unlock_error:
    globus_i_ftp_client_handle_unlock(handle);

    if(restart_info->source_attr)
    {
	globus_ftp_client_operationattr_destroy(&restart_info->source_attr);
    }
free_source_attr_exit:
    if(restart_info->source_attr)
    {
	globus_libc_free(restart_info->source_attr);
    }
free_source_url_exit:
    globus_libc_free(restart_info->source_url);
destroy_dest_attr_exit:
    if(restart_info->dest_attr)
    {
	globus_ftp_client_operationattr_destroy(&restart_info->dest_attr);
    }
free_dest_attr_exit:
    if(restart_info->dest_attr)
    {
	globus_libc_free(restart_info->dest_attr);
    }
free_dest_url_exit:
    globus_libc_free(restart_info->dest_url);
free_restart_exit:
    globus_libc_free(restart_info);
error_exit:
    return globus_error_put(err);
}
/* globus_ftp_client_plugin_restart_operation() */

/**
 * Abort a transfer operation.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing transfer operation 
 * to be aborted. When this happens, all plugins will be notified by
 * their abort callbacks. Once those are processed, the complete
 * callback will be called for all plugins, and then for the user's
 * callback.
 *
 * The complete callback will indicate that the transfer did not
 * complete successfully.
 *
 * @param handle
 *        The handle which is associated with the transfer.
 */
globus_result_t
globus_ftp_client_plugin_abort(
    globus_ftp_client_handle_t *		handle)
{
    return globus_ftp_client_abort(handle);
}
/* globus_ftp_client_plugin_abort() */

/**
 * Add data channels to an existing put transfer.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing transfer operation 
 * to have additional data channels acquired if the attribute set
 * allows it.
 *
 * @param handle
 *        The handle which is associated with the transfer.
 * @param num_channels
 *        The number of channels to add to the transfer.
 * @param stripe
 *        The stripe number to have the channels added to.
 *
 * @note Do the plugins need to be notified when this happens?
 */
globus_result_t
globus_ftp_client_plugin_add_data_channels(
    globus_ftp_client_handle_t *		handle,
    unsigned int				num_channels,
    unsigned int				stripe)
{
    globus_result_t				result;
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char *				myname = "globus_ftp_client_plugin_add_data_channels";

    i_handle = *handle;
    globus_i_ftp_client_handle_lock(i_handle);
    
    if(i_handle->state == GLOBUS_FTP_CLIENT_HANDLE_ABORT ||
       i_handle->state == GLOBUS_FTP_CLIENT_HANDLE_RESTART ||
       i_handle->state == GLOBUS_FTP_CLIENT_HANDLE_FAILURE)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] Handle not transferring data\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	goto err_exit;
    }

    if(i_handle->op != GLOBUS_FTP_CLIENT_PUT)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] Cannot add data channels for 3rd party transfer\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);
	goto err_exit;
    }
    else
    {
	result = globus_ftp_control_data_add_channels(
	    i_handle->dest->control_handle,
	    num_channels,
	    stripe);

	goto result_exit;
    }

 err_exit:
    result = globus_error_put(err);
    globus_i_ftp_client_handle_unlock(i_handle);

 result_exit:
    return result;
}
/* globus_ftp_client_plugin_add_data_channels() */

/**
 * Remove data channels from an existing put transfer.
 * @ingroup globus_ftp_client_plugins
 *
 * This function will cause the currently executing transfer operation 
 * to have data channels removed,  if the attribute set
 * allows it.
 *
 * @param handle
 *        The handle which is associated with the transfer.
 * @param num_channels
 *        The number of channels to remove from the transfer.
 * @param stripe
 *        The stripe number to have the channels removed from.
 *
 * @note Do the plugins need to be notified when this happens?
 */
globus_result_t
globus_ftp_client_plugin_remove_data_channels(
    globus_ftp_client_handle_t *		handle,
    unsigned int				num_channels,
    unsigned int				stripe)
{
    globus_result_t				result;
    globus_object_t *				err;
    globus_i_ftp_client_handle_t *		i_handle;
    static char *				myname = "globus_ftp_client_plugin_remove_data_channels";

    globus_i_ftp_client_handle_lock(i_handle);
    
    if(i_handle->state == GLOBUS_FTP_CLIENT_HANDLE_ABORT ||
       i_handle->state == GLOBUS_FTP_CLIENT_HANDLE_RESTART ||
       i_handle->state == GLOBUS_FTP_CLIENT_HANDLE_FAILURE)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] Handle not transferring data\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);

	goto err_exit;
    }

    if(i_handle->op != GLOBUS_FTP_CLIENT_PUT)
    {
	err = globus_error_construct_string(
	    GLOBUS_FTP_CLIENT_MODULE,
	    GLOBUS_NULL,
	    "[%s] Cannot remove data channels for 3rd party transfer\n",
	    GLOBUS_FTP_CLIENT_MODULE->module_name,
	    myname);
	goto err_exit;
    }
    else
    {
	result = globus_ftp_control_data_remove_channels(
	    i_handle->source->control_handle,
	    num_channels,
	    stripe);

	goto result_exit;
    }

 err_exit:
    result = globus_error_put(err);

result_exit:
    globus_i_ftp_client_handle_unlock(i_handle);

    return result;
}
/* globus_ftp_client_plugin_remove_data_channels() */

#ifndef GLOBUS_DONT_DOCUMENT_INTERNAL
/*--------------------------------------------------------------------------
 * Local/Internal Functions.
 *--------------------------------------------------------------------------
 */
/*@{*/
/**
 * Plugin notification functions
 * @ingroup globus_ftp_client_plugins
 *
 * These function notify all interested plugins that an event related
 * to the current transfer for the handle is happening. Event
 * notification is delivered to a plugin only if the command_mask
 * associated with the plugin indicates that the plugin is interested
 * in the event, and the plugin supports the operation.
 */
void
globus_i_ftp_client_plugin_notify_delete(
    globus_i_ftp_client_handle_t *		handle,
    const char *				url,
    globus_i_ftp_client_operationattr_t *	attr)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);
	
	if(plugin->delete_func)
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->delete_func)(plugin,
				  handle->handle,
				  url,
				  &attr,
				  GLOBUS_FALSE);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }
    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}

void
globus_i_ftp_client_plugin_notify_mkdir(
    globus_i_ftp_client_handle_t *		handle,
    const char *				url,
    globus_i_ftp_client_operationattr_t *	attr)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->mkdir_func)
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->mkdir_func)(plugin,
				 handle->handle,
				 url,
				 &attr,
				 GLOBUS_FALSE);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}

void
globus_i_ftp_client_plugin_notify_rmdir(
    globus_i_ftp_client_handle_t *		handle,
    const char *				url,
    globus_i_ftp_client_operationattr_t *	attr)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->rmdir_func)
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->rmdir_func)(plugin,
				 handle->handle,
				 url,
				 &attr,
				 GLOBUS_FALSE);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }
    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}

void
globus_i_ftp_client_plugin_notify_move(
    globus_i_ftp_client_handle_t *		handle,
    const char *				source_url,
    const char *				dest_url,
    globus_i_ftp_client_operationattr_t *	attr)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);
	if(plugin->move_func)
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->move_func)(plugin,
				handle->handle,
				source_url,
				dest_url,
				&attr,
				GLOBUS_FALSE);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}

void
globus_i_ftp_client_plugin_notify_verbose_list(
    globus_i_ftp_client_handle_t *		handle,
    const char *				url,
    globus_i_ftp_client_operationattr_t *	attr)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);
	if(plugin->vlist_func)
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->vlist_func)(plugin,
				 handle->handle,
				 url,
				 &attr,
				 GLOBUS_FALSE);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }
    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}

void
globus_i_ftp_client_plugin_notify_list(
    globus_i_ftp_client_handle_t *		handle,
    const char *				url,
    globus_i_ftp_client_operationattr_t *	attr)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->list_func)
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->list_func)(plugin,
				handle->handle,
				url,
				&attr,
				GLOBUS_FALSE);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}


void
globus_i_ftp_client_plugin_notify_get(
    globus_i_ftp_client_handle_t *		handle,
    const char *				url,
    globus_i_ftp_client_operationattr_t *	attr,
    const globus_ftp_client_restart_marker_t *	restart)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->get_func)
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->get_func)(plugin,
			       handle->handle,
			       url,
			       &attr,
			       GLOBUS_FALSE);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}


void
globus_i_ftp_client_plugin_notify_put(
    globus_i_ftp_client_handle_t *		handle,
    const char *				url,
    globus_i_ftp_client_operationattr_t *	attr,
    const globus_ftp_client_restart_marker_t *	restart)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;


    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->put_func)
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->put_func)(plugin,
			       handle->handle,
			       url,
			       &attr,
			       GLOBUS_FALSE);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}

void
globus_i_ftp_client_plugin_notify_transfer(
    globus_i_ftp_client_handle_t *		handle,
    const char *				source_url,
    globus_i_ftp_client_operationattr_t *	source_attr,
    const char *				dest_url,
    globus_i_ftp_client_operationattr_t *	dest_attr,
    const globus_ftp_client_restart_marker_t *	restart)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;


    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->transfer_func)
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->transfer_func)(plugin,
				    handle->handle,
				    source_url,
				    &source_attr,
				    dest_url,
				    &dest_attr,
				    GLOBUS_FALSE);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}

void
globus_i_ftp_client_plugin_notify_connect(
    globus_i_ftp_client_handle_t *		handle,
    const globus_url_t *			url)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);
	
	if(plugin->move_func)
	{
	    if(plugin->connect_func &&
	       PLUGIN_SUPPORTS_OP(handle->op, plugin) &&
	       (plugin->command_mask &
		GLOBUS_FTP_CLIENT_CMD_MASK_CONTROL_ESTABLISHMENT)) 
	    {
		if(!unlocked)
		{
		    globus_i_ftp_client_handle_unlock(handle);
		    unlocked = GLOBUS_TRUE;
		}
		(plugin->connect_func)(plugin,
				       handle->handle,
				       url);
	    }
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_connect() */

void
globus_i_ftp_client_plugin_notify_authenticate(
    globus_i_ftp_client_handle_t *		handle,
    const globus_url_t *			url,
    const globus_ftp_control_auth_info_t *	auth_info)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;
    
    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->auth_func &&
	   PLUGIN_SUPPORTS_OP(handle->op, plugin) &&
	   (plugin->command_mask &
	    GLOBUS_FTP_CLIENT_CMD_MASK_CONTROL_ESTABLISHMENT))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->auth_func)(plugin,
				handle->handle,
				url,
				auth_info);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }
    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_authenticate() */

void
globus_i_ftp_client_plugin_notify_read(
    globus_i_ftp_client_handle_t *		handle,
    const globus_byte_t *			buffer,
    globus_size_t				buffer_length)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;
    
    handle->notify_in_progress++;


    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->read_func &&
	   PLUGIN_SUPPORTS_OP(handle->op, plugin))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->read_func)(plugin,
				handle->handle,
				buffer,
				buffer_length);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_read() */

void
globus_i_ftp_client_plugin_notify_write(
    globus_i_ftp_client_handle_t *		handle,
    const globus_byte_t *			buffer,
    globus_size_t				buffer_length,
    globus_off_t				offset,
    globus_bool_t				eof)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;
    
    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->write_func &&
	   PLUGIN_SUPPORTS_OP(handle->op, plugin))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->write_func)(plugin,
				 handle->handle,
				 buffer,
				 buffer_length,
				 offset,
				 eof);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_write() */

void
globus_i_ftp_client_plugin_notify_command(
    globus_i_ftp_client_handle_t *		handle,
    const globus_url_t *			url,
    globus_ftp_client_plugin_command_mask_t	command_mask,
    const char *				command_name)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->command_func &&
	   PLUGIN_SUPPORTS_OP(handle->op, plugin) &&
	   (plugin->command_mask & command_mask))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->command_func)(plugin,
				   handle->handle,
				   url,
				   command_name);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_command() */

void
globus_i_ftp_client_plugin_notify_response(
    globus_i_ftp_client_handle_t *		handle,
    const globus_url_t *			url,
    globus_ftp_client_plugin_command_mask_t	command_mask,
    globus_object_t *				error,
    const globus_ftp_control_response_t *	ftp_response)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;
    
    handle->notify_in_progress++;


    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->response_func &&
	   PLUGIN_SUPPORTS_OP(handle->op, plugin) &&
	   (plugin->command_mask & command_mask))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->response_func)(plugin,
				    handle->handle,
				    url,
				    error,
				    ftp_response);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_response() */

void
globus_i_ftp_client_plugin_notify_fault(
    globus_i_ftp_client_handle_t *		handle,
    const globus_url_t *			url,
    globus_object_t *				error)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;
    
    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->fault_func &&
	   PLUGIN_SUPPORTS_OP(handle->op, plugin))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->fault_func)(plugin,
				 handle->handle,
				 url,
				 error);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_fault() */

void
globus_i_ftp_client_plugin_notify_complete(
    globus_i_ftp_client_handle_t *		handle)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;
    
    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->complete_func &&
	   PLUGIN_SUPPORTS_OP(handle->op, plugin))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->complete_func)(plugin,
				    handle->handle);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_complete() */

void
globus_i_ftp_client_plugin_notify_data(
    globus_i_ftp_client_handle_t *		handle,
    globus_object_t *				error,
    const globus_byte_t *			buffer,
    globus_size_t				buffer_length,
    globus_off_t				offset,
    globus_bool_t				eof)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;
    
    handle->notify_in_progress++;

    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->data_func &&
	   PLUGIN_SUPPORTS_OP(handle->op, plugin))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    (plugin->data_func)(plugin,
				handle->handle,
				error,
				buffer,
				buffer_length,
				offset,
				eof);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_data() */

void
globus_i_ftp_client_plugin_notify_abort(
    globus_i_ftp_client_handle_t *		handle)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    if(handle->notify_in_progress)
    {
	handle->notify_abort = GLOBUS_TRUE;

	return;
    }
    handle->notify_in_progress++;


    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(plugin->abort_func &&
	   PLUGIN_SUPPORTS_OP(handle->op, plugin))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }

	    (plugin->abort_func)(plugin,
				 handle->handle);
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_restart)
    {
	handle->notify_restart = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_restart(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_abort() */

void
globus_i_ftp_client_plugin_notify_restart(
    globus_i_ftp_client_handle_t *		handle)
{
    globus_ftp_client_plugin_t *		plugin;
    globus_list_t *				tmp;
    globus_bool_t				unlocked = GLOBUS_FALSE;

    if(handle->notify_in_progress)
    {
	handle->notify_restart = GLOBUS_TRUE;
	return;
    }
    handle->notify_in_progress++;


    tmp = handle->attr.plugins;
    while(!globus_list_empty(tmp))
    {
	plugin = (globus_ftp_client_plugin_t *) globus_list_first(tmp);
	tmp = globus_list_rest(tmp);

	if(PLUGIN_SUPPORTS_OP(handle->op, plugin))
	{
	    if(!unlocked)
	    {
		globus_i_ftp_client_handle_unlock(handle);
		unlocked = GLOBUS_TRUE;
	    }
	    if(handle->op == GLOBUS_FTP_CLIENT_GET)
	    {
		(plugin->get_func)(plugin,
				   handle->handle,
				   handle->restart_info->source_url,
				   &handle->restart_info->source_attr,
				   GLOBUS_TRUE);
	    }
	    else if(handle->op == GLOBUS_FTP_CLIENT_LIST)
	    {
		(plugin->vlist_func)(plugin,
				     handle->handle,
				     handle->restart_info->source_url,
				     &handle->restart_info->source_attr,
				     GLOBUS_TRUE);
	    }
	    else if(handle->op == GLOBUS_FTP_CLIENT_NLST)
	    {
		(plugin->list_func)(plugin,
				    handle->handle,
				    handle->restart_info->source_url,
				    &handle->restart_info->source_attr,
				    GLOBUS_TRUE);
	    }
	    else if(handle->op == GLOBUS_FTP_CLIENT_DELETE)
	    {
		(plugin->delete_func)(plugin,
				      handle->handle,
				      handle->restart_info->source_url,
				      &handle->restart_info->source_attr,
				      GLOBUS_TRUE);
	    }
	    else if(handle->op == GLOBUS_FTP_CLIENT_MKDIR)
	    {
		(plugin->mkdir_func)(plugin,
				     handle->handle,
				     handle->restart_info->source_url,
				     &handle->restart_info->source_attr,
				     GLOBUS_TRUE);
	    }
	    else if(handle->op == GLOBUS_FTP_CLIENT_RMDIR)
	    {
		(plugin->rmdir_func)(plugin,
				     handle->handle,
				     handle->restart_info->source_url,
				     &handle->restart_info->source_attr,
				     GLOBUS_TRUE);
	    }
	    else if(handle->op == GLOBUS_FTP_CLIENT_MOVE)
	    {
		(plugin->move_func)(plugin,
				    handle->handle,
				    handle->restart_info->source_url,
				    handle->restart_info->dest_url,
				    &handle->restart_info->source_attr,
				    GLOBUS_TRUE);
	    }
	    else if (handle->op == GLOBUS_FTP_CLIENT_PUT)
	    {
		(plugin->put_func)(plugin,
				   handle->handle,
				   handle->restart_info->dest_url,
				   &handle->restart_info->dest_attr,
				   GLOBUS_TRUE);
	    }
	    else
	    {
		(plugin->transfer_func)(plugin,
					handle->handle,
					handle->restart_info->source_url,
					&handle->restart_info->source_attr,
					handle->restart_info->dest_url,
					&handle->restart_info->dest_attr,
					GLOBUS_TRUE);
	    }
	}
    }
    if(unlocked)
    {
	globus_i_ftp_client_handle_lock(handle);
    }

    handle->notify_in_progress--;
    if(handle->notify_abort)
    {
	handle->notify_abort = GLOBUS_FALSE;

	globus_i_ftp_client_plugin_notify_abort(handle);
    }
}
/* globus_i_ftp_client_plugin_notify_restart() */
/*@}*/
#endif
