/**************************************************************************
*
* Program:      Single Remote Service Request - Server
*               single_rsr_server.c
*
* Written at:   High-Performance Computing Laboratory
*               Department of Computer Science
*               Northern Illinois University
*               DeKalb, IL
*
* Original Author:          Nicholas T. Karonis
* Date Written:             2/97
* Updated to Client/Server: Nicholas T. Karonis
* Date Updated:             2/98
*
* Description:
*
*     This server program, along with its matching client program 
*     (single_rsr_client.c) contains the *minimum* amount of code 
*     required to start a server, then start a client, attach the 
*     client to the server, and finally send a single message from 
*     the client to the server.
* 
*     The server in this example is designed to service exactly one client.
*     Once that connection has been established, no other clients may
*     attach to this server.  Further, once the client/server connection
*     has been established and the session runs to completion, BOTH this
*     server and the client terminate.  
*     
*     This is unlike traditional client/server applications where a server
*     may service a number of clients simultaneously and where the server
*     does not terminate after servicing exactly one client.  But, this
*     is an example whose purpose is to demonstrate the Nexus attach and
*     detach functionality.  Extending this example to allow multiple 
*     clients and establish an ever-present server should be straightforward.
*
*     Attachment in Nexus is performed by one process "allowing attachments",
*     globus_nexus_allow_attach(). This returns a <hostname> and <port> from
*     which we can construct a URL, an attach ULR.  For example, the attach URL 
*     constructed by a hostname=olympus and port=44431 would look like this:
*
*                     x-nexus://olympus:44431/
*    
*     It is this attach URL that the client process must pass (as a character
*     string) to globus_nexus_attach() when making a request to attach to us.
*     When such a request is made, our attachment approval function,
*     attach_requested() is automatically (asynchronously) executed.  It
*     is that routine where we decide whether or not to allow attachment.
*     If permission is granted, we give the requesting process a startpoint
*     to ourself.  Note that we do not get a startpoint to the process
*     we grant attach permission to.  If they want to give us a startpoint
*     they can do so by using the startpoint we just gave them.
* 
*     We can also stop listening for attachments by calling 
*     glbous_nexus_disallow_attach(), although we do not do so in this program.
*
*     This program first calls globus_nexus_allow_attach() to register the
*     attachment approval function, attach_requested(), and to get the
*     necessary hostname+port number to construct the URL.  It then
*     constructs the URL, prints it to stdout, and waits for the 
*     client to first attach to us (thereby giving the client a Nexus 
*     startpoint that points back to us) and then using that startpoint,
*     the client sends us a single remote service request (rsr) passing us 
*     an integer.  
*
*     The waiting is implemented using a monitor and a condition variable.  
*     The monitor's condition variable is signalled by the handler routine
*     (rcv_handler) that gets asynchronously called when the client's
*     rsr arrives.  In that handler routine the server extracts the integer
*     value from the buffer and prints it to stdout.
*
* To run this program:
*
*     This server program must be started before starting the client program
*     single_rsr_client.  This program does not require any command line
*     arguments.  Start the program by simply typing its name:
*
*     olympus% single_rsr_server 
*
*     This should generate a line of output that specifies the URL that you
*     must specify as a command line argument to the client program 
*     single_rsr_client.  That line of output should look something like this:
*
*     olympus:c0:t0:p26566: attach url: x-nexus://olympus:48365/
*
*     You should then start the client program (single_rsr_client) passing
*     it the URL above and an integer on the command line.
*
*     olympus% single_rsr_client x-nexus://olympus:48365/ 123
*
*     The client will attach to us and send us a message with a single integer.
*     We will receive that message, extract the integer from the incoming
*     buffer, and print it to stdout.
*
* Output:
*
*     olympus:c0:t0:p26566: attach url: x-nexus://olympus:48365/
*     olympus:c0:t0:p26566: Single Rsr server received 123
*
**************************************************************************/

#include <stdio.h>
#include <globus_common.h>
#include <globus_nexus.h>

/***************************/
/*                         */
/* Nexus Handler Functions */
/*                         */
/***************************/

static void rcv_handler(globus_nexus_endpoint_t *endpoint, 
			globus_nexus_buffer_t   *buffer, 
			globus_bool_t           is_non_threaded_handler);

/*********************************/
/*                               */
/* Nexus Handler Functions Table */
/*                               */
/*********************************/

static globus_nexus_handler_t handlers[] = 
{ 
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED, rcv_handler}
};

/*******************/
/*                 */
/* Data Structures */
/*                 */
/*******************/

/* Monitor                                                   */
/*                                                           */
/* Nexus programs can implement a monitor with a data        */
/* structure as illustrated below.  All monitors implemented */
/* in this way must have a 'globus_mutex_t' field to provide */
/* mutually exclusive access to the monitor.                 */
/*                                                           */
/* In this application we also need a monitor condition      */
/* variable and two flags.                                   */
/*                                                           */
/* Explanation of why 'volatile' is required on some fields: */
/*                                                           */
/* Some compilers, under certain optimization switches,      */
/* attempt to cache the value in a register thus avoiding a  */
/* memory reference.  By declaring the variable 'volatile',  */
/* we force compiler to  NOT cache value in register but     */
/* instead reload the value from  memory each time is is     */
/* referenced.                                               */
/*                                                           */
/* So, if a variable is shared by multiple threads such that */
/* it is written to by one thread and read by another, then  */
/* that variable must be declared volatile.  However, you    */
/* notice that two of the variables                          */
/*                                                           */
/*    globus_mutex_t mutex;                                  */
/*    globus_cond_t  cond;                                   */
/*                                                           */
/* both meet the multiple thread read/write criteria and yet */
/* neither of them are declared volatile.  This is because   */
/* the dataypes globus_mutex_t and globus_cond_t are defined */
/* within the Nexus implemenation (opaque to the user) and   */
/* Nexus has assumed the responsibility of taking the steps  */
/* necessary to assure that any potential compiler-induced   */
/* caching problems will be avoided.  Nexus does not assume  */
/* that responsibility for all its datatypes, only these     */
/* two.  This is because, by their very nature, these two    */
/* datatypes are intended to be shared by multiple threads.  */
/* All other Nexus-supplied datatypes should be explicitly   */
/* declared volatile where appropriate, as we do in this     */
/* example with our two monitor flags (each a nexus_bool_t). */
/*                                                           */
/* The general rule of thumb is, any variable that is not a  */
/* globus_mutex_t or a globus_cond_t and is shared by        */
/* multiple threads such that it is written to by one thread */
/* and read by another must be declared volatile.            */

typedef struct
{
    /*****************************************************/
    /* a 'globus_mutex_t' is required of ALL monitors to */
    /* control mutually exclusive access to the monitor  */
    /*****************************************************/

    /* 'volatile' not required here, handled   */
    /* in the implementation of globus_mutex_t */

    globus_mutex_t mutex;

    /*********************************/
    /* monitor data for this program */
    /*********************************/

    /* 'volatile' not required here, handled  */
    /* in the implementation of globus_cond_t */

    globus_cond_t  cond;

    /* 'volatile' required for both of these fields because */
    /* each is written by one thread and read by another    */

    volatile globus_bool_t  connection_established;
    volatile globus_bool_t  done;

} monitor_t;

/********************/
/*                  */
/* Global Variables */
/*                  */
/********************/

static globus_nexus_endpointattr_t     GlobalEndpointAttr;
static globus_nexus_endpoint_t         GlobalEndpoint;
static monitor_t                       Monitor;

/*******************/
/*                 */
/* Local Functions */
/*                 */
/*******************/

static int attach_requested(void *arg, char *url, globus_nexus_startpoint_t *sp);

/********/
/*      */
/* MAIN */
/*      */
/********/

int main(int argc, char **argv)
{

    int rc;
    char *attach_host;
    unsigned short attach_port;
    char attach_url[1024];

    /**************************************************/
    /* Activation of the Nexus module is required at  */
    /* the beginning of all Nexus programs            */
    /**************************************************/

    if ( globus_module_activate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	fprintf(stderr,"Nexus Module activation failed \n");
	exit(1);
    }

    /******************/
    /* Initialization */
    /******************/

    globus_nexus_endpointattr_init(&GlobalEndpointAttr);
    globus_nexus_endpointattr_set_handler_table(
	&GlobalEndpointAttr,
	handlers,
	sizeof(handlers)/sizeof(globus_nexus_handler_t));
    
    globus_nexus_endpoint_init(&GlobalEndpoint, &GlobalEndpointAttr);
    globus_nexus_endpoint_set_user_pointer(&GlobalEndpoint,(void *)(&Monitor));

    /**************************/
    /* Initialize the monitor */
    /**************************/

    /* initializing the mutex that controls access to the monitor */
    globus_mutex_init(&(Monitor.mutex), (globus_mutexattr_t *) GLOBUS_NULL);
    /* initializing the condition variable */
    globus_cond_init(&(Monitor.cond), (globus_condattr_t *) GLOBUS_NULL);

    /* entering the monitor and clearing the flags */
    globus_mutex_lock(&(Monitor.mutex));
    Monitor.connection_established = GLOBUS_FALSE;
    Monitor.done = GLOBUS_FALSE;
    globus_mutex_unlock(&(Monitor.mutex));

    /***********************/
    /* allowing attachment */
    /***********************/

    /*
     *
     * By calling nexus_allow_attach() we allow other Nexus programs
     * to attach to us.  nexus_allow_attach() tells us the hostname 
     * (attach_host) and port number (attach_port) that we will start 
     * listening for attachments.
     *
     * We use attach_host and attach_port to construct a URL that attach 
     * clients need to use to attach to us.
     *
     * In calling nexus_allow_attach() we also name an approval function, in
     * this case attach_requested(), which is a user-supplied function that
     * gets called every time a client requests an attachment to us.
     *
     */

    /* allow other Nexus programs to attach to us */
    attach_port = 0;
    rc = globus_nexus_allow_attach(&attach_port,     /* port            */
				   &attach_host,     /* host            */
				   attach_requested, /* approval_func() */
				   GLOBUS_NULL);
    if (rc != 0)
    {
	/* libc functions are usually not re-entrant. Re-entrant functions   */
	/* are provided. They perform the necessary locks. But be award of   */
	/* potential dead-lock if miss-used.                                 */
	/* The  functions globus_libc_lock() and globus_libc_unlock() can    */
	/* also be used arround standard libc function                       */
       globus_libc_fprintf(
	   stderr,
	   "ERROR: globus_nexus_allow_attach() failed: rc=%d\n",
	   rc);
       if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
       {
	   globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
       }
       exit(1);
    } /* endif */

    globus_libc_sprintf(
	attach_url,
	"x-nexus://%s:%hu/",
	attach_host,
	attach_port);

    /* advertising URL for attachment */
    globus_libc_printf("attach url: %s\n", attach_url);

    /*************************************************************************/
    /* wait (using monitor) for single_rsr_client to attach and send number. */
    /* first our attach_requested() gets called (when the client attaches to */
    /* us) and then our rcv_handler() gets called (when the client sends us  */
    /* an rsr).  it is rcv_handler() that signals the monitor's condition    */
    /* variable and allows us to proceed.                                    */
    /*************************************************************************/

    /* the implementation of globus_cond_wait() and globus_cond_signal()     */
    /* makes it possible for globus_cond_wait() to experience a 'false      */
    /* wakeup', i.e., return without having had a globus_cond_signal()      */
    /* applied to it.                                                      */
    /*                                                                     */
    /* this is why we must wait on a condition variable in the manner      */
    /* we do below, with a loop and a globus_bool_t done, rather than       */
    /* simply:                                                             */
    /*     globus_mutex_lock(&(Monitor.mutex));                             */
    /*     globus_cond_wait(&(Monitor.cond), &(Monitor.mutex));             */
    /*     globus_mutex_unlock(&(Monitor.mutex));                           */

    globus_mutex_lock(&(Monitor.mutex));
    while (!(Monitor.done))
    {
        globus_cond_wait(&(Monitor.cond), &(Monitor.mutex));
    } /* endwhile */
    globus_mutex_unlock(&(Monitor.mutex));

    /************/
    /* Clean-up */
    /************/

    globus_mutex_destroy(&(Monitor.mutex));
    globus_cond_destroy(&(Monitor.cond));

    globus_nexus_endpoint_destroy(&GlobalEndpoint);
    globus_nexus_endpointattr_destroy(&GlobalEndpointAttr);

    if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	exit(1);
    }

    exit(0);

} /* end main() */

/*******************/
/*                 */
/* Local Functions */
/*                 */
/*******************/

/*
 * attach_requested()
 *
 * This is the attachment approval function.  When the server started
 * allowing attachments (nexus_allow_attach()), it specified this approval 
 * function.
 *
 * This function gets automatically (asynchronously) called when a
 * client calls nexus_attach() specifying this server's URL.
 *
 * The job of all attachment approval functions is to decide whether to
 * allow the requesting process to "attach" to this one.  If the 
 * attachment is granted, this function must bind the startpoint
 * passsed as the third arg to a local endpoint and return 0.
 * If attachment is denied, this function simply returns a non-zero value.
 *
 * In our example program we have decided to allow at most one connection
 * at a time.  In a more traditional client/server application where the
 * server might service multiple clients simultaneously, the logic for
 * this routine should be modified in the obvious way.
 *
 */

static int attach_requested(void *arg, char *url, globus_nexus_startpoint_t *sp)
{
     int rc;

    globus_mutex_lock(&(Monitor.mutex));
    if (Monitor.connection_established)
    {
        /* another single_rsr_client has already connected to us */
        rc = 1;
    }
    else
    {
        /* this is the first single_rsr_client */
        rc = 0;
        Monitor.connection_established = NEXUS_TRUE;
    } /* endif */
    globus_mutex_unlock(&(Monitor.mutex));

     /* binding received startpoint to one of our endpoints */
     if (rc == 0)
     {
         globus_nexus_startpoint_bind(sp, &GlobalEndpoint);
     } /* endif */

     return rc;

} /* end attach_requested() */

/***************************/
/*                         */
/* Nexus Handler Functions */
/*                         */
/***************************/

/*
 * rcv_handler()
 */
static void rcv_handler(globus_nexus_endpoint_t *   endpoint, 
			globus_nexus_buffer_t *     buffer,
			globus_bool_t               is_non_threaded_handler)
{
    monitor_t *mp;
    int i; 
	
    /****************************************************/
    /* extracting pointer to monitor from this endpoint */
    /****************************************************/

    mp = (monitor_t *) globus_nexus_endpoint_get_user_pointer(endpoint);

    /* extracting int from buffer */
    globus_nexus_get_int(buffer, &i, 1);

    /* printing received value */
    globus_libc_printf("Single Rsr server received %d\n", i);

    /* clean-up */

    /* destroying received buffer */
    globus_nexus_buffer_destroy(buffer);

    /**************************************************/
    /* setting the monitor's flag and signalling it's */
    /* cond variable to wake up the slave in main()   */
    /**************************************************/

    /* the implementation of globus_cond_wait() and globus_cond_signal()  */
    /* makes it possible for globus_cond_wait() to experience a 'false    */
    /* wakeup', i.e., return without having had a globus_cond_signal()    */
    /* applied to it.                                                     */
    /*                                                                    */
    /* this is why we must signal a condition variable in the manner      */
    /* we do below, with a globus_bool_t done, rather than simply:        */
    /*     globus_mutex_lock(&(mp->mutex));                               */
    /*     globus_cond_signal(&(mp->cond));                               */
    /*     globus_mutex_unlock(&(mp->mutex));                             */
   
    globus_mutex_lock(&(mp->mutex));
    mp->done = NEXUS_TRUE;
    globus_cond_signal(&(mp->cond));
    globus_mutex_unlock(&(mp->mutex));

} /* end rcv_handler() */
