/**************************************************************************
*
* Program:      Single Remote Service Request - Client
*               single_rsr_client.c
*
* Written at:   High-Performance Computing Laboratory
*               Department of Computer Science
*               Northern Illinois University
*               DeKalb, IL
*
* Original Author:          Nicholas T. Karonis
* Date Written:             2/97
* Updated to Client/Server: Nicholas T. Karonis
* Date Updated:             2/98
* Updated to Client       : Mei-Hui Su (fixed a race condition)
* Date Updated:             5/2000
*
* Description:
*
*     This client program, along with it's matching server program 
*     (single_rsr_server.c) contains the *minimum* amount of code 
*     required to start a server, then start a client, attach the 
*     client to the server, and finally send a single message from 
*     the client to the server.
* 
*     Before running this program you must start its server, single_rsr_server.
*     This client program requires two command line arguments, a URL (explained
*     below in "To run this program:") and an integer.  It attaches to
*     the server using the command line URL, places command line integer 
*     into a Nexus buffer and sends it to the server.
*
*     Sending a message is called initiating a "remote service request" (rsr).
*     An rsr invokes a specified function (called a handler routine, or simply 
*     handler) on the receiving process.  In this program, the rsr instructs 
*     the receiving process (the server) to execute the function rcv_handler(),
*     which is a routine that is found in single_rsr_server.c.
*
*     The receiving process extracts the integer from the buffer and writes
*     the command line integer to stdout.
*
* To run this program:
*
*     Before running this client program you must start the server, 
*     single_rsr_server.  When the server is started, it will print to stdout
*     the URL this client program must use to attach to the server.  Starting 
*     the server will produce an output line looking something like this:
*
*     olympus% single_rsr_server
*     olympus:c0:t0:p26566: attach url: x-nexus://olympus:48365/
*
*     To run this client program, you must pass the above URL and an integer
*     on the command line.  So, using the URL in this example, you would
*     run this program like this:
*
*     olympus% single_rsr_client x-nexus://olympus:48365/ 123
*
*     It will send the integer "123" to the server.
*
* Output:
*
*     If everything went OK, this program will not produce any output.
*     The server should first print the URL when it is started, and
*     then it should print the integer it received from this client,
*     in this example the integer "123".
*    
**************************************************************************/

#include <stdio.h>
#include <globus_common.h>
#include <globus_nexus.h>

/* The server has a table of handler routines.  That table has only one       */
/* entry, the handler routine to receive the integer this client program      */
/* sends to it via the remote service request (rsr).  When sending an rsr     */
/* you must specify which handler routine to execute on the receiving side by */
/* this by specifying the index into the handler table on the receiving side  */
/* side, in this case the index into the handler table is 0.                  */

#define RCV_HANDLER_ID 0

/********/
/*      */
/* MAIN */
/*      */
/********/

int main(int argc, char **argv)
{
    char attach_url[1024];
    globus_nexus_startpoint_t server_sp;
    globus_nexus_buffer_t buffer;
    int i;
    int rc;

    /**************************************************/
    /* Activation of the Nexus module is required at  */
    /* the beginning of all Nexus programs            */
    /**************************************************/

    if ( globus_module_activate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	fprintf(stderr,"Nexus Module activation failed \n");
	exit(1);
    }

    /**************************************************/
    /* check for correct command prompt usage and get */
    /* the attach_url specified in the command line   */
    /**************************************************/

    if (argc == 3 && !strncmp(argv[1], "x-nexus://", strlen("x-nexus://")))
    {
	strcpy(attach_url, argv[1]);
	i = atoi(argv[2]);
    }
    else
    {
	/* libc functions are usually not re-entrant. Re-entrant functions   */
	/* are provided. They perform the necessary locks. But be award of   */
	/* potential dead-lock if miss-used.                                 */
	/* The  functions globus_libc_lock() and globus_libc_unlock() can    */
	/* also be used arround standard libc function                       */
	globus_libc_fprintf(stderr, "usage: %s <attach_url> <int>\n", argv[0]);
	if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
	{
	    globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	}
	exit(1);
    } /* endif */

    /***********************/
    /* attaching to server */
    /***********************/

    /*
     *
     * Attachment in Nexus is done by making a request to attach to another
     * process designated by a URL.  If the request is granted, we get a 0
     * return code and a startpoint that points back to the granting process.
     *
     */

    rc = globus_nexus_attach(attach_url, &server_sp);

    /* if things went OK, we now have a startpoint */
    /* (server_sp) to the single_rsr_server        */

    if (rc != 0)
    {
        globus_libc_fprintf(
	    stderr, 
	    "ERROR: Could not attach to server using url: %s rc %d\n", 
	    attach_url, rc);
	if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
	{
	    globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	}
	exit(1);
    } /* endif */

    /**************************/
    /* remote service request */
    /**************************/

    /* initializing the buffer to hold a single integer */
    globus_nexus_buffer_init(&buffer, globus_nexus_sizeof_int(1), 0);

    /* placing single integer into buffer */
    globus_nexus_put_int(&buffer, &i, 1);

    /* sending buffer to single_rsr_server and invoking rcv_handler() there */
    globus_nexus_send_rsr(&buffer,        /* buffer */
			  &server_sp,     /* destination */
			  RCV_HANDLER_ID, /* handler id */
			  GLOBUS_TRUE,     /* destroy buffer when done */
			  GLOBUS_FALSE);   /* called from non-threaded handler */

    globus_nexus_startpoint_flush(&server_sp); /* wait till the message is out */
  
    /************/
    /* Clean-up */
    /************/

    globus_nexus_startpoint_destroy(&server_sp);

    if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	exit(1);
    }

    exit(0);

} /* end main() */
