/**************************************************************************
*
* Program:      Remote Service Request With Reply - Client
*               rsr_with_reply_client.c
*
* Written at:   High-Performance Computing Laboratory
*               Department of Computer Science
*               Northern Illinois University
*               DeKalb, IL
*
* Original Author:          Nicholas T. Karonis
* Date Written:             2/97
* Updated to Client/Server: Peter Lane
* Date Updated:             2/98
*
* Description:
*
*     This client program, along with it's matching server program 
*     (single_rsr_server.c) contains the *minimum* amount of code 
*     required to start a server, then start a client, attach the 
*     client to the server, have the client send a message to the
*     server and wait for the server's reply.
* 
*     Before running this program you must start its server, 
*     rsr_with_reply_server.  This client program requires three command line 
*     arguments, a URL (explained below in "To run this program:") and two 
*     integers.  It first attaches to the server using the command line URL. 
*     It then creates a endpoint/startpoint pair and places the startpoint 
*     along with 
*     the two command line integers into a Nexus buffer,
*     and sends it to the server.  It then waits for a reply from 
*     the server.  This waiting is implemented by creating a monitor
*     and placing a flag and condition variable inside.  The client 
*     process enters the monitor, clears the flag, and waits on the
*     monitor's condition variable.
*
*     Sending a message is called initiating a "remote service request" (rsr).
*     An rsr invokes a specified function (called a handler routine, or simply 
*     handler) on the receiving process.  In this program, the rsr instructs 
*     the the server to execute the function addem_handler(), which is a 
*     routine that is found in rsr_with_reply_server.c.
*
*     The server extracts the client's startpoint (to be used as a reply
*     startpoint) and both integers from the buffer and calculates their sum.
*     It then places the sum into another Nexus buffer and initiates an
*     rsr back to the client which invokes the get_answer_handler() (located
*     in this file).
*
*     The get_answer_handler() extracts the sum from the buffer.  It then
*     enters the monitor, places the sum in the monitor's data field, 
*     sets the flag, and signals the condition variable.
*
*     Once the condition variable is signaled and the flag has been set, 
*     the sum may be extracted from the monitor's data field and the original 
*     operands and sum are printed.
*
*     In this program we chose to place the address of the monitor variable
*     in the endpoint rather than simply making the monitor variable a 
*     global variable.  This choice was made somewhat arbitrarily.  The
*     main motivation was to illustrate how to set a user pointer in an
*     endpoint.  See the Ring example where we chose to make the monitor
*     a global variable instead.
*
* To run this program:
*
*     Before running this client program you must start the server, 
*     rsr_with_reply_server.  When the server is started, it will print to 
*     stdout the URL this client program must use to attach to the server.  
*     Starting the server will produce an output line looking something like 
*     this:
*
*     olympus% rsr_with_reply_server
*     olympus:c0:t0:p26566: attach url: x-nexus://olympus:48365/
*
*     To run this client program, you must pass the above URL and two integers
*     on the command line.  So, using the URL in this example, you would
*     run this program like this:
*
*     olympus% rsr_with_reply_client x-nexus://olympus:48365/ 1 2
*
*     It will send the integers "1" and "2" to the server.  The server
*     will then add 1+2 and return the sum back to the client.  After
*     receiving the sum from the server, the client will extract the
*     sum from the buffer and print it to stdout.
*
* Output:
*
*    1 + 2 = 3
*
**************************************************************************/

#include <stdio.h>
#include <globus_common.h>
#include <globus_nexus.h>

/* The server has a table of handler routines.  That table has only one       */
/* entry, the handler routine to receive the two integers this client program */
/* sends to it via the remote service request (rsr).  When sending an rsr     */
/* you must specify which handler routine to execute on the receiving side by */
/* this by specifying the index into the handler table on the receiving side  */
/* side, in this case the index into the handler table is 0.                  */

#define ADDEM_HANDLER_ID 0

/**************************/
/*                        */
/* Nexus Handler Function */
/*                        */
/**************************/

static void get_answer_handler(globus_nexus_endpoint_t * endpoint,
			       globus_nexus_buffer_t * buffer,
			       globus_bool_t is_non_threaded_handler);

/********************************/
/*                              */
/* Nexus Handler Function Table */
/*                              */
/********************************/

static globus_nexus_handler_t handlers[] =
{
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED, get_answer_handler}
};

/*******************/
/*                 */
/* Data Structures */
/*                 */
/*******************/

/* Monitor                                                   */
/*                                                           */
/* Nexus programs can implement a monitor with a data        */
/* structure as illustrated below.  All monitors implemented */
/* in this way must have a 'globus_mutex_t' field to provide  */
/* mutually exclusive access to the monitor.                 */
/*                                                           */
/* In this application we also need a monitor condition      */
/* variable, a flag, and an integer.                         */
/*                                                           */
/* Explanation of why 'volatile' is required on some fields: */
/*                                                           */
/* Some compilers, under certain optimization switches,      */
/* attempt to cache the value in a register thus avoiding a  */
/* memory reference.  By declaring the variable 'volatile',  */
/* we force compiler to  NOT cache value in register but     */
/* instead reload the value from  memory each time is is     */
/* referenced.                                               */
/*                                                           */
/* So, if a variable is shared by multiple threads such that */
/* it is written to by one thread and read by another, then  */
/* that variable must be declared volatile.  However, you    */
/* notice that two of the variables                          */
/*                                                           */
/*    globus_mutex_t mutex;                                  */
/*    globus_cond_t  cond;                                   */
/*                                                           */
/* both meet the multiple thread read/write criteria and yet */
/* neither of them are declared volatile.  This is because   */
/* the dataypes globus_mutex_t and globus_cond_t are defined */
/* within the Nexus implemenation (opaque to the user) and   */
/* Nexus has assumed the responsibility of taking the steps  */
/* necessary to assure that any potential compiler-induced   */
/* caching problems will be avoided.  Nexus does not assume  */
/* that responsibility for all its datatypes, only these     */
/* two.  This is because, by their very nature, these two    */
/* datatypes are intended to be shared by multiple threads.  */
/* All other Nexus-supplied datatypes should be explicitly   */
/* declared volatile where appropriate, as we do in this     */
/* example with our monitor flag (a globus_bool_t).           */
/*                                                           */
/* The general rule of thumb is, any variable that is not a  */
/* globus_mutex_t or a globus_cond_t and is shared by multiple*/
/* threads such that it is written to by one thread and read */
/* by another must be declared volatile.                     */


typedef struct _monitor_t
{
    /*****************************************************/
    /* a 'globus_mutex_t' is required of ALL monitors to */
    /* control mutually exclusive access to the monitor  */
    /*****************************************************/

    /* 'volatile' not required here, handled   */
    /* in the implementation of globus_mutex_t */

    globus_mutex_t mutex;

    /*********************************/
    /* monitor data for this program */
    /*********************************/

    /* 'volatile' not required here, handled  */
    /* in the implementation of globus_cond_t */

    globus_cond_t cond;

    /* 'volatile' required for both of these fields because */
    /* each is written by one thread and read by another    */

    volatile globus_bool_t done;
    volatile int sum;

} monitor_t;

/********/
/*      */
/* MAIN */
/*      */
/********/

int main(int argc, char **argv)
{
    char                         attach_url[1024];
    globus_nexus_buffer_t        buffer;
    int                          i;
    int                          rc;
    int                          x, y, sum;
    globus_nexus_endpointattr_t  clientEndpointAttr;
    globus_nexus_endpoint_t      clientEndpoint;
    globus_nexus_startpoint_t    clientStartpoint;
    globus_nexus_startpoint_t    serverStartpoint;
    monitor_t                    replyMonitor;

    /**************************************************/
    /* Activation of the Nexus module is required at  */
    /* the beginning of all Nexus programs            */
    /**************************************************/

    if ( globus_module_activate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	fprintf(stderr,"Nexus Module activation failed \n");
	exit(1);
    }

    /**************************************************/
    /* check for correct command prompt usage and get */
    /* the attach_url specified in the command line   */
    /**************************************************/

    if (argc == 4 && !strncmp(argv[1], "x-nexus://", strlen("x-nexus://")))
    {
	strcpy(attach_url, argv[1]);
	x = atoi(argv[2]);
	y = atoi(argv[3]);
    }
    else
    {
	/* libc functions are usually not re-entrant. Re-entrant functions   */
	/* are provided. They perform the necessary locks. But be award of   */
	/* potential dead-lock if miss-used.                                 */
	/* The  functions globus_libc_lock() and globus_libc_unlock() can    */
	/* also be used arround standard libc function                       */
	globus_libc_fprintf(
	    stderr,
	    "usage: %s <attach_url> <int> <int>\n",
	    argv[0] );
	/* quit now that we have notified the user of the error */
	if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
	{
	    globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	}
	exit(1);
    } /* endif */

    /***********************/
    /* attaching to server */
    /***********************/

    /*
     *
     * Attachment in Nexus is done by making a request to attach to another
     * process designated by a URL.  If the request is granted, we get a 0
     * return code and a startpoint that points back to the granting process.
     *
     */

    rc = globus_nexus_attach( attach_url, &serverStartpoint );

    /* if things went OK, we now have a startpoint */
    /* to the server, serverStartpoint.            */

    if( rc != 0 )
    {
	globus_libc_fprintf(
	    stderr, 
	    "ERROR: Could not attach to server using url: %s rc %d\n", 
	    attach_url, rc
	    );
	/* quit now that we have notified the user of the error */
	if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
	{
	    globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	}
	exit(1);
    } /* endif */

    /***************************/
    /* Endpoint Initialization */
    /***************************/

    globus_nexus_endpointattr_init( &clientEndpointAttr );
    globus_nexus_endpointattr_set_handler_table(
	&clientEndpointAttr,
	handlers,
	sizeof( handlers ) / sizeof( globus_nexus_handler_t )
	);
    globus_nexus_endpoint_init( &clientEndpoint, &clientEndpointAttr );
    /* set up reply our endpoint to point to the reply monitor */
    globus_nexus_endpoint_set_user_pointer(
	&clientEndpoint,
	(void *)(&replyMonitor)
	);
    /* binding a startpoint to our endpoint */
    globus_nexus_startpoint_bind( &clientStartpoint, &clientEndpoint );

    /***************************************************************/
    /* Initialize the monitor that will be used for signalling     */
    /* that we have successfully received the sum from ther server */
    /***************************************************************/

    /* initializing the mutex that controls access to the monitor */
    globus_mutex_init( &(replyMonitor.mutex), (globus_mutexattr_t *) GLOBUS_NULL );
    /* initializing the condition variable */
    globus_cond_init( &(replyMonitor.cond), (globus_condattr_t *) GLOBUS_NULL );

    /* entering the monitor and clearing the flags */
    globus_mutex_lock( &(replyMonitor.mutex) );
    replyMonitor.done = GLOBUS_FALSE;
    globus_mutex_unlock( &(replyMonitor.mutex) );

    /**************************/
    /* remote service request */
    /**************************/

    /* initializing the buffer to hold the clientStartpoint and two integers */
    /* Note: there is no difference between (nexus_sizeof_int(1) * 2) and    */
    /*       simply nexus_sizeof_int(2).                                     */

    globus_nexus_buffer_init(
	&buffer,
	globus_nexus_sizeof_startpoint( &clientStartpoint, 1 )
	+ (globus_nexus_sizeof_int(1) * 2),
	0);

    /* placing the startpoint and two integers into the buffer */

    /* by calling nexus_put_startpoint_transfer() destroys our */
    /* local copy clientStartpoint.  this saves us the trouble */
    /* of having to explicitly call nexus_startpoint_destroy() */
    /* later.                                                  */

    /* nexus_startpoint_destroy() serves our purposes here because */
    /* we do not need to send clientStartpoint anywhere else.  if  */
    /* we did need to send copies clientStartpoint to many places, */
    /* then we would use nexus_put_startpoint() to place copies of */
    /* it into all the buffers (which does *not* destroy it) and   */
    /* then explicitly destroy it at the end with                  */
    /* nexus_startpoint_destroy()                                  */

    globus_nexus_put_startpoint_transfer( &buffer, &clientStartpoint, 1 );
    globus_nexus_put_int( &buffer, &x, 1 );
    globus_nexus_put_int( &buffer, &y, 1 );

    /* sending buffer to the server and invoking addem_handler() there */
    globus_nexus_send_rsr( &buffer,              /* buffer */
			   &serverStartpoint,    /* destination */
			   ADDEM_HANDLER_ID,     /* handler id */
			   GLOBUS_TRUE,    /* destroy buffer when done */
			   GLOBUS_FALSE ); /* called from non-threaded
						    handler */

    /****************************************************************/
    /* wait on the monitor condition variable until the reply comes */
    /****************************************************************/

    /* the implementation of globus_cond_wait() and globus_cond_signal()  */
    /* makes it possible for globus_cond_wait() to experience a 'false    */
    /* wakeup', i.e., return without having had a globus_cond_signal()    */
    /* applied to it.                                                     */
    /*                                                                    */
    /* this is why we must wait on a condition variable in the manner     */
    /* we do below, with a loop and a globus_bool_t done, rather than     */
    /* simply:                                                            */
    /*     globus_mutex_lock(&(monitor.mutex));                           */
    /*     globus_cond_wait(&(monitor.cond), &(monitor.mutex));           */
    /*     globus_mutex_unlock(&(monitor.mutex));                         */

    globus_mutex_lock( &(replyMonitor.mutex) );
    	while( !(replyMonitor.done) ) {
        	globus_cond_wait( &(replyMonitor.cond), &(replyMonitor.mutex) );
   	 } /* endwhile */
    globus_mutex_unlock( &(replyMonitor.mutex) );

    /* at this point, flag in monitor has been set     */
    /* ... OK to extract sum from monitor's data field */

    /*******************************/
    /* print out the reply message */
    /*******************************/

    /* lock the monitor so that we can access the sum */
    globus_mutex_lock( &(replyMonitor.mutex) );
    {
	sum = replyMonitor.sum;
    }
    globus_mutex_unlock( &(replyMonitor.mutex) );
    /* I take care to not have this IO inside the locked area above to avoid */
    /* potential dead-lock (it would not be a dead lock in this simple       */
    /* example, but lets do it right)                                        */
    globus_libc_printf( "%d + %d = %d\n\n", x, y, sum );

    /************/
    /* Clean-up */
    /************/

    /* monitor clean-up */
    globus_mutex_destroy( &(replyMonitor.mutex) );
    globus_cond_destroy( &(replyMonitor.cond) );

    /*endpoint clean-up */
    nexus_endpoint_destroy( &clientEndpoint );
    nexus_endpointattr_destroy( &clientEndpointAttr );

    /* startpoint clean-up */
    nexus_startpoint_destroy( &serverStartpoint );

    /* NOTE: we do not call nexus_startpoint_destroy(&clientStartpoint)  */
    /*       here becuase the clientStartpoint was destroyed when we     */
    /*       placed it into the buffer when we called                    */
    /*       nexus_put_startpoint_transfer()                             */

    if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	exit(1);
    }

    exit(0);

} /* end main() */

/***************************/
/*                         */
/* Nexus Handler Functions */
/*                         */
/***************************/

/*
 * get_answer_handler()
 */
static void
get_answer_handler(globus_nexus_endpoint_t * endpoint,
		   globus_nexus_buffer_t   * buffer,
		   globus_bool_t             is_non_threaded_handler )
{
    monitor_t *mp;
    int sum;

    /****************************************************/
    /* extracting pointer to monitor from this endpoint */
    /****************************************************/

    mp = (monitor_t *) globus_nexus_endpoint_get_user_pointer(endpoint);

    /******************************************/
    /* extracting answer from message buffer  */
    /******************************************/

    globus_nexus_get_int(buffer, &sum, 1);

    /************/
    /* clean-up */
    /************/

    globus_nexus_buffer_destroy(buffer);

    /***********************************************/
    /* placing sum into monitor, setting flag, and */
    /* signalling cond variable to wake up main()  */
    /***********************************************/

    /* the implementation of globus_cond_wait() and globus_cond_signal()  */
    /* makes it possible for globus_cond_wait() to experience a 'false    */
    /* wakeup', i.e., return without having had a globus_cond_signal()    */
    /* applied to it.                                                     */
    /*                                                                    */
    /* this is why we must signal a condition variable in the manner      */
    /* we do below, with a globus_bool_t done, rather than simply:        */
    /*     globus_mutex_lock(&(mp->mutex));                               */
    /*     mp->sum = sum;                                                 */
    /*     globus_cond_signal(&(mp->cond));                               */
    /*     globus_mutex_unlock(&(mp->mutex));                             */
   
    globus_mutex_lock(&(mp->mutex));
    {
	mp->sum = sum;
	mp->done = GLOBUS_TRUE;
	globus_cond_signal(&(mp->cond));
    }
    globus_mutex_unlock(&(mp->mutex));

} /* end get_answer_handler() */





