/**************************************************************************

 Program: nexus_myjob_ring

 This program shows how to use "globus_gram_myjob" to implement a simple
 nexus communication in ring between a set of process, and then use this ring
 to transfer a simple message, using Nexus.

 In this example, the process of rank 0 initiates the comunication by sending
 a message to the process of rank 1.
 The process of rank 1 forward this message to the next process (rank+1) and so
 forth until the "last process (of rank size-1) which forward the message to
 the first process (rank 0). The ring is now closed.
 
How to run this program:

 In order to let globus set-up a correct communication environment for all
 the process of the job executed by this program, this program should be
 started using GRAM.
 
 Amoung the several possibilities existing to do so, the easiest is
 probably to use globusrun, as in the example below:
 
#promtp> globusrun -r pitcairn.mcs.anl.gov-fork -- '&( directory = /usr/local/globus/Examples/nexus_myjob_ring/ )( executable = nexus_myjob_ring )( stdout = my_std_out)( stderr = my_std_err )( count = 5 )(arguments = "Test me")'

    It will create a job containing 5 process, exchanging the message "Test me".

 See the README.html file for more detailed instructions.

Note:

 - Althought this program does not use threads, we use the reentrant version
   of "libc" functions supplied with globus: globus_libc_*. 

 CVS Information: 
     $Source: /home/globdev/CVS/globus-current/Globus/Examples/nexus_myjob_ring/nexus_myjob_ring.c,v $
     $Date: 1998/11/30 16:56:09 $
     $Revision: 1.7 $
     $Author: bester $

**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

/* You must include those file to use nexus, gram_myjob and gass :       */
#include "globus_nexus.h"
#include "globus_gram_myjob.h"
#include "globus_common.h"

/* to keep track of the job size and of our rank in the job */
int    rank;
int    size;

/* message to send */
char *  message;

/*****************************************************************************
 *                         
 * Nexus Handler Functions
 *
 * In this simple example, we have only one handler. In nexus, a handler is a
 * which is executed on a remote process when one send it a
 * "Remote service request" or rsr. A remote service request encapsulate also
 * the data associated with the request, and sent by the process requesting the
 * service.
 *
 * In our example, the data sent is the message to pass around and the only
 * "service" we request to the remote process is to forward the message
 * to the next process. (and got_message_handler implement this functionnality)
 *
 * We therefor have only one handler. Nexus allow you to
 * have several. Each handler must have a reference in the
 * Nexus Handler Functions Table (see below).
 *                         
 *****************************************************************************/
static void
got_message_handler(globus_nexus_endpoint_t *endpoint,
		    globus_nexus_buffer_t *buffer,
		    globus_bool_t is_nonthreaded_handler);
#define GOT_MESSAGE_HANDLER_ID 0

/******************************************************************************
 *                               
 * Nexus Handler Functions Table 
 *
 * This table associate each nexus handler with a handler_id, a unique number
 * identifying each "remote service"
 * This table will be used in the program to initialise an endpoint, and make
 * it abble to trigger the handlers listed in this table when it "receive"
 * a remote servrice request.
 *****************************************************************************/
static globus_nexus_handler_t handler_table[] =
{
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,
     got_message_handler},
};
#define HANDLER_TABLE_SIZE 1


/******************************************************************************
 * nexus endpoint and start points
 */

/* endpoint used to receive messages from the previous process in the ring    */
static globus_nexus_endpointattr_t	epattr;
static globus_nexus_endpoint_t		endpoint;

/*
 * startpoint bound to the end point above. It will be send to the previous
 * process using "myjob", to allow it to send us messages.
 */
static globus_nexus_startpoint_t        sp;

/*
 * We will receive this start point from the next process, and use it to
 * send it the message we want to pass around
 */
static globus_nexus_startpoint_t        next_proc_sp;

/*
 * Conditional variable used to let the nexus handler inform the program
 * when it has finished its job
 */
globus_mutex_t mutex;
globus_cond_t cond;
globus_bool_t done;

/******************************************************************************

  Function : send_msg_using_nexus()

  Send a string using nexus_send_rsr. It also send the lenght of the string.
  Note that using a similar technic, we could send any type supported by nexus,
  (we could for example exchange start point to create a double ring...) 
    
  Parameter:
      In: message       : pointer to the string to send
      
  Globval variable used:
      In: next_proc_sp  : this start point (global variable) is used to
                          send the message
  Return value:
      GLOBUS_SUCCESS or the error code set by globus_nexus_send_rsr()
  
******************************************************************************/
int
send_msg_using_nexus(char * message)
{
    globus_nexus_buffer_t buffer;
    int                   buf_size;
    int                   message_len;
    int                   rc;
    
    message_len = strlen(message);

    /*
     * get the size of a nexus_buffer necessary to send the message lenght (int)
     * and the message itself
     */
    buf_size =
	globus_nexus_sizeof_int(1) +     
	globus_nexus_sizeof_char(message_len);
    /*
     * Initialise the buffer
     */
    globus_nexus_buffer_init(&buffer,buf_size , 0);
    
    /* put the message lenght in the buffer */
    globus_nexus_put_int(&buffer, &message_len, 1);
    
    /* put the message in the buffer */
    globus_nexus_put_char(&buffer,
			  message,
			  message_len);     
    globus_libc_printf("%d: Message size to send %d\n",rank,message_len);

    /*
     * send the buffer
     * Note the handler_id GOT_MESSAGE_HANDLER_ID which will trigger the
     * corresponding function in the remote process
     */
    rc = globus_nexus_send_rsr(
	&buffer,
	&next_proc_sp,
	GOT_MESSAGE_HANDLER_ID,
	GLOBUS_TRUE,              /* please, destroy the buffer when sent       */
	GLOBUS_FALSE);            /* not called from a non threaded handler     */
				  /* actually, not called from a handler at all */
    
    return(rc);

} /* send_msg_using_nexus() */

/******************************************************************************

  Function : got_message_handler()

  This is the function triggered when I receive the message from the previous
  process in the ring. It extracts the message from the buffer, and then
  forwards it to the next process in the ring.  
  
******************************************************************************/
static void
got_message_handler(globus_nexus_endpoint_t *endpoint,
		    globus_nexus_buffer_t *buffer,
		    globus_bool_t is_nonthreaded_handler)
{

    char * received;
    int    received_lenght;
    int    rc;
    
    globus_libc_printf("%d: Got a NEXUS remote service call\n",rank);

    /* extract the lenght of the message */
    globus_nexus_get_int(buffer, &received_lenght, 1);
    globus_libc_printf("%d: Message lenght : %d\n", rank, received_lenght);
    
    received = globus_malloc(received_lenght+1);
    if (received == GLOBUS_NULL)
    {
	globus_libc_fprintf(stderr,
			    "%d: ERROR : Could not allocate memory for incomming message\n",rank);
	
    }

    /* extract the message */
    globus_nexus_get_char(buffer, received, received_lenght);

    /*
     * Since I do not transmit the termination char,
     * I add it to print the message
     */
    received[received_lenght] = '\0';
    globus_libc_printf("%d: Message received: %s\n",rank,received);
    
    
    if (rank !=0)
    {
	/* forward the message */
	if ( rc = send_msg_using_nexus(received) != GLOBUS_SUCCESS)
	{
	    globus_libc_fprintf(
		stderr,
		"%d: ERROR Sending message using nexus, error code; %d!\n",
		rank,
		rc);
	}
    }
    else
    {
	if (strcmp(received, message))
	{
	    globus_libc_fprintf(
		stderr,
		"%d: ERROR Received incorrect message !\n",rank);
	}
	else
	{
	    globus_libc_fprintf(
		stdout,
		"%d: Received the correct message !\n",rank);
	}    
    }
    globus_free(received);
   

    /* signal the I have receive the message, and that the job can terminate */
    globus_mutex_lock(&mutex);
    {
	done = GLOBUS_TRUE;
	globus_cond_signal(&cond);
    }
    globus_mutex_unlock(&mutex);
 
    
}
/******************************************************************************

  Function : send_sp(sp, next)

  Send a start point to the "next" process in the job,
       using "globus_gram_myjob_myjob".
       
  Important notice: Due to some limitation (buffer size) and some latency
                    overhead, "myjob" should not be used for general
		    communication. Use "myjob" only to establish the
		    communication, as in this example, by exchanging some
		    startpoint between the process of the job and create the
		    job topology. Afterward, use those startpoints and nexus
		    to communicate.

  In this function, we do not use a nexus buffer (created
  with globus_nexus_buffer_init() but a user buffer. We must therefor take care
  to:
  - Use the correct set of function (globus_nexus_user_*) when necessary.
  - Send the data format corresponding to our architachture, so the receiver
    can convert the data to its own format.

  Parameter:

  In:  sp   : startpointto send
       next : rank of the process to send the job to.
  
  
******************************************************************************/
void
send_sp(globus_nexus_startpoint_t * sp,
	int                         next)
{
    globus_byte_t *        sp_buf = GLOBUS_NULL;
    int                    buf_size;
    globus_byte_t *        buf_pt;
    globus_byte_t          format;
    int                    rc;
    
    sp_buf = globus_malloc(GLOBUS_GRAM_MYJOB_MAX_BUFFER_LENGTH);
    if (sp_buf == GLOBUS_NULL)
    {
	globus_libc_fprintf(stderr,
			    "%d: malloc() failed.\n",
			    rank);
	globus_gram_myjob_kill();
    }
    
    /*
     * get the size of a nexus_buffer necessary to send the data format specific
     * to this architechture (see comment above) and the startpoint it self
     */
    buf_size =
	1 +
	globus_nexus_sizeof_startpoint(sp, 1);
    
    if (buf_size >= GLOBUS_GRAM_MYJOB_MAX_BUFFER_LENGTH)
    {
	globus_libc_fprintf(
	    stderr,
	    "%d: Startpoint too big to be transfered using myjob !\n",
	    rank);
	globus_gram_myjob_kill();
    }
    
    /* !! globus_nexus_user_put_startpoint_transfer()  will
     *    automatically increment its user_buffer argument 
     *    to point to the next unused byte in user_buffer:
     *    We must take care to NOT use directly sp_buff
     *    (or sp_buff will not point anymore to the allocated buffer
     *    and the programm wil crash whenwe call free()...)
     */
    buf_pt = sp_buf;

    /* put the format in the buffer */
    *buf_pt++ = (globus_byte_t) globus_nexus_user_format();
    
    /* put the startpoint */ 
    globus_nexus_user_put_startpoint_transfer(&buf_pt, sp, 1);   

    printf("%d: Send my startpoint\n", rank);
    
    rc = globus_gram_myjob_send(next,
				sp_buf,
				buf_size);
				
    if ( rc != GLOBUS_GRAM_MYJOB_SUCCESS)
    {
	globus_libc_fprintf(stderr,
			    "globus_gram_myjob_send() failed, rc=0x%08x.\n",
			    rc);
	globus_gram_myjob_kill();
    }

    /* the buffer has been sent using "myjob": we can destroy it */
    globus_free(sp_buf);

} /* send_sp() */

/******************************************************************************

  Function : receive_sp(sp)
  
    Wait for a "my_job" message and extract a start point out of the message.
    This function is just a wrapper around
    globus_gram_myjob_receive().
    
  Parameter:
     Out:
         sp: startpoint we received.
  
******************************************************************************/
void
receive_sp(globus_nexus_startpoint_t *  sp)
{
    int msg_sz;
    int rc;
    globus_byte_t *     buf_pt;
    globus_byte_t       format;
    globus_byte_t *     sp_buf = GLOBUS_NULL;

    
    /* buffer used to receive the messages */
    sp_buf = globus_malloc(GLOBUS_GRAM_MYJOB_MAX_BUFFER_LENGTH);
    if (sp_buf == GLOBUS_NULL)
    {
	globus_libc_fprintf(stderr,
			    "%d: malloc() failed.\n",
			    rank);
	globus_gram_myjob_kill();
    }

    /* wait for the message */
    rc = globus_gram_myjob_receive(sp_buf,
				   &msg_sz);
    if ( rc != GLOBUS_GRAM_MYJOB_SUCCESS)
    {
	globus_libc_fprintf(stderr,
			    "%d: globus_gram_myjob_receive() failed, "
			    "rc=0x%08x.\n",
			    rank,
			    rc);
	globus_gram_myjob_kill();
    }
    
    globus_libc_fprintf(stdout, "%d: Received startpoint \n",
			rank);
    /*
     * Now extract the start point from the "myjob" buffer
     */
    buf_pt = sp_buf;
    /*
     * first extract the data format, since we are using the "user bufer"
     * functions, we have to take care of this.
     * You can notice that in send_msg_using_nexus() we did not used
     * the "user buffer" nexus function, and therefor we did not need to
     * worry about the data format: nexus take car of it
     */
    format = (int) *buf_pt++;
    /*
     * now use the format to extract/convert the startpoint
     */
    globus_nexus_user_get_startpoint(&buf_pt, sp, 1, format);

    globus_free(sp_buf);
    
} /* receive_sp() */


/******************************************************************************
  Main :

  It takes one argument, the string to send around the ring.
  (If the string contain any blank, it should be double quoted...)
******************************************************************************/
int
main(
    int					argc,
    char *				argv[])
{    
    int	       	        rc;
    
    /*
     * parse the simple arguments
     */
    if (argc != 2)
    {
	globus_libc_printf(
	    "Usage : nexus_myjob_ring <message inside quotes>\n");
	exit(1);
    }
    message = argv[1];
    
    /***********************************************************************
     * Modules initialisation
     */
    
    /* This is mandatory in order to use "globus_nexus" */
    if ((rc = globus_module_activate(GLOBUS_NEXUS_MODULE)) != GLOBUS_SUCCESS)
    {
	globus_libc_printf(
	    "Module activation for \"globus_nexus\" failed, rc=0x%08x.\n", rc);
	exit(1);
    }
    
    /* This is mandatory in order to use "globus_myjob" */
    if ((rc = globus_module_activate(GLOBUS_GRAM_MYJOB_MODULE))
	!= GLOBUS_SUCCESS)
    {
	globus_libc_printf(
	    "Module activation for \"globus_myjob\" failed, rc=0x%08x.\n", rc);
	globus_gram_myjob_kill();
    }

    /***********************************************************************
     * Get informations concerning the job : Size of the job and my rank
     * in the job
     */

    /*
     * Get the SIZE of the job this process is participating in
     */
    if ((rc = globus_gram_myjob_size(&size)) != GLOBUS_GRAM_MYJOB_SUCCESS)
    {
	globus_libc_fprintf(stderr,
			    "globus_gram_myjob_size() failed, rc=0x%08x.\n",
			    rc);
	globus_gram_myjob_kill();
    }

    /*
     * Get the RANK of the job this process is participating in
     */
    if ((rc = globus_gram_myjob_rank(&rank)) != GLOBUS_GRAM_MYJOB_SUCCESS)
    {
	globus_libc_fprintf(stderr,
			    "globus_gram_myjob_rank() failed, rc=0x%08x.\n",
			    rc);
	globus_gram_myjob_kill();
    }

    globus_libc_fprintf(
	stdout,
	"\n\tJob of rank %d \n\tTotal number of jobs (job size): %d\n",
	rank,
	size);

    /***********************************************************************
     * Nexus communication :
     */

    /*
     * Create an end point
     */
    globus_nexus_endpointattr_init(&epattr);
    globus_nexus_endpointattr_set_handler_table(&epattr,
						handler_table,
						HANDLER_TABLE_SIZE);
    globus_nexus_endpoint_init(&endpoint, &epattr);

    /*
     * Bind to this endpoint the startpoint we want to send to the previous
     * process in the job.
     */
    globus_nexus_startpoint_bind(&sp, &endpoint);

    /*
     * Initialise a conditional variable, and the mutex protecting the
     * conditional variable.
     */
    globus_mutex_init(&mutex, (globus_mutexattr_t *) NULL);
    globus_cond_init(&cond, (globus_condattr_t *) NULL);
    done = GLOBUS_FALSE;

    /*
     * Now the communication part is starting
     */
    
    /*
     * "communicate with other process only if there is
     * other process in the job (job size > 1)
     */
    if (size > 1) 
    {

	/***************************************************************
        * We must first setup the "nexus" ring 
	****************************************************************/
	
	/*
	 * To get all the communication setup in parallel, I use an even/odd
	 * send/receive, and odd/even receive/send algorithm:
	 *
	 * - all the job with a rank even are sending their start point first
	 *   and then they wait-receive the start point from their "next" job
	 * - all the job with a rank odd first wait-receive a startpoint
	 *   from their "next" job, and then they send their start point
	 */
	if ((rank>>1)<<1 == rank) /* even job number */
	{
	    /*
	     * send a start point binded to my endpoint to the "previous
	     * process in the job, the process of rank "rank-1"
	     * (if I am the process of rank 0, the previous process is
	     * the last one in the job (rank "size-1") to close the ring. 
	     */
	    send_sp(&sp, (rank==0)?size-1:rank-1);

	    /*
	     * wait until I receive a startpoint from the next process in the 
	     * job
	     */
	    receive_sp(&next_proc_sp);    
	}
	else /* odd job number */
	{
	    /*
	     * wait until I receive a startpoint from the next process in the 
	     * job
	     */
	    receive_sp(&next_proc_sp);
	    /*
	     * send a start point binded to my endpoint to the "previous
	     * process in the job
	     */	    
	    send_sp(&sp, rank-1);
	}
	
	/***************************************************************
         * The "nexus" ring is setup, we use it to circulate some data
	 ***************************************************************/

	/*
	 * A process has to start the ring communication. We chose the
	 * process of rank 0, but it is not mandatory.
	 */
	if (rank == 0)
	{
	    
	    globus_libc_printf("Message to send around: %s\n",message);
	    
	    if ( rc = send_msg_using_nexus(message) != GLOBUS_SUCCESS)
	    {
		globus_libc_fprintf(
		    stderr,
		    "%d: ERROR Sending message using nexus, error code; %d!\n",
		    rank,
		    rc);
	    }
	    /* wait for the message to be passed around until it comes back
	     * to me, when the last process in the job will send it to me.
	     * This will trigger my handler: I do not need to do
	     * anything here but wait it is done (see globus_cond_wait below)
	     */
	}
	
	/*
	 * if I am not the process of rank 0, I will get the message from
	 * the previous process some time, and it will trigger my handler
	 * wich will forward it to the next process: I do not need to do
	 * anything here but wait it is done
	 */

	/*
	 * Wait the job is done: When my handler will have received
	 * the message and forwarded it to the next process in the job, it
	 * will signal "cond" and "wake-me-up"
	 * (see handler got_message_handler() for details)
	 */
	globus_mutex_lock(&mutex);
	{
	    while (!done)
	    {
		globus_cond_wait(&cond, &mutex);
	    }
	}
	globus_mutex_unlock(&mutex);
	/*
	 * Here I know the job is done
	 */
    }
    

    /*
     * This process has received and send the message.
     * It has finished its task and can exit after some cleaning.
     */

    /*
     * Clean the end point
     */
    globus_nexus_endpoint_destroy(&endpoint);
    globus_nexus_endpointattr_destroy(&epattr);

    /*
     * clean the start points
     * Note that if I had used globus_nexus_put_startpoint_transfer()
     * and not the "user buffer" function, the local startpoint sp would have
     * been destroyed automatically when transferd in the send buffer
     */    
    globus_nexus_startpoint_destroy(&next_proc_sp);

    /*
     * Note : The transfert of the sp startpoint does destroy it:
     * we should Not destroy it again !
     */
    
    /*
     * clean the conditionnal variable used for synchronisation
     */
    globus_mutex_destroy(&mutex);
    globus_cond_destroy(&cond);
    

    /*************************************************************************
     * Now we HAVE TO deactivate all the module we have used.
     */

    /* deactivate "globus_myjob" */
    if ((rc = globus_module_deactivate(GLOBUS_GRAM_MYJOB_MODULE))
	 != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(
	    stderr,
	    "Module activation for \"globus_myjob\" failed, rc=0x%08x.\n", rc);
	globus_gram_myjob_kill();
    }
    /* deactivate "globus_nexus" */
    if ((rc = globus_module_deactivate(GLOBUS_NEXUS_MODULE))
	 != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(stderr,
	    "Module activation for \"globus_nexus\" failed, rc=0x%08x.\n", rc);
    }

    /* say ByeBye... */
    globus_libc_fprintf(stdout, "%d: test complete.\n", rank);

    exit(0);
}




