/**************************************************************************

 Program: nexus_duroc_ring

 This program shows how to use "globus_duroc_runtime" to implement a simple
 nexus communication in ring between a set of process, and then use this ring
 to transfer a simple message, using Nexus.

 In this example, the process of rank 0 in the subjob 1 initiates the
 comunication by sending a message to the process of rank 1 in the subjob 1,
 or it it is "alone it ist subjob, it will send the message to the next sub
 job, process of rank 0 in the subjob 2.
 
 This process will forward the message to the next process (rank+1 or
 eventually in the next subjob, with rank 0)) and so forth until the "last
 process (of rank size-1) which forward the message to the first process (rank
 0 subjob 1).

 The ring is now closed.
 
How to run this program:

 In order to let globus set-up a correct communication environment for all
 the process of the job executed by this program, this program should be
 started using GRAM (gram_duroc).
 
 Amoung the several possibilities existing to do so, the easiest is
 probably to use globusrun, as in the example below:
 
#promtp> globusrun -f spec

with spec being the name of a file containing a duroc job specification, for example:

+(
   &( directory = /usr/globus/Examples/nexus_duroc_ring )
    ( executable = nexus_duroc_ring )
    ( stdout = my_std_out )
    ( stderr = my_std_err )
    ( count = 2 )
    ( arguments = "Test me")
    ( label= my_subjob1 )
    ( resourceManagerContact = "machineone.mcs.anl.gov:8713:/C=US/O=Globus/O=Argonne National Laboratory/OU=MCS/CN=machineone.mcs.anl.gov-fork" )
 )
 (
   &( directory = /usr/globus/Examples/nexus_duroc_ring )
    ( executable = nexus_duroc_ring )
    ( stdout = my_std_out )
    ( stderr = my_std_err )
    ( count = 5 )
    ( arguments = "Test me" )
    ( label= my_subjob2 )
    ( resourceManagerContact = "machinetwo.mcs.anl.gov:8713:/C=US/O=Globus/O=Argonne National Laboratory/OU=MCS/CN=machinetwo.mcs.anl.gov-fork" )
 )

    It will create a job containning 2 subjobs (labeled my_sunbjob1 and my_subjob2, for example)
    my_sunbjob1 contains 2 process and each process runs on the
    machine "machineone".
    my_sunbjob2 contains 5 process and each process runs on the
    machine "machinetwo".
    
    They will all exchange the message "Test me".

 See the README.html file for more detailed instructions.

Note:

 - Althought this program does not use threads, we use the reentrant version
   of "libc" functions supplied with globus: globus_libc_*. 

 CVS Information: 
     $Source: /home/globdev/CVS/globus-current/Globus/Examples/nexus_duroc_ring/nexus_duroc_ring.c,v $
     $Date: 1998/10/27 19:44:03 $
     $Revision: 1.3 $
     $Author: tedesco $

**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

/* You must include those file to use nexus, duroc and gass :       */
#include "globus_nexus.h"
#include "globus_gram_myjob.h"
#include "globus_duroc_runtime.h"
#include "globus_common.h"

/* to keep track of the number of remote subjob */ 
int   nb_remote_subjob;           
/* to remember our own  subjob address */
int   own_subjob_address;
int   *array_of_subjobs;

/* to know my relative position in the subjob */
int i_am_the_first_job_master= GLOBUS_FALSE;
int previous_job_master_address;

/* to keep track of the subjob size and of our rank in the subjob */
int    rank;
int    size;

/* message to send */
char *  message;
char * tag = "anything";  /* we do not make use of this tag in this example */

/*****************************************************************************
 *                         
 * Nexus Handler Functions
 *
 * In this simple example, we have only one handler. In nexus, a handler is a
 * function which is executed on a remote process when someone send it a
 * "Remote service request" or rsr. A remote service request encapsulate also
 * the data associated with the request, and sent by the process requesting the
 * service.
 *
 * In our example, the data sent is the message to pass around and the only
 * "service" we request to the remote process is to forward the message
 * to the next process. (and got_message_handler implement this functionnality)
 *
 * We therefor have only one handler. Nexus allow you to
 * have several. Each handler must have a reference in the
 * Nexus Handler Functions Table (see below).
 *                         
 *****************************************************************************/
static void
got_message_handler(globus_nexus_endpoint_t *endpoint,
		    globus_nexus_buffer_t *buffer,
		    globus_bool_t is_nonthreaded_handler);
#define GOT_MESSAGE_HANDLER_ID 0

/******************************************************************************
 *                               
 * Nexus Handler Functions Table 
 *
 * This table associate each nexus handler with a handler_id, a unique number
 * identifying each "remote service"
 * This table will be used in the program to initialise an endpoint, and make
 * it abble to trigger the handlers listed in this table when it "receive"
 * a remote servrice request.
 *****************************************************************************/
static globus_nexus_handler_t handler_table[] =
{
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,
     got_message_handler},
};
#define HANDLER_TABLE_SIZE 1


/******************************************************************************
 * nexus endpoint and start points
 */

/* endpoint used to receive messages from the previous process in the ring    */
static globus_nexus_endpointattr_t	epattr;
static globus_nexus_endpoint_t		endpoint;

/*
 * startpoint bound to the end point above. It will be send to the previous
 * process using "duroc", to allow it to send us messages.
 */
static globus_nexus_startpoint_t        sp;

/*
 * We will receive this start point from the next process in the same subjob,
 * or from the first process in the next subjob, and use it to
 * send it the message we want to pass around
 */
static globus_nexus_startpoint_t        next_proc_sp;
static globus_nexus_startpoint_t        next_subjob_sp;

/*
 * Conditional variable used to let the nexus handler inform the program
 * when it has finished its job
 */
globus_mutex_t mutex;
globus_cond_t cond;
globus_bool_t done;

/******************************************************************************

  Function : send_msg_using_nexus()

  Send a string using nexus_send_rsr. It also send the lenght of the string.
  Note that using a similar technic, we could send any type supported by nexus,
  (we could for example exchange start point to create a double ring...) 
    
  Parameter:
      In: message       : pointer to the string to send
      
  Globval variable used:
      In: next_proc_sp  : this start point (global variable) is used to
                          send the message
  Return value:
      GLOBUS_SUCCESS or the error code set by globus_nexus_send_rsr()
  
******************************************************************************/
int
send_msg_using_nexus(char * message,
		     globus_bool_t is_nonthreaded_handler)
{
    globus_nexus_buffer_t buffer;
    int                   buf_size;
    int                   message_len;
    int                   rc;
    
    message_len = strlen(message);

    /*
     * get the size of a nexus_buffer necessary to send the message lenght (int)
     * and the message itself
     */
    buf_size =
	globus_nexus_sizeof_int(1) +     
	globus_nexus_sizeof_char(message_len);
    /*
     * Initialise the buffer
     */
    globus_nexus_buffer_init(&buffer,buf_size , 0);
    
    /* put the message lenght in the buffer */
    globus_nexus_put_int(&buffer, &message_len, 1);
    
    /* put the message in the buffer */
    globus_nexus_put_char(&buffer,
			  message,
			  message_len);     
    globus_libc_printf("%d: Message size to send %d\n",rank,message_len);

    /*
     * send the buffer
     * Note the handler_id GOT_MESSAGE_HANDLER_ID which will trigger the
     * corresponding function in the remote process
     */
    rc = globus_nexus_send_rsr(
	&buffer,
	&next_proc_sp,
	GOT_MESSAGE_HANDLER_ID,
	GLOBUS_TRUE,              /* please, destroy the buffer when sent   */
	is_nonthreaded_handler);            
    
    return(rc);

} /* send_msg_using_nexus() */

/******************************************************************************

  Function : got_message_handler()

  This is the function triggered when I receive the message from the previous
  process in the ring. It extracts the message from the buffer, and then
  forwards it to the next process in the ring.  
  
******************************************************************************/
static void
got_message_handler(globus_nexus_endpoint_t *endpoint,
		    globus_nexus_buffer_t *buffer,
		    globus_bool_t is_nonthreaded_handler)
{

    char * received;
    int    received_lenght;
    int    rc;
    
    globus_libc_printf("%d: Got a NEXUS remote service call\n",rank);

    /* extract the lenght of the message */
    globus_nexus_get_int(buffer, &received_lenght, 1);
    globus_libc_printf("%d: Message lenght : %d\n", rank, received_lenght);
    
    received = globus_malloc(received_lenght+1);
    if (received == GLOBUS_NULL)
    {
	globus_libc_fprintf(stderr,
			    "%d: ERROR : Could not allocate memory for incomming message\n",rank);
	
    }

    /* extract the message */
    globus_nexus_get_char(buffer, received, received_lenght);

    /*
     * Since I do not transmit the termination char,
     * I add it to print the message
     */
    received[received_lenght] = '\0';
    globus_libc_printf("%d: Message received: %s\n",rank,received);
    
    
    if (!i_am_the_first_job_master)
    {
	/* forward the message */
	/*
	 *  called_from_nonthread_handler is set to the value I get as input in
	 *  this handler: is_nonthreaded_handler
	 */
	if ( rc = send_msg_using_nexus(received, is_nonthreaded_handler)
	     != GLOBUS_SUCCESS)
	{
	    globus_libc_fprintf(
		stderr,
		"%d: ERROR Sending message using nexus, error code; %d!\n",
		rank,
		rc);
	}
    }
    else
    {
	if (strcmp(received, message))
	{
	    globus_libc_fprintf(
		stderr,
		"%d: ERROR Received incorrect message !\n",rank);
	}
	else
	{
	    globus_libc_fprintf(
		stdout,
		"%d: Received the correct message !\n",rank);
	}    
    }
    globus_free(received);
   

    /* signal the I have receive the message, and that the job can terminate */
    globus_mutex_lock(&mutex);
    {
	done = GLOBUS_TRUE;
	globus_cond_signal(&cond);
    }
    globus_mutex_unlock(&mutex);
 
    
}
/******************************************************************************

  Function : intra_job_send_sp(sp, next)

  Send a start point to the "next" process inside the subjob,
       using "globus_duroc_runtime".
       
  Important notice: Due to some limitation (buffer size) and some latency
                    overhead, "duroc" should not be used for general
		    communication. Use "duroc" only to establish the
		    communication, as in this example, by exchanging some
		    startpoint between the process of the job and create the
		    job topology. Afterward, use those startpoints and nexus
		    to communicate.

  In this function, we do not use a nexus buffer (created
  with globus_nexus_buffer_init() but a user buffer. We must therefor take care
  to:
  - Use the correct set of function (globus_nexus_user_*) when necessary.
  - Send the data format corresponding to our architachture, so the receiver
    can convert the data to its own format.

  Parameter:

  In:  sp   : startpointto send
       next : rank of the process to send the job to.
  
  
******************************************************************************/
void
intra_job_send_sp(globus_nexus_startpoint_t * sp,
	int                         next)
{
    globus_byte_t *        sp_buf = GLOBUS_NULL;
    int                    buf_size;
    globus_byte_t *        buf_pt;
    globus_byte_t          format;
    int                    rc;
    
    sp_buf = globus_malloc(GLOBUS_GRAM_MYJOB_MAX_BUFFER_LENGTH);
    if (sp_buf == GLOBUS_NULL)
    {
	globus_libc_fprintf(stderr,
			    "%d: malloc() failed.\n",
			    rank);
	globus_gram_myjob_kill();
    }
    
    /*
     * Get the size of a nexus_buffer necessary to send the data format 
     * specificto this architechture (see comment above) and the
     * startpoint itself.
     */
    buf_size =
	1 +
	globus_nexus_sizeof_startpoint(sp, 1);
    
    if (buf_size >= GLOBUS_GRAM_MYJOB_MAX_BUFFER_LENGTH)
    {
	globus_libc_fprintf(
	    stderr,
	    "%d: Startpoint too big to be transfered using myjob !\n",
	    rank);
	globus_gram_myjob_kill();
    }
    
    /* !! globus_nexus_user_put_startpoint_transfer()  will
     *    automatically increment its user_buffer argument 
     *    to point to the next unused byte in user_buffer:
     *    We must take care to NOT use directly sp_buff
     *    (or sp_buff will not point anymore to the allocated buffer
     *    and the programm wil crash whenwe call free()...)
     */
    buf_pt = sp_buf;

    /* put the format in the buffer */
    *buf_pt++ = (globus_byte_t) globus_nexus_user_format();
    
    /* put the startpoint */ 
    globus_nexus_user_put_startpoint_transfer(&buf_pt, sp, 1);   

    printf("%d: Send my startpoint to process %d in my subjob\n", rank, next);

    /*
    rc = globus_gram_myjob_send(next,
				sp_buf,
				buf_size);
				*/
    globus_duroc_runtime_intra_subjob_send(
	next,
	tag,
	buf_size,
	sp_buf);
    /*
    if ( rc != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(stderr,
			    "globus_gram_myjob_send() failed, rc=0x%08x.\n",
			    rc);
	globus_gram_myjob_kill();
    }
    */

    /* the buffer has been sent using "myjob": we can destroy it */
    globus_free(sp_buf);

} /* intra_job_send_sp() */

/******************************************************************************

  Function : inter_job_send_sp(sp, next)

  Send a start point to the "next" subjob,
       (to the process of rank 0 in the next subjob; also called the "master"
       of the next subjob) using "globus_duroc_runtime".

       This function is similar to intra_job_send_sp(): Please refere to
       the latter for mor detailed information.
 
******************************************************************************/
void
inter_job_send_sp(globus_nexus_startpoint_t * sp,
	int                         next)
{
    globus_byte_t *        sp_buf = GLOBUS_NULL;
    int                    buf_size;
    globus_byte_t *        buf_pt;
    globus_byte_t          format;
    int                    rc;
    
    sp_buf = globus_malloc(GLOBUS_GRAM_MYJOB_MAX_BUFFER_LENGTH);
    if (sp_buf == GLOBUS_NULL)
    {
	globus_libc_fprintf(stderr,
			    "%d: malloc() failed.\n",
			    rank);
	globus_gram_myjob_kill();
    }
    
    /*
     * Get the size of a nexus_buffer necessary to send the data format 
     * specificto this architechture (see comment above) and the startpoint
     * itself.
     */
    buf_size =
	1 +
	globus_nexus_sizeof_startpoint(sp, 1);
    
    if (buf_size >= GLOBUS_GRAM_MYJOB_MAX_BUFFER_LENGTH)
    {
	globus_libc_fprintf(
	    stderr,
	    "%d: Startpoint too big to be transfered using myjob !\n",
	    rank);
	globus_gram_myjob_kill();
    }
    
    /* !! globus_nexus_user_put_startpoint_transfer()  will
     *    automatically increment its user_buffer argument 
     *    to point to the next unused byte in user_buffer:
     *    We must take care to NOT use directly sp_buff
     *    (or sp_buff will not point anymore to the allocated buffer
     *    and the programm wil crash whenwe call free()...)
     */
    buf_pt = sp_buf;

    /* put the format in the buffer */
    *buf_pt++ = (globus_byte_t) globus_nexus_user_format();
    
    /* put the startpoint */ 
    globus_nexus_user_put_startpoint_transfer(&buf_pt, sp, 1);   

    printf("%d: INTER Send my startpoint to %d\n", rank, next);

    /*
    rc = globus_gram_myjob_send(next,
				sp_buf,
				buf_size);
				*/
    globus_duroc_runtime_inter_subjob_send(
	next,
	tag,
	buf_size,
	sp_buf);
    /*
    if ( rc != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(stderr,
			    "globus_gram_myjob_send() failed, rc=0x%08x.\n",
			    rc);
	globus_gram_myjob_kill();
    }
    */

    /* the buffer has been sent using "myjob": we can destroy it */
    globus_free(sp_buf);

} /* inter_job_send_sp() */

/******************************************************************************

  Function : intra_job_receive_sp(sp)
  
    Wait for an intra job  "duroc" message (message comming from a process
    inside the same subjob) and extract a start point out of the message.
    
    This function is just a wrapper around
    globus_duroc_runtime_intra_subjob_receive().
    
  Parameter:
     Out:
         sp: startpoint we received.
  
******************************************************************************/
void
intra_job_receive_sp(globus_nexus_startpoint_t *  sp)
{
    int msg_sz;
    int rc;
    globus_byte_t *     buf_pt;
    globus_byte_t       format;
    globus_byte_t *     sp_buf = GLOBUS_NULL;

    
    /* buffer used to receive the messages */
    /* Note: 
     * The "intra" subjob version of receive does NOT allocate the buffer,
     * (the inter subjob version does the allocation, see intra_job_receive())
     */
    sp_buf = globus_malloc(GLOBUS_GRAM_MYJOB_MAX_BUFFER_LENGTH);
    if (sp_buf == GLOBUS_NULL)
    {
	globus_libc_fprintf(stderr,
			    "%d: malloc() failed.\n",
			    rank);
	globus_gram_myjob_kill();
    }

    /* wait for the message */
    globus_duroc_runtime_intra_subjob_receive(
	tag,
	&msg_sz,				
	sp_buf);
    /*
    if ( rc != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(stderr,
			    "%d: globus_gram_myjob_receive() failed, "
			    "rc=0x%08x.\n",
			    rank,
			    rc);
	globus_gram_myjob_kill();
    }
    */
    
    globus_libc_fprintf(stdout, "%d: Received startpoint \n",
			rank);
    /*
     * Now extract the start point from the "myjob" buffer
     */
    buf_pt = sp_buf;
    /*
     * first extract the data format, since we are using the "user bufer"
     * functions, we have to take care of this.
     * You can notice that in send_msg_using_nexus() we did not used
     * the "user buffer" nexus function, and therefor we did not need to
     * worry about the data format: nexus take car of it
     */
    format = (int) *buf_pt++;
    /*
     * now use the format to extract/convert the startpoint
     */
    globus_nexus_user_get_startpoint(&buf_pt, sp, 1, format);

    globus_free(sp_buf);
    
} /* intra_job_receive_sp() */

/******************************************************************************

  Function : inter_job_receive_sp(sp)
  
    Wait for an inter job "duroc" message (message comming from the master
    of an other subjob) and extract a start point out of the message.
    
    This function is just a wrapper around
    globus_duroc_runtime_inter_subjob_receive().
    
  Parameter:
     Out:
         sp: startpoint we received.
  
******************************************************************************/
void
inter_job_receive_sp(globus_nexus_startpoint_t *  sp)
{
    int msg_sz;
    int rc;
    globus_byte_t *     buf_pt;
    globus_byte_t       format;
    globus_byte_t *     sp_buf = GLOBUS_NULL;

    /* Note: 
     * The "inter" subjob version of receive does the buffer allocation,
     * (not the intra version, see intra_job_receive())
     */
    globus_duroc_runtime_inter_subjob_receive(
	tag,
	&msg_sz,				
	&sp_buf);
    
    globus_libc_fprintf(stdout, "%d: INTER Received startpoint \n",
			rank);
    /*
     * Now extract the start point from the "myjob" buffer
     */
    buf_pt = sp_buf;
    /*
     * first extract the data format, since we are using the "user bufer"
     * functions, we have to take care of this.
     * You can notice that in send_msg_using_nexus() we did not used
     * the "user buffer" nexus function, and therefor we did not need to
     * worry about the data format: nexus take car of it
     */
    format = (int) *buf_pt++;
    /*
     * now use the format to extract/convert the startpoint
     */
    globus_nexus_user_get_startpoint(&buf_pt, sp, 1, format);

    globus_free(sp_buf);
    
} /* receive_sp() */

/******************************************************************************
  Main :

  It takes one argument, the string to send around the ring.
  (If the string contain any blank, it should be double quoted...)
******************************************************************************/
int
main(
    int					argc,
    char *				argv[])
{
    /* "general" purpose return code */
    int	rc;
    
    /*
     * parse the simple arguments
     */
    if (argc != 2)
    {
	globus_libc_printf(
	    "Usage : nexus_duroc_ring <message inside quotes>\n");
	exit(1);
    }
    message = argv[1];
    
    /***********************************************************************
     * Modules initialisation
     */
    
    /* This is mandatory in order to use "globus_nexus" */
    if ((rc = globus_module_activate(GLOBUS_NEXUS_MODULE)) != GLOBUS_SUCCESS)
    {
	globus_libc_printf(
	    "Module activation for \"globus_nexus\" failed, rc=0x%08x.\n", rc);
	exit(1);
    }
    
    /* This is mandatory in order to use "globus_myjob" */
    if ((rc = globus_module_activate(GLOBUS_GRAM_MYJOB_MODULE))
	!= GLOBUS_SUCCESS)
    {
	globus_libc_printf(
	    "Module activation for \"globus_myjob\" failed, rc=0x%08x.\n", rc);
	globus_gram_myjob_kill();
    }

    /* This is mandatory in order to use "globus_duroc_runtime" */
    if ((rc = globus_module_activate(GLOBUS_DUROC_RUNTIME_MODULE))
	!= GLOBUS_SUCCESS)
    {
	globus_libc_printf(
	    "Module activation for \"globus_duroc_runtime\" failed, rc=0x%08x.\n", rc);
	globus_gram_myjob_kill();
    }


    /***********************************************************************
     * When using duroc, in order to wait that all the job have been started
     * correctly, the first thing to do is always to wait at the barrier.
     */
     globus_duroc_runtime_barrier ();

    /***********************************************************************
     * Get informations concerning the subjob : Size of the subjob and my 
     * rank in the subjob
     */

    /*
     * Get the SIZE of the subjob this process is participating in
     */
    if ((rc = globus_duroc_runtime_intra_subjob_size(&size)) != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(stderr,
			    "globus_duroc_runtime_intra_subjob_size() failed, rc=0x%08x.\n",
			    rc);
	globus_gram_myjob_kill();
    }

    /*
     * Get the RANK of the job this process is participating in
     */
    if ((rc = globus_duroc_runtime_intra_subjob_rank(&rank)) != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(stderr,
			    "globus_duroc_runtime_intra_subjob_rank() failed, rc=0x%08x.\n",
			    rc);
	globus_gram_myjob_kill();
    }

    globus_libc_fprintf(
	stdout,
	"\n\tJob of rank %d \n\tTotal number of jobs (subjob size): %d\n",
	rank,
	size);

    /*
     * Only a master process (process of rank 0) can require the job structure
     */
    if (rank == 0)
    {
	int max_address;
	int min_address;
	int i;
	
	/******************************************************************
	 * Now I can ask for the number subjob in the job, my address in the
	 * job and the array containing the other job addresses.
	 */
	if ((rc = globus_duroc_runtime_inter_subjob_structure (
	    &own_subjob_address, 
	    &nb_remote_subjob,                    
	    &array_of_subjobs)) != GLOBUS_SUCCESS)
	{
	    globus_libc_fprintf(
		stderr,
		"globus_duroc_runtime_inter_subjob_structure() "\
		"failed, rc=0x%08x.\n",
		rc);
	    globus_gram_myjob_kill();
	}

	/* Since the address of all the submaster are not contiguous,
	 * I need to work a little to findout:
	 * - What is the address of the previous job master (I order jobs
	 *  from the one with the smallest address to the one with the biggest
	 *  address, although addresses are not contiguous)
	 * - If I am the first job master (I define the first job master
	 * as the job master with the smallest address.)
	 */
	max_address= own_subjob_address;
	min_address= own_subjob_address;
	previous_job_master_address= own_subjob_address;
	
	for (i=0; i< nb_remote_subjob; i++)
	{
	    if (array_of_subjobs[i] > max_address)
	    {
		max_address= array_of_subjobs[i];
	    }
	    if (array_of_subjobs[i] < min_address)
	    {
		min_address= array_of_subjobs[i];
	    }
	    if (array_of_subjobs[i] < own_subjob_address)
	    {
		if (previous_job_master_address== own_subjob_address )
		{
		    previous_job_master_address=array_of_subjobs[i];
		}
		else
		{
		    if (array_of_subjobs[i] > previous_job_master_address)
		    {
			previous_job_master_address = array_of_subjobs[i]; 
		    }
		}
	    }
	}   
	if (previous_job_master_address == own_subjob_address)
	{
	    previous_job_master_address = max_address;
	}
	if (own_subjob_address == min_address )
	{
	    i_am_the_first_job_master= GLOBUS_TRUE;
	}
	
	/*
	 * I have all the information I need to close the ring between all
	 * the subjobs, I can free this array:
	 */
	globus_free(array_of_subjobs);
		
	globus_libc_fprintf(
	    stdout,
	    "\n\tSubJob of address %d \n" \
	    "\tTotal number of Subjobs (including me): %d\n"\
	    "\t%s\n",
	    own_subjob_address,
	    nb_remote_subjob + 1,
	    i_am_the_first_job_master?
	    "And I am the first job master\n":
	    "And I am NOT the first job master\n");
    }
    
    /***********************************************************************
     * Nexus communication :
     */

    /*
     * Create an end point
     */
    globus_nexus_endpointattr_init(&epattr);
    globus_nexus_endpointattr_set_handler_table(&epattr,
						handler_table,
						HANDLER_TABLE_SIZE);
    globus_nexus_endpoint_init(&endpoint, &epattr);

    /*
     * Bind to this endpoint the startpoint we want to send to the previous
     * process in the job.
     */
    globus_nexus_startpoint_bind(&sp, &endpoint);

    /*
     * Initialise a conditional variable, and the mutex protecting the
     * conditional variable.
     */
    globus_mutex_init(&mutex, (globus_mutexattr_t *) NULL);
    globus_cond_init(&cond, (globus_condattr_t *) NULL);
    done = GLOBUS_FALSE;

    /*
     * Now the communication part is starting
     */
    
    /*
     * "communicate with other process only if there is
     * other process 
     */
    if (size == 1 && nb_remote_subjob == 0)
    {
	globus_libc_fprintf(
	    stderr,
	    "I am the only process in a uniq subjob: I have nobody to communicate with...abort\n");
	goto myend; /* avoid indentation... */
	
    }
    

    /***************************************************************
     * We must first setup the "nexus" ring inside each subjob
     ****************************************************************/
    if (size > 1) 
    {
	
	/*
	 * To get all the communication setup in parallel, I use an even/odd
	 * send/receive, and odd/even receive/send algorithm:
	 *
	 * - all the job with a rank even are sending their start point first
	 *   and then they wait-receive the start point from their "next" job
	 * - all the job with a rank odd first wait-receive a startpoint
	 *   from their "next" job, and then they send their start point
	 */
	if ((rank>>1)<<1 == rank) /* even job number */
	{
	    /*
	     * send a start point binded to my endpoint to the "previous
	     * process in the job, the process of rank "rank-1"
	     * (if I am the process of rank 0, the previous process is
	     * the last one in the job (rank "size-1") to close the ring. 
	     */
	    intra_job_send_sp(&sp, (rank==0)?size-1:rank-1);

	    /*
	     * wait until I receive a startpoint from the next process in the 
	     * job
	     */
	    intra_job_receive_sp(&next_proc_sp);    
	}
	else /* odd job number */
	{
	    /*
	     * wait until I receive a startpoint from the next process in the 
	     * job
	     */
	    intra_job_receive_sp(&next_proc_sp);
	    /*
	     * send a start point binded to my endpoint to the "previous
	     * process in the job
	     */	    
	    intra_job_send_sp(&sp, rank-1);
	}
    }
    else
    {
	next_proc_sp = sp;
    }
        
    /***************************************************************
     * Each sub job has established its own ring,
     * lets close the big ring, between subjobs.
     *
     */
    if ( nb_remote_subjob > 0 && rank == 0)
    {
	globus_libc_fprintf(
	    stdout,
	    "I am a master and there is other subjobs (\"small rings\"):"\
	    "I want to link them together to close the whole ring\n");

	/* since subjob addresses are not contiguous, it is not easy to
	 * use the same algorithm than for the small rings (odd/even)
	 *
	 * We use a simpler methode where the first job master send its
	 * startpoint first to its "previous job master".
	 *
	 * All the other job master wait to receive a startpoint.
	 *
	 * When a job master receive a startpoint, it send its own
	 * startpoint to its "previous job master" and so forth.
	 * Until the first job master, which is now waiting to receive
	 * a startpoint, does receive it. The "big ring" is then closed. 
	 * 
	 */  
	if ( i_am_the_first_job_master )
	{
	    
	    inter_job_send_sp(&next_proc_sp, previous_job_master_address);
	    
	    inter_job_receive_sp(&next_subjob_sp);

	    next_proc_sp = next_subjob_sp;
	}
	else 
	{	    
	    inter_job_receive_sp(&next_subjob_sp);

	    inter_job_send_sp(&next_proc_sp, previous_job_master_address);

	    next_proc_sp = next_subjob_sp;
	}
    }
    /*
     * else, if I am the only subjob, I do not need to do any thing, the
     * ring is already closed. if I am not rank 0 (not a job master) I do
     * nothing: the job master of my subjob does take care to close the ring.
     */


	
    /***************************************************************
     * The "nexus" ring is setup, we use it to circulate some data
     ***************************************************************/

    /*
     * We choose to start the communication with "the first job master"
     * because it is an easy choice, but it is not mandatory.
     * (you should take care to check correctly that the message has finished
     * to go around, if you use an other startpoint. (see how  this is
     * checked in got_message_handler())
     */
    if (i_am_the_first_job_master)
    {
	
	globus_libc_printf("Message to send around: %s\n",message);

	/*
	 *  Since I am not in a handler, called_from_nonthread_handler
	 *  must be false
	 */
	if ( rc = send_msg_using_nexus(message,GLOBUS_FALSE) != GLOBUS_SUCCESS)
	{
	    globus_libc_fprintf(
		stderr,
		"%d: ERROR Sending message using nexus, error code; %d!\n",
		rank,
		rc);
	}
	/* wait for the message to be passed around until it comes back
	 * to me, when the last process in the job will send it to me.
	 * This will trigger my handler: I do not need to do
	 * anything here but wait it is done (see globus_cond_wait below)
	 */
    }
    
    /*
     * if I am not the process of rank 0, I will get the message from
     * the previous process some time, and it will trigger my handler
     * wich will forward it to the next process: I do not need to do
     * anything here but wait it is done
     */
    
    /*
     * Wait the job is done: When my handler will have received
     * the message and forwarded it to the next process in the job, it
     * will signal "cond" and "wake-me-up"
     * (see handler got_message_handler() for details)
     */
    globus_mutex_lock(&mutex);
    {
	while (!done)
	{
	    globus_cond_wait(&cond, &mutex);
	}
    }
    globus_mutex_unlock(&mutex);
    /*
     * Here I know the job is done
     */
    
myend:
    /*
     * This process has received and send the message.
     * It has finished its task and can exit after some cleaning.
     */

    /*
     * Clean the end point
     */
    globus_nexus_endpoint_destroy(&endpoint);
    globus_nexus_endpointattr_destroy(&epattr);

    /*
     * clean the start points
     * Note that if I had used globus_nexus_put_startpoint_transfer()
     * and not the "user buffer" function, the local startpoint sp would have
     * been destroyed automatically when transferd in the send buffer
     */    
    globus_nexus_startpoint_destroy(&next_proc_sp);

    /*
     * Note : The transfert of the sp startpoint does destroy it:
     * we should Not destroy it again !
     *
     * Note also that when next_subjob_sp has been used to receive a remote
     * startpoint, next_subjob_sp has been assigned to next_proc_sp, and
     * the previous globus_nexus_startpoint_destroy destroyed both of them
     * (Well, they are actually a uniq startpoint that both of them refer)
     * 
     */
    
    /*
     * clean the conditionnal variable used for synchronisation
     */
    globus_mutex_destroy(&mutex);
    globus_cond_destroy(&cond);

    
    /*************************************************************************
     * Now we HAVE TO deactivate all the module we have used.
     */

    /* deactivate "globus_duroc_runtime" */
    if ((rc = globus_module_deactivate(GLOBUS_DUROC_RUNTIME_MODULE))
	 != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(
	    stderr,
	    "Module activation for \"globus_duroc_runtime\" failed, rc=0x%08x.\n", rc);
	globus_gram_myjob_kill();
    }
    /* deactivate "globus_myjob" */
    if ((rc = globus_module_deactivate(GLOBUS_GRAM_MYJOB_MODULE))
	 != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(
	    stderr,
	    "Module activation for \"globus_myjob\" failed, rc=0x%08x.\n", rc);
	globus_gram_myjob_kill();
    }
    /* deactivate "globus_nexus" */
    if ((rc = globus_module_deactivate(GLOBUS_NEXUS_MODULE))
	 != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(stderr,
	    "Module activation for \"globus_nexus\" failed, rc=0x%08x.\n", rc);
    }

    /* say ByeBye... */
    globus_libc_fprintf(stdout, "%d: test complete.\n", rank);

    return(0);
}




