/**************************************************************************
*
* Program:      Nexus-Talk Server
*               talk_server.c
*
* Written at:   High-Performance Computing Laboratory
*               Department of Computer Science
*               Northern Illinois University
*               DeKalb, IL
*
* Authors:      Robert Bach
*               John Bresnahan
*
* Date Written: 2/97
*
* Description:
* 
*     This is the Nexus version of the Unix "talk" application.  As such, it
*     is a curses-based program.  It is intended to be an example of how
*     to use the Nexus "attach" functionality in a client/server application.
* 
*     The server in this example is designed to service exactly one client.
*     Once that connection has been established, no other clients may
*     attach to this server.  Further, once the client/server connection
*     has been established and the session runs to completion, upon
*     disconnecting BOTH the server and the client terminate.  
*     
*     This is unlike traditional client/server applications where a server
*     my service a number of clients simultaneously and where the server
*     does not terminate after servicing exactly one client.  But, this
*     is an example whose purpose is to demonstrate the Nexus attach and
*     detach functionality.  Extending this example to allow multiple 
*     clients and establish an omnipresent server should be straightforward.
*
*     Attachment in Nexus is performed by one process "allowing attachments",
*     globus_nexus_allow_attach(). This returns a <hostname> and <port> from 
*     which we can construct a URL, an attach ULR. For example, the attach URL 
*     constructed by a hostname=olympus and port=44431 would look like this:
*
*                     x-nexus://olympus:44431/
*     
*     It is this attach URL that the other process must pass (as a character
*     string) to globus_nexus_attach() when making a request to attach to us.
*     When such a request is made, our attachment approval function,
*     attach_requested() is automatically (asynchronously) executed.  It
*     is that routine where we decide whether or not to allow attachment.
*     If permission is granted, we give the requesting process a startpoint
*     to ourself.  Note that we do not get a startpoint to the process
*     we grant attach permission to.  If they want to give us a startpoint
*     they can do so by using the startpoint we just gave them.
* 
*     We can also stop listening for attachments by calling 
*     globus_nexus_disallow_attach().
*
* To run this program:
* 
*     talk_server
*
*     The talk session is terminated when either side enters control-x.
* 
* Output:
*
*     A curses-based talk session.
*
**************************************************************************/

#include <stdio.h>
#include <string.h>
#include <globus_common.h>
#include <globus_nexus.h>
#include <curses.h>
#include "display.h"

/***************************/
/*                         */
/* Nexus Handler Functions */
/*                         */
/***************************/

static void attach_handler(
    globus_nexus_endpoint_t * endpoint, 
    globus_nexus_buffer_t   * buffer, 
    globus_bool_t             is_non_threaded_handler);
static void receive_char_handler(
    globus_nexus_endpoint_t * endpoint, 
    globus_nexus_buffer_t   * buffer,
    globus_bool_t             is_non_threaded_handler);

/*********************************/
/*                               */
/* Nexus Handler Functions Table */
/*                               */
/*********************************/

#define ATTACH_HANDLER_ID       0
#define RECEIVE_CHAR_HANDLER_ID 1

static globus_nexus_handler_t handlers[] = 
{ 
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED, attach_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED, receive_char_handler}
};

/*******************/
/*                 */
/* Data Structures */
/*                 */
/*******************/

/* Monitor                                                 */
/*                                                         */
/* we can simulate a monitor by defining a struct that has */
/* at least a globus_mutex_t to provide mutually exclusive  */
/* access to the monitor.                                  */
/* in this application we also need a monitor condition    */
/* variable and two flags.                                 */

typedef struct
{
    /* for mutually exclusive access to monitor */
    globus_mutex_t mutex;

    /* a monitor condition variable and flag */
    globus_cond_t  cond;
    globus_bool_t  done;
    globus_bool_t  connection_established;
} monitor_t;

/********************/
/*                  */
/* Global Variables */
/*                  */
/********************/

static globus_nexus_endpointattr_t	EpAttr;
static globus_nexus_endpoint_t 	        GlobalEndpoint;

static globus_nexus_startpoint_t	TalkClientSP;
static monitor_t		        Monitor;
static globus_bool_t		        RemoteKill;

static struct UserScreen 	        User1;
static struct UserScreen 	        User2;

/*******************/
/*                 */
/* Local Functions */
/*                 */
/*******************/

static int attach_requested(void *arg,
			    char *url,
			    globus_nexus_startpoint_t *sp);
static void allow_attachments(unsigned short *attach_port,
			      char *attach_url);
static globus_bool_t write_attach_file(char *attach_url);


/*
 * abort_talk()
 * Desactivate Nexus module and exit with error
 * Desactivation of modules before exiting is mandatory.
 */
void abort_talk()
{
    if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE)
	 != GLOBUS_SUCCESS )
    {
	globus_libc_fprintf(
	    stderr,
	    "Nexus Module DEactivation failed \n");
    }
    exit(1);
} /* abort_talk() */

/********/
/*      */
/* MAIN */
/*      */
/********/

int main(int argc, char **argv)
{
    int i, ch;
    globus_nexus_buffer_t buffer;
    char attach_url[1024];
    unsigned short attach_port;

    /**************************************************/
    /* Activation of the Nexus module is required at  */
    /* the beginning of all Nexus programs            */
    /**************************************************/

    if ( globus_module_activate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	fprintf(stderr,"Nexus Module activation failed \n");
	exit(1);
    }

    /******************/
    /* Initialization */
    /******************/

    /* initializing an endpointattr */
    globus_nexus_endpointattr_init(&EpAttr);
    globus_nexus_endpointattr_set_handler_table(&EpAttr, 
				    handlers, 
				    sizeof(handlers)/sizeof(globus_nexus_handler_t));

    /* initializing endpoint ... no particular address */
    globus_nexus_endpoint_init(&GlobalEndpoint, &EpAttr);
    
    /**********************************************************/
    /* Initialize the monitor that will be used for signaling */
    /* that the talk_client has successfully attached to us.  */
    /**********************************************************/

    /* initializing the mutex that controls access to the monitor */
    globus_mutex_init(&(Monitor.mutex), (globus_mutexattr_t *) GLOBUS_NULL);
    /* initializing the condition variable */
    globus_cond_init(&(Monitor.cond), (globus_condattr_t *) GLOBUS_NULL);

    /* entering the monitor and clearing the flags */
    globus_mutex_lock(&(Monitor.mutex));
    Monitor.done = GLOBUS_FALSE;
    Monitor.connection_established = GLOBUS_FALSE;
    globus_mutex_unlock(&(Monitor.mutex));

    /* initializing remote kill flag */
    RemoteKill = GLOBUS_FALSE;

    /***********************/
    /* allowing attachment */
    /***********************/

    /* allow other Nexus programs to attach to us */
    allow_attachments(&attach_port, attach_url);
        
    /* advertising URL for attachment in URL_FILE */
    if (!write_attach_file(attach_url))
    {
	globus_libc_fprintf(stderr,
			    "ERROR: Could not write file: %s\n", URL_FILE);
	globus_libc_fprintf(stderr,
			    "File may already exist.\n");
	abort_talk();
    } /* endif */

    InitScreen();
    IntroScreen(attach_url);

    /**************************************************/
    /* wait (using monitor) for talk_client to attach */
    /**************************************************/

    /* the implementation of globus_cond_wait() and globus_cond_signal()     */
    /* makes it possible for globus_cond_wait() to experience a 'false      */
    /* wakeup', i.e., return without having had a globus_cond_signal()      */
    /* applied to it.                                                      */
    /*                                                                     */
    /* this is why we must wait on a condition variable in the manner      */
    /* we do below, with a loop and a globus_bool_t done, rather than       */
    /* simply:                                                             */
    /*     globus_mutex_lock(&(Monitor.mutex));                             */
    /*     globus_cond_wait(&(Monitor.cond), &(Monitor.mutex));             */
    /*     globus_mutex_unlock(&(Monitor.mutex));                           */

    globus_mutex_lock(&(Monitor.mutex));
    while (!(Monitor.done))
    {
	globus_cond_wait(&(Monitor.cond), &(Monitor.mutex));
    } /* endwhile */
    globus_mutex_unlock(&(Monitor.mutex));

    /* at this point, flag in monitor has been set   */
    /* ... the talk_client has successfully attached */

    /******************************************************/
    /* talk session --- ends when either side hits cntl-X */
    /******************************************************/

    StartTalk(&User1, &User2, attach_url);
    
    ch = 0;
    while (ch != CTRL_X && !RemoteKill)
    {
	ch = getchar();
	fflush(stdin);
	if(!RemoteKill)
	{ 
	    /* displaying single character on screen */
	    AddChar(&User1, ch);

	    /* sending single character to talk_client */
	    globus_nexus_buffer_init(&buffer, globus_nexus_sizeof_int(1), 0);
	    globus_nexus_put_int(&buffer, &ch, 1);
	    globus_nexus_send_rsr(&buffer, 
			&TalkClientSP, 
			RECEIVE_CHAR_HANDLER_ID, 
			GLOBUS_TRUE, 
			GLOBUS_FALSE);
	} /* endif */
    } /* endwhile */

    /************/ 
    /* Shutdown */
    /************/ 

    endwin();
    ThankYou();

    globus_nexus_disallow_attach(attach_port);

    /********************/ 
    /* Initial Clean-up */
    /********************/ 

    if (unlink(URL_FILE))
       globus_libc_printf("\nError unlinking file\n");

    globus_nexus_startpoint_destroy(&TalkClientSP);

    globus_mutex_destroy(&(Monitor.mutex));
    globus_cond_destroy(&(Monitor.cond));

    /*********************/
    /* Standard Clean-up */
    /*********************/

    globus_nexus_endpoint_destroy(&GlobalEndpoint);
    globus_nexus_endpointattr_destroy(&EpAttr);

    if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	exit(1);
    }

    exit(0);

} /* end main() */

/*******************/
/*                 */
/* Local Functions */
/*                 */
/*******************/

/*
 * attach_requested()
 *
 * This is the attachment approval function.  When the server started
 * allowing attachments (nexus_allow_attach()), it specified this approval 
 * function.
 *
 * This function gets automatically (asynchronously) called when a
 * client calls globus_nexus_attach() specifying this server's URL.
 *
 * The job of all attachment approval functions is to decide whether to
 * allow the requesting process to "attach" to this one.  If the 
 * attachment is granted, this function must bind the startpoint
 * passsed as the third arg to a local endpoint and return 0.
 * If attachment is denied, this function simply returns a non-zero value.
 *
 * In our example program we have decided to allow at most one connection
 * at a time.  In a more traditional client/server application where the
 * server might service multiple clients simultaneously, the logic for
 * this routine should be modified in the obvious way.
 *
 */
static int
attach_requested(void *arg, char *url, globus_nexus_startpoint_t *sp)
{
     int rc;

    globus_mutex_lock(&(Monitor.mutex));
    if (Monitor.connection_established)
    {
	/* another talk_client has already connected to us */
	rc = 1;
    }
    else
    {
	/* this is the first talk_client */
	rc = GLOBUS_SUCCESS;
	Monitor.connection_established = GLOBUS_TRUE;
    } /* endif */
    globus_mutex_unlock(&(Monitor.mutex));

     /* binding received startpoint to one of our endpoints */
     if (rc == GLOBUS_SUCCESS)
     {
	 globus_nexus_startpoint_bind(sp, &GlobalEndpoint);
     } /* endif */

     return rc;

} /* end attach_requested() */

/*
 * allow_attachments()
 *
 * By calling globus_nexus_allow_attach() we allow other Nexus programs
 * to attach to us.  globus_nexus_allow_attach() tells us the hostname (attach_host)
 * and port number (attach_port) that it will start listening for attachments.
 * We use attach_host and attach_port to construct a URL that attach 
 * clients need to use to attach to us.
 *
 * In calling globus_nexus_allow_attach() we also name an approval function, in
 * this case attach_requested(), which is a user-supplied function that
 * gets called every time a client requests an attachment to us.
 *
 */
static void allow_attachments(unsigned short *attach_port, char *attach_url)
{
    int rc;
    char *attach_host;

    /* allow other Nexus programs to attach to us */
    *attach_port = 0;
    rc = globus_nexus_allow_attach(
	attach_port,      /* port            */
	&attach_host,     /* host            */
	attach_requested, /* approval_func() */
	GLOBUS_NULL);
    if (rc != GLOBUS_SUCCESS)
    {
       /* libc functions are usually not re-entrant. Re-entrant functions   */
       /* are provided. They perform the necessary locks. But be award of   */
       /* potential dead-lock if miss-used.                                 */
       /* The  functions globus_libc_lock() and globus_libc_unlock() can    */
       /* also be used arround standard libc function                       */
       globus_libc_fprintf(
	   stderr,

	   "ERROR: globus_nexus_allow_attach() failed: rc=%d\n",
	   rc);
       if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
       {
	   globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
       }
       exit(1);
    } /* endif */

    globus_libc_sprintf(attach_url,
			"x-nexus://%s:%hu/",
			attach_host,
			*attach_port);

} /* end allow_attachments() */

/*
 * write_attach_file()
 */
static globus_bool_t write_attach_file(char *attach_url)
{
    FILE *fp;
    globus_bool_t rc;

    /* libc functions are usually not re-entrant. Re-entrant functions   */
    /* are provided. They perform the necessary locks. But be award of   */
    /* potential dead-lock if miss-used.                                 */
    /* The  functions globus_libc_lock() and globus_libc_unlock() can    */
    /* also be used arround standard libc function                       */
    globus_libc_lock();
    {
	if ((fp = fopen(URL_FILE, "w")) == (FILE *) GLOBUS_NULL)
	    rc = GLOBUS_FALSE;
	else
	{
	    rc = GLOBUS_TRUE;
	    
	    fprintf(fp, "%s\n", attach_url);
	    
	    fclose(fp);
	} /* endif */
    }
    globus_libc_unlock();

    return rc;

} /* end write_attach_file() */

/***************************/
/*                         */
/* Nexus Handler Functions */
/*                         */
/***************************/

/*
 * attach_handler()
 */
static void
attach_handler(globus_nexus_endpoint_t * endpoint, 
	       globus_nexus_buffer_t   * buffer, 
	       globus_bool_t             is_non_threaded_handler)
{
    globus_nexus_buffer_t reply_buffer;

    /* extracting talk_client startpoint from buffer */
    globus_nexus_get_startpoint(buffer, &TalkClientSP, 1);

    /* replying to talk_client with empty message */
    /* ACK, invoking reply_handler() there.       */
    globus_nexus_buffer_init(&reply_buffer, 0, 0);
    globus_nexus_send_rsr(&reply_buffer, 
			  &TalkClientSP, 
			  ATTACH_HANDLER_ID, 
			  GLOBUS_TRUE, 
			  GLOBUS_FALSE);

    /* clean-up */
    globus_nexus_buffer_destroy(buffer);

    /* the implementation of globus_cond_wait() and globus_cond_signal()  */
    /* makes it possible for globus_cond_wait() to experience a 'false   */
    /* wakeup', i.e., return without having had a globus_cond_signal()   */
    /* applied to it.                                                   */
    /*                                                                  */
    /* this is why we must signal a condition variable in the manner    */
    /* we do below, with a globus_bool_t done, rather than simply:       */
    /*     globus_mutex_lock(&(Monitor.mutex));                          */
    /*     globus_cond_signal(&(Monitor.cond));                          */
    /*     globus_mutex_unlock(&(Monitor.mutex));                        */

    globus_mutex_lock(&(Monitor.mutex));
    Monitor.done = GLOBUS_TRUE;
    globus_cond_signal(&(Monitor.cond));
    globus_mutex_unlock(&(Monitor.mutex));      

} /* end attach_handler() */

/*
 * receive_char_handler()
 */
static void
receive_char_handler(globus_nexus_endpoint_t * endpoint, 
		     globus_nexus_buffer_t   * buffer,
		     globus_bool_t             is_non_threaded_handler)
{
    int ch;

    /* extracting character from buffer */
    globus_nexus_get_int(buffer, &ch, 1);  

    if (ch == CTRL_X)
    { 
	/* remote user hit ctrl-X */
	RemoteKill = GLOBUS_TRUE;
	PartyLeft(&User1); 
    }
    else
	/* writing character to screen */
	AddChar(&User2, ch);

    /* clean-up */
    globus_nexus_buffer_destroy(buffer);

} /* end receive_char_handler() */


