/**************************************************************************
*
* Program:      Nexus-Talk Client
*               talk_client.c
*
* Written at:   High-Performance Computing Laboratory
*               Department of Computer Science
*               Northern Illinois University
*               DeKalb, IL
*
* Authors:      Robert Bach
*               John Bresnahan
*
* Date Written: 2/97
*
* Description:
*
*     This is the Nexus version of the Unix "talk" application.  As such, it
*     is a curses-based program.  It is intended to be an example of how
*     to use the Nexus "attach" functionality in a client/server application.
* 
*     This is the talk client application.  Before running the client the 
*     talk server must be started first.  The "welcome" screen of the
*     talk server displays the attach URL that clients wishing to attach
*     to it must use.
*
*     Once the talk client has that URL, it uses it to ask the talk server
*     for a connection using globus_nexus_attach().  The talk server may grant
*     the request, in which case globus_nexus_attach() returns with 0 and we
*     get a startpoint that points back to the server.  If the request
*     is denied, globus_nexus_attach() returns with a non-zero value.
*
*     Note that if the request is granted, the client has a startpoint
*     that points back to the server.  The server does not have a startpoint
*     that points back to the client.  If the client wishes the server
*     to have a startpoint that points back to the client, it may use
*     the startpoint it just received that points to the server to send
*     a startpoint to the server.
*
*     This program needs the attach URL generated by the talk server.
*     When the talk server starts executing it not only displays the
*     attach URL on its "welcome" screen, but also writes it to
*     /tmp/talk_url as well.  If the talk client can access that same
*     file, then it will.  If the talk client cannot access that file
*     (e.g., it's being executed on a remote machine), then the user
*     may pass the attach URL to the talk client as a command line argument.
*
* To run this program:
*
*         If the talk client has access to /tmp/talk_url created 
*         by the talk server.
*
*             talk_client
*
*     or
*
*         If the talk client does not have access to /tmp/talk_url
*         created by the talk server.  Then the attach URL displayed
*         on the talk server "welcome" screen must be passed as a
*         command line argument.  For example, if the attach URL
*         is "x-nexus://olympus:44431/"
*
*             talk_client -a x-nexus://olympus:44431/
*
*     The talk session is terminated when either side enters control-x.
*
* Output:
*
*     A curses-based talk session.
*
**************************************************************************/

#include <stdio.h>
#include <string.h>
#include <globus_common.h>
#include <globus_nexus.h>
#include <curses.h>
#include "display.h"

/***************************/
/*                         */
/* Nexus Handler Functions */
/*                         */
/***************************/

static void reply_handler(
    globus_nexus_endpoint_t * endpoint, 
    globus_nexus_buffer_t   * buffer,
    globus_bool_t             is_non_threaded_handler);
static void receive_char_handler(
    globus_nexus_endpoint_t * endpoint, 
    globus_nexus_buffer_t   * buffer, 
    globus_bool_t             is_non_threaded_handler);

/*********************************/
/*                               */
/* Nexus Handler Functions Table */
/*                               */
/*********************************/

#define REPLY_HANDLER_ID        0
#define RECEIVE_CHAR_HANDLER_ID 1

static globus_nexus_handler_t handlers[] = 
{ 
    {NEXUS_HANDLER_TYPE_NON_THREADED, reply_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, receive_char_handler}
};

/*******************/
/*                 */
/* Data Structures */
/*                 */
/*******************/

/* Monitor                                                 */
/*                                                         */
/* we can simulate a monitor by defining a struct that has */
/* at least a globus_mutex_t to provide mutually exclusive */
/* access to the monitor.                                  */
/* in this application we also need a monitor condition    */
/* variable and a flag.                                    */

typedef struct
{
    /* for mutually exclusive access to monitor */
    globus_mutex_t mutex;

    /* a monitor condition variable and flag */
    globus_cond_t cond;
    globus_bool_t done;
} monitor_t;

/********************/
/*                  */
/* Global Variables */
/*                  */
/********************/

static globus_nexus_endpointattr_t	EpAttr;
static globus_nexus_endpoint_t		GlobalEndpoint;

static globus_nexus_startpoint_t	TalkServerSP;
static monitor_t		        Monitor;
globus_bool_t			        RemoteKill;

static struct UserScreen 	        User1;
static struct UserScreen 	        User2;

/*******************/
/*                 */
/* Local Functions */
/*                 */
/*******************/

static void attach_to_server(char *attach_url);
static globus_bool_t read_attach_file(char *url, char *file);

/*
 * abort_talk()
 * Desactivate Nexus module and exit with error
 * Desactivation of modules before exiting is mandatory.
 */
void abort_talk()
{
    if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE)
	 != GLOBUS_SUCCESS )
    {
	globus_libc_fprintf(
	    stderr,
	    "Nexus Module DEactivation failed \n");
    }
    exit(1);
} /* abort_talk() */


/********/
/*      */
/* MAIN */
/*      */
/********/

int main(int argc, char **argv)
{
    globus_nexus_startpoint_t my_sp;
    int i, ch;
    globus_nexus_buffer_t buffer;
    char attach_url[1024];

    /**************************************************/
    /* Activation of the Nexus module is required at  */
    /* the beginning of all Nexus programs            */
    /**************************************************/

    if ( globus_module_activate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	fprintf(stderr,"Nexus Module activation failed \n");
	exit(1);
    }

    /******************/
    /* Initialization */
    /******************/
    /* initializing an endpointattr */
    globus_nexus_endpointattr_init(&EpAttr);
    globus_nexus_endpointattr_set_handler_table(&EpAttr, 
				    handlers, 
				    sizeof(handlers)/sizeof(globus_nexus_handler_t));

    /* initializing endpoint ... no particular address */
    globus_nexus_endpoint_init(&GlobalEndpoint, &EpAttr);
 
    /* getting attach_url */

    /* checking for optional [-a attach_url] */
    if (argc > 1)
    {
	if (argc != 3 || strcmp(argv[1], "-a"))
	{
	/* libc functions are usually not re-entrant. Re-entrant functions   */
	/* are provided. They perform the necessary locks. But be award of   */
	/* potential dead-lock if miss-used.                                 */
	/* The  functions globus_libc_lock() and globus_libc_unlock() can    */
	/* also be used arround standard libc function                       */
	    globus_libc_fprintf(
		stderr,
		"usage: %s [-a attach_url]\n",
		argv[0]);
	    abort_talk();
	}
	else
	{
	    /* optional [-a attach_url] supplied */
	    strcpy(attach_url, argv[2]);
	} /* endif */
    } 
    else
    {
	/* no command line args ... must get attach_url from URL_FILE */
	if (!read_attach_file(attach_url, URL_FILE))
	{
	    globus_libc_fprintf(
		stderr, 
		"ERROR: Could not read attach_url from file: %s\n", 
		URL_FILE);
	    globus_libc_fprintf(
		stderr, 
		"talk_server may not be running on this file system.\n");
	    globus_libc_fprintf(
		stderr, 
		"If you know your party's attach_url type: %s -a attach_url\n",
		argv[0]);
	    globus_libc_fprintf(
		stderr,
		"usage: %s <attach_url> <int>\n",
		argv[0]);
	    abort_talk();
	    exit(1);
	} /* endif */
    } /* endif */

    /* initializing remote kill flag */
    RemoteKill = GLOBUS_FALSE;

    /* Setup monitor that I'll suspend on */
    globus_mutex_init(&(Monitor.mutex), (globus_mutexattr_t *) GLOBUS_NULL);
    globus_cond_init(&(Monitor.cond), (globus_condattr_t *) GLOBUS_NULL);

    /* entering the monitor and clearing the flag */
    globus_mutex_lock(&(Monitor.mutex));
    Monitor.done = GLOBUS_FALSE;
    globus_mutex_unlock(&(Monitor.mutex));

    /*************************/
    /* attach to talk_server */
    /*************************/

    attach_to_server(attach_url);

    /* I now have a startpoint to the talk_server, TalkServerSP */

    /*********************************/
    /* send talk_server a startpoint */
    /* to myself and wait for ACK    */
    /*********************************/

    globus_nexus_startpoint_bind(&my_sp, &GlobalEndpoint);

    globus_nexus_buffer_init(&buffer, globus_nexus_sizeof_startpoint(&my_sp, 1), 0);
    globus_nexus_put_startpoint_transfer(&buffer, &my_sp, 1);

    /* sending startpoint to talk_server and invoking attach_handler() there */
    globus_nexus_send_rsr(&buffer, 
		    &TalkServerSP, 
		    REPLY_HANDLER_ID, 
		    GLOBUS_TRUE, 
		    GLOBUS_FALSE);

    /* waiting for ACK */

    /* the implementation of globus_cond_wait() and globus_cond_signal()    */
    /* makes it possible for globus_cond_wait() to experience a 'false      */
    /* wakeup', i.e., return without having had a globus_cond_signal()      */
    /* applied to it.                                                       */
    /*                                                                      */
    /* this is why we must wait on a condition variable in the manner       */
    /* we do below, with a loop and a globus_bool_t done, rather than       */
    /* simply:                                                              */
    /*     globus_mutex_lock(&(Monitor.mutex));                             */
    /*     globus_cond_wait(&(Monitor.cond), &(Monitor.mutex));             */
    /*     globus_mutex_unlock(&(Monitor.mutex));                           */

    globus_mutex_lock(&(Monitor.mutex));
    while (!(Monitor.done))
    {
       globus_cond_wait(&(Monitor.cond), &(Monitor.mutex));
    } /* endwhile */
    globus_mutex_unlock(&(Monitor.mutex));

    /* at this point, flag in monitor has been set      */
    /* ... we have successfully attached to talk_server */
    /* and have sent him a startpoint to ourself        */

    /******************************************************/
    /* talk session --- ends when either side hits cntl-X */
    /******************************************************/

    InitScreen();
    StartTalk(&User1, &User2, attach_url);
  
    ch = 0;
    while (ch != CTRL_X && !RemoteKill)
    {
	ch = getchar();
	fflush(stdin);
	if(!RemoteKill)
	{ 
	    /* displaying single character on screen */
	    AddChar(&User1, ch);

            /* sending single character to talk_server */
	    globus_nexus_buffer_init(&buffer, globus_nexus_sizeof_int(1), 0);
	    globus_nexus_put_int(&buffer, &ch, 1);
	    globus_nexus_send_rsr(&buffer, 
			&TalkServerSP, 
			RECEIVE_CHAR_HANDLER_ID, 
			GLOBUS_TRUE, 
			GLOBUS_FALSE);
	} /* endif */
    } /* endwhile */

    /************/ 
    /* Shutdown */
    /************/ 

    endwin();
    ThankYou();

    /********************/ 
    /* Initial Clean-up */
    /********************/ 

    globus_nexus_startpoint_destroy(&TalkServerSP);

    globus_mutex_destroy(&(Monitor.mutex));
    globus_cond_destroy(&(Monitor.cond));

    /*********************/
    /* Standard Clean-up */
    /*********************/

    globus_nexus_endpoint_destroy(&GlobalEndpoint);
    globus_nexus_endpointattr_destroy(&EpAttr);
    
    if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	exit(1);
    }

    exit(0);

} /* end main() */

/*******************/
/*                 */
/* Local Functions */
/*                 */
/*******************/

/*
 * attach_to_server()
 *
 * Attachment in Nexus is done by making a request to attach to another
 * process designated by a URL.  If the request is granted, we get a 0
 * return code and a startpoint that points back to the granting process.
 *
 * NOTE: We get a startpoint to the process we just attached to, they
 *       do not get a startpoint back to us.  If we need them to have
 *       a startpoint back to us, we need to send them one using the
 *       startpoint we just received ... the one that points back to them.
 *
 * If the request is denied, we get a non-zero return code back.
 *
 */
static void attach_to_server(char *attach_url)
{
    int rc;

    rc = globus_nexus_attach(attach_url, &TalkServerSP);

    /* if things went OK, we now have a startpoint */
    /* to the talk_server, TalkServerSP            */

    if (rc != GLOBUS_SUCCESS)
    {
	globus_libc_fprintf(
	    stderr, 
	    "ERROR: Could not attach to server using url: %s rc %d\n", 
	    attach_url, rc);
	abort_talk();
    } /* endif */

} /* end attach_to_server() */

/*
 * read_attach_file()
 */
static globus_bool_t read_attach_file(char *attach_url, char *file)
{
    FILE *fp;
    globus_bool_t rc;

    /* libc functions are usually not re-entrant. Re-entrant functions   */
    /* are provided. They perform the necessary locks. But be award of   */
    /* potential dead-lock if miss-used.                                 */
    /* The  functions globus_libc_lock() and globus_libc_unlock() can    */
    /* also be used arround standard libc function                       */
    globus_libc_lock();
    {
	if ((fp = fopen(file, "r")) == (FILE *) GLOBUS_NULL)
	    rc = GLOBUS_FALSE;
	else
	{
	    rc = GLOBUS_TRUE;
	    
	    fscanf(fp, "%s", attach_url);

	    fclose(fp);
	} /* endif */
    }
    globus_libc_unlock();

    return rc;

} /* end read_attach_file() */

/***************************/
/*                         */
/* Nexus Handler Functions */
/*                         */
/***************************/

/*
 * reply_handler()
 */
static void reply_handler(globus_nexus_endpoint_t *endpoint, 
			  globus_nexus_buffer_t *buffer,
			  globus_bool_t is_non_threaded_handler)
{
    /* clean-up */
    globus_nexus_buffer_destroy(buffer);

    /* the implementation of globus_cond_wait() and globus_cond_signal()  */
    /* makes it possible for globus_cond_wait() to experience a 'false   */
    /* wakeup', i.e., return without having had a globus_cond_signal()   */
    /* applied to it.                                                   */
    /*                                                                  */
    /* this is why we must signal a condition variable in the manner    */
    /* we do below, with a globus_bool_t done, rather than simply:       */
    /*     globus_mutex_lock(&(Monitor.mutex));                          */
    /*     globus_cond_signal(&(Monitor.cond));                          */
    /*     globus_mutex_unlock(&(Monitor.mutex));                        */

    globus_mutex_lock(&(Monitor.mutex));
    Monitor.done = GLOBUS_TRUE;
    globus_cond_signal(&(Monitor.cond));
    globus_mutex_unlock(&(Monitor.mutex));   

} /* end reply_handler() */

/*
 * receive_char_handler()
 */
static void receive_char_handler(
    globus_nexus_endpoint_t * endpoint, 
    globus_nexus_buffer_t   * buffer, 
    globus_bool_t             is_non_threaded_handler)
{
    int ch;

    /* extracting character from buffer */
    globus_nexus_get_int(buffer, &ch, 1);

    if (ch == CTRL_X) 
    {
        /* remote user hit ctrl-X */
	RemoteKill = GLOBUS_TRUE;
	PartyLeft(&User1);
    }
    else
        /* writing character to screen */
	AddChar(&User2, ch);

    /* clean-up */
    globus_nexus_buffer_destroy(buffer);

} /* end receive_char_handler() */






