
/********************************************************************
 *
 * This file implements thread-safe utility routines
 *
 ********************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <assert.h>

#include "globus_duroc_common.h"
#include "globus_common.h"
#include "globus_gram_client.h"

#include "nexus.h"

#include "utils.h"

int
globus_duroc_error_is_gram_client_error (int error_code)
{
  return ( error_code >= GLOBUS_DUROC_ERROR_GRAM_ERROR_BASE );
}

int
globus_duroc_error_get_gram_client_error (int error_code)
{
  return ( error_code - GLOBUS_DUROC_ERROR_GRAM_ERROR_BASE );
}

const char *
globus_duroc_error_string (int err)
{
  const char * result;

  switch (err) {
  case GLOBUS_DUROC_SUCCESS: 
    result = "GLOBUS_SUCCESS"; break;
  case GLOBUS_DUROC_ERROR_INVALID_REQUEST: 
    result = "GLOBUS_DUROC_ERROR_INVALID_REQUEST"; break;
  case GLOBUS_DUROC_ERROR_INVALID_PARAMETER: 
    result = "GLOBUS_DUROC_ERROR_INVALID_PARAMETER"; break;
  case GLOBUS_DUROC_ERROR_DUCT_FAILED: 
    result = "GLOBUS_DUROC_ERROR_DUCT_FAILED"; break;
  case GLOBUS_DUROC_ERROR_INIT_FAILED: 
    result = "GLOBUS_DUROC_ERROR_INIT_FAILED"; break;
  case GLOBUS_DUROC_ERROR_UNKNOWN_LABEL: 
    result = "GLOBUS_DUROC_ERROR_UNKNOWN_LABEL"; break;
  case GLOBUS_DUROC_ERROR_NEXUS_FAILED: 
    result = "GLOBUS_DUROC_ERROR_NEXUS_FAILED"; break;
  case GLOBUS_DUROC_ERROR_MALLOC_FAILED: 
    result = "GLOBUS_DUROC_ERROR_MALLOC_FAILED"; break;
  case GLOBUS_DUROC_ERROR_INTERNAL_FAILURE: 
    result = "GLOBUS_DUROC_ERROR_INTERNAL_FAILURE"; break;
  case GLOBUS_DUROC_ERROR_INVALID_RSL: 
    result = "GLOBUS_DUROC_ERROR_INVALID_RSL"; break;
  case GLOBUS_DUROC_ERROR_INVALID_MULTIREQ: 
    result = "GLOBUS_DUROC_ERROR_INVALID_MULTIREQ"; break;
  case GLOBUS_DUROC_ERROR_GRAM_FAILED: 
    result = "GLOBUS_DUROC_ERROR_GRAM_FAILED"; break;
  case GLOBUS_DUROC_ERROR_INVALID_MANAGER_CONTACT: 
    result = "GLOBUS_DUROC_ERROR_INVALID_MANAGER_CONTACT"; break;
  case GLOBUS_DUROC_ERROR_ALREADY_RELEASED: 
    result = "GLOBUS_DUROC_ERROR_ALREADY_RELEASED"; break;
  case GLOBUS_DUROC_ERROR_ALREADY_CANCELLED: 
    result = "GLOBUS_DUROC_ERROR_ALREADY_CANCELLED"; break;
  case GLOBUS_DUROC_ERROR_BAD_START_TYPE: 
    result = "GLOBUS_DUROC_ERROR_BAD_START_TYPE"; break;
  case GLOBUS_DUROC_ERROR_BAD_COMMS_TYPE: 
    result = "GLOBUS_DUROC_ERROR_BAD_COMMS_TYPE"; break;
  case GLOBUS_DUROC_ERROR_NOT_INITIALIZED: 
    result = "GLOBUS_DUROC_ERROR_NOT_INITIALIZED"; break;
  case GLOBUS_DUROC_ERROR_INVALID_OPERATION: 
    result = "GLOBUS_DUROC_ERROR_INVALID_OPERATION"; break;
  case GLOBUS_DUROC_ERROR_DUPLICATE_SUBJOB_LABEL: 
    result = "GLOBUS_DUROC_ERROR_DUPLICATE_SUBJOB_LABEL"; break;
  case GLOBUS_DUROC_ERROR_PROTOCOL_VERSION_MISMATCH: 
    result = "GLOBUS_DUROC_ERROR_PROTOCOL_VERSION_MISMATCH"; break;
  case GLOBUS_DUROC_ERROR_INVALID_CHECKIN: 
    result = "GLOBUS_DUROC_ERROR_INVALID_CHECKIN"; break;
  default:
    if ( globus_duroc_error_is_gram_client_error (err) ) {
      result = globus_gram_client_error_string (
		globus_duroc_error_get_gram_client_error (err));
    }
    else {
      result = "GLOBUS_FAILURE"; 
    }
    break;
  }

  return result;
}


int utils_sprintf(char *s, const char *format,  ...)
{
  va_list pa;

  int res;

  nexus_stdio_lock ();

  va_start(pa, format);
  res = vsprintf (s, format, pa);
  va_end(pa);

  nexus_stdio_unlock ();

  return res;
}

int utils_fprintf(FILE *fp, const char *format,  ...)
{
  va_list pa;

  int res;

  nexus_stdio_lock ();

  va_start(pa, format);
  res = vfprintf (fp, format, pa);
  va_end(pa);

  nexus_stdio_unlock ();

  return res;
}

int utils_strlen (const char *str)
{
  int len = 0;

  while (str[len] != '\0') len++;

  return len;
}

char *utils_strdup (const char *src)
{
  int i, len;
  char *dst;

  if (src==NULL) return NULL;

  for (len=0; src[len]!='\0'; len++) /* count */;

  dst = globus_malloc (sizeof(char)*(len + 1));
  assert (dst!=NULL);

  for (i=0; i<len; i++) {
    dst[i] = src[i];
  }
  dst[len] = '\0';

  return dst;
}

int utils_streq (const char *s1, const char *s2)
/* assume ASCII */
{
  int i=0;
  
  while ( (s1[i] != '\0')
	  && (s2[i] != '\0') ) {
    if ( s1[i] != s2[i] ) return 0;

    i++;
  }

  if ( s1[i] != s2[i] ) return 0;

  return 1;
}

#define s_abs_int(x) \
(((x) < 0) ? (0 - (x)) : (x))

#define s_alpha(c) \
     ( ( ((c) >= 'a') && ((c) <= 'z') ) \
       || ( ((c) >= 'A') && ((c) <= 'Z') ) )

#define s_down_case(c) \
     ( ( ((c) >= 'A') && ((c) <= 'Z') ) \
       ? ((c) - ('A' - 'a')) \
       : (c) )

int utils_streq_ci (const char *s1, const char *s2)
{
  int i=0;
  
  while ( (s1[i] != '\0')
	  && (s2[i] != '\0') ) {
    if ( s_alpha (s1[i]) && s_alpha (s2[i]) ) {
      if ( s_down_case (s1[i]) != s_down_case (s2[i]) )
	return 0;
    }
    else if ( s1[i] != s2[i] ) return 0;

    i++;
  }

  if ( s1[i] != s2[i] ) return 0;

  return 1;
}

void utils_debug (int pred_flag, const char*format, ...)
{
  va_list pa;

  if ( pred_flag ) {
    va_start(pa, format);
    nexus_stdio_lock ();
    vfprintf (stderr,
	      format, pa);
    nexus_stdio_unlock ();
    va_end(pa);
  }
}

