
#include "globus_common.h"
#include "lber.h"
#include "ldap.h"
#include "globus_rsl.h"
#include <string.h>

#define GRID_INFO_HOST "pitcairn.mcs.anl.gov"
#define GRID_INFO_PORT "6666"
#define GRID_INFO_BASEDN "o=Grid"

/*
 * forward declaration
 */

char *   read_rsl_from_file(char *);
void     substitute_hostname_with_contact(globus_rsl_t *  request);
char *   get_contact(char *);

int 
main( 
    int      argc, 
    char*    argv[] )
{
    int              rc;
    char *           rsl;
    char *           newrsl;
    globus_rsl_t *   request;

    /*
     * Module activation
     */
    rc = globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);

    /*
     * read the RSL from file pointed to by command-line argument.
     */
    if (argc < 2)
    {
	globus_libc_printf("provide argument <rsl file>\n");
	exit(1);
    }

    rsl = read_rsl_from_file(argv[1]);
    globus_assert(rsl != GLOBUS_NULL);

    /*
     * convert the RSL string to the internal tree representation
     */
    request = globus_rsl_parse(rsl);
    globus_libc_free(rsl);
    globus_assert(request != GLOBUS_NULL);

    /*
     * do some substituion
     */
    substitute_hostname_with_contact(request);

    /*
     * unparse to get the RSL on string format again
     */
    newrsl = globus_rsl_unparse(request);
    globus_libc_printf("%s\n", newrsl);

    /*
     * we are done.
     */
    globus_libc_free(newrsl);
    globus_libc_free(request);
    globus_module_deactivate_all();
    return 0;
}
/* main() */


/*
 * substitute_hostname_with_contact()
 *
 * substitutes the contactHostName  
 *
 * Parameters:
 *
 *    request - a request structure obtained by 
 *              parsing a  RSL string
 *
 * Returns:
 * 
 *    The request argument is changed within this function 
 *
 */


void 
substitute_hostname_with_contact(
    globus_rsl_t *  request)
{
    globus_list_t *          list;         /* temp variable */
    globus_rsl_t *           first;        /* temp variable */
    char *                   attribute;
    char *                   value;
    char *                   contact;
    globus_rsl_value_t *     tmp_value;
    globus_rsl_t *           tmp_rsl;
    globus_list_t *          tmp_list;
    
    /*
     * if the request is a multirequest, run this function repeatedly 
     * over the list of requests
     */
    if (globus_rsl_is_boolean_multi(request))
    {
	list = (globus_list_t *) globus_rsl_boolean_get_operand_list(request);
	while (!globus_list_empty(list))
	{
	    substitute_hostname_with_contact(globus_list_first(list));
	    list=globus_list_rest(list);
	}
	return;
    }

    if (globus_rsl_is_boolean(request))
    {
	list = globus_rsl_boolean_get_operand_list(request);
	while ( !globus_list_empty(list) )
	{
	    first = (globus_rsl_t *) globus_list_first(list);
	    if ( globus_rsl_is_boolean(first) )
	    {
		substitute_hostname_with_contact(first);
	    }

	    attribute = globus_rsl_relation_get_attribute(first);

	    if (attribute &&
		!strcasecmp(attribute, "contactHostName"))
	    {
		tmp_value = globus_rsl_relation_get_single_value(first);
		if (tmp_value)
		{
		    value = globus_rsl_value_literal_get_string(tmp_value);
		    contact = get_contact(value);
		    if (contact)
		    {
			/*
			 * Replace "first" with a new RSL relation with
			 * attribute "resourceManagerContact" and a
			 * value sequence with a single entry 'contact'.
			 */
			tmp_list = GLOBUS_NULL;
			globus_list_insert( 
			    &tmp_list, 
			    globus_rsl_value_make_literal(contact) );

			tmp_rsl = 
			    globus_rsl_make_relation( 
				GLOBUS_RSL_EQ,
				"resourceManagerContact",
				globus_rsl_value_make_sequence(tmp_list) );

			globus_list_replace_first( list, tmp_rsl );
		    }
		}
	    }
	    list = globus_list_rest(list);
	}
    }
}
/* substitute_manager_with_contact() */



/*
 * read_rsl_from_file()
 *
 * reads a rsl string from a file
 *
 * Paramters:
 * 
 *     filename - a string containing the file name
 *
 * Returns:
 *
 *     A pointer to the read rsl string on success.
 *     Memory for this string is allocated in the function 
 *     and needs to be freed by the caller
 * 
 *     GLOBUS_NULL on failure
 */

char *  
read_rsl_from_file(
    char *    filename)
{
    char *  req;
    int     fd, i;
    char    c;
    globus_off_t   len = 0;
    
    fd = globus_libc_open(filename, O_RDONLY);
    if (fd < 0)
    {
	globus_libc_fprintf(stderr, "cannot open file %s!\n", filename);
	return GLOBUS_NULL;
    }
    
    len = globus_libc_lseek(fd, 0, SEEK_END);
    globus_libc_lseek(fd, 0, SEEK_SET);
    
    req = (char *) globus_libc_malloc (sizeof(char) * (len + 1));
    i=0;
    
    while ( (i<=len)  && read(fd, &c, 1) > 0)
    {
	req[i++] = c;
    }
    req[i] = '\0';
    globus_libc_close(fd);
    
    return req;
}
/* read_rsl_from_file() */


/*
 * get_contact()
 *
 * queries the LDAP server (MDS) for records corresponding to a
 * job manager and returns a string containing the contact field
 * of the found records
 *
 * Paramters:
 *    
 *    manager_hn - the hostname on which the jobmanager runs
 *
 * Returns:
 * 
 *    exit(1) upon any ldap error
 *
 *    a string containing the contact field 
 *
 *    GLOBUS_NULL if no records were found
 */

 
char *  
get_contact( 
    char *    manager_hn )
{
    LDAP *           ldap_server;
    LDAPMessage *    reply;
    LDAPMessage *    entry;
    char *           attrs[2];
    char *           search_filter;
    char *           server   = GRID_INFO_HOST;
    int              port     = atoi(GRID_INFO_PORT);
    char *           base_dn  = GRID_INFO_BASEDN;

    /*
     * list of attributes that we want included in the search result.
     */
    attrs[0] = "contact";
    attrs[1] = GLOBUS_NULL;

    /*
     * Open connection to LDAP server
     */
    if ((ldap_server = ldap_open(server, port)) == GLOBUS_NULL)
    {
	ldap_perror(ldap_server, "ldap_open");
	exit(1);
    }

    /*
     * Bind to LDAP server
     */
    if (ldap_simple_bind_s(ldap_server, "", "") != LDAP_SUCCESS)
    {
	ldap_perror(ldap_server, "ldap_simple_bind_s");
	ldap_unbind(ldap_server);
	exit(1);
    }

#define search_format "(&(objectclass=GlobusServiceJobManager)(hn=%s))"

    search_filter = globus_libc_malloc( strlen(search_format) +
					strlen(manager_hn)  + 5);
    globus_libc_sprintf(search_filter, search_format, manager_hn);

    if (ldap_search_s( ldap_server,
		       base_dn,
		       LDAP_SCOPE_SUBTREE,
		       search_filter,
		       attrs,
		       0,
		       &reply) != LDAP_SUCCESS)
    {
	ldap_perror(ldap_server, "ldap_search");
	ldap_unbind(ldap_server);
	exit(1);
    }

    /*
     * go through the entries returned by the LDAP server. for each
     * entry, we must search for the right attribute and then get the
     * value associated with it.
     */
    for ( entry = ldap_first_entry(ldap_server, reply);
	  entry != GLOBUS_NULL;
	  entry = ldap_next_entry(ldap_server, entry) )
    {
	BerElement *    ber;
	char**          values;
	char *          attr;
	char *          answer = GLOBUS_NULL;

	for (attr = ldap_first_attribute(ldap_server,entry,&ber);
	     attr != NULL;
	     attr = ldap_next_attribute(ldap_server,entry,ber) )
	{
	    if (strcmp(attr, "contact") == 0)
	    {
		values = ldap_get_values(ldap_server, entry, attr);
		answer = strdup(values[0]);
		ldap_value_free(values);
		break;
	    }
	}
	if (answer)  /* found it: close connection and return value */
	{
	    ldap_unbind(ldap_server);
	    globus_libc_free(search_filter);
	    return answer;
	}
    }

    /*
     * no contact was found -- close connection and return null
     */
    ldap_unbind(ldap_server);
    globus_libc_free(search_filter);
    return GLOBUS_NULL;
}
/* get_contact() */







