/*
 * single_rsr_server.c
 *
 * trimmed-down version of the file that is shipped with the globus source
 * code, in the Examples directory.
 */

#include "globus_common.h"
#include "globus_nexus.h"

/*
 * Nexus Handler Functions 
 */

static void rcv_handler(globus_nexus_endpoint_t *endpoint, 
			globus_nexus_buffer_t   *buffer, 
			globus_bool_t           is_non_threaded_handler);

/*
 * Nexus Handler Functions Table 
 */

static globus_nexus_handler_t handlers[] = 
{ 
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED, rcv_handler}
};

/*
 * Data Structures 
 */

/* Monitor                                                   */
/*                                                           */
/* Nexus programs can implement a monitor with a data        */
/* structure as illustrated below.  All monitors implemented */
/* in this way must have a 'globus_mutex_t' field to provide */
/* mutually exclusive access to the monitor.                 */
/*                                                           */
/* In this application we also need a monitor condition      */
/* variable and two flags.                                   */
/*                                                           */
/* Explanation of why 'volatile' is required on some fields: */
/*                                                           */
/* Some compilers, under certain optimization switches,      */
/* attempt to cache the value in a register thus avoiding a  */
/* memory reference.  By declaring the variable 'volatile',  */
/* we force compiler to  NOT cache value in register but     */
/* instead reload the value from  memory each time is is     */
/* referenced.                                               */
/*                                                           */
/* So, if a variable is shared by multiple threads such that */
/* it is written to by one thread and read by another, then  */
/* that variable must be declared volatile.  However, you    */
/* notice that two of the variables                          */
/*                                                           */
/*    globus_mutex_t mutex;                                  */
/*    globus_cond_t  cond;                                   */
/*                                                           */
/* both meet the multiple thread read/write criteria and yet */
/* neither of them are declared volatile.  This is because   */
/* the dataypes globus_mutex_t and globus_cond_t are defined */
/* within the Nexus implemenation (opaque to the user) and   */
/* Nexus has assumed the responsibility of taking the steps  */
/* necessary to assure that any potential compiler-induced   */
/* caching problems will be avoided.  Nexus does not assume  */
/* that responsibility for all its datatypes, only these     */
/* two.  This is because, by their very nature, these two    */
/* datatypes are intended to be shared by multiple threads.  */
/* All other Nexus-supplied datatypes should be explicitly   */
/* declared volatile where appropriate, as we do in this     */
/* example with our two monitor flags (each a nexus_bool_t). */
/*                                                           */
/* The general rule of thumb is, any variable that is not a  */
/* globus_mutex_t or a globus_cond_t and is shared by        */
/* multiple threads such that it is written to by one thread */
/* and read by another must be declared volatile.            */

typedef struct
{
    /*
     * a 'globus_mutex_t' is required of ALL monitors to 
     * control mutually exclusive access to the monitor  
     */

    /* 'volatile' not required here, handled   */
    /* in the implementation of globus_mutex_t */

    globus_mutex_t mutex;

    /*
     * monitor data for this program 
     */

    /* 'volatile' not required here, handled  */
    /* in the implementation of globus_cond_t */

    globus_cond_t  cond;

    /* 'volatile' required for both of these fields because */
    /* each is written by one thread and read by another    */

    volatile globus_bool_t  connection_established;
    volatile globus_bool_t  done;

} monitor_t;

/*
 *        
 * Global Variables
 *        
 */

static globus_nexus_endpointattr_t     GlobalEndpointAttr;
static globus_nexus_endpoint_t         GlobalEndpoint;
static monitor_t                       Monitor;

/*
 *
 * Local Functions 
 *
 */

static int attach_requested(void *arg, char *url, globus_nexus_startpoint_t *sp);

/*
 *
 * MAIN 
 *
 */

int 
main(
    int       argc, 
    char *    argv[])
{
    int rc;
    char *attach_host;
    unsigned short attach_port;
    char attach_url[1024];

    /*
     * Module activation: we are using the common and the nexus module
     */

    rc = globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);

    rc = globus_module_activate(GLOBUS_NEXUS_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);

    /*
     * Initialization 
     */

    globus_nexus_endpointattr_init(&GlobalEndpointAttr);
    globus_nexus_endpointattr_set_handler_table(
	&GlobalEndpointAttr,
	handlers,
	sizeof(handlers)/sizeof(globus_nexus_handler_t));
    
    globus_nexus_endpoint_init(&GlobalEndpoint, &GlobalEndpointAttr);
    nexus_endpoint_set_user_pointer(&GlobalEndpoint, (void *) (&Monitor));

    /*
     * Initialize the monitor
     */

    /* initializing the mutex that controls access to the monitor */

    globus_mutex_init(&(Monitor.mutex), (globus_mutexattr_t *) GLOBUS_NULL);

    /* initializing the condition variable */

    globus_cond_init(&(Monitor.cond), (globus_condattr_t *) GLOBUS_NULL);

    /* entering the monitor and clearing the flags */

    globus_mutex_lock(&(Monitor.mutex));
    {
	Monitor.connection_established = GLOBUS_FALSE;
	Monitor.done = GLOBUS_FALSE;
    }
    globus_mutex_unlock(&(Monitor.mutex));

    /*
     * allowing attachment
     */

    /*
     *
     * By calling nexus_allow_attach() we allow other Nexus programs
     * to attach to us.  nexus_allow_attach() tells us the hostname 
     * (attach_host) and port number (attach_port) that we will start 
     * listening for attachments.
     *
     * We use attach_host and attach_port to construct a URL that attach 
     * clients need to use to attach to us.
     *
     * In calling nexus_allow_attach() we also name an approval function, in
     * this case attach_requested(), which is a user-supplied function that
     * gets called every time a client requests an attachment to us.
     *
     */

    /* allow other Nexus programs to attach to us */

    attach_port = 0;
    rc = globus_nexus_allow_attach(&attach_port,     /* port            */
				   &attach_host,     /* host            */
				   attach_requested, /* approval_func() */
				   GLOBUS_NULL);
    if (rc != 0)
    {
	/* libc functions are usually not re-entrant. Re-entrant functions   */
	/* are provided. They perform the necessary locks. But be aware of   */
	/* potential dead-lock if miss-used.                                 */
	/* The  functions globus_libc_lock() and globus_libc_unlock() can    */
	/* also be used arround standard libc function                       */
       globus_libc_fprintf(
	   stderr,
	   "ERROR: globus_nexus_allow_attach() failed: rc=%d\n",
	   rc);
       globus_module_deactivate_all();
       exit(1);
    } /* endif */

    globus_libc_sprintf(
	attach_url,
	"x-nexus://%s:%hu/",
	attach_host,
	attach_port);

    /* advertising URL for attachment */
    globus_libc_printf("attach url: %s\n", attach_url);

    /*
     * wait (using monitor) for single_rsr_client to attach and send number. 
     * first our attach_requested() gets called (when the client attaches to 
     * us) and then our rcv_handler() gets called (when the client sends us  
     * an rsr).  it is rcv_handler() that signals the monitor's condition    
     * variable and allows us to proceed.                                    
     */

    globus_mutex_lock(&(Monitor.mutex));
    {
	while (!(Monitor.done))
	{
	    globus_cond_wait(&(Monitor.cond), &(Monitor.mutex));
	}
    }
    globus_mutex_unlock(&(Monitor.mutex));

    /*
     * Clean-up 
     */

    globus_mutex_destroy(&(Monitor.mutex));
    globus_cond_destroy(&(Monitor.cond));

    globus_nexus_endpoint_destroy(&GlobalEndpoint);
    globus_nexus_endpointattr_destroy(&GlobalEndpointAttr);

    globus_module_deactivate_all();
    exit(0);
} /* end main() */


/*
 *
 * Local Functions 
 *
 */

/*
 * attach_requested()
 *
 * This is the attachment approval function.  When the server started
 * allowing attachments (nexus_allow_attach()), it specified this approval 
 * function.
 *
 * Parameters:
 *   
 *    arg - used for propagating arguments through the *attach_allow call
 *
 *    url - the url used by globus_nexus_attach() to attach to this port
 *
 *    sp - an uninitialized startpoint 
 *
 * Returns:
 *    
 *    GLOBUS_SUCCESS on success
 *
 *    -1 on failure
 */

static int 
attach_requested(
    void *                         arg, 
    char *                         url, 
    globus_nexus_startpoint_t *    sp)
{
     int rc;

    globus_mutex_lock(&(Monitor.mutex));
    if (Monitor.connection_established)
    {
        /* another single_rsr_client has already connected to us */
        rc = -1;
    }
    else
    {
        /* this is the first single_rsr_client */
        rc = GLOBUS_SUCCESS;
        Monitor.connection_established = NEXUS_TRUE;
    } /* endif */
    globus_mutex_unlock(&(Monitor.mutex));

     /* binding received startpoint to one of our endpoints */
     if (rc == GLOBUS_SUCCESS)
     {
         globus_nexus_startpoint_bind(sp, &GlobalEndpoint);
     } /* endif */

     return rc;

} 
/* end attach_requested() */


/*
 *
 * Nexus Handler Functions 
 *
 */

/*
 * rcv_handler()
 *
 * handler used for receiving data
 *
 * Paramters:
 *
 *    endpoint - endpoint associated with communication
 *
 *    buffer - buffer from which to receive
 *
 *    is_non_threaded_handler - true if handler is expected 
 *                              to be non-threaded
 */

static void 
rcv_handler(
    globus_nexus_endpoint_t *    endpoint, 
    nexus_buffer_t *             buffer,
    globus_bool_t                is_non_threaded_handler)
{
    monitor_t *mp;
    int i; 
	
    /*
     * extracting pointer to monitor from this endpoint 
     */

    mp = (monitor_t *) globus_nexus_endpoint_get_user_pointer(endpoint);

    /* extracting int from buffer */
    globus_nexus_get_int(buffer, &i, 1);

    /* printing received value */
    globus_libc_printf("Single Rsr server received %d\n", i);

    /* clean-up */

    /* destroying received buffer */
    globus_nexus_buffer_destroy(buffer);

    /*
     * setting the monitor's flag and signalling it's 
     * cond variable to wake up the slave in main()   
     */

    globus_mutex_lock(&(mp->mutex));
    {
	mp->done = NEXUS_TRUE;
	globus_cond_signal(&(mp->cond));
    }
    globus_mutex_unlock(&(mp->mutex));

} 
/* end rcv_handler() */
