
#include <stdio.h>
#include <globus_common.h>
#include <globus_nexus.h>


#define GET_ANSWER_HANDLER_ID 0

/*
 *
 * Nexus Handler Functions
 *
 */

static void addem_handler(globus_nexus_endpoint_t *endpoint, 
			  globus_nexus_buffer_t   *buffer,
			  globus_bool_t           is_non_threaded_handler);

/*
 *
 * Nexus Handler Functions Table 
 *
 */

static globus_nexus_handler_t handlers[] =
{ 
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED, addem_handler}
};


/* Monitor                                                   */
/*                                                           */
/* Nexus programs can implement a monitor with a data        */
/* structure as illustrated below.  All monitors implemented */
/* in this way must have a 'globus_mutex_t' field to provide */
/* mutually exclusive access to the monitor.                 */
/*                                                           */
/* In this application we also need a monitor condition      */
/* variable and two flags.                                   */
/*                                                           */


typedef struct _monitor_t
{
    globus_mutex_t mutex;
    /*
     * monitor data for this program 
     */
    globus_cond_t cond;

    volatile globus_bool_t connection_established;
    volatile globus_bool_t done;
} monitor_t;


/*
 *
 * Global Variables
 *                 
 */

static globus_nexus_endpointattr_t     serverEndpointAttr;
static globus_nexus_endpoint_t         serverEndpoint;
static globus_nexus_startpoint_t       clientStartpoint;
static monitor_t                       Monitor;


/*
 *
 * Local Functions
 *                
 */

static int 
attach_requested(
    void *    arg,
    char *    url,
    globus_nexus_startpoint_t *sp);

/*
 *
 * MAIN
 *
 */

int
main(
    int       argc, 
    char *    argv[])
{

    int rc;
    char *attach_host;
    unsigned short attach_port;
    char attach_url[1024];

    /*
     * Activation of the Nexus module is required at
     * the beginning of all Nexus programs          
     */

    if ( globus_module_activate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	fprintf(stderr,"Nexus Module activation failed \n");
	exit(1);
    }

    /*
     * Endpoint Initialization 
     */

    globus_nexus_endpointattr_init( &serverEndpointAttr );

    globus_nexus_endpointattr_set_handler_table(
	&serverEndpointAttr,
	handlers,
	sizeof(handlers)/sizeof(globus_nexus_handler_t));

    globus_nexus_endpoint_init( &serverEndpoint,
				&serverEndpointAttr );
    /*
     * Note : The monitor does not need to be global, but could have been
     * dynamically allocated: its address is passed around using the
     * user_pointer of the endpoint
     */

    globus_nexus_endpoint_set_user_pointer( &serverEndpoint,
					    (void *)(&Monitor));

    /*
     * Initialize the monitor 
     */

    /* initializing the mutex that controls access to the monitor */
    
    globus_mutex_init( &(Monitor.mutex), (globus_mutexattr_t *) GLOBUS_NULL );
    
    /* initializing the condition variable */
    
    globus_cond_init( &(Monitor.cond), (globus_condattr_t *) GLOBUS_NULL );

    /* entering the monitor and clearing the flags */
    
    globus_mutex_lock( &(Monitor.mutex) );
    Monitor.connection_established = GLOBUS_FALSE;
    Monitor.done = GLOBUS_FALSE;
    globus_mutex_unlock( &(Monitor.mutex) );

    /*
     * allowing attachment 
     */

    /*
     *
     * By calling globus_nexus_allow_attach() we allow other Nexus programs
     * to attach to us.  globus_nexus_allow_attach() tells us the hostname 
     * (attach_host)and port number (attach_port) that we will start 
     * listening for attachments.
     *
     * We use attach_host and attach_port to construct a URL that attach 
     * clients need to use to attach to us.
     *
     * In calling globus_nexus_allow_attach() we also name an approval function,
     * in this case attach_requested(), which is a user-supplied function that
     * gets called every time a client requests an attachment to us.
     *
     */

    /* allow other Nexus programs to attach to us */
    attach_port = 0;
    rc = globus_nexus_allow_attach(&attach_port,     /* port            */
				   &attach_host,     /* host            */
				   attach_requested, /* approval_func() */
				   GLOBUS_NULL);
    if (rc != 0)
    {
       /* libc functions are usually not re-entrant. Re-entrant functions   */
       /* are provided. They perform the necessary locks. But be award of   */
       /* potential dead-lock if miss-used.                                 */
       /* The  functions globus_libc_lock() and globus_libc_unlock() can    */
       /* also be used arround standard libc function                       */
       globus_libc_fprintf(
	   stderr,

	   "ERROR: globus_nexus_allow_attach() failed: rc=%d\n",
	   rc);
       if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
       {
	   globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
       }
       exit(1);
    } /* endif */

    globus_libc_sprintf(attach_url,
			"x-nexus://%s:%hu/",
			attach_host,
			attach_port);

    /* advertising URL for attachment */
    globus_libc_printf("attach url: %s\n", attach_url);

    /*
     * wait (using monitor) for single_rsr_client to attach and
     * send numbers.
     */
    globus_mutex_lock(&(Monitor.mutex));
    {
	while (!(Monitor.done))
	{
	    globus_cond_wait(&(Monitor.cond), &(Monitor.mutex));
	}
    }
    globus_mutex_unlock(&(Monitor.mutex));

    /*
     * Clean-up 
     */

    globus_mutex_destroy(&(Monitor.mutex));
    globus_cond_destroy(&(Monitor.cond));

    globus_nexus_endpoint_destroy(&serverEndpoint);
    globus_nexus_endpointattr_destroy(&serverEndpointAttr);

    
    if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	exit(1);
    }

    exit(0);

} 
/* end main() */


/*
 *
 * Local Functions 
 *                 
 */


/*
 * attach_requested()
 *
 * This is the attachment approval function.  When the server started
 * allowing attachments (nexus_allow_attach()), it specified this approval 
 * function.
 *
 * Parameters:
 *   
 *    arg - used for propagating arguments through the *attach_allow call
 *
 *    url - the url used by globus_nexus_attach() to attach to this port
 *
 *    sp - an uninitialized startpoint 
 *
 * Returns:
 *    
 *    GLOBUS_SUCCESS on success
 *
 *    -1 on failure
 */

static int
attach_requested(
    void *                         arg,
    char *                         url,
    globus_nexus_startpoint_t *    sp)
{
    int connected;

    globus_mutex_lock( &(Monitor.mutex) );
    if ( Monitor.connection_established )
    {
	/* another talk_client has already connected to us */
	connected = -1;
    }
    else
    {
	/* this is the first talk_client */
	connected = GLOBUS_SUCCESS;
	Monitor.connection_established = GLOBUS_TRUE;
	
	/* binding received startpoint to one of our endpoints */
	globus_nexus_startpoint_bind( sp, &serverEndpoint );
    } /* endif */
    globus_mutex_unlock( &(Monitor.mutex) );
    
    return connected;
    
} 
/* end attach_requested() */


/*
 * 
 * Nexus Handler Functions 
 * 
 */

/*
 * addem_handler()
 *
 * handler that receives two integers,adds them and 
 * sends the result back to the client. 
 *
 * Paramters:
 *
 *    endpoint - endpoint associated with communication
 *
 *    buffer - buffer from which to receive
 *
 *    is_non_threaded_handler - true if handler is expected 
 *                              to be non-threaded
 */

static void
addem_handler( 
    globus_nexus_endpoint_t *    endpoint, 
    globus_nexus_buffer_t *      buffer,
    globus_bool_t                is_non_threaded_handler )
{
    monitor_t *mp;
    globus_nexus_startpoint_t reply_done_sp;
    globus_nexus_buffer_t reply_buffer;
    int operands[2]; 
    int sum;
	
    /*
     * extracting pointer to monitor from this endpoint
     */

    mp = (monitor_t *) globus_nexus_endpoint_get_user_pointer(endpoint);

    /*
     * extracting reply_done_sp and operands from buffer 
     */

    /* note that items from buffer *must* be extracted in         */
    /* the same order that they were placed into the buffer.      */
    /* note also that the two operands may be extracted using     */
    /* a *single* get by specifying a space big enough to extract */
    /* them and the fact that we want "2" of them.                */

    globus_nexus_get_startpoint(buffer, &reply_done_sp, 1);
    globus_nexus_get_int(buffer, operands, 2);

    globus_libc_printf("server: client asked what %d + %d is\n",
		       operands[0],
		       operands[1] );
    
    /*
     * calculating value of function 
     */

    sum = operands[0] + operands[1];

    /*
     * remote service request 
     */

    /* initializing buffer to hold single integer */

    globus_nexus_buffer_init(&reply_buffer, nexus_sizeof_int(1), 0);

    /* placing sum into buffer */

    globus_nexus_put_int(&reply_buffer, &sum, 1);

    /* sending buffer to original process, using reply_done_sp    */
    /* that we just extracted from the buffer, and invoking       */
    /* get_answer_handler() there                                 */

    globus_nexus_send_rsr( &reply_buffer, 
			   &reply_done_sp, 
			   GET_ANSWER_HANDLER_ID,
			   GLOBUS_TRUE,
			   is_non_threaded_handler );

    /*
     * clean-up 
     */

    globus_nexus_buffer_destroy(buffer);
    globus_nexus_startpoint_destroy( &reply_done_sp );

    /*
     * setting the monitor's flag and signalling it's
     * cond variable to wake up the slave in main()  
     */

    /* the implementation of globus_cond_wait() and globus_cond_signal()  */
    /* makes it possible for globus_cond_wait() to experience a 'false   */
    /* wakeup', i.e., return without having had a globus_cond_signal()   */
    /* applied to it.                                                   */
    /*                                                                  */
    /* this is why we must signal a condition variable in the manner    */
    /* we do below, with a globus_bool_t done, rather than simply:       */
    /*     globus_mutex_lock(&(mp->mutex));                              */
    /*     globus_cond_signal(&(mp->cond));                              */
    /*     globus_mutex_unlock(&(mp->mutex));                            */
   
    globus_mutex_lock(&(mp->mutex));
    mp->done = GLOBUS_TRUE;
    globus_cond_signal(&(mp->cond));
    globus_mutex_unlock(&(mp->mutex));

} 
/* end addem_handler() */


