#include <stdio.h>
#include <globus_common.h>
#include <globus_nexus.h>


#define ADDEM_HANDLER_ID 0

/*
 *
 * Nexus Handler Function 
 *
 */

static void get_answer_handler(globus_nexus_endpoint_t * endpoint,
			       globus_nexus_buffer_t * buffer,
			       globus_bool_t is_non_threaded_handler);

/*
 *
 * Nexus Handler Function Table 
 *
 */

static globus_nexus_handler_t handlers[] =
{
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED, get_answer_handler}
};

/*
 *
 * Data Structures 
 *
 */

/* Monitor                                                   */
/*                                                           */
/* Nexus programs can implement a monitor with a data        */
/* structure as illustrated below.  All monitors implemented */
/* in this way must have a 'globus_mutex_t' field to provide  */
/* mutually exclusive access to the monitor.                 */


typedef struct _monitor_t
{
    globus_mutex_t mutex;
    globus_cond_t cond;

    /* 'volatile' required for both of these fields because */
    /* each is written by one thread and read by another    */

    volatile globus_bool_t done;
    volatile int sum;
} monitor_t;


/*
 *
 * MAIN
 *
 */

int 
main(
    int       argc, 
    char *    argv[])
{
    char                         attach_url[1024];
    globus_nexus_buffer_t        buffer;
    int                          i;
    int                          rc;
    int                          x, y, sum;
    globus_nexus_endpointattr_t  clientEndpointAttr;
    globus_nexus_endpoint_t      clientEndpoint;
    globus_nexus_startpoint_t    clientStartpoint;
    globus_nexus_startpoint_t    serverStartpoint;
    monitor_t                    replyMonitor;

    /*
     * Activation of the Nexus module is required at 
     * the beginning of all Nexus programs           
     */

    if ( globus_module_activate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	fprintf(stderr,"Nexus Module activation failed \n");
	exit(1);
    }

    /*
     * check for correct command prompt usage and get 
     * the attach_url specified in the command line   
     */

    if (argc == 4 && !strncmp(argv[1], "x-nexus://", strlen("x-nexus://")))
    {
	strcpy(attach_url, argv[1]);
	x = atoi(argv[2]);
	y = atoi(argv[3]);
    }
    else
    {
	globus_libc_fprintf(
	    stderr,
	    "usage: %s <attach_url> <int> <int>\n",
	    argv[0] );

	/* quit now that we have notified the user of the error */

	if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
	{
	    globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	}
	exit(1);
    } /* endif */

    /*
     * attaching to server 
     */

    /*
     * Attachment in Nexus is done by making a request to attach to another
     * process designated by a URL.  If the request is granted, we get a 0
     * return code and a startpoint that points back to the granting process.
     */

    rc = globus_nexus_attach( attach_url, &serverStartpoint );

    /* if things went OK, we now have a startpoint */
    /* to the server, serverStartpoint.            */

    if( rc != 0 )
    {
	globus_libc_fprintf(
	    stderr, 
	    "ERROR: Could not attach to server using url: %s rc %d\n", 
	    attach_url, rc
	    );
	/* quit now that we have notified the user of the error */
	if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
	{
	    globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	}
	exit(1);
    } /* endif */

    /*
     * Endpoint Initialization
     */

    globus_nexus_endpointattr_init( &clientEndpointAttr );
    globus_nexus_endpointattr_set_handler_table(
	&clientEndpointAttr,
	handlers,
	sizeof( handlers ) / sizeof( nexus_handler_t )
	);
    globus_nexus_endpoint_init( &clientEndpoint, &clientEndpointAttr );
    /* set up reply our endpoint to point to the reply monitor */
    
    globus_nexus_endpoint_set_user_pointer(
	&clientEndpoint,
	(void *)(&replyMonitor)
	);
    /* binding a startpoint to our endpoint */
    
    globus_nexus_startpoint_bind( &clientStartpoint, &clientEndpoint );

    /*
     * Initialize the monitor that will be used for signalling    
     * that we have successfully received the sum from ther server
     */

    /* initializing the mutex that controls access to the monitor */
    
    globus_mutex_init( &(replyMonitor.mutex), 
		       (globus_mutexattr_t *) GLOBUS_NULL );

    /* initializing the condition variable */
    
    globus_cond_init( &(replyMonitor.cond), 
		      (globus_condattr_t *) GLOBUS_NULL );

    /* entering the monitor and clearing the flags */

    globus_mutex_lock( &(replyMonitor.mutex) );
    replyMonitor.done = GLOBUS_FALSE;
    globus_mutex_unlock( &(replyMonitor.mutex) );

    /*
     * remote service request
     */

    /* initializing the buffer to hold the clientStartpoint and two integers */
    /* Note: there is no difference between (nexus_sizeof_int(1) * 2) and    */
    /*       simply nexus_sizeof_int(2).                                     */

    globus_nexus_buffer_init(
	&buffer,
	globus_nexus_sizeof_startpoint( &clientStartpoint, 1 )
	+ (globus_nexus_sizeof_int(1) * 2),
	0);

    /*
     * Placing the startpoint and two integers into the buffer
     *
     * note: startpoint_transfer() is destructing, so we don't need
     * to clean up the startpoint later.
     */
    
    globus_nexus_put_startpoint_transfer( &buffer, &clientStartpoint, 1 );
    globus_nexus_put_int( &buffer, &x, 1 );
    globus_nexus_put_int( &buffer, &y, 1 );

    /* sending buffer to the server and invoking addem_handler() there */
    
    globus_nexus_send_rsr( &buffer,              /* buffer */
			   &serverStartpoint,    /* destination */
			   ADDEM_HANDLER_ID,     /* handler id */
			   GLOBUS_TRUE,    /* destroy buffer when done */
			   GLOBUS_FALSE ); /* called from non-threaded
						    handler */

    /*
     * wait on the monitor condition variable until the reply comes 
     */

    globus_mutex_lock( &(replyMonitor.mutex) );
    {
	while( !(replyMonitor.done) )
	{
	    globus_cond_wait( &(replyMonitor.cond), &(replyMonitor.mutex) );
	}
    }
    globus_mutex_unlock( &(replyMonitor.mutex) );


    /* lock the monitor so that we can access the sum */

    globus_mutex_lock( &(replyMonitor.mutex) );
    {
	sum = replyMonitor.sum;
    }
    globus_mutex_unlock( &(replyMonitor.mutex) );

    globus_libc_printf( "client: server returned %d + %d = %d\n\n",
			x,
			y,
			sum );

    /*
     * Clean-up 
     */

    /* monitor clean-up */
    globus_mutex_destroy( &(replyMonitor.mutex) );
    globus_cond_destroy( &(replyMonitor.cond) );

    /*endpoint clean-up */
    nexus_endpoint_destroy( &clientEndpoint );
    nexus_endpointattr_destroy( &clientEndpointAttr );

    /* startpoint clean-up */
    nexus_startpoint_destroy( &serverStartpoint );

    if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    {
	globus_libc_fprintf(stderr,"Nexus Module DEactivation failed \n");
	exit(1);
    }

    exit(0);

} 
/* end main() */

/*
 *
 * Nexus Handler Functions 
 *
 */


/*
 * get_answer_handler()
 *
 * handler used for receiving the computed sum
 *
 * Paramters:
 *
 *    endpoint - endpoint associated with communication
 *
 *    buffer - buffer from which to receive
 *
 *    is_non_threaded_handler - true if handler is expected 
 *                              to be non-threaded
 */

static void
get_answer_handler(
    globus_nexus_endpoint_t *    endpoint,
    globus_nexus_buffer_t   *    buffer,
    globus_bool_t                is_non_threaded_handler )
{
    monitor_t *mp;
    int sum;
    
    mp = (monitor_t *) globus_nexus_endpoint_get_user_pointer(endpoint);
    
    /*
     * extracting answer from message buffer  
     */

    globus_nexus_get_int(buffer, &sum, 1);

    /*
     * clean-up
     */

    globus_nexus_buffer_destroy(buffer);

    /*
     * placing sum into monitor, setting flag, and 
     * signalling cond variable to wake up main()  
     */
    
    globus_mutex_lock(&(mp->mutex));
    {
	mp->sum = sum;
	mp->done = GLOBUS_TRUE;
	globus_cond_signal(&(mp->cond));
    }
    globus_mutex_unlock(&(mp->mutex));

} 
/* end get_answer_handler() */





