/*
 * ldaptest.c
 */

#include "globus_common.h"
#include "lber.h"
#include "ldap.h"
#include <string.h>

#define GRID_INFO_HOST "pitcairn.mcs.anl.gov"
#define GRID_INFO_PORT "6666"
#define GRID_INFO_BASEDN "o=Grid"

/*
 * Define the query
 */

#define search_format "(&(objectclass=GlobusServiceJobManager)(hn=%s)(service=%s))"

 
/*
 * Forward declaration 
 */

char *  get_ldap_attribute( char *  search_filter,
			    char *  attribute    );

/*
 * Usage info
 */

static char *  USAGE=
"\n ldaptest -h <hostname> [-s <servicename>] \n" ;

/*
 * main() : use the function above to search for a resource manager running
 *          on <hostname>, and get it to return the resource manager
 *          contact string.
 */


int 
main(
    int       argc, 
    char *    argv[])
{
    int       rc;
    char *    filter;
    char *    result;
    char *    hostname = GLOBUS_NULL;
    char *    service = GLOBUS_NULL;
    char      c;

    /*
     * module activation
     */
    rc = globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);

    /*
     * parse arguments
     */
    
    while ((c=getopt(argc,argv,"h:s:")) != EOF)
    {
      switch(c)
	{
	case 'h':
	  hostname = globus_libc_strdup(optarg);
	  break;
	case 's':
	  service = globus_libc_strdup(optarg);
	  break;
	default:
	  globus_libc_printf("unknown flag -%c\n%s",(char) c, USAGE);
	  globus_module_deactivate_all();
	  exit(1);
        }
    }
    
    if(hostname==GLOBUS_NULL)
    {
      globus_libc_printf("A hostname needs to be specified\n%s", USAGE);
      globus_module_deactivate_all();
      exit(1);
    } 
    
    /*
     * construct the LDAP search filter
     */

    if(service!=GLOBUS_NULL)
    {
      filter = globus_libc_malloc(strlen(search_format) + strlen(hostname) 
				  + strlen(service) + 5);
      				
      globus_libc_sprintf(filter, search_format, hostname, service);
    }
    else
    {
      filter = globus_libc_malloc(strlen(search_format) 
				  + strlen(hostname) + 5);
      globus_libc_sprintf(filter, search_format, hostname, "*");
    }


    /*
     * do the search, print result.
     */
    result = get_ldap_attribute( filter, "contact" );
    
    globus_libc_printf("search result : %s\n",
		       (result) ? result : "<not found>" );

    globus_libc_free(filter);
    globus_libc_free(result);

    /*
     * we are done
     */
    rc = globus_module_deactivate_all();
    globus_assert(rc == GLOBUS_SUCCESS);

    return 0;
}
/* main */


/*
 * get_ldap_attribute()
 *
 * queries the LDAP server (MDS) for records using the search filter 
 * and returns a string containing the <attribute> field of the found
 * records
 *
 * Paramters:
 *    
 *    search-filter - a string containing a ldap query 
 *  
 *    attribute - the attribute of the found records that
 *                is to be returned
 *
 * Returns:
 * 
 *    exit(1) upon any ldap error
 *
 *    a string containing the <attribute> field of any records passing 
 *    the search filter
 *
 *    GLOBUS_NULL if no records were found
 */


char *  
get_ldap_attribute( 
    char *    search_filter,
    char *    attribute    )
{
    LDAP *           ldap_server;
    LDAPMessage *    reply;
    LDAPMessage *    entry;
    char *           attrs[2];
    char *           server   = GRID_INFO_HOST;
    int              port     = atoi(GRID_INFO_PORT);
    char *           base_dn  = GRID_INFO_BASEDN;

    /*
     * list of attributes that we want included in the search result.
     */
    attrs[0] = attribute;
    attrs[1] = GLOBUS_NULL;

    /*
     * Open connection to LDAP server
     */
    if ((ldap_server = ldap_open(server, port)) == GLOBUS_NULL)
    {
	ldap_perror(ldap_server, "ldap_open");
	exit(1);
    }

    /*
     * Bind to LDAP server
     */
    if (ldap_simple_bind_s(ldap_server, "", "") != LDAP_SUCCESS)
    {
	ldap_perror(ldap_server, "ldap_simple_bind_s");
	ldap_unbind(ldap_server);
	exit(1);
    }

    if (ldap_search_s( ldap_server,
		       base_dn,
		       LDAP_SCOPE_SUBTREE,
		       search_filter,
		       attrs,
		       0,
		       &reply) != LDAP_SUCCESS)
    {
	ldap_perror(ldap_server, "ldap_search");
	ldap_unbind(ldap_server);
	exit(1);
    }

    /*
     * go through the entries returned by the LDAP server. for each
     * entry, we must search for the right attribute and then get the
     * value associated with it.
     */
    
    for ( entry = ldap_first_entry(ldap_server, reply);
	  entry != GLOBUS_NULL;
	  entry = ldap_next_entry(ldap_server, entry) )
    {
	BerElement *    ber;
	char**          values;
	char *          attr;
	char *          answer = GLOBUS_NULL;

	for (attr = ldap_first_attribute(ldap_server,entry,&ber);
	     attr != NULL;
	     attr = ldap_next_attribute(ldap_server,entry,ber) )
	{
	    if (strcmp(attr, attribute) == 0)
	    {
		values = ldap_get_values(ldap_server, entry, attr);
		answer = strdup(values[0]);
		ldap_value_free(values);
		break;
	    }
	}
	if (answer)  /* found it: close connection and return value */
	{
	    ldap_unbind(ldap_server);
	    return answer;
	}
    }

    /*
     * no contact was found -- close connection and return null
     */
    ldap_unbind(ldap_server);
    return GLOBUS_NULL;
}
/* get_ldap_attribute */




