/*
 * raw-connect.c 
 *
 * this program connects a host/port given by [-h <host>] -p <port> 
 * and sends a message to the server process.
 */

#include "globus_common.h"

#include <sys/types.h>
#include <sys/socket.h>

/* forward declarations */
int establish_connection(char *  hostname, int port);

static char * USAGE =
"\nraw-connect [-h host] -p port\n"
"host defaults to localhost if not given\n\n";


int 
main( 
    int       argc, 
    char *    argv[] )
{
    int                    rc;                    /* return code storage */
    char *                 host;                  /* server host         */
    int                    port;                  /* server port         */
    int                    fd;                    /* socket file desc    */
    char                   msgbuf[GREETINGSIZE];
    int                    c;
    extern char *          optarg;
    
    /*
     * Check arguments
     */
    rc = globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);

    port = -1;
    host = "localhost";
    while ((c=getopt(argc,argv,"h:p:")) != EOF)
    {
        switch(c)
	{
	case 'h':
	    host = globus_libc_strdup(optarg);
	    break;
	case 'p':
	    port = atoi(optarg);
	    break;
	default:
            globus_libc_printf("unknown flag -%c\n%s",(char) c, USAGE);
            globus_module_deactivate_all();
            exit(1);
	}
    }

    if (port <= 0)
    {
	globus_libc_printf("must provide port number!\n%s", USAGE);
	globus_module_deactivate(GLOBUS_COMMON_MODULE);
	exit(1);
    }

    /*
     * Establish a TCP connection with hostname argv[1], port argv[2]
     */
    fd = establish_connection(host, port);
    if (fd < 0)
    {
	globus_libc_fprintf(stderr,"connect failed\n");
	globus_module_deactivate_all();
	exit(1);
    }
	
    /*
     * Have a connection!
     */
    globus_libc_sprintf(msgbuf,
			"Hello via raw sockets from pid %d\n",
			globus_libc_getpid());

    globus_libc_write(fd,msgbuf,GREETINGSIZE);
    
    globus_libc_close(fd);

    globus_libc_printf("client: sent greeting over pure socket connection\n");
    
    globus_module_deactivate_all();
    return 0;
}
/* main */




/*
 * establish_connection()
 *
 * establishes a TCP connection to hostname at port
 * 
 * Parameters:
 *   
 *   hostname - the host to contact
 *   port     - the port to contact
 * 
 * Returns:
 * 
 * The filedescriptor of the local socket if successful
 *
 * -1 if an error occurred
 *
 */

int 
establish_connection(
    char *    hostname, 
    int       port)
{
    int                    fd;
    struct sockaddr_in     my_addr;      /* INET address structure      */
    struct hostent *       hp;
    struct hostent         result;
    int                    herrno;
    char                   buf[1024];

    hp = globus_libc_gethostbyname_r( hostname,
				      &result,
				      buf,
				      1024,
				      &herrno );
    if (!hp)
	return -1;

    /* Get a socket */
    if ((fd = socket(AF_INET, SOCK_STREAM, 0))  <  0)
	return -1;
    
    /* Set up address structure */
    bzero((char *)&my_addr, sizeof(my_addr));
    my_addr.sin_family = AF_INET;
    memcpy( &(my_addr.sin_addr.s_addr),
	    hp->h_addr_list[0],
	    sizeof(my_addr.sin_addr.s_addr) );
    my_addr.sin_port = htons(port);

    if ((connect( fd, (struct sockaddr *) &my_addr, sizeof(my_addr))) < 0)
	return -1;
    
    return fd;
}
/* establish_connection */












