/**
 * @example listen.c 
 *
 * This program demonstrates how to create a simple server program
 * using Globus I/O. It covers listener establishment, accepting new
 * connections, reading data, and closing a handle.
 *
 * Command line options allow the user of this program to enable
 * Globus I/O security on the connection, as well as set some socket
 * options. This are done using  @link attr Globus I/O attributes
 * @endlink.
 *
 * In order to use this program with the connect.c program, the
 * security attributes must be compatible. Mismatching authentication,
 * authorization, or protection attributes will result in errors.
 */

#include "globus_io.h"

static char *  USAGE =
"\nlisten [Connection] [Authentication] [Authorization] [Message Protection]\n"
"   Connection\n"
"       <noopt>           System defaults\n"
"       -r N              Set socket recv buffer to N bytes\n"
"       -s N              Set socket send buffer to N bytes\n"
"       -k                Send keepalive messages\n"
"       -n                Set TCP nodelay\n"
"   Authentication\n"
"       <noopt>           No authentication\n"
"       -a                Turn on authentication\n"
"   Authorization\n"
"       <noopt>           Don't authorize (default)\n"
"       -u self           authorize my own dn\n"
"       -u <dn>           authorize remote id \"<dn>\"\n"
"   Message wrapping\n"
"       <noopt>           No Message protection (default)\n"
"       -m ssl            Turn on SSL wire protocol\n"             
"       -m gssapi         Turn on GSSAPI wrap/unwrap\n"
"   Data protection\n"
"       <noopt>           No Message protection (default for unwrapped\n"
"                         messages)\n"
"       -d safe           Turn on integrity checking (default for wrapped\n"
"                         messages)\n"
"       -d private        Turn on encryption\n\n";

int 
main( 
    int       argc, 
    char *    argv[] )
{
    globus_result_t                        result;
    unsigned short                         port;   
    globus_io_handle_t                     listener_handle;
    globus_io_handle_t                     conn_handle;
    globus_io_attr_t                       attr;
    globus_byte_t                          client_message[GREETINGSIZE];
    globus_size_t                          bytes_read;
    globus_io_secure_authorization_data_t  auth_data;
    globus_io_secure_authorization_mode_t  auth_mode;
    int                                    c;
    int                                    rc;
    extern char *                          optarg;
        

    /******************************************************************
     *
     * Parse arguments. Setup TCP attributes as we go along.
     *
     ******************************************************************/

    rc = globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);
    rc = globus_module_activate(GLOBUS_IO_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);
    
    globus_io_tcpattr_init( &attr );
    globus_io_secure_authorization_data_initialize(&auth_data);

    while ((c=getopt(argc,argv,"s:r:nkau:m:d:")) != EOF)
    {
        switch(c)
        {
	case 'r':
	case 's':
	    globus_assert( atoi(optarg) > 0 );
	    if (c == 'r')
		globus_io_attr_set_socket_rcvbuf( &attr, atoi(optarg));
	    else
		globus_io_attr_set_socket_sndbuf( &attr, atoi(optarg));
	    break;
	case 'k':
	    globus_io_attr_set_socket_keepalive( &attr, GLOBUS_TRUE );
	    break;
	case 'n':
	    globus_io_attr_set_tcp_nodelay( &attr, GLOBUS_TRUE );
	    break;
	case 'a':
	    globus_io_attr_set_secure_authentication_mode(
		&attr,
		GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI, 
		GSS_C_NO_CREDENTIAL);
            break;
	case 'u':
	    if (strcmp(optarg,"self") == 0)
		auth_mode = GLOBUS_IO_SECURE_AUTHORIZATION_MODE_SELF;
	    else
	    {
		auth_mode = GLOBUS_IO_SECURE_AUTHORIZATION_MODE_IDENTITY;
		globus_io_secure_authorization_data_set_identity(
		    &auth_data,
		    optarg);
	    }
	    globus_io_attr_set_secure_authorization_mode(
		&attr,
		auth_mode,
		&auth_data);
            break;
        case 'm':
	    if (strcmp(optarg,"ssl") == 0)
	    {
		globus_io_attr_set_secure_channel_mode(
		    &attr,
		    GLOBUS_IO_SECURE_CHANNEL_MODE_SSL_WRAP);
	    }
	    if (strcmp(optarg,"gssapi") == 0)
	    {
		globus_io_attr_set_secure_channel_mode(
		    &attr,
		    GLOBUS_IO_SECURE_CHANNEL_MODE_GSI_WRAP);
	    }
            break;
        case 'd':
            if (strcmp(optarg,"safe") == 0)
            {
		globus_io_attr_set_secure_protection_mode(
		    &attr,
		    GLOBUS_IO_SECURE_PROTECTION_MODE_SAFE);
	    }
	    if (strcmp(optarg,"private") == 0)
	    {
		globus_io_attr_set_secure_protection_mode(
		    &attr,
		    GLOBUS_IO_SECURE_PROTECTION_MODE_PRIVATE);
	    }
	    break;
        default:
            globus_libc_printf("unknown flag -%c\n%s",(char) c, USAGE);
            globus_io_tcpattr_destroy(&attr);
            globus_module_deactivate_all();
            exit(1);
        }
    }

    /******************************************************************
     *
     * Create a listener.
     * Let system choose a port to listen on. NOTE: we can define an
     * interval by setting env var GLOBUS_TCP_PORT_RANGE=min,max
     *
     ******************************************************************/

    port = 0;
    result = globus_io_tcp_create_listener( &port,       /* returns port     */
                                            2,           /* backlog 2 conns. */
                                            &attr,
                                            &listener_handle);
    if (result != GLOBUS_SUCCESS)
    {
	globus_libc_printf("server: error when creating TCP listener()\n");
	globus_io_tcpattr_destroy(&attr);
	globus_module_deactivate_all();
	exit(1);
    }

    globus_libc_printf("Listening on port %d\n", port);
    
    /*
     * Start listener.
     */
    result = globus_io_tcp_listen(&listener_handle);
    if (result != GLOBUS_SUCCESS)
    {
	globus_libc_printf("server: something happened in tcp_listen()\n");
	globus_io_tcpattr_destroy(&attr);
	globus_module_deactivate_all();
	exit(1);
    }

    result = globus_io_tcp_accept(&listener_handle,
                                  &attr,
                                  &conn_handle);
    if (result != GLOBUS_SUCCESS)
    {
	globus_object_t *  err = globus_error_get(result);
	if (globus_object_type_match(
		GLOBUS_IO_ERROR_TYPE_AUTHENTICATION_FAILED,
	        globus_object_get_type(err)))
	{
	    globus_libc_printf("server: authentication failed\n");
	}
	else if (globus_object_type_match(
		   GLOBUS_IO_ERROR_TYPE_AUTHORIZATION_FAILED,
	           globus_object_get_type(err)))
	{
	    globus_libc_printf("server: authorization failed\n");
	}
	else if (globus_object_type_match(
	           GLOBUS_IO_ERROR_TYPE_NO_CREDENTIALS,
	           globus_object_get_type(err)))
	{
	    globus_libc_printf("server: failure: no credentials\n");
	}
	else
	{
	    globus_libc_printf("server: accept() failed\n");
	}
	globus_io_close(&listener_handle);
	globus_io_tcpattr_destroy(&attr);
	globus_object_free(err);
	globus_module_deactivate_all();
	exit(1);
    }
    
    /*
     * We are done with the listener.
     */
    globus_io_close(&listener_handle);
    
    /*
     * The client sends a message which contains a null-terminated
     * ASCII greeting.
     */
    result = globus_io_read( &conn_handle,
                             client_message,
                             GREETINGSIZE,
                             GREETINGSIZE,
                             &bytes_read );
    if (result != GLOBUS_SUCCESS)
    {
	globus_object_t *  err = globus_error_get(result);

	globus_libc_printf("server: read %d bytes (out of %d)\n",
			   bytes_read,
			   GREETINGSIZE);
	globus_io_close(&conn_handle);
	globus_io_tcpattr_destroy(&attr);
	globus_object_free(err);
	globus_module_deactivate_all();
	exit(1);
    }
	
    
    globus_io_close(&conn_handle);
    globus_io_tcpattr_destroy(&attr);

    globus_libc_printf("server: got a message: %s\n", (char *) client_message);

    globus_module_deactivate_all();

    return 0;
}
/* main */




