/**
 * @example connect.c
 *
 * This example demonstrates how to create a simple client program
 * using Globus I/O. It covers connection establishment, writing data,
 * and closing a handle.
 *
 * Command line options allow the user of this program to enable
 * Globus I/O security on the connection, as well as set some socket
 * options. This are done using  @link attr Globus I/O attributes
 * @endlink.
 *
 * In order to use this program with the listen.c program, the
 * security attributes must be compatible. Mismatching authentication,
 * authorization, or protection attributes will result in errors.
 */

#include "globus_io.h"

static char *  USAGE =
"\nconnect [Connection] [Authentication] [Authorization] [Message Protection]\n"
"   Connection\n"
"       -p port           Port to connect to                   [required]\n"
"       -h host           Remote host to connect to            [localhost]\n"
"       -r N              Set socket recv buffer to N bytes\n"
"       -s N              Set socket send buffer to N bytes\n"
"       -k                Send keepalive messages\n"
"       -n                Set TCP nodelay\n"
"   Authentication\n"
"       <noopt>           No authentication                    [default]\n"
"       -a                Turn on authentication\n"
"   Authorization\n"
"       <noopt>           No authorization                     [default]\n"
"       -u self           authorize my own dn\n"
"       -u <dn>           authorize remote id \"<dn>\"\n"
"   Message wrapping\n"
"       <noopt>           No Message protection                [default]\n"
"       -m ssl            Turn on SSL wire protocol\n"             
"       -m gssapi         Turn on GSSAPI wrap/unwrap\n"
"   Data protection\n"
"       <noopt>           No Message protection (default for unwrapped\n"
"                         messages)\n"
"       -d safe           Turn on integrity checking (default for wrapped\n"
"                         messages)\n"
"       -d private        Turn on encryption\n\n";

int 
main( 
    int       argc, 
    char *    argv[] )
{
    int                                    rc;
    int                                    c;
    extern char *                          optarg;
    int                                    port;     
    char *                                 hostname; 
    globus_byte_t                          message[GREETINGSIZE];
    globus_io_handle_t                     handle; 
    globus_io_attr_t                       attr;
    globus_result_t                        result;
    globus_size_t                          bytes_written;
    globus_io_secure_authorization_data_t  auth_data;
    globus_io_secure_authorization_mode_t  auth_mode;
	                   
    
    /******************************************************************
     *
     * Parse arguments. Setup TCP attributes as we go along.
     *
     ******************************************************************/
    
    rc = globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);
    rc = globus_module_activate(GLOBUS_IO_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);
    
    globus_io_tcpattr_init(&attr);
    globus_io_secure_authorization_data_initialize(&auth_data);

    port = -1;
    hostname = "localhost";
    while ((c=getopt(argc,argv,"h:p:s:r:nkau:m:d:")) != EOF)
    {
	switch(c)
	{
	case 'h':
	    hostname = globus_libc_strdup(optarg);
	    break;
	case 'p':
	case 'r':
	case 's':
	    globus_assert( atoi(optarg) > 0 );
	    if (c == 'p')
		port = atoi(optarg);
	    else if (c == 'r')
		globus_io_attr_set_socket_rcvbuf( &attr, atoi(optarg));
	    else
		globus_io_attr_set_socket_sndbuf( &attr, atoi(optarg));
	    break;
	case 'k':
	    globus_io_attr_set_socket_keepalive( &attr, GLOBUS_TRUE );
	    break;
	case 'n':
	    globus_io_attr_set_tcp_nodelay( &attr, GLOBUS_TRUE );
	    break;
	case 'a':
	    globus_io_attr_set_secure_authentication_mode(
		&attr,
		GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI, 
		GSS_C_NO_CREDENTIAL);
            break;
	case 'u':
	    if (strcmp(optarg,"self") == 0)
		auth_mode = GLOBUS_IO_SECURE_AUTHORIZATION_MODE_SELF;
	    else
	    {
		auth_mode = GLOBUS_IO_SECURE_AUTHORIZATION_MODE_IDENTITY;
		globus_io_secure_authorization_data_set_identity(
		    &auth_data,
		    optarg);
	    }
	    globus_io_attr_set_secure_authorization_mode(
		&attr,
		auth_mode,
		&auth_data);
            break;
        case 'm':
	    if (strcmp(optarg,"ssl") == 0)
	    {
		globus_io_attr_set_secure_channel_mode(
		    &attr,
		    GLOBUS_IO_SECURE_CHANNEL_MODE_SSL_WRAP);
	    }
	    if (strcmp(optarg,"gssapi") == 0)
	    {
		globus_io_attr_set_secure_channel_mode(
		    &attr,
		    GLOBUS_IO_SECURE_CHANNEL_MODE_GSI_WRAP);
	    }
            break;
        case 'd':
            if (strcmp(optarg,"safe") == 0)
            {
		globus_io_attr_set_secure_protection_mode(
		    &attr,
		    GLOBUS_IO_SECURE_PROTECTION_MODE_SAFE);
	    }
	    if (strcmp(optarg,"private") == 0)
	    {
		globus_io_attr_set_secure_protection_mode(
		    &attr,
		    GLOBUS_IO_SECURE_PROTECTION_MODE_PRIVATE);
	    }
	    break;
        default:
            globus_libc_printf("unknown flag -%c\n%s",(char) c, USAGE);
            globus_io_tcpattr_destroy(&attr);
            globus_module_deactivate_all();
            exit(1);
	}
    }

    if (port <= 0)
    {
	globus_libc_printf("provide a portnumber!\n%s", USAGE);
	globus_module_deactivate_all();
	exit(1);
    }

    /******************************************************************
     *
     * Establish a TCP connection
     *
     ******************************************************************/
    result = globus_io_tcp_connect( hostname,
				    (unsigned short)port,
				    &attr,
				    &handle );
    if (result != GLOBUS_SUCCESS)
    {
	globus_object_t *  err = globus_error_get(result);
	if (globus_object_type_match(
		GLOBUS_IO_ERROR_TYPE_AUTHENTICATION_FAILED,
	        globus_object_get_type(err)))
	{
	    globus_libc_printf("client: authentication failed\n");
	}
	else if (globus_object_type_match(
		   GLOBUS_IO_ERROR_TYPE_AUTHORIZATION_FAILED,
	           globus_object_get_type(err)))
	{
	    globus_libc_printf("client: authorization failed\n");
	}
	else if (globus_object_type_match(
	           GLOBUS_IO_ERROR_TYPE_NO_CREDENTIALS,
	           globus_object_get_type(err)))
	{
	    globus_libc_printf("client: failure: no credentials\n");
	}
	else
	{
	    globus_libc_printf("client: accept() failed\n");
	}
	
	globus_object_free(err);
	
	exit(1);
    }
	
    /*
     * Have a connection! Send a greeting and then quit.
     */
    globus_libc_sprintf( (char *) message,
			 "Hello from client pid %ld\n",
			 globus_libc_getpid() );

    globus_io_write( &handle,
		     message,
		     GREETINGSIZE,
		     &bytes_written );
    
    globus_io_close(&handle);
    globus_io_tcpattr_destroy(&attr);

    globus_libc_printf("client: succeeded to connect to server\n");

    globus_module_deactivate_all();

    return 0;
}
/* main */


