
/*
 * tokens.c 
 *
 * this file contains support routines for listen.c and connect.c 
 * which send and receive a token
 *
 */

#include "globus_common.h"

/*
 * send_token()
 *
 * sends the token length followed by the token to a fd 
 * in network byte order
 *
 * Parameters:
 *    
 *    arg - the fd to send to
 *
 *    token - the token to send
 *
 *    token_length - the length of the token to send
 *
 * Returns:
 *    
 *    0 upon success
 *
 *   -1 upon failure
 *
 */

int 
send_token(
    void *    arg, 
    void *    token, 
    size_t    token_length)
{
    size_t			num_written = 0;
    ssize_t			n_written;
    int 			fd = *( (int *) arg );
    unsigned char		token_length_buffer[4];

    /* encode the token length in network byte order: 4 byte, big endian */
    token_length_buffer[0] = (unsigned char) ((token_length >> 24) & 0xff);
    token_length_buffer[1] = (unsigned char) ((token_length >> 16) & 0xff);
    token_length_buffer[2] = (unsigned char) ((token_length >>  8) & 0xff);
    token_length_buffer[3] = (unsigned char) ((token_length      ) & 0xff);

    /* send the token length */

    while(num_written < 4)
    {
	n_written = globus_libc_write(fd,
				      token_length_buffer + num_written,
				      4 - num_written);
	if(n_written < 0)
	{
	    if(errno == EINTR)
		continue;
	    else
		return -1;
	}
	else
	    num_written += n_written;
    }

    /* send the token */

    num_written = 0;
    while(num_written < token_length)
    {
	n_written = globus_libc_write(fd,
				      ((globus_byte_t *)token) + num_written,
				      token_length - num_written);
	if(n_written < 0)
	{
	    if(errno == EINTR)
		continue;
	    else
		return -1;
	}
	else
	    num_written += n_written;
    }

    return 0;
}
/* send_token */


/*
 * get_token()
 *
 * gets the token length followed by the token from a fd 
 * in network byte order
 *
 * Parameters:
 *    
 *    arg - the fd to receive from
 *
 *    token - mechanism for passing the token, space for
 *            the token will be allocated in this function
 *
 *    token_length - the length of the token received
 *
 * Returns:
 *    
 *    0 upon success
 *
 *   -1 upon failure
 *
 */

int 
get_token(
    void *arg, 
    void ** token, 
    size_t * token_length)
{
    size_t			num_read = 0;
    ssize_t			n_read;
    int 			fd = *( (int *) arg );
    unsigned char		token_length_buffer[4];

    /* read the token length */

    while(num_read < 4)
    {
	n_read = globus_libc_read(fd,
				  token_length_buffer + num_read,
				  4 - num_read);
	if(n_read < 0)
	{
	    if(errno == EINTR)
		continue;
	    else
		return -1;
	}
	else
	    num_read += n_read;
    }
    num_read = 0;
    /* decode the token length from network byte order: 4 byte, big endian */

    *token_length  = (((size_t) token_length_buffer[0]) << 24) & 0xffff;
    *token_length |= (((size_t) token_length_buffer[1]) << 16) & 0xffff;
    *token_length |= (((size_t) token_length_buffer[2]) <<  8) & 0xffff;
    *token_length |= (((size_t) token_length_buffer[3])      ) & 0xffff;

    if(*token_length > 1<<24)
    {
	/* token too large */
	return -1;
    }

    /* allocate space for the token */

    *((void **)token) = (void *) globus_libc_malloc(*token_length);

    if(*token == NULL)
    {
	return -1;
    }

    /* receive the token */

    num_read = 0;
    while(num_read < *token_length)
    {
	n_read = globus_libc_read(fd,
				  ((globus_byte_t *) (*token)) + num_read,
				  (*token_length) - num_read);
	if(n_read < 0)
	{
	    if(errno == EINTR)
		continue;
	    else
		return -1;
	}
	else
	    num_read += n_read;
    }

    return 0;
}
/* get_token */
 





