/*
 * listen.c 
 *
 * this program listens on a port and establish a security context with
 * the connecting process
 */

#include "globus_common.h"
#include "gssapi.h"

#include <sys/types.h>
#include <sys/socket.h>


/*
 * Forward declarations
 */

int setup_listener();   /* sets up a TCP server socket */

int accept_security_context(int client_fd,
			    char **name,
			    gss_ctx_id_t * context);

/*
 * these two are in tokens.c
 */


extern int 
send_token(void *arg, void * token, size_t  token_length);

extern int 
get_token(void *arg, void ** token, size_t * token_length);


int 
main( 
    int       argc, 
    char *    argv[] )
{
    int                    rc;           /* return code storage         */
    int                    listener_fd;  /* listener socket file desc   */
    int                    client_fd;    /* connecting socket file desc */
    struct sockaddr_in     client_addr;  /* INET address structure      */
    int                    client_addr_len;
    char *		   name = NULL;
    gss_ctx_id_t	   context = GSS_C_NO_CONTEXT;

    /*
     * Create socket using the Internet domain, sequencial, reliable,
     * two-way handshake based stream with variable stringlength.
     * Use the default protocol.
     *
     * In other words: get me a TCP socket. :)
     */

    rc = globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);

    listener_fd = setup_listener();
    globus_assert(listener_fd >= 0);
    
    /*
     * Wait for a connection
     */

    client_addr_len = sizeof(client_addr);
    bzero((char *) &client_addr, client_addr_len);
    client_fd = -1;

    while( client_fd < 0 )
    {
        fd_set		   fds;
	FD_ZERO(&fds);
	FD_SET(listener_fd, &fds);

	rc = select(listener_fd+1, &fds, NULL, NULL, NULL);

	globus_assert(rc >= 0);

	client_fd = accept(listener_fd, 
			   (struct sockaddr *) &client_addr,
			   &client_addr_len);
    }

    /*
     * Have a connection!
     */
    close(listener_fd);

    /*
     * do the server-side security protocol
     */
    if(accept_security_context(client_fd, &name, &context) != 0)
    {
	close(client_fd);
	globus_module_deactivate(GLOBUS_COMMON_MODULE);
	return 1;
    }
    
    globus_libc_printf("Got a connection on fd %d from\n\"%s\"\n",
		       (int) client_fd,
		       name);

    close(client_fd);
    globus_module_deactivate(GLOBUS_COMMON_MODULE);

    return 0;
}
/* main */



/*
 * setup_listener()
 *
 * sets up a listening socket and prints the portnumber the socket 
 * is connected to
 *
 * Returns:
 *
 *    the socket fd upon success
 *
 *    -1 upon failure 
 */

int 
setup_listener()
{
    int                    listener_fd;  /* listener socket file desc   */
    int                    len;
    struct sockaddr_in     my_addr;      /* INET address structure      */

    if ((listener_fd = socket(AF_INET, SOCK_STREAM, 0))  <  0)
	return -1;

    len = sizeof(my_addr);

    /* Clear structure */
    bzero((char *)&my_addr, len);

    /* Set up address structure: any host, wildcard port */
    my_addr.sin_family = AF_INET;
    my_addr.sin_addr.s_addr = INADDR_ANY;
    my_addr.sin_port = 0;

    /* Bind the socket */
    if (bind(listener_fd, (struct sockaddr *) &my_addr, len)  < 0)
	return -1;

    /* put the socket in LISTEN state. Backlog 2 connections. */
    if (listen(listener_fd, 2) < 0)
	return -1;

    /* get some socket info (including what port the system gave us */
    if (getsockname(listener_fd, (struct sockaddr *)&my_addr, &len) < 0)
	return -1;

    globus_libc_printf("Listening on port %d\n", ntohs(my_addr.sin_port));

    return listener_fd;
}
/* setup_listener() */


/* 
 * accept_security_context()
 *
 * sets up a security context with a client 
 *
 * Parameters:
 * 
 *    client_fd - fd of the socket the client is 
 *                connected to
 *    
 *    name - used for returning the name of the client
 *    
 *    context - a security context identifier, 
 *              modified in the function
 *
 * Returns:
 * 
 *    0 upon success
 *
 *   -1 upon failure
 */


int 
accept_security_context(
    int             client_fd,
    char**          name,
    gss_ctx_id_t *  context)
{
    OM_uint32			major_status;
    OM_uint32			minor_status;
    OM_uint32			minor_status2;
    OM_uint32			ret_flags = 0;
    int				user_to_user = 0;
    int				token_status = 0;
    gss_cred_id_t		credential = GSS_C_NO_CREDENTIAL;
    gss_name_t			client_name = GSS_C_NO_NAME;

    gss_buffer_desc		input_token_desc  = GSS_C_EMPTY_BUFFER;
    gss_buffer_t    		input_token       = &input_token_desc;
    gss_buffer_desc 		output_token_desc = GSS_C_EMPTY_BUFFER;
    gss_buffer_t    		output_token      = &output_token_desc;
    gss_buffer_desc		name_buffer;
    gss_OID			mech_type = GSS_C_NO_OID;
    OM_uint32			time_ret;

    /* acquire our credential */
    
    major_status = gss_acquire_cred(&minor_status,
				    GSS_C_NO_NAME,
				    GSS_C_INDEFINITE,
				    GSS_C_NO_OID_SET,
				    GSS_C_BOTH,
				    &credential,
				    NULL,
				    NULL);

    if(major_status != GSS_S_COMPLETE)
    {
	globus_libc_printf("Failed to acquire credentials\n");

	return -1;
    }

    do
    {
	token_status = get_token(&client_fd,
				 &input_token->value,
				 &input_token->length);
	if(token_status != 0)
	{
	    major_status = GSS_S_DEFECTIVE_TOKEN|GSS_S_CALL_INACCESSIBLE_READ;
	    break;
	}

	major_status = gss_accept_sec_context(&minor_status,
					      context,
					      credential,
					      input_token,
					      GSS_C_NO_CHANNEL_BINDINGS,
					      &client_name,
					      &mech_type,
					      output_token,
					      &ret_flags,
					      &time_ret,
					      NULL /*delegated cred handle*/);
	if(output_token->length != 0)
	{
	    token_status = send_token(&client_fd,
				      output_token->value,
				      output_token->length);
	    if(token_status != 0)
	    {
		major_status =
		    GSS_S_DEFECTIVE_TOKEN|GSS_S_CALL_INACCESSIBLE_WRITE;
	    }
	    gss_release_buffer(&minor_status2,
			       output_token);
	}
	if (GSS_ERROR(major_status))
	{
	    if (context != GSS_C_NO_CONTEXT)
	    {
		gss_delete_sec_context(&minor_status2,
				       context,
				       GSS_C_NO_BUFFER);
		break;
	    }
	}
	if (input_token->length >0)
	{
	    globus_libc_free(input_token->value); /* alloc done by get_token */
	    input_token->length = 0;
	}
    } while(major_status & GSS_S_CONTINUE_NEEDED);

    if (input_token->length >0)
    {
	globus_libc_free(input_token->value); /* alloc done by get_token */
	input_token->length = 0;
    }

    /* authentication failed */
    if(major_status != GSS_S_COMPLETE)
    {
	globus_libc_printf("Failed to establish security context (accept).");
	return -1;
    }

    /* authentication succeeded, figure out who it is */
    if (major_status == GSS_S_COMPLETE)
    {
	major_status = gss_display_name(&minor_status,
					client_name,
					&name_buffer,
					NULL);
	*name = name_buffer.value;
    }

    gss_release_name(&minor_status2, &client_name);

    return 0;
}
/* accept_security_context() */

