/*
 * connect.c 
 *
 * this program connects a port given by argv[1] and establishes a security
 * context with the server process.
 */

#include "globus_common.h"
#include "gssapi.h"

#include <sys/types.h>
#include <sys/socket.h>

/* forward declarations */

int establish_connection(char *  hostname, int port);
int init_sec_context(int fd, gss_ctx_id_t * context);

/*
 * these two are in tokens.c
 */

extern int 
send_token(void *arg, void * token, size_t  token_length);

extern int 
get_token(void *arg, void ** token, size_t * token_length);


int 
main( 
    int       argc,
    char *    argv[] )
{
    int                    rc;           /* return code storage         */
    int                    port;         /* server port                 */
    int                    fd;           /* socket file desc            */
    gss_ctx_id_t	   context = GSS_C_NO_CONTEXT;	 /* security context */

    /*
     * Check arguments
     */
    
    rc = globus_module_activate(GLOBUS_COMMON_MODULE);

    globus_assert(rc == GLOBUS_SUCCESS);

    port = -1;
    if (argc == 3)
	port = atoi(argv[2]);
    
    if (port <= 0)
    {
	globus_libc_printf("provide two arguments (hostname + port)!\n");
	globus_module_deactivate(GLOBUS_COMMON_MODULE);
	exit(1);
    }

    /*
     * Establish a TCP connection with hostname argv[1], port argv[2]
     */

    fd = establish_connection(argv[1], port);
    globus_assert(fd > 0);
	
    /*
     * Have a connection!
     */

    globus_libc_printf("initalizing security context\n");
    init_sec_context(fd, &context);

    close(fd);
    globus_module_deactivate(GLOBUS_COMMON_MODULE);
	
    return 0;
}
/* main */



/*
 * establish_connection()
 *
 * establishes a TCP connection to hostname at port
 * 
 * Parameters:
 *   
 *   hostname - the host to contact
 *   port     - the port to contact
 * 
 * Returns:
 * 
 * The filedescriptor of the local socket if successful
 *
 * -1 if an error occurred
 *
 */


int 
establish_connection(
    char *    hostname, 
    int       port)
{
    int                    fd;
    struct sockaddr_in     my_addr;      /* INET address structure      */
    struct hostent *       hp;
    struct hostent         result;
    int                    herrno;
    char                   buf[1024];

    /* resolve the hostname */
    hp = globus_libc_gethostbyname_r( hostname, 
				      &result,
				      buf,
				      1024,
				      &herrno );
    if (!hp)
	return -1;

    /* Get a socket */
    if ((fd = socket(AF_INET, SOCK_STREAM, 0))  <  0)
	return -1;
    
    /* Set up address structure */
    bzero((char *)&my_addr, sizeof(my_addr));
    my_addr.sin_family = AF_INET;
    memcpy( &(my_addr.sin_addr.s_addr),
	    hp->h_addr_list[0],
	    sizeof(my_addr.sin_addr.s_addr) );
    my_addr.sin_port = htons(port);

    
    /* connect */
    if ((connect( fd, (struct sockaddr *) &my_addr, sizeof(my_addr))) < 0)
	return -1;
    
    return fd;
}
/* establish_connection() */


/*
 * init_sec_context()
 *
 * initialize a security context between client (us) and server
 * 
 * Parameters:
 * 
 *   client_fd - the fd of the client socket
 *
 *   context - the context identifier
 *
 * Returns:
 *
 *   0 upon success   
 * 
 *   -1 upon failure 
 */

int 
init_sec_context(
    int               client_fd, 
    gss_ctx_id_t *    context)
{
    
    OM_uint32			major_status = 0;
    OM_uint32			minor_status = 0;
    OM_uint32			minor_status2 = 0;
    gss_cred_id_t		credential = GSS_C_NO_CREDENTIAL;
    OM_uint32			req_flags  = GSS_C_MUTUAL_FLAG;
    OM_uint32			ret_flags  = 0;
    int				token_status = 0;
    gss_name_t			target_name = GSS_C_NO_NAME;
    globus_bool_t		context_established = GLOBUS_FALSE;
    gss_OID *			actual_mech_type = NULL;
    OM_uint32			time_ret;

    gss_buffer_desc		input_token_desc  = GSS_C_EMPTY_BUFFER;
    gss_buffer_t    		input_token       = &input_token_desc;
    gss_buffer_desc 		output_token_desc = GSS_C_EMPTY_BUFFER;
    gss_buffer_t    		output_token      = &output_token_desc;

    /* acquire the users credentials */

    major_status = gss_acquire_cred(&minor_status,
				    GSS_C_NO_NAME,
				    GSS_C_INDEFINITE,
				    GSS_C_NO_OID_SET,
				    GSS_C_BOTH,
				    &credential,
				    NULL,
				    NULL);

    if(major_status != GSS_S_COMPLETE)
    {
	globus_libc_printf("Failed to acquire credentials\n");
	return -1;
    }

    /* extract the name associated with the creds */

    major_status = gss_inquire_cred(&minor_status,
				    credential,
				    &target_name,
				    NULL, /* lifetime */
				    NULL, /* credential usage */
				    NULL /* mechanisms */);

    if(major_status != GSS_S_COMPLETE)
    {
	globus_libc_printf("Failed to determine my name\n");
	return -1;
    }

    /* establish the security context with the server */

    while(!context_established)
    {
	major_status = gss_init_sec_context(&minor_status,
					    credential,
					    context,
					    target_name,
					    GSS_C_NO_OID, /*mech_type*/
					    req_flags,
					    0, /* default time */
					    GSS_C_NO_CHANNEL_BINDINGS,
					    input_token,
					    actual_mech_type,
					    output_token,
					    &ret_flags,
					    &time_ret);
	/* free any token we've just processed */
	if(input_token->length > 0)
	{
	    globus_libc_free(input_token->value);
	    input_token->length = 0;
	}
	/* and send any new token to the server */
	if(output_token->length != 0)
	{
	    token_status = send_token(&client_fd,
				      output_token->value,
				      output_token->length);
	    if(token_status != 0)
	    {
		major_status =
		    GSS_S_DEFECTIVE_TOKEN|GSS_S_CALL_INACCESSIBLE_WRITE;
		gss_release_buffer(&minor_status,
				   output_token);
	    }
	}
	if (GSS_ERROR(major_status))
	{
	    if (*context != GSS_C_NO_CONTEXT)
	    {
		gss_delete_sec_context(&minor_status2,
				       context,
				       GSS_C_NO_BUFFER);
		break;
	    }
	}
	if(major_status & GSS_S_CONTINUE_NEEDED)
	{
	    token_status = get_token(&client_fd,
				     &input_token->value,
				     &input_token->length);
	    if(token_status != 0)
	    {
		major_status = 
		    GSS_S_DEFECTIVE_TOKEN | GSS_S_CALL_INACCESSIBLE_READ;
		break;
	    }
	}
	else
	{
	    context_established = 1;
	}
    } /* while() */

    if(input_token->length > 0)
    {
	globus_libc_free(input_token->value); /* alloc done by get_token */
        input_token->value = NULL;
	input_token->length = 0;
    }

    if (target_name != GSS_C_NO_NAME)
    {
        gss_release_name(&minor_status2,&target_name);
    }

    if(major_status != GSS_S_COMPLETE)
    {
	globus_libc_printf("Failed to establish security context (init)\n");
	return -1;
    }
    
    /* release the credential */

    major_status = gss_release_cred(&minor_status,
				    &credential);

    if(major_status != GSS_S_COMPLETE)
    {
	globus_gss_assist_display_status(
	    stdout,
	    "Failed to release credential: ",
	    major_status,
	    minor_status,
	    token_status);
	return -1;
    }

    return 0;
}
/* init_sec_context() */



