/*
 * listen.c 
 *
 * this program listens on a port and establish a security context with
 * the connecting process
 */

#include "globus_common.h"
#include "globus_gss_assist.h"

#include <sys/types.h>
#include <sys/socket.h>


/*
 * Forward declarations
 */

int setup_listener();   /* sets up a TCP server socket */

int accept_security_context(int client_fd,
			    char **name,
			    gss_ctx_id_t * context);

/*
 * these two are in tokens.c
 */

extern int 
send_token(void *arg, void * token, size_t  token_length);

extern int 
get_token(void *arg, void ** token, size_t * token_length);



int 
main( 
    int       argc, 
    char *    argv[] )
{
    int                    rc;           /* return code storage         */
    int                    listener_fd;  /* listener socket file desc   */
    int                    client_fd;    /* connecting socket file desc */
    struct sockaddr_in     client_addr;  /* INET address structure      */
    int                    client_addr_len;
    char *		   name = NULL;
    gss_ctx_id_t	   context;

    /*
     * Create socket using the Internet domain, sequencial, reliable,
     * two-way handshake based stream with variable stringlength.
     * Use the default protocol.
     *
     * In other words: get me a TCP socket. :)
     */

    rc = globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);

    listener_fd = setup_listener();
    globus_assert(listener_fd >= 0);
    
    /*
     * Wait for a connection
     */

    client_addr_len = sizeof(client_addr);
    bzero((char *) &client_addr, client_addr_len);
    client_fd = -1;

    while( client_fd < 0 )
    {
        fd_set		   fds;
	FD_ZERO(&fds);
	FD_SET(listener_fd, &fds);

	rc = select(listener_fd+1, &fds, NULL, NULL, NULL);

	globus_assert(rc >= 0);

	client_fd = accept(listener_fd, 
			   (struct sockaddr *) &client_addr,
			   &client_addr_len);
    }

    /*
     * Have a connection!
     */
    close(listener_fd);

    /*
     * do the server-side security protocol
     */
    if(accept_security_context(client_fd, &name, &context) != 0)
    {
	close(client_fd);
	globus_module_deactivate(GLOBUS_COMMON_MODULE);	
	return 0;
    }
    
    globus_libc_printf("Got a connection on fd %d from\n\"%s\"\n",
		       (int) client_fd,
		       name);

    globus_libc_close(client_fd);
    globus_module_deactivate(GLOBUS_COMMON_MODULE);
	
    return 0;
}
/* main */



/*
 * setup_listener()
 *
 * sets up a listening socket and prints the portnumber the socket 
 * is connected to
 *
 * Returns:
 *
 *    the socket fd upon success
 *
 *    -1 upon failure 
 */

int 
setup_listener()
{
    int                    listener_fd;  /* listener socket file desc   */
    int                    len;
    struct sockaddr_in     my_addr;      /* INET address structure      */

    if ((listener_fd = socket(AF_INET, SOCK_STREAM, 0))  <  0)
	return -1;

    len = sizeof(my_addr);

    /* Clear structure */
    bzero((char *)&my_addr, len);

    /* Set up address structure: any host, wildcard port */
    my_addr.sin_family = AF_INET;
    my_addr.sin_addr.s_addr = INADDR_ANY;
    my_addr.sin_port = 0;

    /* Bind the socket */
    if (bind(listener_fd, (struct sockaddr *) &my_addr, len)  < 0)
	return -1;

    /* put the socket in LISTEN state. Backlog 2 connections. */
    if (listen(listener_fd, 2) < 0)
	return -1;
    
    if (getsockname(listener_fd, (struct sockaddr *)&my_addr, &len) < 0)
	return -1;

    globus_libc_printf("Listening on port %d\n", ntohs(my_addr.sin_port));

    return listener_fd;
}
/* setup_listener() */


/* 
 * accept_security_context()
 *
 * sets up a security context with a client 
 *
 * Parameters:
 * 
 *    client_fd - fd of the socket the client is 
 *                connected to
 *    
 *    name - used for returning the name of the client
 *    
 *    context - a security context identifier, 
 *              modified in the function
 *
 * Returns:
 * 
 *    0 upon success
 *
 *   -1 upon failure
 */

int 
accept_security_context(
    int               client_fd,
    char **           name,
    gss_ctx_id_t *    context)
{
    OM_uint32			major_status;
    OM_uint32			minor_status;
    OM_uint32			ret_flags = 0;
    int				user_to_user = 0;
    int				token_status = 0;
    gss_cred_id_t		credential = GSS_C_NO_CREDENTIAL;

    /* acquire our credentials */

    major_status = globus_gss_assist_acquire_cred(&minor_status,
					          GSS_C_BOTH,
					          &credential);
    if(major_status != GSS_S_COMPLETE)
    {
	globus_gss_assist_display_status(stdout,
					 "Failed to acquire credentials:",
					 major_status,
					 minor_status,
					 0);
	return -1;
    }

    /* accept the security context request from the client */

    major_status =
	globus_gss_assist_accept_sec_context(&minor_status,
	                                     context,
				             credential,
				             name,
				             &ret_flags,
				             &user_to_user,
				             &token_status,
					     GLOBUS_NULL,
				             get_token,
				             (void *) &client_fd,
				             send_token,
				             (void *) &client_fd);

    if(major_status != GSS_S_COMPLETE)
    {
	globus_gss_assist_display_status(
	    stdout,
	    "Failed to establish security context (accept):",
	    major_status,
	    minor_status,
	    token_status);
	return -1;
    }
    return 0;
}
/* accept_security_context() */
