/*
 * connect.c 
 *
 * this program connects a port given by argv[1] and establishes a security
 * context with the server process.
 */

#include "globus_common.h"
#include "globus_gss_assist.h"

#include <sys/types.h>
#include <sys/socket.h>

/* forward declarations */
int establish_connection(char *  hostname, int port);

int initialize_security_context(int fd, gss_ctx_id_t * context);

/*
 * these two are in tokens.c
 */
extern int 
send_token(void *arg, void * token, size_t  token_length);

extern int 
get_token(void *arg, void ** token, size_t * token_length);


int 
main( 
    int       argc, 
    char *    argv[] )
{
    int                    rc;           /* return code storage         */
    int                    port;         /* server port                 */
    int                    fd;           /* socket file desc            */
    gss_ctx_id_t	   context = GSS_C_NO_CONTEXT;	 /* security context */
    
    /*
     * Check arguments
     */
    
    rc=globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);

    port = -1;
    if (argc == 3)
	port = atoi(argv[2]);

    if (port <= 0)
    {
	globus_libc_printf("provide two arguments (hostname + port)!\n");
	globus_module_deactivate(GLOBUS_COMMON_MODULE);
	exit(1);
    }

    /*
     * Establish a TCP connection with hostname argv[1], port argv[2]
     */
    fd = establish_connection(argv[1], port);
    globus_assert(fd > 0);
	
    /*
     * Have a connection!
     */

    init_sec_context(fd, &context);

    globus_libc_close(fd);

    globus_module_deactivate_all();
    return 0;
}
/* main */



/*
 * establish_connection()
 *
 * establishes a TCP connection to hostname at port
 * 
 * Parameters:
 *   
 *   hostname - the host to contact
 *   port     - the port to contact
 * 
 * Returns:
 * 
 * The filedescriptor of the local socket if successful
 *
 * -1 if an error occurred
 *
 */

int 
establish_connection(
    char *    hostname, 
    int       port)
{
    int                    fd;
    struct sockaddr_in     my_addr;      /* INET address structure      */
    struct hostent *       hp;
    struct hostent         result;
    int                    herrno;
    char                   buf[1024];

    /* resolve the hostname */
    hp = globus_libc_gethostbyname_r( hostname,
				      &result,
				      buf,
				      1024,
				      &herrno );
    if (!hp)
	return -1;

    /* Get a socket */
    if ((fd = socket(AF_INET, SOCK_STREAM, 0))  <  0)
	return -1;
    
    /* Set up address structure */
    bzero((char *)&my_addr, sizeof(my_addr));
    my_addr.sin_family = AF_INET;
    memcpy( &(my_addr.sin_addr.s_addr),
	    hp->h_addr_list[0],
	    sizeof(my_addr.sin_addr.s_addr) );
    my_addr.sin_port = htons(port);

    /* connect */

    if ((connect( fd, (struct sockaddr *) &my_addr, sizeof(my_addr))) < 0)
	return -1;
    
    return fd;
}
/* establish_connection() */


/*
 * init_sec_context()
 *
 * initialize a security context between client (us) and server
 * 
 * Parameters:
 * 
 *   client_fd - the fd of the client socket
 *
 *   context - the context identifier
 *
 * Returns:
 *
 *   0 upon success   
 * 
 *   -1 upon failure 
 */


int 
init_sec_context(
    int               client_fd, 
    gss_ctx_id_t *    context)
{
    OM_uint32			major_status = 0;
    OM_uint32			minor_status = 0;
    gss_cred_id_t		credential = GSS_C_NO_CREDENTIAL;
    OM_uint32			req_flags  = GSS_C_MUTUAL_FLAG;
    OM_uint32			ret_flags  = 0;
    int				token_status = 0;

    /* acquire our credentials */

    major_status = globus_gss_assist_acquire_cred(&minor_status,
					          GSS_C_BOTH,
					          &credential);

    if(major_status != GSS_S_COMPLETE)
    {
	globus_gss_assist_display_status(stdout,
					 "Failed to acquire credentials: ",
					 major_status,
					 minor_status,
					 0);
	return -1;
    }

    /* initialize the security context */

    major_status = 
	globus_gss_assist_init_sec_context(&minor_status,
					   credential,
					   context,
					   NULL,
					   req_flags,
					   &ret_flags,
					   &token_status,
					   get_token,
					   (void *) &client_fd,
					   send_token,
					   (void *) &client_fd);

    if(major_status != GSS_S_COMPLETE)
    {
	globus_gss_assist_display_status(
	    stdout,
	    "Failed to establish security context (init): ",
	    major_status,
	    minor_status,
	    token_status);
	return -1;
    }

    /* release the credentials */

    major_status = gss_release_cred(&minor_status,
				    &credential);
    if(major_status != GSS_S_COMPLETE)
    {
	globus_gss_assist_display_status(
	    stdout,
	    "Failed to release credential: ",
	    major_status,
	    minor_status,
	    token_status);
	return -1;
    }
    return 0;
}
/* init_sec_context() */





