/*
* threads.c
*
* this program shows how to make use of the portable thread, mutex, and
* libc libraries that are included in the globus_common module.
*
* main()      : starts argv[1] number of threads, wait for completion
* my_thread() : code executed by each thread
*
*/

/* module header file(s) */

#include "globus_common.h"

/*
*  "key" is a "shared resource" : it needs to be protected by a mutex 
*
*  In addition to a protecting mutex, we also make use of a condition to
*  signal to other threads when the value of 'key' is changed.
*/

volatile int              key = 0;     /* volatile needed when optimized */
static globus_mutex_t     key_mutex;
static globus_cond_t      key_cond;

/* forward declarations */

void * my_thread(void *	v);


int 
main( 
    int	argc, 
    char *	argv[] )
{
    
    int		       rc;
    int                i;
    int                n_threads;
    int *              nump;
    globus_bool_t      done;
    globus_thread_t    thread;
    
    /*
    * Module activation. We only use the common module
    */
    
    rc = globus_module_activate(GLOBUS_COMMON_MODULE);
    globus_assert(rc == GLOBUS_SUCCESS);

    /*
    * Get the number of threads to start from the command-line
    */
    
    n_threads = 0;
    if (argc == 2)
    n_threads = atoi(argv[1]);

    if (n_threads < 1)
    {
	globus_libc_printf("need an argument (no of threads to start)!\n");
	exit(1);
    }

    /* 
    * Initialize the mutex that will ensure correct access to the 
    * shared resource "key", defined above. Also initialize the 
    * condition structure.
    */
    
    globus_mutex_init( &key_mutex,
		       (globus_mutexattr_t *) GLOBUS_NULL );
    
    globus_cond_init( &key_cond,
		      (globus_condattr_t *) GLOBUS_NULL);

    /*
    * Start up the threads
    */
    for (i=0; i<n_threads; i++)
    {
	/*
	 * Allocate space and store the thread number there. 
	 */
	nump = (int *) globus_libc_malloc(sizeof(int));
	*nump = i;
	
	rc = globus_thread_create( &thread,        /* work storage           */
				   GLOBUS_NULL,    /* default stack size etc */
				   my_thread,      /* thread func            */
				   (void *) nump ); /* thread arg            */
	globus_assert(rc == GLOBUS_SUCCESS);
    }
	     
    /*
     * Master process just sits and waits until all threads have updated
     * the key, that is when the value of key is equal to n_threads
     */
	     
    globus_mutex_lock(&key_mutex);
    {
	while (key < n_threads)
	{
	    globus_cond_wait(&key_cond, &key_mutex);
	}
    }	
    globus_mutex_unlock(&key_mutex);
    
    /*
     * All threads are done -- can access the key directly
     */
    
    globus_libc_printf("master process: key = %d\n", key);
	     
    /*
     * We are done.
     */
    
    globus_mutex_destroy( &key_mutex );
    globus_cond_destroy(  &key_cond  );
    
    rc = globus_module_deactivate(GLOBUS_COMMON_MODULE);
    globus_assert( rc == GLOBUS_SUCCESS );	
    
    
    return 0;
}



/*  my_thread()
 *
 *  The globus thread function interface is defined as
 *  "(void *) (globus_thread_func_t)(void *)"
 *  
 * Paramters:
 * 
 * v - pointer to the id of the thread
 *
 *
 */

void *
my_thread(
    void *	v)
{
    int *  my_num;
    long   sleep;
    
    /*
     * thread number is provided as an argument
     */
    my_num = (int *)(v);
    
    /* sleep "for a while" before accessing the key */
    sleep  = 50000 + 75000 * (*my_num % 4);
    
    /* all globus_libc calls are thread safe */
    globus_libc_printf( "thread %d goes to sleep %ld!\n", *my_num, sleep );
    globus_libc_usleep( sleep );
    
    /* increment the value of the shared resource "key" */
    globus_mutex_lock(&key_mutex);
    {
	++key;
	globus_libc_printf( "thread %d: updated key to %d\n", *my_num, key );
	globus_cond_signal(&key_cond);
    }
    globus_mutex_unlock(&key_mutex);
    
    /* the "my_num" points to storage allocated for this thread. free it */
    globus_libc_free(my_num);
    
    /* thread dies when this function exits */
}

    
