/******************************************************************************
Description:

  test_xtp.c
  
  To run this program:
  
      test_xtp <reps> <message>

CVS Information:

  $Source: /home/globdev/CVS/globus-current/Globus/Communication/nexus/tests/miscellaneous/test_xtp.c,v $
  $Date: 1998/08/24 15:31:14 $
  $Revision: 1.1 $
  $State: Exp $
  $Author: toonen $
******************************************************************************/

/******************************************************************************
			     Include header files
******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <arpa/inet.h>
#include "globus_nexus.h"


/******************************************************************************
		       Define module specific constants
******************************************************************************/

#define DEBUG_LEVEL 2

extern int _nx_debug_level;

/*
 * FT_ENABLE - enable testing of fault tolerance
 */
#undef FT_ENABLE

#define NUM_MESSAGES	50
#define USLEEP_INTERVAL	10000
#define MESSAGE_SIZE	(128)


/******************************************************************************
			 Define module specific macros
******************************************************************************/
#if defined(DEBUG_LEVEL)
#   define debug_printf(level, message)		\
    {						\
        if ((level) <= (DEBUG_LEVEL))		\
        {					\
            nexus_printf message;		\
        }					\
    }
#else
#   define debug_printf(level, message)
#endif


/******************************************************************************
			       Type definitions
******************************************************************************/
typedef struct
{
    globus_mutex_t			mutex;
    globus_cond_t			cond;
    volatile int			cnt;
} Test_Xtp_monitor_t;    
    

/******************************************************************************
			  Module specific prototypes
******************************************************************************/
static void
Test_Xtp_handle_Msg(
    nexus_endpoint_t *			endpoint, 
    nexus_buffer_t *			recv_buf,
    globus_bool_t			is_non_threaded_handler);

void Test_Xtp_handle_Unknown(
    nexus_endpoint_t *			endpoint,
    nexus_buffer_t *			buffer,
    int					handler_id);

static int
Test_Xtp_handle_FT_Event(
    void *				user_arg,
    int					fault_code);


/******************************************************************************
			  Initialized handler tables
******************************************************************************/
static nexus_handler_t Test_Xtp_handler_table[] =
{
    {NEXUS_HANDLER_TYPE_NON_THREADED,
     (nexus_handler_func_t) Test_Xtp_handle_Msg}
};

#define Test_Xtp_handle_Msg_HID 0


/******************************************************************************
Function:	main()

Description:	main test program

Parameters:	argc		number of command line arguments
		argv		pointers to the actual arguments

Returns:	0 = success, !0 = failure
******************************************************************************/
int
main(int argc,
     char **argv)
{
    int					args_reps;
    char *				args_message;

    globus_bool_t			do_sender;
    globus_bool_t			do_receiver;

    nexus_endpointattr_t		endpoint_attr;
    nexus_proto_info_xtp_t		proto_info;  
    nexus_endpoint_t			endpoint;
    nexus_endpoint_t			endpoint2;
    nexus_startpoint_t			startpoint;

    nexus_buffer_t			buffer;
    int					message_size;

    Test_Xtp_monitor_t			monitor;
    int					cnt;

    int					rc;

    /*
     * Initialize Nexus
     */
#if 0
    rc = nexus_init(
	&argc, 
	&argv,
	"NEXUS_ARGS",
	"nx",
	NULL);
    if (rc != NEXUS_SUCCESS
	&& rc != NEXUS_ERROR_ALREADY_INITIALIZED)
    {
	nexus_printf("main(): nexus_init() failed with rc=%d\n", rc);
	return(1);
    }
#else
    _nx_debug_level = 0;
    rc = globus_module_activate(GLOBUS_NEXUS_MODULE);
    if (rc != GLOBUS_SUCCESS)
    {
	printf("globus_module_activate() failed with rc=%d\n", rc);
	exit(0);
    }
#endif

    /*
     * Enable fault tolerance if we are testing that
     *
     * NOTE: fault tolerance in Nexus means that Nexus will detect the fault,
     * put itself back in a reasonable state so it can continue, and the notify
     * the user that the fault has occurred.  It does not, however, provide the
     * user with a toolset to help their code tolerate the fault.  In other
     * words, knowing what messages the remote nodes have received and
     * processed is completely left to the user.
     */
#   if defined(FT_ENABLE)
    {
	globus_nexus_enable_fault_tolerance(handle_ft_event, NULL);
    }
#   endif

    /*
     * Process commmand line arguments.
     */
    if (argc != 4)
    {
	nexus_printf("usage: test_xtp <mode> <reps> <message>\n");
	globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	globus_module_deactivate_all();
	return(2);
    }

    NexusMalloc(
	main(),
	args_message,
	char *,
	strlen(argv[3]));

    if (strcmp(argv[1], "r") == 0) {
	do_sender = NEXUS_FALSE;
	do_receiver = NEXUS_TRUE;
    }
    else if (strcmp(argv[1], "s") == 0) {
	do_sender = NEXUS_TRUE;
	do_receiver = NEXUS_FALSE;
    }
    else if ((strcmp(argv[1], "rs") == 0) || (strcmp(argv[1], "sr") == 0)) {
	do_sender = NEXUS_TRUE;
	do_receiver = NEXUS_TRUE;
    }
    else {
	nexus_printf("invalid mode (one of {s,r,rs,sr})\n");
	globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	globus_module_deactivate_all();
	return(2);
    }

    args_reps = atoi(argv[2]);
    strcpy(args_message, argv[3]);

    debug_printf(1, ("main(): do_sender = %d, do_receiver = %d\n", do_sender, do_receiver));

    /*
     * Initialize monitor structure
     */
    globus_mutex_init(&monitor.mutex, (globus_mutexattr_t *) NULL);
    globus_cond_init(&monitor.cond, (globus_condattr_t *) NULL);
    monitor.cnt = 0;

    if (do_sender) {

	/*
	 * Create a endpoint attribute definition for XTP.  To do this, we need
	 * to register the handler table, the routine to handle bad message 
	 * handler IDs, as well as the information specific to the XTP 
	 * protocol.
	 */
	nexus_endpointattr_init(&endpoint_attr);

	/* fill in proto_info */
	memset((void *) &proto_info, 0, sizeof(proto_info));
	proto_info.sender = NEXUS_TRUE;
	proto_info.diameter = 16;
	proto_info.addr.s_addr = inet_addr("239.0.0.239");
	proto_info.port = 2023;
	proto_info.sreq_freq = 4;

	rc = nexus_endpointattr_set_protocol(
	    &endpoint_attr,
	    NEXUS_PROTO_TYPE_XTP,
	    &proto_info,
	    sizeof(nexus_proto_info_xtp_t));
	if (rc != NEXUS_SUCCESS)
	{
	    nexus_printf(
		"main(): nexus_endpointattr_set_protocol() failed, rc=%d\n",
		rc);
	    globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	    globus_module_deactivate_all();
	    return(3);
	}

	/*
	 * Construct a new endpoint using the endpoint attributes defined
	 * previously.
	 */
	debug_printf(1, ("main(): calling nexus_endpoint_init()\n"));
	rc = nexus_endpoint_init(&endpoint, &endpoint_attr);
	if (rc != NEXUS_SUCCESS)
	{
	    nexus_printf("main(): nexus_endpoint_init() failed, rc=%d\n", rc);
	    globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	    globus_module_deactivate_all();
	    return(4);
	}
	nexus_endpoint_set_user_pointer(&endpoint, &monitor);

	/*
	 * Construct a XTP startpoint from the endpoint
	 */
	debug_printf(1, ("main(): calling nexus_startpoint_bind()\n"));

	rc = nexus_startpoint_bind(&startpoint, &endpoint);
	if (rc != NEXUS_SUCCESS)
	{
	    nexus_printf("main(): nexus_startpoint_bind() failed, rc=%d\n", rc);
	    globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	    globus_module_deactivate_all();
	    return(5);
	}

	/*
	 * We are done with the endpoint attribute information which we previously
	 * defined, so release any resource held by this definition
	 */
	nexus_endpointattr_destroy(&endpoint_attr);

    }

    if (do_receiver) {

	/* 
	 * Need to initialize monitor.cnt very early, before any packets can
	 * hit the receiving host.  Before the endpoint gets created should
	 * be sufficient.
	 */
	monitor.cnt = args_reps;

	/*
	 * Make another endpoint, which we'll use for the receive side
	 * of this association (basically a cut 'n' paste of the original
	 * startpoint. 
	 */
	nexus_endpointattr_init(&endpoint_attr);

	/*
	 * Set handlers.
	 */
	nexus_endpointattr_set_unknown_handler(
	    &endpoint_attr,
	    Test_Xtp_handle_Unknown,
	    NEXUS_HANDLER_TYPE_NON_THREADED);

	nexus_endpointattr_set_handler_table(
	    &endpoint_attr,
	    Test_Xtp_handler_table, 
	    (sizeof(Test_Xtp_handler_table) / sizeof(nexus_handler_t)));

	/* XXX fill in proto_info XXX */
	proto_info.sender = NEXUS_FALSE;
	proto_info.diameter = 16;
	proto_info.addr.s_addr = inet_addr("239.0.0.239");
	proto_info.port = 2023;

	rc = nexus_endpointattr_set_protocol(
	    &endpoint_attr,
	    NEXUS_PROTO_TYPE_XTP,
	    &proto_info,
	    sizeof(nexus_proto_info_xtp_t));
	if (rc != NEXUS_SUCCESS)
	{
	    nexus_printf(
		"main(): nexus_endpointattr_set_protocol() failed, rc=%d\n",
		rc);
	    globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	    globus_module_deactivate_all();
	    return(3);
	}

	/*
	 * Construct a new endpoint using the endpoint attributes defined
	 * previously.  Once this completes, we will start receiving data from the
	 * specified process group.
	 */
	debug_printf(1, ("main(): calling nexus_endpoint_init()\n"));
	rc = nexus_endpoint_init(&endpoint2, &endpoint_attr);
	if (rc != NEXUS_SUCCESS)
	{
	    nexus_printf("main(): nexus_endpoint_init() failed, rc=%d\n", rc);
	    globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	    globus_module_deactivate_all();
	    return(4);
	}
	nexus_endpoint_set_user_pointer(&endpoint2, &monitor);

	nexus_endpointattr_destroy(&endpoint_attr);

    }

    if (do_sender) {

	/*
	 * Send a series of messages 
	 */
	debug_printf(1, ("main(): calling nexus_send_rsr() %d times\n",
			 args_reps));

	message_size = strlen(args_message);
	debug_printf( 1, ("message size is %d\n", message_size ));

	for (cnt = 0; cnt < args_reps; cnt++)
	{
	  sleep(1);

	  debug_printf( 1, ("cnt = %d\n", cnt ));

	    nexus_buffer_init(
		&buffer,
		nexus_sizeof_int(1) + nexus_sizeof_char(message_size),
		0);
	    nexus_put_int(&buffer, &message_size, 1);
	    nexus_put_char(&buffer, args_message, message_size);


	    rc = nexus_send_rsr(&buffer, 
				&startpoint,
				Test_Xtp_handle_Msg_HID,
				NEXUS_TRUE, 
				NEXUS_FALSE);
	    if (rc != NEXUS_SUCCESS)
	    {
		nexus_printf("main(): nexus_send_rsr() failed, rc=%d\n", rc);
		globus_module_deactivate(GLOBUS_NEXUS_MODULE);
		globus_module_deactivate_all();
		return(6);
	    }
	}

    }

    if (do_receiver) {

	nexus_printf("main(): receiver cond_wait loop\n");

	globus_mutex_lock(&monitor.mutex);
	{
	    while (monitor.cnt > 0)
		{
		    globus_cond_wait(&monitor.cond, &monitor.mutex);
		}
	}
	globus_mutex_unlock(&monitor.mutex);

    }

    /*
     * Cleanup 
     */
    if (do_sender) {

	debug_printf(1, ("main(): calling nexus_startpoint_destroy()\n"));
	rc = nexus_startpoint_destroy(&startpoint);
	if (rc != NEXUS_SUCCESS)
	{
	    nexus_printf("main(): nexus_startpoint_destroy() failed, rc=%d\n", rc);
	    globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	    globus_module_deactivate_all();
	    return(7);
	}

	debug_printf(1, ("main(): calling nexus_endpoint_destroy()\n"));
	nexus_endpoint_destroy(&endpoint);
	if (rc != NEXUS_SUCCESS)
	{
	    nexus_printf("main(): nexus_endpoint_destroy() failed, rc=%d\n", rc);
	    globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	    globus_module_deactivate_all();
	    return(8);
	}

    }

    if (do_receiver) {

	debug_printf(1, ("main(): calling nexus_endpoint_destroy()\n"));
	nexus_endpoint_destroy(&endpoint2);
	if (rc != NEXUS_SUCCESS)
	    {
		nexus_printf("main(): nexus_endpoint_destroy() failed, rc=%d\n", rc);
		globus_module_deactivate(GLOBUS_NEXUS_MODULE);
		globus_module_deactivate_all();
		return(8);
	    }

    }

    /*
     * Shutdown Nexus
     */
    debug_printf(1, ("main(): calling globus_module_deactivate(GLOBUS_NEXUS_MODULE)\n"));
    globus_module_deactivate(GLOBUS_NEXUS_MODULE);
    globus_module_deactivate_all();

    debug_printf(1, ("main(): completed successfully()\n"));
    return(0);

} /* end main() */


/******************************************************************************
Function:	Test_Xtp_handle_Msg()

Description:	receive a message

Parameters:	endpoint	
		buffer
		hanlder_id

Returns:	none
******************************************************************************/
void Test_Xtp_handle_Msg(
    nexus_endpoint_t *			endpoint,
    nexus_buffer_t *			buffer,
    int					handler_id)
{
    Test_Xtp_monitor_t *		monitor;
    int					msg_sz;
    char				msg[1024];

    monitor = (Test_Xtp_monitor_t *)
	nexus_endpoint_get_user_pointer(endpoint);

    nexus_get_int(buffer, &msg_sz, 1);
    nexus_get_char(buffer, msg, msg_sz);

    msg[msg_sz] = '\0';

    globus_mutex_lock(&monitor->mutex);
    {
	nexus_printf("Test_Xtp_handle_Msg(): %d-%s\n", monitor->cnt, msg);

	monitor->cnt--;
	globus_cond_signal(&monitor->cond);
    }
    globus_mutex_unlock(&monitor->mutex);

}
/* Test_Xtp_handle_Msg() */


/******************************************************************************
Function:	Test_Xtp_handle_Unknown()

Description:	catch any bad handler invocation requests

Parameters:	endpoint	
		buffer
		hanlder_id

Returns:	none
******************************************************************************/
void Test_Xtp_handle_Unknown(
    nexus_endpoint_t *			endpoint,
    nexus_buffer_t *			buffer,
    int					handler_id)
{
    nexus_printf("Test_Xtp_handle_Unknown(): handler_id=%d\n", handler_id);
}
/* Test_Xtp_handle_Unknown() */


/******************************************************************************
Function:	Test_Xtp_handle_FT_Event

Description:	catch any faults that happen and print them out

Parameters:	user_arg	(not used)
		fault_code	code identifying the fault which occurred

Returns:	zero - cleanup and proceed
******************************************************************************/
static int
Test_Xtp_handle_FT_Event(
    void *				user_arg,
    int					fault_code)
{
    nexus_printf("Test_Xtp_handle_FT_Event(), code=%d\n", fault_code);

    return(0);
}
/* Test_Xtp_handle_FT_Event() */
